package ohd.hseb.ohdfewsadapter;

import java.io.File;
import java.util.TimeZone;

import ohd.hseb.time.DateTime;
import ohd.hseb.util.CodeTimer;
import ohd.hseb.util.Logger;
import ohd.hseb.util.fews.Diagnostics;
import ohd.hseb.util.fews.FewsAdapterDAO;
import ohd.hseb.util.fews.FewsXMLParser;
import ohd.hseb.util.fews.IDriver;
import ohd.hseb.util.fews.NwsrfsDataTypeMappingReader;
import ohd.hseb.util.fews.OHDConfigurationInfo;
import ohd.hseb.util.fews.OHDConfigurationInfoReader;
import ohd.hseb.util.fews.OHDConstants;
import ohd.hseb.util.fews.Parameters;
import ohd.hseb.util.fews.RunInfo;
import ohd.hseb.util.io.ExceptionParser;

/**
 * The FewsAdapter class is the connection between FEWS and OHD supplied models. FEWS invokes the main method of this
 * class. The model to be run is instantiated and the parameters,state and time series data files passed by FEWS are
 * parsed and values populated in appropriate objects within the model to be run. The model is executed and the results
 * and diagnostics are written to XML (conforming to FEWS pi_* schemas).
 * 
 * @author FewsPilot team
 */
public class OHDFewsAdapter
{
    private String _fewsDataWorkLocation = null;
    private String _diagnosticsFile;
    private final Logger _logger = new Diagnostics();

    private TimeZone _timeZone; // will be set according to the tz offset double value in input fews xml <timeZone>
    private IDriver _driver;
    private RunInfo _runInfo = null;

    /* -------- for record each period time consuming ------------- */
    private final CodeTimer _stopWatch = new CodeTimer();

    public OHDFewsAdapter()
    {
        // empty constructor, used in main
    }

    /**
     * This is the entry point for FEWS
     * 
     * @param args - 1 arguments required; fully qualified path and filename to run info file
     * @throws Exception - true exceptional conditions (not caused by the data/model) are propagated upwards.
     */
    public static void main(final String[] args)
    {

        final OHDFewsAdapter fewsAdapter = new OHDFewsAdapter();

        fewsAdapter._stopWatch.start();// start the stop watch

        try
        {
            fewsAdapter.parseXmlFilesAndSetupDriver(args);

            //access nonsrc/version/ohdcommonchps_config.xml: nonsrc must be in Eclipse's classpath; "version/ohdcommonchps_config.xml" is packed in ohdcommonchps.jar
            final OHDConfigurationInfoReader configReader = new OHDConfigurationInfoReader(fewsAdapter._logger,
                                                                                           "version/"
                                                                                               + OHDConstants.OHD_FEWSADAPTER_CONFIG);
            final OHDConfigurationInfo configInfo = configReader.getVersionControl();
            fewsAdapter._logger.log(Logger.INFO, "Using OHD Fews Adapter version: " + configInfo.getVersion() + " ("
                + configInfo.getDate() + ")");

            /** ----------------------run the model -------------------------- */
            fewsAdapter._driver.executeDriver();

            /**
             * --------log timing states ------------------
             */
            fewsAdapter._stopWatch.stop();
            final StringBuffer logTime = new StringBuffer();
            logTime.append(OHDConstants.NEW_LINE)
                   .append("******************************Timing Stats*************************")
                   .append(OHDConstants.NEW_LINE);
            logTime.append("Total run time: ")
                   .append(fewsAdapter._stopWatch.getElapsedTimeInSeconds())
                   .append(" second(s)")
                   .append(OHDConstants.NEW_LINE);
            logTime.append("from initial state time ")
                   .append(DateTime.getDateTimeStringFromLong(fewsAdapter._driver.getInitialStateTime(),
                                                              fewsAdapter._timeZone))
                   .append(" to ")
                   .append(DateTime.getDateTimeStringFromLong(fewsAdapter._driver.getState().getDateTime(),
                                                              fewsAdapter._timeZone))
                   .append(" with current time at ")
                   .append(DateTime.getDateTimeStringFromLong(fewsAdapter._driver.getSwitchFromObservedToForecastTime(),
                                                              fewsAdapter._timeZone))
                   .append(OHDConstants.NEW_LINE);
            logTime.append("***********************************************************************")
                   .append(OHDConstants.NEW_LINE);
            fewsAdapter._logger.log(Logger.DEBUG, logTime.toString());

        }
        catch(final Throwable e)
        {
        	e.printStackTrace();
            /*
             * inserted "ERROR in OHDFewsAdapter: " into diag.xml so that junit error test can detect it
             */
            fewsAdapter._logger.log(Logger.FATAL, OHDConstants.ERROR_MESSAGE_PREFIX);

            fewsAdapter._logger.log(Logger.FATAL, ExceptionParser.multiLineStackTrace(e));
        }
        finally
        {
            try
            {
                /**
                 * ---------------------- write diagnostics out to diagnostics file --------------------------
                 */
                FewsAdapterDAO.writeLog(fewsAdapter._logger, fewsAdapter._diagnosticsFile);
            }
            catch(final Exception e)
            {
                System.err.println("ERROR: The diagnostics could not be printed to the diagnostic file: "
                    + e.getMessage());
                System.err.println("The contents that were to be logged are for diagnostics:"
                    + ((Diagnostics)fewsAdapter._logger).getListOfDiagnosticsAsString());
            }
        }
    } // close main

    private boolean canReadFile(final String fileName)
    {
        boolean result = false;

        final File file = new File(fileName);
        if(file.canRead())
        {
            result = true;
        }

        return result;
    }

    private synchronized void parseXmlFilesAndSetupDriver(final String[] args) throws Exception
    {
        // String header = "OHDFewsAdapter.defineModelRunInfoAndSetupModel() " + Thread.currentThread().getId() + ": ";

        final FewsXMLParser xmlParser = new FewsXMLParser(_logger);

        if(args != null && args.length == 1)
        {
            String run_info_file_name = args[0];

            if(run_info_file_name == null)
            {
                throw new Exception("run info file name does not exist!");
            }

            run_info_file_name = run_info_file_name.trim();

            //  if(!new File(run_info_file_name).canRead())
            if(!canReadFile(run_info_file_name))
            {
                throw new Exception(" Cannot read from the run info file: " + run_info_file_name);
            }

            _runInfo = new RunInfo(_logger);

            xmlParser.parseRunInfoFile(run_info_file_name, _runInfo);
            //  System.out.println(header + "run_info_file_name =  " + run_info_file_name);
            //  System.out.println(header + "_runInfo.getOutputTimeSeriesFile() =  " + _runInfo.getOutputTimeSeriesFile());

            _diagnosticsFile = _runInfo.getDiagnosticFile(); //since now, diag.xml is available

            _runInfo.validatRunInfo();

            _runInfo.validateOptionalRunInfo();

            getFewsDriver().setRunInfo(_runInfo);

            final String printDebugInfo = _runInfo.getProperties().getProperty(OHDConstants.PRINT_DEBUG_INFO);

            if(printDebugInfo != null)
            {
                try
                {
                    final int debugLevel = Integer.valueOf(printDebugInfo);
                    _logger.setPrintDebugInfo(debugLevel);
                }
                catch(final Exception e)
                {
                    _logger.log(Logger.DEBUG,
                                "A problem ocurrs when reading the print debug information flag from run file "
                                    + e.getMessage());
                }
                _runInfo.logRunInfo(Logger.DEBUG);

            }

            final String outputLocationId = _runInfo.getProperties().getProperty(OHDConstants.OUTPUT_LOCATION_ID);
            if(outputLocationId != null)
            {
                _logger.log(Logger.DEBUG, "Output Location id from Run info file [" + outputLocationId + "]");
            }

            _fewsDataWorkLocation = _runInfo.getWorkDir();

            // Initialize the data type mapping file before parsing time series files
            NwsrfsDataTypeMappingReader.initializeNwsrfsDataTypeMap(_logger);

            /*
             * Parse the input time series files(potentially multiple files) and the single input parameter file when
             * and only when run_info.xml has only one <inputParameterFile>. Otherwise, skip parsing(e.g. FFH and
             * Gridded FFG). Later, FFGDriver will do it. The individual input MAT ts location id has to be set to its
             * parent directory(area id). The parameter file for FFH and Gridded FFG is pointed to by the run info
             * property "ffgParameters".
             */
            if(_runInfo.getInputParametersFileList() != null && _runInfo.getInputParametersFileList().size() == 1)
            {
                xmlParser.parseTimeSeries(_runInfo.getInputTimeSeriesFileList(),
                                          _driver.getTsList(),
                                          _driver.getAreMissingValuesAlwaysAllowed(),
                                          _driver.getInitialStateTime());

                if(xmlParser.getTimeSeriesHandler() != null
                    && xmlParser.getTimeSeriesHandler().willTimeSeriesDataBeReadInFromBinaryFile() == true)
                {
                    _driver.setReadAndWriteBinary(true);
                }

                final String parameterFileName = _runInfo.getInputParameterFile();

                xmlParser.parseParameters(parameterFileName, (Parameters)_driver.getParameters());

                _logger.log(Logger.DEBUG, "OHDFewsAdapter parsed input time series files and input parameter file.");

                //if there are input TSs and location Id information is available from them, add location ID information to the logging prepend string
                if(_driver.getTsList().isEmpty() == false)
                {

                    final String locationId = _driver.getTsList().get(0).getLocationId();

                    if(locationId.isEmpty() == false)
                    {//adding locationId to prepend string in logging; if location id information is not available from input TSs, then don't bother

                        final String logPrependStr = ((Diagnostics)_logger).getPrependStr() + " LocationID("
                            + locationId + ")";

                        ((Diagnostics)_logger).setPrependStr(logPrependStr);
                    }
                }

            }
            else
            {
                _logger.log(Logger.DEBUG,
                            "OHDFewsAdapter skipped parsing input time series files and input parameter files");
            }

            /*
             * <inputStateDescriptionFile> could be absent or have multiple presence. Only parse it when and only when
             * there is single one <inputStateDescriptionFile>. For Gridded FFG, skip parsing here since it has multiple
             * <inputStateDescriptionFile>. Later FFGDriver will parse the states meta file pointed to by
             * "griddedFfgStates"(only Gridded FFG has states meta file for the first area case. FFH does not have
             * states meta file.).
             */
            if(_runInfo.getInputStateDescriptionFileList() != null
                && _runInfo.getInputStateDescriptionFileList().size() == 1)
            {
                final String statesMetaXmlFile = _runInfo.getInputStateDescriptionFile();

                xmlParser.parseStatesMetaFileAndLoadState(statesMetaXmlFile, _driver.getState());

                _driver.setOutputStateFile(xmlParser.getStateMetaFileHandler().getOutputStateFileName());

                _logger.log(Logger.DEBUG, "OHDFewsAdapter parsed input states meta file.");
            }
            else
            {
                _logger.log(Logger.DEBUG, "OHDFewsAdapter did not parse input states meta file.");
            }

            _timeZone = _driver.getRunInfo().getTimeZone();
            _driver.getParameters().setLogger(_logger);
            _driver.getState().setLogger(_logger);

            _driver.setWorkDir(_fewsDataWorkLocation);

        }

        // stop to take a reading
        _stopWatch.stop();
        _logger.log(Logger.DEBUG, "Time spent parsing input xml files (input, params, states and run_info): "
            + _stopWatch.getElapsedTimeInSeconds() + " sec."); // timeOnePeriod
        // restart to continue reading
        _stopWatch.restart();
        _driver.setStopWatch(_stopWatch);

    }

    private IDriver getFewsDriver() throws Exception
    {
        if(_driver == null)
        {
            String driverName;

            /*
             * get "model" or "driver" name from run_info.xml properties. We prefer using "driver", but "model" was used
             * in the past. So be backward compatibility.
             */
            if(_runInfo.getProperties().containsKey(OHDConstants.MODEL_STRING))
            {
                driverName = _runInfo.getProperties().getProperty(OHDConstants.MODEL_STRING);
            }
            else if(_runInfo.getProperties().containsKey(OHDConstants.DRIVER_STRING))
            {
                driverName = _runInfo.getProperties().getProperty(OHDConstants.DRIVER_STRING);
            }
            else
            {
                throw new Exception("Run Info properties does not contain either model or driver");
            }

            final Class<?> modelDriverClass = Class.forName(driverName);
            _driver = (IDriver)modelDriverClass.getDeclaredConstructor().newInstance();
            //calls the class(e.g. Snow17ModelDriver.java or SacSmaHTModelDriver.java) default constructor(no parameters passed in);
            _driver.setLogger(_logger);

            //get model name to add to prepend string inside _logger
            String modelName = driverName.substring(driverName.lastIndexOf(".") + 1).toUpperCase(); //stripped all the packages

            if(modelName.contains("DRIVER"))
            {
                modelName = modelName.substring(0, modelName.length() - 6); //further stripped out "DRIVER", so only left model name, like "SNOW17MODEL" etc
            }

            ((Diagnostics)_logger).setPrependStr(modelName);
        }

        return _driver;

    }
}
