package ohd.hseb.util.fews;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.TreeMap;

import ohd.hseb.db.DbTimeHelper;
import ohd.hseb.measurement.RegularTimeSeries;
import ohd.hseb.time.DateTime;
import ohd.hseb.util.CodeTimer;
import ohd.hseb.util.Logger;

public abstract class Driver implements IDriver
{
    protected IState _state;
    protected IParameters _parameters;
    protected RunInfo _runInfo;

    /*
     * the list is filled by FewsXMLParser by parsing the corresponding xml files following FEWS schema. Therefore each
     * RegularTimeSeries in both lists is in fact FewsRegularTimeSeries. However, model computation does not care. Our
     * policy is model computation should be independent, un-related to FEWS.
     */
    protected List<RegularTimeSeries> _tsList = new ArrayList<RegularTimeSeries>();//input RTS(s)

    private String _workDir;

    private String _inputStateFile;
    //for Gridded FFG, rfc template netcdf file defined in states.xml <input> when <inputNetcdfFile> is absent in run_info.xml

    private String _outputStateFile;

    protected CodeTimer _stopWatch;

    protected Logger _logger;

    protected String _outputLocationId; //default is _drivingTs locationId, but can be overwrite by the properties in run_info.xml

    private boolean _isReadAndWriteBinary = false; //if the inputs.xml is binary, outputs.xml will be binary. Here, i set its default to be false. But I think in reality all be binary mode

    /**
     * Flag that specifies if the
     */
    private boolean _areMissingValuesAlwaysAllowed = false;

    protected final Map<String, RegularTimeSeries> _resultMap = new TreeMap<String, RegularTimeSeries>();

    protected abstract void execute() throws Exception;

    /**
     * Default is {@link OHDConstants#NWSRFS_START_OF_DAY}, which is used in forecasting mode. It can be modified by run
     * info property "startHourOfDay" or "startLocalHourOfDay". For MCP3 mode, it needs to be 0. ESP running starts from
     * local 12 o'clock, so for a California basin, it is set to 20Z. Will be validated to make sure it is in synch with
     * input time series.
     */
    protected int _startHourOfDay = OHDConstants.NWSRFS_START_OF_DAY;

    /**
     * Confirm the initial state time from states.xml is same as run_info.xml
     */
    protected void runDriverValidation() throws Exception
    {
        //check the initial state time from parsing states.xml and from run_info.xml are the same 
        if(_state.getDateTime() != getInitialStateTime())
        {
            final String errorMessage = "Initial state time from state meta file ["
                + DateTime.getDateTimeStringFromLong(_state.getDateTime(),OHDConstants.GMT_TIMEZONE) + "] and from run info file ["
                + DateTime.getDateTimeStringFromLong(getInitialStateTime(),OHDConstants.GMT_TIMEZONE) + " ] are not equal";

            _logger.log(Logger.WARNING, errorMessage);
        }
    }

    // write states meta xml file
    protected void writeStatesMetaFile() throws Exception
    {

        FewsAdapterDAO.writeStatesMetaXMLFile(_runInfo.getInputStateDescriptionFile(),
                                              getState(),
                                              _runInfo.getTimeZone(),
                                              _runInfo.getTimeZoneRawOffsetInHours(),
                                              _runInfo.getDaylightSavingObservatingTimeZone(),
                                              _logger);
    }

    /**
     * Write the output TSs to the file, specified in run_info.xml. Note:<br>
     * 1). If the input files are ".fi" format, then the output file will be in ".fi" format too.<br>
     * 2). If the input files(input/inputs.xml and other mods related xml file) are in binary format, then the output
     * file will be in binary format.
     */
    protected void writeOutputTimeseries() throws Exception
    {
        // Change the output file name extension to fi if needed.
        String xmlFileName = _runInfo.getOutputTimeSeriesFile();

        if(_runInfo.getInputTimeSeriesFileList().isEmpty() == false
            && _runInfo.getInputTimeSeriesFileList().get(0).endsWith(".fi")) //some models, e.g. baseflow does not have inputs.xml
        {
            xmlFileName = xmlFileName.substring(0, xmlFileName.lastIndexOf(".")) + ".fi";
        }

        final boolean isOutputInBinary = isReadAndWriteBinary();

        FewsAdapterDAO.writeTimeSeriesToFewsXML(isOutputInBinary,
                                                this.getFewsOutputTimeseriesList(),
                                                xmlFileName,
                                                _logger,
                                                _runInfo.getTimeZone(),
                                                _runInfo.getDaylightSavingObservatingTimeZone());
    }

    /**
     * Returns a list of {@link FewsRegularTimeSeries}. Get individual RegularTimeSeries obj from resultTimeserisMap and
     * convert it to FewsRegularTimeSeries obj, returns the list of them.
     */
    public ArrayList<FewsRegularTimeSeries> getFewsOutputTimeseriesList() throws Exception
    {
        final Iterator<Map.Entry<String, RegularTimeSeries>> ite = _resultMap.entrySet().iterator();

        // for holding final version of time series
        final ArrayList<FewsRegularTimeSeries> resultList = new ArrayList<FewsRegularTimeSeries>();
        // convert all the values in resultMap(RegularTimeSeries) to
        // FewsRegularTimeSeries
        while(ite.hasNext())
        {
            final RegularTimeSeries resultTs = ite.next().getValue();

            final FewsRegularTimeSeries finalFewsResultTs = new FewsRegularTimeSeries(resultTs.getStartTime(),
                                                                                      resultTs.getEndTime(),
                                                                                      resultTs.getIntervalInHours(),
                                                                                      resultTs.getMeasuringUnit());

            final String timeSeriesType = resultTs.getTimeSeriesType();
            finalFewsResultTs.setTimeSeriesType(timeSeriesType);
            finalFewsResultTs.setType(FewsRegularTimeSeries.getFewsTimeCode(NwsrfsDataTypeMappingReader.getTimeCode(timeSeriesType,
                                                                                                                    _logger)));
            //finalFewsResultTs.setQualifierId(resultTs.getQualifierId());
            finalFewsResultTs.setQualifierIds(resultTs.getQualifierIds());

            // Added to output ensemble related information.
            finalFewsResultTs.setEnsembleId(resultTs.getEnsembleId());
            finalFewsResultTs.setEnsembleMemberIndex(resultTs.getEnsembleMemberIndex());

            //some models don't have neither _drivingTs or properties outputLocationId in run_info.xml, e.g. some ens models
            if(_outputLocationId != null)
            {
                finalFewsResultTs.setLocationId(_outputLocationId);
            }
            else
            {
                finalFewsResultTs.setLocationId(resultTs.getLocationId());
            }

            if(getTsList() != null && !getTsList().isEmpty())
            {
                finalFewsResultTs.setLongName(((FewsRegularTimeSeries)getTsList().get(0)).getLongName());
                finalFewsResultTs.setSourceOrganization(((FewsRegularTimeSeries)getTsList().get(0)).getSourceOrganization());
                finalFewsResultTs.setSourceSystem(((FewsRegularTimeSeries)getTsList().get(0)).getSourceSystem());
                finalFewsResultTs.setFileDescription(((FewsRegularTimeSeries)getTsList().get(0)).getFileDescription());
                finalFewsResultTs.setIntervalTimeStepTimes(((FewsRegularTimeSeries)getTsList().get(0)).getIntervalTimeStepTimes());
            }

            finalFewsResultTs.setMeasurementList(resultTs.getMeasurementList());
            resultList.add(finalFewsResultTs);

        } // close while loop

        return resultList;
    }

    /**
     * Construct a composite key({@link #_outputLocationId} + rts.getTimeSeriesType() + rts.getIntervalInHours()), then
     * insert into {@link #_resultMap}. If the composite key is identical, the previous entry will be replaced. However,
     * this allows the same type RTS with different intervals exist simultaneously in the map.
     * 
     * @param rts
     */
    protected void addTStoResultMap(final RegularTimeSeries rts)
    {
        _resultMap.put(OHDUtilities.getCompositeKeyInResultMap(rts), rts);
    }

    /**
     * Returns the run start time in run_info.xml, maybe not same as the time in states.xml, which will be checked and
     * logged. The information in run_info.xml determines the true run start time, not states.xml.
     */
    public long getInitialStateTime()
    {
        return _runInfo.getRunStartTimeLong();
    }

    public long getComputationEndTime()
    {
        return _runInfo.getRunEndTimeLong();
    }

    public RunInfo getRunInfo()
    {
        return _runInfo;
    }

    public void setParameters(final Parameters params)
    {
        _parameters = params;
    }

    public IParameters getParameters()
    {
        return _parameters;
    }

    public IState getState()
    {
        return _state;
    }

    public void setState(final State state)
    {
        _state = state;
    }

    /**
     * Return LSTCMPDY. If _switchFromObservedToForecastTime has not been set in run info file then return time0 of
     * model run. In case _runInfo is null, return 0 to avoid NullPointerException(for FFH, GriddedFFG purpose).
     */
    public long getSwitchFromObservedToForecastTime()
    {
        if(_runInfo != null)
        {
            return _runInfo.getRunLastObservationTimeLong();
        }

        return 0;
    }

    public List<RegularTimeSeries> getTsList()
    {
        return _tsList;
    }

    public void setTsList(final List<RegularTimeSeries> ts)
    {
        _tsList = ts;
    }

    public Logger getLogger()
    {
        return _logger;
    }

    public void setLogger(final Logger logger)
    {
        _logger = logger;
    }

    public Properties getDriverProperties()
    {
        return _runInfo.getProperties();
    }

    public void setRunInfo(final RunInfo runInfo)
    {
        _runInfo = runInfo;
    }

    public String getWorkDir()
    {
        return _workDir;
    }

    public void setWorkDir(final String workDir)
    {
        _workDir = workDir;
    }

    protected String getOutputStateFile()
    {
        return _outputStateFile;
    }

    public void setOutputStateFile(final String outputStateFile)
    {
        _outputStateFile = outputStateFile;
    }

    protected String getInputStateFile()
    {
        return _inputStateFile;
    }

    public void setInputStateFile(final String inputStateFile)
    {
        _inputStateFile = inputStateFile;
    }

    public CodeTimer getStopWatch()
    {
        return _stopWatch;
    }

    public void setStopWatch(final CodeTimer stopWatch)
    {
        _stopWatch = stopWatch;
    }

    public void setReadAndWriteBinary(final boolean b)
    {
        _isReadAndWriteBinary = b;
    }

    public boolean isReadAndWriteBinary()
    {
        return _isReadAndWriteBinary;
    }

    public void setAreMissingValuesAlwaysAllowed(final boolean b)
    {
        this._areMissingValuesAlwaysAllowed = b;
    }

    public boolean getAreMissingValuesAlwaysAllowed()
    {
        return this._areMissingValuesAlwaysAllowed;
    }

    /**
     * Returns an int which is the start of the day. This value is from the properties file. If not present, the default
     * value {@link OHDConstants#NWSRFS_START_OF_DAY} will be returned.
     */
    public int getStartHourOfDay()
    {
        return _startHourOfDay;
    }

}
