package ohd.hseb.util.fews;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.TimeZone;

import javax.xml.XMLConstants;
import javax.xml.stream.XMLStreamException;
import javax.xml.stream.XMLStreamReader;
import javax.xml.stream.XMLStreamWriter;

import ohd.hseb.grid.NetcdfConstants;
import ohd.hseb.measurement.MeasuringUnit;
import ohd.hseb.measurement.RegularTimeSeries;
import ohd.hseb.time.DateTime;
import ohd.hseb.util.CodeTimer;
import ohd.hseb.util.Logger;
import ohd.hseb.util.MathHelper;
import ohd.hseb.util.fews.GroupOfParameters.ValidPeriod;
import ohd.hseb.util.fews.OHDUtilities.StateLocation;
import ohd.hseb.util.io.EndianConvertingOutputStream;
import ohd.hseb.util.xml.OHDXmlUtils;

public class FewsAdapterDAO
{

    {
        new CodeTimer();
    }

    public static final ThreadLocal<Boolean> _isTimeseriesEquidistant = new ThreadLocal<Boolean>()
    {
        @Override
        public Boolean initialValue()
        {
            return true;
        }
    };

    /**
     * Parse the xml file containing FewsRegularTimeSeries and obtain a list of RegularTimeSeries(in fact,
     * FewsRegularTimeSeries). The RegularTimeSeries contained in the input fewsXMLFileName file maybe in non-GMT time
     * zone, with the offset information in the element <timezone>. The returned list of RTS is in GMT time, since
     * RegularTimeSeries and FewsRegularTimeSeries only know GMT time.
     * 
     * @param fewsXMLFileName
     * @param logger
     * @param areMissingValuesAlwaysAllowed Passed into TimeSeriesHandler prior to usage.
     * @return a list of RegularTimeSeries(in fact, FewsRegularTimeSeries)
     * @throws ModelException
     */
    public synchronized static List<RegularTimeSeries> getListOfRTSFromFewsXML(final String fewsXMLFileName,
                                                                               final Logger logger,
                                                                               final boolean areMissingValuesAlwaysAllowed) throws Exception
    {
        List<RegularTimeSeries> resultTsList;
        XMLStreamReader reader = null;
        BufferedInputStream inputStream = null;
        try
        {
            FewsXmlValidation.validateXmlFileAgainsXMLSchema(fewsXMLFileName,
                                                             OHDFewsAdapterConstants.PI_TIMESERIES_XML_SCHEMA_TAG,
                                                             logger);

            resultTsList = new ArrayList<RegularTimeSeries>();
            final TimeSeriesHandler tsHandler = new TimeSeriesHandler(resultTsList, logger);
            tsHandler.initiateForBinaryFileReadingIfAvailable(new File(fewsXMLFileName));
            tsHandler.setAreMissingValuesAlwaysAllowed(areMissingValuesAlwaysAllowed);

            final File inputFile = new File(fewsXMLFileName);
            inputStream = new BufferedInputStream(new FileInputStream(inputFile));

            reader = OHDXmlUtils.getStreamReader(inputStream, fewsXMLFileName);
            tsHandler.find(reader);

        }
        catch(final Exception e)
        {
            final String message = "Error in parsing Fews XML time series file: " + fewsXMLFileName;
            throw (new Exception(e.getMessage() + ". " + message));
        }
        finally
        {
            try
            {
                reader.close();
                inputStream.close();
            }
            catch(final XMLStreamException sre)
            {
                //TODO: log error
            }
            catch(final IOException ioe)
            {
                // TODO: log error
            }

        }

        return resultTsList;
    }

    /**
     * Assumes false for areMissingValuesAlwaysAllowed.
     */
    public synchronized static List<RegularTimeSeries> getListOfRTSFromFewsXML(final String fewsXMLFileName,
                                                                               final Logger logger) throws Exception
    {
        return getListOfRTSFromFewsXML(fewsXMLFileName, logger, true);
    }

    /***
     * overloading method to keep backward compatibility
     * 
     * @param fewsTsList
     * @param outputFileName
     * @param logger
     * @param timeZone
     * @throws Exception
     */
    public synchronized static void writeTimeSeriesToFewsXML(final List<FewsRegularTimeSeries> fewsTsList,
                                                             final String outputFileName,
                                                             final Logger logger,
                                                             final TimeZone timeZone) throws Exception
    {
        final String daylightSavingObservatingTimeZone = null;
        writeTimeSeriesToFewsXML(false, fewsTsList, outputFileName, logger, timeZone, daylightSavingObservatingTimeZone);
    }

    /**
     * overloading method to keep backward compatibility
     * 
     * @param binary
     * @param fewsTsList
     * @param fileName
     * @param logger
     * @param timeZone
     * @throws Exception
     */
    public synchronized static void writeTimeSeriesToFewsXML(final boolean binary,
                                                             final List<FewsRegularTimeSeries> fewsTsList,
                                                             final String fileName,
                                                             final Logger logger,
                                                             final TimeZone timeZone) throws Exception
    {
        final String daylightSavingObservatingTimeZone = null;
        writeTimeSeriesToFewsXML(binary, fewsTsList, fileName, logger, timeZone, daylightSavingObservatingTimeZone);
    }

    /**
     * Output {@link FewsRegularTimeSeries} objects in list to text format fews xml file. See another similar method
     * {@link #writeTimeSeriesToFewsXML(boolean, List, String, Logger, TimeZone)} which let user choose in binary or
     * text format.
     */
    public synchronized static void writeTimeSeriesToFewsXML(final List<FewsRegularTimeSeries> fewsTsList,
                                                             final String outputFileName,
                                                             final Logger logger,
                                                             final TimeZone timeZone,
                                                             final String daylightSavingObservatingTimeZone) throws Exception
    {
        writeTimeSeriesToFewsXML(false, fewsTsList, outputFileName, logger, timeZone, daylightSavingObservatingTimeZone);
    }

    /**
     * Output {@link FewsRegularTimeSeries} objects in list to text format fews xml file or binary format. Note for a
     * special case: if the Measurement value is double NaN, which is produced by 0.0/0.0, then output it as 0.0. Each
     * event time will be in the time zone passed in as parameter.
     * 
     * @param binary - if true, output the list of {@link FewsRegularTimeSeries} to binary format; if false, to text
     *            format
     * @param fewsTsList - list of fews time series objects.
     * @param fileName - Name of the file in to which the time series list is to be written
     * @param logger - The log
     * @param timeZone - The time zone
     * @throws Exception
     */
    public synchronized static void writeTimeSeriesToFewsXML(final boolean binary,
                                                             final List<FewsRegularTimeSeries> fewsTsList,
                                                             final String fileName,
                                                             final Logger logger,
                                                             final TimeZone timeZone,
                                                             final String daylightSavingObservatingTimeZone) throws Exception
    {
        //Initialize the binary output stream.
        EndianConvertingOutputStream binaryOutputStream = null;

        String encoding = "utf-8";
        if(binary)
        {
            String binFileName = null;
            if(fileName.endsWith(".xml"))
            {
                binFileName = fileName.substring(0, fileName.length() - 4) + ".bin";
            }
            else if(fileName.endsWith(".fi"))
            {
                binFileName = fileName.substring(0, fileName.length() - 3) + ".bin";
                encoding = "finf";
            }
            else
            {
                logger.log(Logger.WARNING, "Output file name " + fileName
                    + " does not end in .xml.  Confused... not outputting in binary.");
            }
            final File binFile = new File(binFileName);
            binaryOutputStream = new EndianConvertingOutputStream(new BufferedOutputStream(new FileOutputStream(binFile)));
            logger.log(Logger.DEBUG, "Time series data will be written to binary file " + binFileName);
        }

        // Set up output stream for fast infoset document
        final File outputFile = new File(fileName);
        final BufferedOutputStream outputStream = new BufferedOutputStream(new FileOutputStream(outputFile));

        final XMLStreamWriter writer = OHDXmlUtils.getStreamWriter(outputStream, fileName);

        try
        {
            // Create new document
            //   writer.writeStartDocument("utf-8", "1.0");
            writer.writeStartDocument(encoding, "1.0");
            // Create State element
            writer.writeStartElement("TimeSeries");
            writer.writeAttribute("version", fewsTsList.get(0).getVersion());
            writer.writeAttribute("xsi:schemaLocation", "http://www.wldelft.nl/fews/PI pi_timeseries.xsd");
            writer.writeAttribute("xmlns:xsi", XMLConstants.W3C_XML_SCHEMA_INSTANCE_NS_URI);
            writer.writeAttribute("xmlns", "http://www.wldelft.nl/fews/PI");

            if(daylightSavingObservatingTimeZone != null && !daylightSavingObservatingTimeZone.isEmpty())
            {
                // Create daylightSavingObservingTimeZone element                                          
                writer.writeStartElement("daylightSavingObservingTimeZone");
                writer.writeCharacters(daylightSavingObservatingTimeZone);
                writer.writeEndElement(); // end of stateIdNode                
            }
            else
            {
                //to get a double offset value for <timeZone> element and add <timeZone> element to TimeSeries element
                writer.writeStartElement("timeZone");
                writer.writeCharacters(Double.toString(OHDUtilities.getTimeZoneRawOffsetInHours(timeZone)));
                writer.writeEndElement(); // end of timeZoneElement
            }

            FewsRegularTimeSeries fewsTs;

            final int numOfTSs = fewsTsList.size();

            for(int i = 0; i < numOfTSs; i++)
            { //one loop for one TS

                fewsTs = fewsTsList.get(i);

                final double tsMissingValue = fewsTs.getMissingValue();

                //check if TS type is specified in a unit map file; if so, change to new unit
                //get fewsUnit from NwsrfsDataTypeMapping
                final String dataType = fewsTs.getTimeSeriesType();

                final String fewsUnitStr = NwsrfsDataTypeMappingReader.getFewsUnit(dataType, logger);
                final String nwsrfsUnitStr = NwsrfsDataTypeMappingReader.getNwsrfsUnit(dataType, logger);

                if((fewsUnitStr != null) && !(fewsUnitStr.equalsIgnoreCase(nwsrfsUnitStr)))
                {
                    logger.log(Logger.DEBUG, "Converting " + dataType + " TS unit from "
                        + fewsTs.getMeasuringUnit().toString() + " to " + fewsUnitStr + ".");
                    fewsTs.setMeasuringUnit(MeasuringUnit.getMeasuringUnit(fewsUnitStr.trim()));
                }
                else
                {
                    logger.log(Logger.DEBUG,
                               "NO need to convert data because NWSRFS and FEWS units are the same for dataytpe"
                                   + dataType);
                }

                writer.writeStartElement("series");
                //<series> element has elements: one header and many event
                writer.writeStartElement("header");

                writeWholeElement(writer, "type", fewsTs.getType().getTypeName());
                writeWholeElement(writer, "locationId", fewsTs.getLocationId());
                writeWholeElement(writer, "parameterId", dataType); //parameterId element is equivalent to NWSRFS datatype

                /*----------------- set up header element ------------------*/

                //write out the qualifier id only if different from the location id
//                if(fewsTs.getQualifierId() != ""
//                    && fewsTs.getQualifierId().equalsIgnoreCase(fewsTs.getLocationId()) == false)
//                {
//                    writer.writeStartElement("qualifierId");
//                    writer.writeCharacters(fewsTs.getQualifierId());
//                    writer.writeEndElement(); // end of qualifierId
//                }
//write out the qualifier id only if different from the location id
                if(fewsTs.getQualifierIds() != null && fewsTs.getQualifierIds().size() > 0)
                {
                    if(fewsTs.getQualifierIds().size() > 1)
                    {
                        for(final String qualifierId: fewsTs.getQualifierIds())
                        {
                            writer.writeStartElement("qualifierId");
                            writer.writeCharacters(qualifierId);
                            writer.writeEndElement(); // end of qualifierId

                        }
                    }
                    else
                    {
                        final String qualifierId = fewsTs.getQualifierIds().get(0);
                        if(qualifierId.isEmpty() == false
                            && (qualifierId.equalsIgnoreCase(fewsTs.getLocationId()) == false))
                        {
                            writer.writeStartElement("qualifierId");
                            writer.writeCharacters(qualifierId);
                            writer.writeEndElement(); // end of qualifierId
                        }
                    }
                }

                //Write out the ensemble id if not null. 
                //Write out the ensmebleMemberIndex if not zero or if ensembleId was output.
                boolean ensembleIdOutput = false;
                if(fewsTs.getEnsembleId().isEmpty() == false
                    && !fewsTs.getEnsembleId().equalsIgnoreCase(FewsRegularTimeSeries.NO_ENSEMBLE_ID))
                {
                    writer.writeStartElement("ensembleId");
                    writer.writeCharacters(fewsTs.getEnsembleId());
                    writer.writeEndElement(); // end of ensembleId
                    ensembleIdOutput = true;
                }
                if((fewsTs.getEnsembleMemberIndex() > 0) || ensembleIdOutput)
                {
                    writer.writeStartElement("ensembleMemberIndex");
                    writer.writeCharacters(Integer.toString(fewsTs.getEnsembleMemberIndex()));
                    writer.writeEndElement(); // end of ensembleMemberIndex
                }

                //set timeStep element in header
                writer.writeEmptyElement("timeStep");
                if(_isTimeseriesEquidistant.get() == false)
                {
                    writer.writeAttribute("unit", "nonequidistant");
                }
                else
                {
                    if(fewsTs.getIntervalTimeStepTimes() != null && !fewsTs.getIntervalTimeStepTimes().isEmpty())
                    {
                        writer.writeAttribute("times", fewsTs.getIntervalTimeStepTimes());
                    }
                    else
                    {
                        //in case some time step < 1 hour, we use minute
                        writer.writeAttribute("unit", "minute");
                        writer.writeAttribute("multiplier", String.valueOf(fewsTs.getIntervalInHours() * 60));
                    }
                }

                // set startDate element in header
                FewsAdapterDAO.writeDateTimeElement(writer, "startDate", fewsTs.getStartTime(), timeZone);

                // set endDate element in header
                FewsAdapterDAO.writeDateTimeElement(writer, "endDate", fewsTs.getEndTime(), timeZone);

                writer.writeStartElement("missVal");
                writer.writeCharacters(Double.toString(tsMissingValue));
                writer.writeEndElement(); // end of missVal

                // optional elements, only set if exist

                writeWholeElementIfNotEmpty(writer, "longName", fewsTs.getLongName());
                writeWholeElementIfNotEmpty(writer, "stationName", fewsTs.getName());

                writeWholeElementIfNotEmpty(writer, "lat", fewsTs.getLatitude());
                writeWholeElementIfNotEmpty(writer, "lon", fewsTs.getLongitude());
                writeWholeElementIfNotEmpty(writer, "x", fewsTs.getX());
                writeWholeElementIfNotEmpty(writer, "y", fewsTs.getY());
                writeWholeElementIfNotEmpty(writer, "z", fewsTs.getZ());

                writeWholeElementIfNotEmpty(writer, "units", fewsTs.getMeasuringUnit().getName());
                writeWholeElementIfNotEmpty(writer, "sourceOrganisation", fewsTs.getSourceOrganization());
                writeWholeElementIfNotEmpty(writer, "sourceSystem", fewsTs.getSourceSystem());
                writeWholeElementIfNotEmpty(writer, "fileDescription", fewsTs.getFileDescription());
                writeWholeElementIfNotEmpty(writer, "creationDate", fewsTs.getCreationDate());
                writeWholeElementIfNotEmpty(writer, "creationTime", fewsTs.getCreationTime());

                if(fewsTs.getThresholdsList() != null && fewsTs.getThresholdsList().size() > 0)
                {
                    writer.writeStartElement("thresholds");
                    for(final TimeSeriesThreshold thresholds: fewsTs.getThresholdsList())
                    {
                        if(thresholds != null)
                        {
                            writer.writeEmptyElement("highLevelThreshold");
                            writer.writeAttribute("id", thresholds.getId());
                            writer.writeAttribute("name", thresholds.getName());
                            writer.writeAttribute("value",
                                                  Double.toString(thresholds.getValue(fewsTs.getMeasuringUnit())));
                        }
                    }
                    writer.writeEndElement();
                }

                writer.writeEndElement(); // end of header Element;

                /*----------------- finished setting up header element ------------------*/

                double mValue = 0.0;

                final int numOfMeasurements = fewsTs.getMeasurementCount();

                for(int j = 0; j < numOfMeasurements; j++)
                {//one loop for one event

                    mValue = fewsTs.getMeasurementByIndex(j).getValue();

                    //For binary output, just output the value and go on to the next.
                    if(binaryOutputStream != null)
                    {
                        binaryOutputStream.writeFloatSwap((float)mValue);
                        continue;
                    }

                    if(_isTimeseriesEquidistant.get() == false && mValue == tsMissingValue)
                    {
                        continue;
                    }

                    FewsAdapterDAO.writeDateTimeElement(writer, "event", fewsTs.getMeasurementTimeByIndex(j), timeZone);

                    if(Double.isNaN(mValue))
                    {
                        throw new Exception("When writing out TS, the measurement value is NaN.");
                    }
                    else
                    {
                        //normally, keep 5 digits after decimal point
                        writer.writeAttribute("value", Double.toString(MathHelper.roundToNDecimalPlaces(mValue, 5)));
                    }

                    //we don't use flag, but is required by Schema, so hard-coded the flag value to "1" 
                    writer.writeAttribute("flag", "1");

                    // end of writing event element

                } //close for loop(one event, one loop)

                writer.writeEndElement(); // end of seriesElement;

                logger.log(Logger.DEBUG, "Output the time series of " + dataType + " to " + fileName);

            } //close for loop(one ts, one loop)

            writer.writeEndElement(); // end of timeSeriesElement
            writer.writeEndDocument();
            writer.flush();
            writer.close();

            outputStream.close();

            FewsXmlValidation.validateXmlFileAgainsXMLSchema(fileName,
                                                             OHDFewsAdapterConstants.PI_TIMESERIES_XML_SCHEMA_TAG,
                                                             logger);
        }
        finally
        {
            if(binaryOutputStream != null)
            {
                binaryOutputStream.flush();
                binaryOutputStream.close();
            }
            if(binaryOutputStream != null)
            {
                outputStream.flush();
                outputStream.close();
            }
        }

        // reset this static flag to true; in case it was changed
        _isTimeseriesEquidistant.set(true);
        logger.log(Logger.DEBUG, "The TS list has been written to " + fileName);
    }

    /**
     * This is a convenience method used to reduce repetitive code. Given and element name and value, it writes out the
     * whole element
     * 
     * @param writer - XMLStreamWriter
     * @param elementName
     * @param elementValue
     * @throws XMLStreamException
     */
    public synchronized static void writeWholeElement(final XMLStreamWriter writer,
                                                      final String elementName,
                                                      final String elementValue) throws XMLStreamException
    {
        writer.writeStartElement(elementName);
        writer.writeCharacters(elementValue);
        writer.writeEndElement(); // end of creationTime
    }

    /**
     * This is a convenience method used to reduce repetitive code. It writes out an XML element if the element's value
     * is not null or an empty string
     * 
     * @param writer
     * @param elementName
     * @param elementValue
     * @throws XMLStreamException
     */
    public synchronized static void writeWholeElementIfNotEmpty(final XMLStreamWriter writer,
                                                                final String elementName,
                                                                final Object elementValue) throws XMLStreamException
    {
        if((elementValue != null) && (elementValue.toString().length() > 0))
        {
            writeWholeElement(writer, elementName, elementValue.toString());
        }
    }

    /**
     * This is a convenience method used to reduce repetitive code. It writes out a time related element(e.g.
     * "startDateTime", "endDateTime", "time0", "lastObservationDateTime"). One example is:<br>
     * <dateTimeElementName date="1988-11-01" time="24:00:00"/><br>
     * Note: Even though the date and time string do not show time zone information, they have been converted to that
     * time zone. If the parameter time zone is null, it is taken as GMT.
     * 
     * @param writer
     * @param dateTimeElementName
     * @param dateTimeValue
     * @param timeZone
     * @throws XMLStreamException
     */
    private synchronized static void writeDateTimeElement(final XMLStreamWriter writer,
                                                          final String dateTimeElementName,
                                                          final long dateTimeValue,
                                                          final TimeZone timeZone) throws XMLStreamException
    {
        //   String header = "FewAdapterDAO.java.writeDateTimeElement(): ";

        //DateTime dateTime = SimpleDateAndTime.getDateTimeObjectFromLong(dateTimeValue, timeZone);
        writer.writeEmptyElement(dateTimeElementName);
        //writer.writeAttribute("date", dateTime.getDateString();
        //writer.writeAttribute("time", dateTime.gettimeString();

        //  _timer.restart();

        final DateTime simpleDateAndTime = new DateTime(dateTimeValue, timeZone);
        final String dateString = simpleDateAndTime.getDateString();
        final String timeString = simpleDateAndTime.getTimeString();

        //  System.out.println(header + "dateString = " + dateString + " timeString = " + timeString);

        writer.writeAttribute("date", dateString);
        writer.writeAttribute("time", timeString);

//        writer.writeAttribute("date", SimpleDateAndTime.getDateStringFromLong(dateTimeValue, timeZone));
//        writer.writeAttribute("time", SimpleDateAndTime.getTimeStringFromLong(dateTimeValue, timeZone));
//        
// _timer.stop(header + "Total time so far for both date and time parsing: ");
    }

    /**
     * @param resultTimeseriesList
     * @param outputLocationId
     * @param fileDescription
     * @param logger
     * @param timeZone
     * @return
     * @throws Exception
     */
    private static ArrayList<FewsRegularTimeSeries> getFewsOutputTimeseriesList(final List<RegularTimeSeries> resultTimeseriesList,
                                                                                final String outputLocationId,
                                                                                final String fileDescription,
                                                                                final Logger logger,
                                                                                final TimeZone timeZone) throws Exception
    {

        //for holding final version of time series
        final ArrayList<FewsRegularTimeSeries> resultList = new ArrayList<FewsRegularTimeSeries>();
        //convert all the values in (RegularTimeSeries) to FewsRegularTimeSeries
        for(final RegularTimeSeries resultTs: resultTimeseriesList)
        {
            final FewsRegularTimeSeries finalFewsResultTs = new FewsRegularTimeSeries(resultTs.getStartTime(),
                                                                                      resultTs.getEndTime(),
                                                                                      resultTs.getIntervalInHours(),
                                                                                      resultTs.getMeasuringUnit());

            if(fileDescription != null)
            {
                finalFewsResultTs.setFileDescription(fileDescription);
            }

            final String timeSeriesType = resultTs.getTimeSeriesType();
            finalFewsResultTs.setTimeSeriesType(timeSeriesType);
            finalFewsResultTs.setType(FewsRegularTimeSeries.getFewsTimeCode(NwsrfsDataTypeMappingReader.getTimeCode(timeSeriesType,
                                                                                                                    logger)));
            //finalFewsResultTs.setQualifierId(resultTs.getQualifierId());
            finalFewsResultTs.setQualifierIds(resultTs.getQualifierIds());

            //Added to output ensemble related information.
            finalFewsResultTs.setEnsembleId(resultTs.getEnsembleId());
            finalFewsResultTs.setEnsembleMemberIndex(resultTs.getEnsembleMemberIndex());

            if(outputLocationId != null && outputLocationId.trim().length() > 0)
            {
                finalFewsResultTs.setLocationId(outputLocationId);
            }
            else
            {
                finalFewsResultTs.setLocationId(resultTs.getLocationId());
            }

            if(!resultTimeseriesList.isEmpty())
            {
                finalFewsResultTs.setLongName(((FewsRegularTimeSeries)resultTimeseriesList.get(0)).getLongName());
                finalFewsResultTs.setSourceOrganization(((FewsRegularTimeSeries)resultTimeseriesList.get(0)).getSourceOrganization());
                finalFewsResultTs.setSourceSystem(((FewsRegularTimeSeries)resultTimeseriesList.get(0)).getSourceSystem());
                finalFewsResultTs.setFileDescription(((FewsRegularTimeSeries)resultTimeseriesList.get(0)).getFileDescription());
            }

            for(int j = 0; j < resultTs.getMeasurementCount(); j++)
            {
                finalFewsResultTs.setMeasurementByIndex(resultTs.getMeasurementByIndex(j), j);
            }
            resultList.add(finalFewsResultTs);

        } //close while loop

        return resultList;
    }

    /**
     * Overloading method
     * 
     * @param fewsTsList
     * @param outputFileName
     * @param logger
     * @param timeZone
     * @param binaryFlag
     * @param daylightSavingObservatingTimeZone
     * @throws Exception
     */
    public synchronized static void writeRegularTimeSeriesToFewsXML(final List<RegularTimeSeries> fewsTsList,
                                                                    final String outputFileName,
                                                                    final Logger logger,
                                                                    final TimeZone timeZone,
                                                                    final boolean binaryFlag,
                                                                    final String daylightSavingObservatingTimeZone) throws Exception
    {
        writeTimeSeriesToFewsXML(binaryFlag,
                                 getFewsOutputTimeseriesList(fewsTsList, null, null, logger, timeZone),
                                 outputFileName,
                                 logger,
                                 timeZone,
                                 daylightSavingObservatingTimeZone);
    }

    /**
     * @param fewsTsList
     * @param outputFileName
     * @param logger
     * @param timeZone
     * @param binaryFlag
     * @throws Exception
     */
    public synchronized static void writeRegularTimeSeriesToFewsXML(final List<RegularTimeSeries> fewsTsList,
                                                                    final String outputFileName,
                                                                    final Logger logger,
                                                                    final TimeZone timeZone,
                                                                    final boolean binaryFlag) throws Exception
    {
        final String daylightSavingObservatingTimeZone = null;
        writeTimeSeriesToFewsXML(binaryFlag,
                                 getFewsOutputTimeseriesList(fewsTsList, null, null, logger, timeZone),
                                 outputFileName,
                                 logger,
                                 timeZone,
                                 daylightSavingObservatingTimeZone);
    }

    /**
     * Writes the states XML file (conforming to FEWS pi_state.xsd),not the actual states file itself. This XML file
     * contains metadata about the states file, including the time for which the state is valid.
     * 
     * @param states - the object representing the state for a given model
     * @throws Exception
     */
    public synchronized static void writeStatesMetaXMLFile(final String fileName,
                                                           final IState state,
                                                           final TimeZone timeZone,
                                                           final int timeZoneRawOffsetInHours,
                                                           final Logger logger) throws Exception
    {
        final String daylightSavingObservatingTimeZone = null;
        writeStatesMetaXMLFile(fileName,
                               state,
                               timeZone,
                               timeZoneRawOffsetInHours,
                               daylightSavingObservatingTimeZone,
                               logger);

    }

    /**
     * Writes the states XML file (conforming to FEWS pi_state.xsd),not the actual states file itself. This XML file
     * contains metadata about the states file, including the time for which the state is valid.
     * 
     * @param states - the object representing the state for a given model
     * @throws Exception
     */
    public synchronized static void writeStatesMetaXMLFile(final String fileName,
                                                           final IState state,
                                                           final TimeZone timeZone,
                                                           final int timeZoneRawOffsetInHours,
                                                           final String daylightSavingObservatingTimeZone,
                                                           final Logger logger) throws Exception
    {
        String encoding = "utf-8";

        if(fileName.endsWith(".fi"))
        {
            encoding = "finf";
        }

        // Set up output stream for fast infoset document
        final File outputFile = new File(fileName);
        BufferedOutputStream outputStream = null;

        XMLStreamWriter writer = null;

        try
        {

            outputStream = new BufferedOutputStream(new FileOutputStream(outputFile));
            writer = OHDXmlUtils.getStreamWriter(outputStream, fileName);
            // We are always expecting the value to be warm for Id. For that reason we hard code it.
            //final String stateIdStr = modelState.getId();
            final String stateIdStr = "warm";

            // We are always expecting the value to be warm for Name. For that reason we hard code it.
            //final String stateNameStr = modelState.getName();
            final String stateNameStr = "warm";

            //reset it with the time of latest state
            final long fewsTsTime = state.getDateTime();

            // Create new document
            //       writer.writeStartDocument("utf-8", "1.0");
            writer.writeStartDocument(encoding, "1.0");

            // Create State element
            writer.writeStartElement("State");
            writer.writeAttribute("version", state.getVersion());
            writer.writeAttribute("xsi:schemaLocation", "http://www.wldelft.nl/fews/PI pi_state.xsd");
            writer.writeAttribute("xmlns:xsi", XMLConstants.W3C_XML_SCHEMA_INSTANCE_NS_URI);
            writer.writeAttribute("xmlns", "http://www.wldelft.nl/fews/PI");
            //writer.writeEndElement();

            // Create stateId element
            writer.writeStartElement("stateId");
            writer.writeCharacters(stateIdStr);
            writer.writeEndElement(); // end of stateId

            // Create stateName element
            writer.writeStartElement("stateName");
            writer.writeCharacters(stateNameStr);
            writer.writeEndElement(); // end of stateName

            if(daylightSavingObservatingTimeZone != null && !daylightSavingObservatingTimeZone.isEmpty())
            {
                // Create daylightSavingObservingTimeZone element
                writer.writeStartElement("daylightSavingObservingTimeZone");
                writer.writeCharacters(daylightSavingObservatingTimeZone);
                writer.writeEndElement(); // end of stateIdNode                
            }
            else
            {
                // Create timeZone element
                writer.writeStartElement("timeZone");
                //to get a offset value for <timeZone> element
                writer.writeCharacters(String.valueOf(timeZoneRawOffsetInHours));
                writer.writeEndElement(); // end of stateIdNode
            }

            // Create dateTime element
            FewsAdapterDAO.writeDateTimeElement(writer, "dateTime", fewsTsTime, timeZone);

            for(final StateLocation stateLocation: state.getStateLocation().values())
            {
                // Create stateName element
                writer.writeStartElement("stateLoc");
                writer.writeAttribute("type", stateLocation.getType());
                //read Location
                writer.writeStartElement("readLocation");
                writer.writeCharacters(stateLocation.getReadLocation());
                writer.writeEndElement(); // end of readLocation
                // write Location
                writer.writeStartElement("writeLocation");
                writer.writeCharacters(stateLocation.getWriteLocation());
                writer.writeEndElement(); // end of writeLocation

                writer.writeEndElement(); // end of statelOC

            }
            writer.writeEndElement(); // end of state
            writer.writeEndDocument(); // end of document

            writer.flush();
        }
        catch(final Exception e)
        {
            final String message = "Error in writing Fews XML State meta data file: " + fileName;
            throw (new Exception(e.getMessage() + ". " + message));
        }
        finally
        {

            writer.close();

            outputStream.close();
        }

        FewsXmlValidation.validateXmlFileAgainsXMLSchema(fileName,
                                                         OHDFewsAdapterConstants.PI_STATE_XML_SCHEMA_TAG,
                                                         logger);

        logger.log(Logger.DEBUG, "wrote states meta data to xml: " + fileName);

    }

//    private static void writeParameters(final Parameters parameter, final XMLStreamWriter writer) throws XMLStreamException
//    {
//        // Create Parameter element
//        writer.writeStartElement("<parameters version=\"");
//        writer.writeAttribute("version", parameter.getVersion());
//        writer.writeAttribute("xsi:schemaLocation", "http://www.wldelft.nl/fews/PI pi_modelparameters.xsd");
//        writer.writeAttribute("xmlns:xsi", XMLConstants.W3C_XML_SCHEMA_INSTANCE_NS_URI);
//        writer.writeAttribute("xmlns", "http://www.wldelft.nl/fews/PI");
//
//        /** -------------output namespace, schemaLocation etc -------------------- */
//
//        for(final GroupOfParameters groups: parameter.getParamsList())
//        {
//
//            writer.writeStartElement("group");
//            if(groups.getId() != null)
//            {
//                writer.writeAttribute("id", groups.getId());
//            }
//            else
//            {
//                writer.writeAttribute("id", OHDConstants.DEFAULT_STRING);
//            }
//            if(groups.getName() != null)
//            {
//                writer.writeAttribute("name", groups.getName());
//            }
//            if(groups.getReadonly() != null && groups.getReadonly().equals(new Boolean(true)))
//            {
//                writer.writeAttribute("readonly", groups.getReadonly().toString());
//            }
//
//            if(groups.getDescription() != null && groups.getDescription().isEmpty() == false)
//            {
//                writer.writeStartElement("description");
//                writer.writeCharacters(groups.getDescription());
//                writer.writeEndElement(); // end of description;
//            }
//            if(groups.getLocationId() != null && groups.getLocationId().isEmpty() == false)
//            {
//                writer.writeStartElement("locationId");
//                writer.writeCharacters(groups.getLocationId());
//                writer.writeEndElement(); // end of locationId;
//            }
//            if(groups.getModel() != null && groups.getModel().isEmpty() == false)
//            {
//                writer.writeStartElement("model");
//                writer.writeCharacters(groups.getModel());
//                writer.writeEndElement(); // end of model;
//            }
//            if(groups.getValidPeriod() != null)
//            {
//                final ValidPeriod vPeriod = groups.getValidPeriod();
//                writer.writeStartElement("validPeriod");
//                if(vPeriod.getTimeZone() != null)
//                {
//                    writer.writeStartElement("timeZone");
//                    writer.writeCharacters(vPeriod.getTimeZone().toString());
//                    writer.writeEndElement(); // end of timeZone;
//                }
//
//                if(vPeriod.getStartDate() != null && vPeriod.getEndDate() != null)
//                {
//                    writer.writeStartElement("startDate");
//                    writer.writeAttribute("date", vPeriod.getStartDate().getDate());
//                    writer.writeAttribute("time", vPeriod.getStartDate().getTime());
//                    writer.writeEndElement(); // end of startDate;
//                    writer.writeStartElement("endDate");
//                    writer.writeAttribute("date", vPeriod.getEndDate().getDate());
//                    writer.writeAttribute("time", vPeriod.getEndDate().getTime());
//                    writer.writeEndElement(); // end of endDate;
//                }
//                else
//                {
//                    if(vPeriod.getValidBeforeDate() != null)
//                    {
//                        writer.writeStartElement("validBeforeDate");
//                        writer.writeAttribute("date", vPeriod.getValidBeforeDate().getDate());
//                        writer.writeAttribute("version", parameter.getVersion());
//                        writer.writeAttribute("time", vPeriod.getValidBeforeDate().getTime());
//                        writer.writeAttribute("version", parameter.getVersion());
//                        writer.writeEndElement(); // end of validBeforeDate;
//                    }
//                    else if(vPeriod.getValidAfterDate() != null)
//                    {
//                        writer.writeStartElement("validAfterDate");
//                        writer.writeAttribute("date", vPeriod.getValidAfterDate().getDate());
//                        writer.writeAttribute("time", vPeriod.getValidAfterDate().getTime());
//                        writer.writeEndElement(); // end of validAfterDate;
//                    }
//                    else if(vPeriod.getStartMonthDay() != null && vPeriod.getEndMonthDay() != null)
//                    {
//                        writer.writeStartElement("startMonthDay");
//                        writer.writeCharacters(vPeriod.getStartMonthDay());
//                        writer.writeEndElement(); // end of startMonthDay;
//                        writer.writeStartElement("endMonthDay");
//                        writer.writeCharacters(vPeriod.getEndMonthDay());
//                        writer.writeEndElement(); // end of endMonthDay;
//                    }
//                    else if(vPeriod.getMonthDay() != null && vPeriod.getMonthDay().size() > 0)
//                    {
//                        for(final String sMonthDay: vPeriod.getMonthDay())
//                        {
//                            writer.writeStartElement("monthDay");
//                            writer.writeCharacters(sMonthDay);
//                            writer.writeEndElement(); // end of monthDay;
//                        }
//                    }
//                    else if(vPeriod.getMonth() != null && vPeriod.getMonth().size() > 0)
//                    {
//                        for(final String sMonth: vPeriod.getMonth())
//                        {
//                            writer.writeStartElement("month");
//                            writer.writeCharacters(sMonth);
//                            writer.writeEndElement(); // end of Month;
//                        }
//                    }
//                    else if(vPeriod.getDay() != null && vPeriod.getDay().size() > 0)
//                    {
//                        for(final String sDay: vPeriod.getMonthDay())
//                        {
//                            writer.writeStartElement("day");
//                            writer.writeCharacters(sDay);
//                            writer.writeEndElement(); // end of Day;
//                        }
//                    }
//                }
//                writer.writeEndElement(); // end of validPeriod;
//            } // end valid period
//
//            if(groups.getParameterMap() != null && groups.getParameterMap().size() > 0)
//            {
//                for(final ParameterType param: groups.getParameterMap().values())
//                {
//                    writer.writeStartElement("parameter");
//                    writer.writeAttribute("id", param.getId());
//                    if(param.getName() != null)
//                    {
//                        writer.writeAttribute("name", param.getName());
//                    }
//                    if(param.getModified() != null && param.getModified().equals(new Boolean(true)))
//                    {
//                        writer.writeAttribute("modified", param.getModified().toString());
//                    }
//                    if(param.getDescription() != null && param.getDescription().isEmpty() == false)
//                    {
//                        writer.writeStartElement("description");
//                        writer.writeCharacters(groups.getDescription());
//                        writer.writeEndElement(); // end of description;
//                    }
//                    if(param.getDblValue() != null)
//                    {
//                        writer.writeStartElement("dblValue");
//                        writer.writeCharacters(param.getDblValue().toString());
//                        writer.writeEndElement(); // end of dblValue;
//                    }
//                    else if(param.getIntValue() != null)
//                    {
//                        writer.writeStartElement("intValue");
//                        writer.writeCharacters(param.getIntValue().toString());
//                        writer.writeEndElement(); // end of intValue)=;
//                    }
//                    else if(param.getBoolValue() != null)
//                    {
//                        writer.writeStartElement("boolValue");
//                        writer.writeCharacters(param.getBoolValue().toString());
//                        writer.writeEndElement(); // end of boolValue;
//                    }
//                    else if(param.getStringValue() != null && param.getStringValue().isEmpty() == false)
//                    {
//                        writer.writeStartElement("stringValue");
//                        writer.writeCharacters(param.getStringValue());
//                        writer.writeEndElement(); // end of stringValue;
//                    }
//                    else if(param.getTable() != null)
//                    {
//                        writer.writeStartElement("table");
//                        if(param.getTable().getColumnIds() != null)
//                        {
//                            writer.writeEmptyElement("columnIds");
//
//                            FewsAdapterDAO.writeParameterTableRowAttributes(writer, param.getTable().getColumnIds());
//
//                        }
//                        if(param.getTable().getColumnTypes() != null)
//                        {
//                            writer.writeEmptyElement("columnTypes");
//
//                            FewsAdapterDAO.writeParameterTableRowAttributes(writer, param.getTable().getColumnTypes());
//
//                        }
//                        if(param.getTable().getRow() != null && param.getTable().getRow().size() > 0)
//                        {
//                            for(final ohd.hseb.util.fews.ParameterType.Row tableRow: param.getTable().getRow())
//                            {
//                                writer.writeEmptyElement("row");
//
//                                FewsAdapterDAO.writeParameterTableRowAttributes(writer, tableRow);
//                            }
//                        }
//                        writer.writeEndElement(); // end of table;
//                    }
//                    writer.writeEndElement(); // end of parameter;
//                }
//            }
//            if(groups.getEnsemble() != null)
//            {
//                writer.writeEmptyElement("ensemble");
//                writer.writeAttribute("", groups.getEnsemble().getId());
//            }
//            writer.writeEndElement(); // end of group;
//        }
//        writer.writeEndElement(); // end of parameters element
//        writer.writeEndDocument();
//    }

    public synchronized static String writeParamsToStringTab(final Parameters parameter, final Logger logger) throws Exception
    {
//        final String paramsToString = null;

        final StringBuilder writer = new StringBuilder("<?xml version=\"1.0\" encoding=\"utf-8\"?>");

        writer.append("<parameters version=\"" + parameter.getVersion()
            + "\" xsi:schemaLocation=\"http://www.wldelft.nl/fews/PI pi_modelparameters.xsd\" xmlns:xsi=\""
            + XMLConstants.W3C_XML_SCHEMA_INSTANCE_NS_URI + "\" ");
        writer.append("xmlns=\"http://www.wldelft.nl/fews/PI\">\n");

        /** -------------output namespace, schemaLocation etc -------------------- */

        for(final GroupOfParameters groups: parameter.getParamsList())
        {

            writer.append("<group ");
            if(groups.getId() != null)
            {
                writer.append("id = \"" + groups.getId() + "\" ");
            }
            else
            {
                writer.append("id=\"" + OHDConstants.DEFAULT_STRING + "\" ");
            }
            if(groups.getName() != null)
            {
                writer.append("name=\"" + groups.getName() + "\" ");
            }
            if(groups.getReadonly() != null && groups.getReadonly().equals(true))
            {
                writer.append("readonly=\"" + groups.getReadonly().toString() + "\" ");
            }
            writer.append(">\n");
            if(groups.getDescription() != null && groups.getDescription().isEmpty() == false)
            {
                writer.append("<description>");
                writer.append(groups.getDescription());
                writer.append("</description>\n"); // end of description;
            }
            if(groups.getLocationId() != null && groups.getLocationId().isEmpty() == false)
            {
                writer.append("<locationId>");
                writer.append(groups.getLocationId());
                writer.append("</locationId>\n"); // end of locationId;
            }
            if(groups.getModel() != null && groups.getModel().isEmpty() == false)
            {
                writer.append("<model>");
                writer.append(groups.getModel());
                writer.append("</model>\n"); // end of model;
            }
            if(groups.getValidPeriod() != null)
            {
                final ValidPeriod vPeriod = groups.getValidPeriod();
                writer.append("<validPeriod>");
                if(vPeriod.getTimeZone() != null)
                {
                    writer.append("<timeZone>");
                    writer.append(vPeriod.getTimeZone().toString());
                    writer.append("</timeZone>\n"); // end of timeZone;
                }

                if(vPeriod.getStartDate() != null && vPeriod.getEndDate() != null)
                {
                    writer.append("<startDate ");
                    writer.append("date=\"" + vPeriod.getStartDate().getDate() + "\" ");
                    ;
                    writer.append("time=\"" + vPeriod.getStartDate().getTime() + "\"/>\n");

                    writer.append("<endDate ");
                    writer.append("date=\"" + vPeriod.getEndDate().getDate() + "\" ");
                    writer.append("time=\"" + vPeriod.getEndDate().getTime() + "\"/>\n");
                }
                else
                {
                    if(vPeriod.getValidBeforeDate() != null)
                    {
                        writer.append("<validBeforeDate ");
                        writer.append("date=\"" + vPeriod.getValidBeforeDate().getDate() + "\" ");
                        writer.append("version=\"" + parameter.getVersion() + "\" ");
                        writer.append("time=\"" + vPeriod.getValidBeforeDate().getTime() + "\" ");
                        writer.append("version=\"" + parameter.getVersion() + "\"/>\n");
                    }
                    else if(vPeriod.getValidAfterDate() != null)
                    {
                        writer.append("<validAfterDate");
                        writer.append("date=\"" + vPeriod.getValidAfterDate().getDate() + "\" ");
                        writer.append("time=\"" + vPeriod.getValidAfterDate().getTime() + "\"/>\n");
                    }
                    else if(vPeriod.getStartMonthDay() != null && vPeriod.getEndMonthDay() != null)
                    {
                        writer.append("<startMonthDay>");
                        writer.append(vPeriod.getStartMonthDay());
                        writer.append("</startMonthDay>\n"); // end of startMonthDay;
                        writer.append("<endMonthDay>");
                        writer.append(vPeriod.getEndMonthDay());
                        writer.append("</endMonthDay>\n"); // end of endMonthDay;
                    }
                    else if(vPeriod.getMonthDay() != null && vPeriod.getMonthDay().size() > 0)
                    {
                        for(final String sMonthDay: vPeriod.getMonthDay())
                        {
                            writer.append("<monthDay>");
                            writer.append(sMonthDay);
                            writer.append("</monthDay>\n"); // end of monthDay;
                        }
                    }
                    else if(vPeriod.getMonth() != null && vPeriod.getMonth().size() > 0)
                    {
                        for(final String sMonth: vPeriod.getMonth())
                        {
                            writer.append("<month>");
                            writer.append(sMonth);
                            writer.append("</month>\n"); // end of Month;
                        }
                    }
                    else if(vPeriod.getDay() != null && vPeriod.getDay().size() > 0)
                    {
                        for(final String sDay: vPeriod.getMonthDay())
                        {
                            writer.append("<day>");
                            writer.append(sDay);
                            writer.append("</day>\n"); // end of Day;
                        }
                    }
                }
                writer.append("</validPeriod>\n"); // end of validPeriod;
            } // end valid period

            if(groups.getParameterMap() != null && groups.getParameterMap().size() > 0)
            {
                for(final ParameterType param: groups.getParameterMap().values())
                {
                    writer.append("<parameter ");
                    writer.append("id=\"" + param.getId() + "\"");
                    if(param.getName() != null)
                    {
                        writer.append(" name=\"").append(param.getName()).append("\"");
                    }
                    if(param.getModified() != null && param.getModified().equals(true))
                    {
                        writer.append(" modified=\"").append(param.getModified()).append("\"");
                    }
                    writer.append(">\n");

                    if(param.getDescription() != null && param.getDescription().isEmpty() == false)
                    {
                        writer.append("<description>");
                        writer.append(groups.getDescription());
                        writer.append("</description>\n");
                        ; // end of description;
                    }
                    if(param.getDblValue() != null)
                    {
                        writer.append("\t<dblValue>");
                        writer.append(param.getDblValue());
                        writer.append("</dblValue>\n"); // end of dblValue;
                    }
                    else if(param.getIntValue() != null)
                    {
                        writer.append("\t<intValue>");
                        writer.append(param.getIntValue());
                        writer.append("</intValue>\n"); // end of intValue)=;
                    }
                    else if(param.getBoolValue() != null)
                    {
                        writer.append("\t<boolValue>");
                        writer.append(param.getBoolValue());
                        writer.append("</boolValue>\n"); // end of boolValue;
                    }
                    else if(param.getStringValue() != null && param.getStringValue().isEmpty() == false)
                    {
                        writer.append("\t<stringValue>");
                        writer.append(param.getStringValue());
                        writer.append("</stringValue>\n"); // end of stringValue;
                    }
                    else if(param.getTable() != null)
                    {
                        writer.append("<table>\n");
                        if(param.getTable().getColumnIds() != null)
                        {
                            writer.append("\t<columnIds ");

                            FewsAdapterDAO.writeParameterTableRowAttributes(writer, param.getTable().getColumnIds());
                            writer.append("/>\n");

                        }
                        if(param.getTable().getColumnTypes() != null)
                        {
                            writer.append("\t<columnTypes ");

                            FewsAdapterDAO.writeParameterTableRowAttributes(writer, param.getTable().getColumnTypes());

                            writer.append("/>\n");

                        }
                        if(param.getTable().getRow() != null && param.getTable().getRow().size() > 0)
                        {
                            for(final ohd.hseb.util.fews.ParameterType.Row tableRow: param.getTable().getRow())
                            {
                                writer.append("\t<row ");

                                FewsAdapterDAO.writeParameterTableRowAttributes(writer, tableRow);
                                writer.append("/>\n");
                            }
                        }
                        writer.append("</table>\n"); // end of table;
                    }
                    writer.append("</parameter>\n"); // end of parameter;
                }
            }
            if(groups.getEnsemble() != null)
            {
                writer.append("<ensemble ");
                writer.append(groups.getEnsemble().getId()).append("/>");
            }
            writer.append("</group>\n"); // end of group;
        }
        writer.append("</parameters>"); // end of parameters element

        return writer.toString();
    }

    public synchronized static String writeParamsToString(final Parameters parameter, final Logger logger) throws Exception
    {
//        final String paramsToString = null;

        final StringBuilder writer = new StringBuilder("<?xml version=\"1.0\" encoding=\"utf-8\"?>");

        writer.append("<parameters version=\"" + parameter.getVersion()
            + "\" xsi:schemaLocation=\"http://www.wldelft.nl/fews/PI pi_modelparameters.xsd\" xmlns:xsi=\""
            + XMLConstants.W3C_XML_SCHEMA_INSTANCE_NS_URI + "\" ");
        writer.append("xmlns=\"http://www.wldelft.nl/fews/PI\">");

        /** -------------output namespace, schemaLocation etc -------------------- */

        for(final GroupOfParameters groups: parameter.getParamsList())
        {

            writer.append("<group ");
            if(groups.getId() != null)
            {
                writer.append("id = \"" + groups.getId() + "\" ");
            }
            else
            {
                writer.append("id=\"" + OHDConstants.DEFAULT_STRING + "\" ");
            }
            if(groups.getName() != null)
            {
                writer.append("name=\"" + groups.getName() + "\" ");
            }
            if(groups.getReadonly() != null && groups.getReadonly().equals(true))
            {
                writer.append("readonly=\"" + groups.getReadonly().toString() + "\" ");
            }
            writer.append(">");
            if(groups.getDescription() != null && groups.getDescription().isEmpty() == false)
            {
                writer.append("<description>");
                writer.append(groups.getDescription());
                writer.append("</description>"); // end of description;
            }
            if(groups.getLocationId() != null && groups.getLocationId().isEmpty() == false)
            {
                writer.append("<locationId>");
                writer.append(groups.getLocationId());
                writer.append("</locationId>"); // end of locationId;
            }
            if(groups.getModel() != null && groups.getModel().isEmpty() == false)
            {
                writer.append("<model>");
                writer.append(groups.getModel());
                writer.append("</model>"); // end of model;
            }
            if(groups.getValidPeriod() != null)
            {
                final ValidPeriod vPeriod = groups.getValidPeriod();
                writer.append("<validPeriod>");
                if(vPeriod.getTimeZone() != null)
                {
                    writer.append("<timeZone>");
                    writer.append(vPeriod.getTimeZone().toString());
                    writer.append("</timeZone>"); // end of timeZone;
                }

                if(vPeriod.getStartDate() != null && vPeriod.getEndDate() != null)
                {
                    writer.append("<startDate ");
                    writer.append("date=\"" + vPeriod.getStartDate().getDate() + "\" ");
                    ;
                    writer.append("time=\"" + vPeriod.getStartDate().getTime() + "\"/>");

                    writer.append("<endDate ");
                    writer.append("date=\"" + vPeriod.getEndDate().getDate() + "\" ");
                    writer.append("time=\"" + vPeriod.getEndDate().getTime() + "\"/>");
                }
                else
                {
                    if(vPeriod.getValidBeforeDate() != null)
                    {
                        writer.append("<validBeforeDate ");
                        writer.append("date=\"" + vPeriod.getValidBeforeDate().getDate() + "\" ");
                        writer.append("version=\"" + parameter.getVersion() + "\" ");
                        writer.append("time=\"" + vPeriod.getValidBeforeDate().getTime() + "\" ");
                        writer.append("version=\"" + parameter.getVersion() + "\"/>");
                    }
                    else if(vPeriod.getValidAfterDate() != null)
                    {
                        writer.append("<validAfterDate");
                        writer.append("date=\"" + vPeriod.getValidAfterDate().getDate() + "\" ");
                        writer.append("time=\"" + vPeriod.getValidAfterDate().getTime() + "\"/>");
                    }
                    else if(vPeriod.getStartMonthDay() != null && vPeriod.getEndMonthDay() != null)
                    {
                        writer.append("<startMonthDay>");
                        writer.append(vPeriod.getStartMonthDay());
                        writer.append("</startMonthDay>"); // end of startMonthDay;
                        writer.append("<endMonthDay>");
                        writer.append(vPeriod.getEndMonthDay());
                        writer.append("</endMonthDay>"); // end of endMonthDay;
                    }
                    else if(vPeriod.getMonthDay() != null && vPeriod.getMonthDay().size() > 0)
                    {
                        for(final String sMonthDay: vPeriod.getMonthDay())
                        {
                            writer.append("<monthDay>");
                            writer.append(sMonthDay);
                            writer.append("</monthDay>"); // end of monthDay;
                        }
                    }
                    else if(vPeriod.getMonth() != null && vPeriod.getMonth().size() > 0)
                    {
                        for(final String sMonth: vPeriod.getMonth())
                        {
                            writer.append("<month>");
                            writer.append(sMonth);
                            writer.append("</month>"); // end of Month;
                        }
                    }
                    else if(vPeriod.getDay() != null && vPeriod.getDay().size() > 0)
                    {
                        for(final String sDay: vPeriod.getMonthDay())
                        {
                            writer.append("<day>");
                            writer.append(sDay);
                            writer.append("</day>"); // end of Day;
                        }
                    }
                }
                writer.append("</validPeriod>"); // end of validPeriod;
            } // end valid period

            if(groups.getParameterMap() != null && groups.getParameterMap().size() > 0)
            {
                for(final ParameterType param: groups.getParameterMap().values())
                {
                    writer.append("<parameter ");
                    writer.append("id=\"" + param.getId() + "\"");
                    if(param.getName() != null)
                    {
                        writer.append(" name=\"").append(param.getName()).append("\"");
                    }
                    if(param.getModified() != null && param.getModified().equals(true))
                    {
                        writer.append(" modified=\"").append(param.getModified()).append("\"");
                    }
                    writer.append(">");

                    if(param.getDescription() != null && param.getDescription().isEmpty() == false)
                    {
                        writer.append("<description>");
                        writer.append(groups.getDescription());
                        writer.append("</description>");
                        ; // end of description;
                    }
                    if(param.getDblValue() != null)
                    {
                        writer.append("<dblValue>");
                        writer.append(param.getDblValue());
                        writer.append("</dblValue>"); // end of dblValue;
                    }
                    else if(param.getIntValue() != null)
                    {
                        writer.append("<intValue>");
                        writer.append(param.getIntValue());
                        writer.append("</intValue>"); // end of intValue)=;
                    }
                    else if(param.getBoolValue() != null)
                    {
                        writer.append("<boolValue>");
                        writer.append(param.getBoolValue());
                        writer.append("</boolValue>"); // end of boolValue;
                    }
                    else if(param.getStringValue() != null && param.getStringValue().isEmpty() == false)
                    {
                        writer.append("<stringValue>");
                        writer.append(param.getStringValue());
                        writer.append("</stringValue>"); // end of stringValue;
                    }
                    else if(param.getTable() != null)
                    {
                        writer.append("<table>");
                        if(param.getTable().getColumnIds() != null)
                        {
                            writer.append("<columnIds ");

                            FewsAdapterDAO.writeParameterTableRowAttributes(writer, param.getTable().getColumnIds());
                            writer.append("/>");

                        }
                        if(param.getTable().getColumnTypes() != null)
                        {
                            writer.append("<columnTypes ");

                            FewsAdapterDAO.writeParameterTableRowAttributes(writer, param.getTable().getColumnTypes());

                            writer.append("/>");

                        }
                        if(param.getTable().getRow() != null && param.getTable().getRow().size() > 0)
                        {
                            for(final ohd.hseb.util.fews.ParameterType.Row tableRow: param.getTable().getRow())
                            {
                                writer.append("<row ");

                                FewsAdapterDAO.writeParameterTableRowAttributes(writer, tableRow);
                                writer.append("/>");
                            }
                        }
                        writer.append("</table>"); // end of table;
                    }
                    writer.append("</parameter>"); // end of parameter;
                }
            }
            if(groups.getEnsemble() != null)
            {
                writer.append("<ensemble ");
                writer.append(groups.getEnsemble().getId()).append("/>");
            }
            writer.append("</group>"); // end of group;
        }
        writer.append("</parameters>"); // end of parameters element

        return writer.toString();
    }

    /**
     * @param parameter
     * @param fileName
     * @param logger
     * @throws Exception
     */
    public synchronized static void writeParamsToXML(final Parameters parameter,
                                                     final String fileName,
                                                     final Logger logger) throws Exception
    {
        // Set up output stream for fast infoset document
        final File outputFile = new File(fileName);
        BufferedOutputStream outputStream = null;

        XMLStreamWriter writer = null;
        try
        {
            outputStream = new BufferedOutputStream(new FileOutputStream(outputFile));

            writer = OHDXmlUtils.getStreamWriter(outputStream, fileName);

            String encoding = "utf-8";

            if(fileName.endsWith(".fi"))
            {
                encoding = "finf";
            }

            //   	writer.writeStartDocument("utf-8", "1.0");
            writer.writeStartDocument(encoding, "1.0");

            // Create State element
            writer.writeStartElement("parameters");
            writer.writeAttribute("version", parameter.getVersion());
            writer.writeAttribute("xsi:schemaLocation", "http://www.wldelft.nl/fews/PI pi_modelparameters.xsd");
            writer.writeAttribute("xmlns:xsi", XMLConstants.W3C_XML_SCHEMA_INSTANCE_NS_URI);
            writer.writeAttribute("xmlns", "http://www.wldelft.nl/fews/PI");

            /** -------------output namespace, schemaLocation etc -------------------- */

            for(final GroupOfParameters groups: parameter.getParamsList())
            {

                writer.writeStartElement("group");
                if(groups.getId() != null)
                {
                    writer.writeAttribute("id", groups.getId());
                }
                else
                {
                    writer.writeAttribute("id", OHDConstants.DEFAULT_STRING);
                }
                if(groups.getName() != null)
                {
                    writer.writeAttribute("name", groups.getName());
                }
                if(groups.getReadonly() != null && groups.getReadonly().equals(true))
                {
                    writer.writeAttribute("readonly", groups.getReadonly().toString());
                }

                if(groups.getDescription() != null && groups.getDescription().isEmpty() == false)
                {
                    writer.writeStartElement("description");
                    writer.writeCharacters(groups.getDescription());
                    writer.writeEndElement(); // end of description;
                }
                if(groups.getLocationId() != null && groups.getLocationId().isEmpty() == false)
                {
                    writer.writeStartElement("locationId");
                    writer.writeCharacters(groups.getLocationId());
                    writer.writeEndElement(); // end of locationId;
                }
                if(groups.getModel() != null && groups.getModel().isEmpty() == false)
                {
                    writer.writeStartElement("model");
                    writer.writeCharacters(groups.getModel());
                    writer.writeEndElement(); // end of model;
                }
                if(groups.getValidPeriod() != null)
                {
                    final ValidPeriod vPeriod = groups.getValidPeriod();
                    writer.writeStartElement("validPeriod");
                    if(vPeriod.getTimeZone() != null)
                    {
                        writer.writeStartElement("timeZone");
                        writer.writeCharacters(vPeriod.getTimeZone().toString());
                        writer.writeEndElement(); // end of timeZone;
                    }

                    if(vPeriod.getStartDate() != null && vPeriod.getEndDate() != null)
                    {
                        writer.writeStartElement("startDate");
                        writer.writeAttribute("date", vPeriod.getStartDate().getDate());
                        writer.writeAttribute("time", vPeriod.getStartDate().getTime());
                        writer.writeEndElement(); // end of startDate;
                        writer.writeStartElement("endDate");
                        writer.writeAttribute("date", vPeriod.getEndDate().getDate());
                        writer.writeAttribute("time", vPeriod.getEndDate().getTime());
                        writer.writeEndElement(); // end of endDate;
                    }
                    else
                    {
                        if(vPeriod.getValidBeforeDate() != null)
                        {
                            writer.writeStartElement("validBeforeDate");
                            writer.writeAttribute("date", vPeriod.getValidBeforeDate().getDate());
                            writer.writeAttribute("version", parameter.getVersion());
                            writer.writeAttribute("time", vPeriod.getValidBeforeDate().getTime());
                            writer.writeAttribute("version", parameter.getVersion());
                            writer.writeEndElement(); // end of validBeforeDate;
                        }
                        else if(vPeriod.getValidAfterDate() != null)
                        {
                            writer.writeStartElement("validAfterDate");
                            writer.writeAttribute("date", vPeriod.getValidAfterDate().getDate());
                            writer.writeAttribute("time", vPeriod.getValidAfterDate().getTime());
                            writer.writeEndElement(); // end of validAfterDate;
                        }
                        else if(vPeriod.getStartMonthDay() != null && vPeriod.getEndMonthDay() != null)
                        {
                            writer.writeStartElement("startMonthDay");
                            writer.writeCharacters(vPeriod.getStartMonthDay());
                            writer.writeEndElement(); // end of startMonthDay;
                            writer.writeStartElement("endMonthDay");
                            writer.writeCharacters(vPeriod.getEndMonthDay());
                            writer.writeEndElement(); // end of endMonthDay;
                        }
                        else if(vPeriod.getMonthDay() != null && vPeriod.getMonthDay().size() > 0)
                        {
                            for(final String sMonthDay: vPeriod.getMonthDay())
                            {
                                writer.writeStartElement("monthDay");
                                writer.writeCharacters(sMonthDay);
                                writer.writeEndElement(); // end of monthDay;
                            }
                        }
                        else if(vPeriod.getMonth() != null && vPeriod.getMonth().size() > 0)
                        {
                            for(final String sMonth: vPeriod.getMonth())
                            {
                                writer.writeStartElement("month");
                                writer.writeCharacters(sMonth);
                                writer.writeEndElement(); // end of Month;
                            }
                        }
                        else if(vPeriod.getDay() != null && vPeriod.getDay().size() > 0)
                        {
                            for(final String sDay: vPeriod.getMonthDay())
                            {
                                writer.writeStartElement("day");
                                writer.writeCharacters(sDay);
                                writer.writeEndElement(); // end of Day;
                            }
                        }
                    }
                    writer.writeEndElement(); // end of validPeriod;
                } // end valid period

                if(groups.getParameterMap() != null && groups.getParameterMap().size() > 0)
                {
                    for(final ParameterType param: groups.getParameterMap().values())
                    {
                        writer.writeStartElement("parameter");
                        writer.writeAttribute("id", param.getId());
                        if(param.getName() != null)
                        {
                            writer.writeAttribute("name", param.getName());
                        }
                        if(param.getModified() != null && param.getModified().equals(true))
                        {
                            writer.writeAttribute("modified", param.getModified().toString());
                        }
                        if(param.getDescription() != null && param.getDescription().isEmpty() == false)
                        {
                            writer.writeStartElement("description");
                            writer.writeCharacters(groups.getDescription());
                            writer.writeEndElement(); // end of description;
                        }
                        if(param.getDblValue() != null)
                        {
                            writer.writeStartElement("dblValue");
                            writer.writeCharacters(param.getDblValue().toString());
                            writer.writeEndElement(); // end of dblValue;
                        }
                        else if(param.getIntValue() != null)
                        {
                            writer.writeStartElement("intValue");
                            writer.writeCharacters(param.getIntValue().toString());
                            writer.writeEndElement(); // end of intValue)=;
                        }
                        else if(param.getBoolValue() != null)
                        {
                            writer.writeStartElement("boolValue");
                            writer.writeCharacters(param.getBoolValue().toString());
                            writer.writeEndElement(); // end of boolValue;
                        }
                        else if(param.getStringValue() != null && param.getStringValue().isEmpty() == false)
                        {
                            writer.writeStartElement("stringValue");
                            writer.writeCharacters(param.getStringValue());
                            writer.writeEndElement(); // end of stringValue;
                        }
                        else if(param.getTable() != null)
                        {
                            writer.writeStartElement("table");
                            if(param.getTable().getColumnIds() != null)
                            {
                                writer.writeEmptyElement("columnIds");

                                FewsAdapterDAO.writeParameterTableRowAttributes(writer, param.getTable().getColumnIds());

                            }
                            if(param.getTable().getColumnTypes() != null)
                            {
                                writer.writeEmptyElement("columnTypes");

                                FewsAdapterDAO.writeParameterTableRowAttributes(writer, param.getTable()
                                                                                             .getColumnTypes());

                            }
                            if(param.getTable().getRow() != null && param.getTable().getRow().size() > 0)
                            {
                                for(final ohd.hseb.util.fews.ParameterType.Row tableRow: param.getTable().getRow())
                                {
                                    writer.writeEmptyElement("row");

                                    FewsAdapterDAO.writeParameterTableRowAttributes(writer, tableRow);
                                }
                            }
                            writer.writeEndElement(); // end of table;
                        }
                        writer.writeEndElement(); // end of parameter;
                    }
                }
                if(groups.getEnsemble() != null)
                {
                    writer.writeEmptyElement("ensemble");
                    writer.writeAttribute("", groups.getEnsemble().getId());
                }
                writer.writeEndElement(); // end of group;
            }
            writer.writeEndElement(); // end of parameters element
            writer.writeEndDocument();
            writer.flush();
        }
        catch(final Exception e)
        {
            final String message = "Error in writing Fews XML Parameter file: " + fileName;
            throw (new Exception(e.getMessage() + ". " + message));
        }
        finally
        {
            writer.close();
            //outputFile.
            outputStream.close();
        }

        FewsXmlValidation.validateXmlFileAgainsXMLSchema(fileName,
                                                         OHDFewsAdapterConstants.PI_MODELPARAMETERS_XML_SCHEMA_TAG,
                                                         logger);
    }

    private static void writeParameterTableRowAttributesIfNotNull(final StringBuilder writer,
                                                                  final String attributeName,
                                                                  final String attributeValue) throws XMLStreamException
    {
        if(attributeValue != null)
        {
            writer.append(attributeName).append("=\"").append(attributeValue).append("\"");
        }
    }

    private static void writeParameterTableRowAttributes(final StringBuilder writer,
                                                         final ohd.hseb.util.fews.ParameterType.Row row)
    {
        try
        {
            //  Need to check for null or an exception will be throw.
            String valueA = row.getA();
            if(valueA == null)
            {
                valueA = "";
            }

            //Always write out A.
            writer.append("A").append("=\"").append(valueA).append("\"");
            //The rest will only be written if not null 		
            writeParameterTableRowAttributesIfNotNull(writer, "B", row.getB());
            writeParameterTableRowAttributesIfNotNull(writer, "C", row.getC());
            writeParameterTableRowAttributesIfNotNull(writer, "D", row.getD());
            writeParameterTableRowAttributesIfNotNull(writer, "E", row.getE());
            writeParameterTableRowAttributesIfNotNull(writer, "F", row.getF());
            writeParameterTableRowAttributesIfNotNull(writer, "G", row.getG());
            writeParameterTableRowAttributesIfNotNull(writer, "H", row.getH());
            writeParameterTableRowAttributesIfNotNull(writer, "I", row.getI());
            writeParameterTableRowAttributesIfNotNull(writer, "J", row.getJ());
            writeParameterTableRowAttributesIfNotNull(writer, "K", row.getK());
            writeParameterTableRowAttributesIfNotNull(writer, "L", row.getL());
            writeParameterTableRowAttributesIfNotNull(writer, "M", row.getM());
            writeParameterTableRowAttributesIfNotNull(writer, "N", row.getN());
            writeParameterTableRowAttributesIfNotNull(writer, "O", row.getO());
            writeParameterTableRowAttributesIfNotNull(writer, "P", row.getP());
            writeParameterTableRowAttributesIfNotNull(writer, "Q", row.getQ());
            writeParameterTableRowAttributesIfNotNull(writer, "R", row.getR());
            writeParameterTableRowAttributesIfNotNull(writer, "S", row.getS());
            writeParameterTableRowAttributesIfNotNull(writer, "T", row.getT());
            writeParameterTableRowAttributesIfNotNull(writer, "U", row.getU());
            writeParameterTableRowAttributesIfNotNull(writer, "V", row.getV());
            writeParameterTableRowAttributesIfNotNull(writer, "W", row.getW());
            writeParameterTableRowAttributesIfNotNull(writer, "X", row.getX());
            writeParameterTableRowAttributesIfNotNull(writer, "Y", row.getY());
            writeParameterTableRowAttributesIfNotNull(writer, "Z", row.getZ());
        }
        catch(final Exception e)
        {
            // TODO
        }
    } // end of writeParameterTableRowAttributes method

    private static void writeParameterTableRowAttributesIfNotNull(final XMLStreamWriter writer,
                                                                  final String attributeName,
                                                                  final String attributeValue) throws XMLStreamException
    {
        if(attributeValue != null)
        {
            writer.writeAttribute(attributeName, attributeValue);
        }
    }

    private static void writeParameterTableRowAttributes(final XMLStreamWriter writer,
                                                         final ohd.hseb.util.fews.ParameterType.Row row)
    {
        try
        {
            //  Need to check for null or an exception will be throw.
            String valueA = row.getA();
            if(valueA == null)
            {
                valueA = "";
            }

            //Always write out A.
            writer.writeAttribute("A", valueA);
            //The rest will only be written if not null         
            writeParameterTableRowAttributesIfNotNull(writer, "B", row.getB());
            writeParameterTableRowAttributesIfNotNull(writer, "C", row.getC());
            writeParameterTableRowAttributesIfNotNull(writer, "D", row.getD());
            writeParameterTableRowAttributesIfNotNull(writer, "E", row.getE());
            writeParameterTableRowAttributesIfNotNull(writer, "F", row.getF());
            writeParameterTableRowAttributesIfNotNull(writer, "G", row.getG());
            writeParameterTableRowAttributesIfNotNull(writer, "H", row.getH());
            writeParameterTableRowAttributesIfNotNull(writer, "I", row.getI());
            writeParameterTableRowAttributesIfNotNull(writer, "J", row.getJ());
            writeParameterTableRowAttributesIfNotNull(writer, "K", row.getK());
            writeParameterTableRowAttributesIfNotNull(writer, "L", row.getL());
            writeParameterTableRowAttributesIfNotNull(writer, "M", row.getM());
            writeParameterTableRowAttributesIfNotNull(writer, "N", row.getN());
            writeParameterTableRowAttributesIfNotNull(writer, "O", row.getO());
            writeParameterTableRowAttributesIfNotNull(writer, "P", row.getP());
            writeParameterTableRowAttributesIfNotNull(writer, "Q", row.getQ());
            writeParameterTableRowAttributesIfNotNull(writer, "R", row.getR());
            writeParameterTableRowAttributesIfNotNull(writer, "S", row.getS());
            writeParameterTableRowAttributesIfNotNull(writer, "T", row.getT());
            writeParameterTableRowAttributesIfNotNull(writer, "U", row.getU());
            writeParameterTableRowAttributesIfNotNull(writer, "V", row.getV());
            writeParameterTableRowAttributesIfNotNull(writer, "W", row.getW());
            writeParameterTableRowAttributesIfNotNull(writer, "X", row.getX());
            writeParameterTableRowAttributesIfNotNull(writer, "Y", row.getY());
            writeParameterTableRowAttributesIfNotNull(writer, "Z", row.getZ());
        }
        catch(final Exception e)
        {
            // TODO
        }
    } // end of writeParameterTableRowAttributes method

    /**
     * Write the logging information to an XML file.
     * 
     * @param logger the Diagnostic object to be save to xml file.
     * @param fileName The name of the file to which to dump the log to
     * @throws Exception any exceptions caught while building/writing XML
     */
    public synchronized static void writeLog(final Logger logger, final String fileName) throws Exception
    {

        final Diagnostics diagnostics = (Diagnostics)logger;
        final String XMLNS = "http://www.wldelft.nl/fews/PI";
        final String XSI = "http://www.w3.org/2001/XMLSchema-instance";
        final String DIAGSCHEMALOCATION = "http://www.wldelft.nl/fews/PI http://fews.wldelft.nl/schemas/version1.0/pi-schemas/pi_diag.xsd";
        final String SCHEMAVERSION = "1.2";
        ArrayList<ohd.hseb.util.fews.Diagnostic> diagList = new ArrayList<ohd.hseb.util.fews.Diagnostic>();
        try
        {
            String encoding = "utf-8";
            if(fileName.endsWith(".fi"))
            {
                encoding = "finf";
            }

            // Set up output stream for fast infoset document
            final File outputFile = new File(fileName);
            final BufferedOutputStream outputStream = new BufferedOutputStream(new FileOutputStream(outputFile));
            final XMLStreamWriter writer = OHDXmlUtils.getStreamWriter(outputStream, fileName);

            writer.writeStartDocument(encoding, "1.0");

            // Create Diagnostic element
            writer.writeStartElement("Diag");
            writer.writeAttribute("xmlns", XMLNS);
            writer.writeAttribute("xmlns:xsi", XSI);
            writer.writeAttribute("xsi:schemaLocation", DIAGSCHEMALOCATION);
            writer.writeAttribute("version", SCHEMAVERSION);

            diagList = diagnostics.getList();

            String message;
            for(int i = 0; i < diagList.size(); i++)
            {
                writer.writeEmptyElement("line");
                writer.writeAttribute("level", Integer.toString(diagList.get(i).getLevel()));
                if(diagList.get(i).getMessage() != null)
                {
                    message = diagList.get(i).getMessage().replace("\n", "\\n");

                    writer.writeAttribute("description", message);
                }
                else
                { //the message is null
                    writer.writeAttribute("description", "");
                }
                //writer.writeEndElement();
            }
            writer.writeEndElement();
            writer.writeEndDocument();
            writer.flush();
            writer.close();

            outputStream.close();

            FewsXmlValidation.validateXmlFileAgainsXMLSchema(fileName, "pi_diag.xsd", logger);
        }
        finally
        {
            //out.flush();
            //out.close();
        }
    }

    /**
     * Write the Meta data Mods information to an XML file.
     * 
     * @param logger the Diagnostic object to be save to xml file.
     * @param fileName The name of the file to which to dump the log to
     * @throws Exception any exceptions caught while building/writing XML
     */
    public synchronized static void writeMetaDataImportMods(final ModelImportMods importMods,
                                                            final String fileName,
                                                            final Logger logger) throws Exception
    {

        final String XMLNS = "http://www.wldelft.nl/fews/PI";
        final String XSI = "http://www.w3.org/2001/XMLSchema-instance";
        final String DIAGSCHEMALOCATION = "http://www.wldelft.nl/fews/PI http://fews.wldelft.nl/schemas/version1.0/pi-schemas/pi_importmods.xsd";
        try
        {
            String encoding = "utf-8";
            if(fileName.endsWith(".fi"))
            {
                encoding = "finf";
            }

            // Set up output stream for fast infoset document
            final File outputFile = new File(fileName);
            final BufferedOutputStream outputStream = new BufferedOutputStream(new FileOutputStream(outputFile));
            final XMLStreamWriter writer = OHDXmlUtils.getStreamWriter(outputStream, fileName);

            writer.writeStartDocument(encoding, "1.0");

            // Create Diagnostic element
            writer.writeStartElement("importMods");
            writer.writeAttribute("xmlns", XMLNS);
            writer.writeAttribute("xmlns:xsi", XSI);
            writer.writeAttribute("xsi:schemaLocation", DIAGSCHEMALOCATION);

            if(importMods.getMetaData().getTimeZone() != null)
            {
                writer.writeStartElement("timeZone");
                writer.writeCharacters(importMods.getMetaData().getTimeZone());
                writer.writeEndElement();
            }
            writer.writeStartElement("metaData");
            writer.writeStartElement("modifierId");
            writer.writeCharacters(importMods.getMetaData().getModifierId());
            writer.writeEndElement();
            if(importMods.getMetaData().getName() != null)
            {
                writer.writeStartElement("name");
                writer.writeCharacters(importMods.getMetaData().getName());
                writer.writeEndElement();
            }

            writer.writeEmptyElement("startTime");
            writer.writeAttribute("date", importMods.getMetaData().getStartTime().getDateString());
            writer.writeAttribute("time", importMods.getMetaData().getStartTime().getTimeString());

            writer.writeEmptyElement("endTime");
            writer.writeAttribute("date", importMods.getMetaData().getEndTime().getDateString());
            writer.writeAttribute("time", importMods.getMetaData().getEndTime().getTimeString());

            writer.writeEmptyElement("validTime");
            writer.writeAttribute("date", importMods.getMetaData().getValidTime().getDateString());
            writer.writeAttribute("time", importMods.getMetaData().getValidTime().getTimeString());

            writer.writeEndElement();

            //Only one of the files type can be in the xml generated.
            writer.writeStartElement("file");
            if(importMods.getFile().getTimeSeriesFile() != null)
            {
                writer.writeStartElement("timeSeriesFile");
                writer.writeCharacters(importMods.getFile().getTimeSeriesFile());
                writer.writeEndElement();
            }
            else
            {
                writer.writeStartElement("parametersFile");
                writer.writeCharacters(importMods.getFile().getParametersFile());
                writer.writeEndElement();
            }
            writer.writeEndElement();

            writer.writeEndElement();
            writer.writeEndDocument();
            writer.flush();
            writer.close();

            outputStream.close();

            FewsXmlValidation.validateXmlFileAgainsXMLSchema(fileName,
                                                             OHDFewsAdapterConstants.PI_IMPORTMODS_XML_SCHEMA_TAG,
                                                             logger);

        }
        finally
        {
        }
    }

    /**
     * Write the logging information to an XML file.
     * 
     * @param logger the Diagnostic object to be save to xml file.
     * @param fileName The name of the file to which to dump the log to
     * @throws Exception any exceptions caught while building/writing XML
     */
    public synchronized static void writeRunInfo(final RunInfo runInfo, final String fileName, final Logger logger) throws Exception
    {

        final String XMLNS = "http://www.wldelft.nl/fews/PI";
        final String XSI = "http://www.w3.org/2001/XMLSchema-instance";
        final String DIAGSCHEMALOCATION = "http://www.wldelft.nl/fews/PI http://fews.wldelft.nl/schemas/version1.0/pi-schemas/pi_run.xsd";
        try
        {
            String encoding = "utf-8";
            if(fileName.endsWith(".fi"))
            {
                encoding = "finf";
            }

            // Set up output stream for fast infoset document
            final File outputFile = new File(fileName);
            final BufferedOutputStream outputStream = new BufferedOutputStream(new FileOutputStream(outputFile));
            final XMLStreamWriter writer = OHDXmlUtils.getStreamWriter(outputStream, fileName);

            writer.writeStartDocument(encoding, "1.0");

            // Create Diagnostic element
            writer.writeStartElement("Run");
            writer.writeAttribute("xmlns", XMLNS);
            writer.writeAttribute("xmlns:xsi", XSI);
            writer.writeAttribute("xsi:schemaLocation", DIAGSCHEMALOCATION);
            writer.writeAttribute("version", runInfo.getVersion());

            if(runInfo.getDaylightSavingObservatingTimeZone() != null
                && !runInfo.getDaylightSavingObservatingTimeZone().isEmpty())
            {
                // Create daylightSavingObservatingTimeZone element
                writer.writeStartElement("daylightSavingObservatingTimeZone");
                writer.writeCharacters(runInfo.getDaylightSavingObservatingTimeZone());
                writer.writeEndElement();
            }
            else
            {
                writer.writeStartElement("timeZone");
                writer.writeCharacters(Double.toString(runInfo.getTimeZoneRawOffsetInHours()));
                writer.writeEndElement();
            }

            FewsAdapterDAO.writeDateTimeElement(writer,
                                                "startDateTime",
                                                runInfo.getRunStartTimeLong(),
                                                runInfo.getTimeZone());

            FewsAdapterDAO.writeDateTimeElement(writer,
                                                "endDateTime",
                                                runInfo.getRunEndTimeLong(),
                                                runInfo.getTimeZone());

            FewsAdapterDAO.writeDateTimeElement(writer, "time0", runInfo.getTime0Long(), runInfo.getTimeZone());

            FewsAdapterDAO.writeDateTimeElement(writer,
                                                "lastObservationDateTime",
                                                runInfo.getRunLastObservationTimeLong(),
                                                runInfo.getTimeZone());

            writer.writeStartElement("workDir");
            writer.writeCharacters(runInfo.getWorkDir());
            writer.writeEndElement();

            writer.writeStartElement("inputParameterFile");
            writer.writeCharacters(runInfo.getInputParameterFile());
            writer.writeEndElement();

            writer.writeStartElement("inputStateDescriptionFile");
            writer.writeCharacters(runInfo.getInputStateDescriptionFile());
            writer.writeEndElement();

            for(final String inputTimeSeriesFileName: runInfo.getInputTimeSeriesFileList())
            {
                writer.writeStartElement("inputTimeSeriesFile");
                writer.writeCharacters(inputTimeSeriesFileName);
                writer.writeEndElement();
            }

            if(runInfo.getInputNetCdfFile() != null)
            {
                writer.writeStartElement("inputNetcdfFile");
                writer.writeCharacters(runInfo.getInputNetCdfFile());
                writer.writeEndElement();
            }

            writer.writeStartElement("outputDiagnosticFile");
            writer.writeCharacters(runInfo.getDiagnosticFile());
            writer.writeEndElement();

            writer.writeStartElement("outputTimeSeriesFile");
            writer.writeCharacters(runInfo.getOutputTimeSeriesFile());
            writer.writeEndElement();

            if(runInfo.getOutputNetCdfFile() != null)
            {
                writer.writeStartElement(NetcdfConstants.OUTPUT_NETCDF_FILE);
                writer.writeCharacters(runInfo.getOutputNetCdfFile());
                writer.writeEndElement();
            }

            writer.writeStartElement("properties");
            for(final String propertyName: runInfo.getProperties().stringPropertyNames())
            {
                final Object value = runInfo.getProperties().getProperty(propertyName);
                String name = null;
                try
                {
                    Integer.parseInt(value.toString());
                    name = "int";
                }
                catch(final NumberFormatException e)
                {
                    name = "string";
                }
                catch(final Exception e)
                {
                    name = "string";
                }
                writer.writeEmptyElement(name);
                writer.writeAttribute("key", propertyName);
                writer.writeAttribute("value", (String)value);

            }
            writer.writeEndElement(); //properties

            writer.writeEndElement();
            writer.writeEndDocument();
            writer.flush();
            writer.close();

            outputStream.close();

            FewsXmlValidation.validateXmlFileAgainsXMLSchema(fileName,
                                                             OHDFewsAdapterConstants.PI_RUN_XML_SCHEMA_TAG,
                                                             logger);

        }
        finally
        {
            //out.flush();
            //out.close();
        }
    }
}
