package ohd.hseb.util.fews;

import java.util.List;

import ohd.hseb.measurement.Measurement;
import ohd.hseb.measurement.MeasuringUnit;
import ohd.hseb.measurement.RegularTimeSeries;
import ohd.hseb.time.DateTime;

/**
 * Class representing a timeseries element received from FEWS. Adds fields to represent inputs received from FEWS not in
 * RegularTimeSeries.
 * 
 * @author FewsPilot team
 */
final public class FewsRegularTimeSeries extends RegularTimeSeries
{

    /**
     * represents the type received from Fews - accumulated/instantaneous
     */
    private FEWS_RTS_TYPE _type = FEWS_RTS_TYPE.INSTANTANEOUS;

    /*
     * The following fields will be saved as received from FEWS, so that they can be written back with the resultant
     * time series - no processing done on these fields
     */
    private String _longName = "";

    private String _sourceOrganization = "";

    private String _sourceSystem = "";

    private String _fileDescription = "";

    private DateTime _creationDateTime;

    private Double _x = null;
    private Double _y = null;
    private Double _z = null;
    private Float _latitude = null;
    private Float _longitude = null;
    private String _version = "1.2"; //set the value to the default.

    private List<TimeSeriesThreshold> _thresholdsList = null;

    /**
     * Sets up a time series with Measurement objects in a list to hold data for each time step
     * 
     * @param startTime of time series (epoch)
     * @param endTime of time series (epoch)
     * @param intervalInHours timestep
     * @param unit unit
     */
    public FewsRegularTimeSeries(final long startTime,
                                 final long endTime,
                                 final int intervalInHours,
                                 final MeasuringUnit unit)
    {
        super(startTime, endTime, intervalInHours, unit, OHDConstants.MISSING_DATA);

        _creationDateTime = new DateTime(System.currentTimeMillis(), OHDConstants.GMT_TIMEZONE);
    }

    /**
     * Copy Constructor
     * 
     * @param ts
     */
    public FewsRegularTimeSeries(final FewsRegularTimeSeries ts)
    {
        super(ts.getStartTime(), ts.getEndTime(), ts.getIntervalInHours(), ts.getMeasuringUnit());
        _longName = ts.getLongName();
        _sourceOrganization = ts.getSourceOrganization();
        _sourceSystem = ts.getSourceSystem();
        _fileDescription = ts.getFileDescription();
        _creationDateTime = ts._creationDateTime.clone();
        _type = ts.getType();
        setLocationId(ts.getLocationId());
        setEnsembleId(ts.getEnsembleId());
        setEnsembleMemberIndex(ts.getEnsembleMemberIndex());
        _timeSeriesType = ts.getTimeSeriesType();
        setName(ts.getName());
        setX(ts.getX());
        setY(ts.getY());
        setZ(ts.getZ());
        setLatitude(ts.getLatitude());
        setLongitude(ts.getLongitude());
        for ( int i = 0 ; i < ts.getMeasurementCount(); i++ )
        {
            Measurement measurement = new Measurement( ts.getMeasurementByIndex( i ) );
            this.setMeasurementByIndex( measurement, i );
        }
    }

    /**
     * Takes a FewsRegularTimeSeries object so it can get FEWS related data and then fills in with measurements from a
     * RegularTimeSeries object Used to translate results from existing models in to a FewsRegularTimeSeries object,
     * other data for which was filled up when data was received from FEWS
     * 
     * @param ts FewsRegularTimeSeries containing metadata
     * @param rts RegularTimeSeries holding actual data
     */
    public FewsRegularTimeSeries(final FewsRegularTimeSeries ts, final RegularTimeSeries rts)
    {
        super(ts.getStartTime(), ts.getEndTime(), ts.getIntervalInHours(), ts.getMeasuringUnit());
        _longName = ts.getLongName();
        _sourceOrganization = ts.getSourceOrganization();
        _sourceSystem = ts.getSourceSystem();
        _fileDescription = ts.getFileDescription();
        _creationDateTime = ts._creationDateTime.clone();
        _type = ts.getType();
        setLocationId(ts.getLocationId());
        //setQualifierId(ts.getQualifierId());
        setQualifierIds(ts.getQualifierIds());
        setEnsembleId(ts.getEnsembleId());
        setEnsembleMemberIndex(ts.getEnsembleMemberIndex());
        _timeSeriesType = rts.getTimeSeriesType();
        setName(rts.getName());
        setX(ts.getX());
        setY(ts.getY());
        setZ(ts.getZ());
        setLatitude(ts.getLatitude());
        setLongitude(ts.getLongitude());
        // If we use previous code, ensmdodels project is affected and does not run.
        for(int i = 0; i < rts.getMeasurementCount(); i++)
        {
            setMeasurementByIndex(rts.getMeasurementByIndex(i), i);
        }
    }

    /**
     * If has not been explicitly assigned, return the default value: the current date in GMT
     */
    public String getCreationDate()
    {
        return _creationDateTime.getDateString();
    }

    /**
     * If has not been explicitly assigned, return the default value: the current time in GMT
     */
    public String getCreationTime()
    {
        return _creationDateTime.getTimeString();
    }

    public void setCreationDateTime(final DateTime simpleDateTime)
    {
        _creationDateTime = simpleDateTime.clone();
    }

    public String getFileDescription()
    {
        return _fileDescription;
    }

    public void setFileDescription(final String description)
    {
        _fileDescription = description;
    }

    public String getLongName()
    {
        return _longName.trim();
    }

    public void setLongName(final String name)
    {
        _longName = name;
    }

    public String getSourceOrganization()
    {
        return _sourceOrganization.trim();
    }

    public void setSourceOrganization(final String organization)
    {
        _sourceOrganization = organization;
    }

    public String getSourceSystem()
    {
        return _sourceSystem.trim();
    }

    public void setSourceSystem(final String system)
    {
        _sourceSystem = system;
    }

    public FEWS_RTS_TYPE getType()
    {
        return _type;
    }

    /**
     * @param type accumulative/instantaneous
     */
    public void setType(final FEWS_RTS_TYPE type)
    {
        this._type = type;
    }

    @Override
    public String toString()
    {
        final StringBuffer buffer = new StringBuffer();

        buffer.append(" Long Name: ").append(getLongName()).append("\n");
        buffer.append(" SourceOrganization: ").append(getSourceOrganization()).append("\n");
        buffer.append(" SourceSystem: ").append(getSourceSystem()).append("\n");
        buffer.append(" Type: ").append(getType()).append("\n");
        buffer.append(" FileDescription: ").append(getFileDescription()).append("\n");
        buffer.append(" CreationDate: ").append(_creationDateTime.getDateString()).append("\n");
        buffer.append(" CreationTime: ").append(_creationDateTime.getTimeString()).append("\n");

        if(getThresholdsList() != null && getThresholdsList().size() > 0)
            for(final TimeSeriesThreshold thresholds: getThresholdsList())
            {
                buffer.append(" Threshold  Id: ").append(thresholds.getId());
                buffer.append(", Name: ").append(thresholds.getName());
                buffer.append(", Value: ").append(thresholds.getValue(getMeasuringUnit())).append("\n");
            }

        buffer.append(super.toString());

        return buffer.toString();
    }

    /**
     * @param timeSeriesType ACCM, MEAN, or INST the following method takes the ohd time code and returns the FEWS time
     *            code (instantaneous or accumulated) FEWS currently does not have a MEAN type, so we use accumulated in
     *            those cases
     */
    public static FEWS_RTS_TYPE getFewsTimeCode(final String timeSeriesType)
    {
        if(timeSeriesType.equals(NwsrfsDataTypeMappingReader.INSTANTANEOUS_TIME_CODE.trim()))
        {
            return FEWS_RTS_TYPE.INSTANTANEOUS;
        }
        else if(timeSeriesType.equals(NwsrfsDataTypeMappingReader.ACCUMULATED_TIME_CODE.trim()))
        {
            return FEWS_RTS_TYPE.ACCUMULATIVE;
        }
        else if(timeSeriesType.equals(NwsrfsDataTypeMappingReader.MEAN_TIME_CODE.trim()))
        {
            return FEWS_RTS_TYPE.MEAN;
        }

        return FEWS_RTS_TYPE.UNKNOWN;

    }

    public List<TimeSeriesThreshold> getThresholdsList()
    {
        return this._thresholdsList;
    }

    public void setTimeSeriesThresholdsList(final List<TimeSeriesThreshold> thresholdsList)
    {
        this._thresholdsList = thresholdsList;
    }

    /**
     * Return the TimeSeriesThreshold object based on the id. If there are none or more than one with the same id or
     * none, throws an Exception.
     */
    public TimeSeriesThreshold getThreshold(final String id) throws Exception
    {
        if(getThresholdsList() == null)
        {
            throw new Exception(id + " thresholds element was not found in " + getTimeSeriesType()
                + " TimeSeries header.");
        }

        TimeSeriesThreshold aThreshold = null;

        final StringBuffer strBufForIds = new StringBuffer();

        int count = 0;
        for(final TimeSeriesThreshold threshold: getThresholdsList())
        {
            if(threshold.getId() != null)
            {
                strBufForIds.append(threshold.getId()).append(" ");

                if(threshold.getId().equalsIgnoreCase((id)))
                {
                    aThreshold = threshold;

                    count++;
                }
            }
        }//close for loop

        if(count == 0)
        {//didn't find it
            throw new Exception(id + " was not found in " + getTimeSeriesType()
                + " TimeSeries header thresholds element[" + strBufForIds.toString().trim() + "].");
        }
        else if(count > 1)
        {//found more than 1 times
            throw new Exception(id + " was found more than one time in " + getTimeSeriesType()
                + " TimeSeries header thresholds element[" + strBufForIds.toString().trim() + "].");
        }

        return aThreshold;
    }

    /**
     * X coordinate of station
     * 
     * @return the x
     */
    public Double getX()
    {
        return _x;
    }

    /**
     * X coordinate of station
     * 
     * @param x the x to set
     */
    public void setX(final Double x)
    {
        this._x = x;
    }

    /**
     * Y coordinate of station
     * 
     * @return the y
     */
    public Double getY()
    {
        return _y;
    }

    /**
     * Y coordinate of station
     * 
     * @param y the y to set
     */
    public void setY(final Double y)
    {
        this._y = y;
    }

    /**
     * Z coordinate of station
     * 
     * @return the z
     */
    public Double getZ()
    {
        return _z;
    }

    /**
     * Z coordinate of station
     * 
     * @param z the z to set
     */
    public void setZ(final Double z)
    {
        this._z = z;
    }

    /**
     * Latitude of station
     * 
     * @return the latitude
     */
    public Float getLatitude()
    {
        return _latitude;
    }

    /**
     * Latitude of station
     * 
     * @param latitude the latitude to set
     */
    public void setLatitude(final Float latitude)
    {
        this._latitude = latitude;
    }

    /**
     * Longitude of station
     * 
     * @return the longitude
     */
    public Float getLongitude()
    {
        return _longitude;
    }

    /**
     * Longitude of station
     * 
     * @param longitude the longitude to set
     */
    public void setLongitude(final Float longitude)
    {
        this._longitude = longitude;
    }

    /**
     * @return the version of the xml file
     */
    public String getVersion()
    {
        return _version;
    }

    /**
     * @param version the version of the xml file to set
     */
    public void setVersion(final String version)
    {
        this._version = version;
    }

}
