package ohd.hseb.util.fews;

import java.io.BufferedInputStream;
import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.util.ArrayList;
import java.util.List;

import javax.xml.stream.XMLInputFactory;
import javax.xml.stream.XMLStreamReader;
import javax.xml.transform.stream.StreamSource;

import ohd.hseb.measurement.RegularTimeSeries;
import ohd.hseb.time.DateTime;
import ohd.hseb.util.Logger;
import ohd.hseb.util.xml.OHDXmlUtils;

import org.xml.sax.SAXException;

/**
 * Sets up an XML parser to parse the run info, parameters, states and time series files
 */
final public class FewsXMLParser
{
    //the three handlers have getters
    private StateMetaFileHandler _stateMetaFileHandler = null;
    private TimeSeriesHandler _timeSeriesHandler = null;
    private RunInfoHandler _runInfoHandler = null;
    private ParameterHandler _parameterHandler = null;
    private final Logger _logger;

    /**
     * FewsAdapter uses this constructor
     * 
     * @param logger Diagnostics object to log errors in
     * @throws Exception
     */
    public FewsXMLParser(final Logger logger) throws Exception
    {
        _logger = logger;
    }

    /**
     * First, validate the xml file against the schema; then parse the xml file and the resulted object is inside
     * fileHandler.
     */
    private void validateAndParseXmlFile(final FewsXmlHandler fileHandler,
                                         final String xmlFileName,
                                         final String xmlSchemaFileName) throws Exception
    {
//        final String header = "FewsXMLParser.validateAndParseXmlFile() " + Thread.currentThread().getId() + ": ";
//        System.out.println(header + " xmlFileName = " + xmlFileName);

        FewsXmlValidation.validateXmlFileAgainsXMLSchema(xmlFileName, xmlSchemaFileName, _logger);

        final File xmlFile = new File(xmlFileName);

        BufferedInputStream bis = null;
        XMLStreamReader xsr = null;

        try
        {
            bis = new BufferedInputStream(new FileInputStream(xmlFile));
            xsr = OHDXmlUtils.getStreamReader(bis, xmlFileName);

            fileHandler.find(xsr); //depending on what fileHandler is, calling various *Handler's find(xsr) method

            _logger.log(Logger.DEBUG, "Parsed " + xmlFileName);
        }
        finally
        {//close inputStream and reader no matter what happens

            if(bis != null)
            {
                bis.close();
            }

            if(xsr != null)
            {
                xsr.close();
            }
        }

    }

    /**
     * Parse run_info.xml and gets all the information into the object runInfo
     */
    public void parseRunInfoFile(final String runInfoFile, final RunInfo runInfo) throws Exception
    {
        _runInfoHandler = new RunInfoHandler(runInfo, _logger);
        this.validateAndParseXmlFile(_runInfoHandler, runInfoFile, OHDFewsAdapterConstants.PI_RUN_XML_SCHEMA_TAG);
    }

    /**
     * Parse a list of time series xml files and place the Regular Time Series objects (conforming to pi_timeseries.xsd)
     * in tsList.
     * 
     * @param initStateTime - used to determine which values in TS to check for missing data. If it is 0, don't use it.
     * @throws Exception
     */
    public void parseTimeSeries(final List<String> tsFileNames,
                                final List<RegularTimeSeries> tsList,
                                final boolean areMissingValuesAlwaysAllowed,
                                final long initStateTime) throws Exception
    {
        for(final String tsFileName: tsFileNames)
        {
            _timeSeriesHandler = new TimeSeriesHandler(tsList, _logger);

            _timeSeriesHandler.setAreMissingValuesAlwaysAllowed(areMissingValuesAlwaysAllowed);

            if(initStateTime != 0)
            {
                _timeSeriesHandler.setInitialStateTime(initStateTime);
            }
            _timeSeriesHandler.initiateForBinaryFileReadingIfAvailable(new File(tsFileName));

            this.validateAndParseXmlFile(_timeSeriesHandler,
                                         tsFileName,
                                         OHDFewsAdapterConstants.PI_TIMESERIES_XML_SCHEMA_TAG);
        }

    } //close method

    /**
     * Parse a string content of a PiTimeseries XML file and place the Regular Time Series objects (conforming to
     * pi_timeseries.xsd) in tsList.
     * 
     * @param tsStringXML - a string PiTimeseries XML file that retrieved from Fews database
     * @param initStateTime -used to determine which values in TS to check for missing data. If it is 0, don't use it.
     * @throws Exception
     */
    public void parseTimeSeriesFromStringXML(final String tsStringXML,
                                             final List<RegularTimeSeries> tsList,
                                             final boolean areMissingValuesAlwaysAllowed,
                                             final long initStateTime) throws Exception
    {
        _timeSeriesHandler = new TimeSeriesHandler(tsList, _logger);

        _timeSeriesHandler.setAreMissingValuesAlwaysAllowed(areMissingValuesAlwaysAllowed);

        if(initStateTime != 0)
        {
            _timeSeriesHandler.setInitialStateTime(initStateTime);
        }

        parseXMLString(tsStringXML, "TIMESERIES");
    }

    /**
     * Parameters parsed and placed in the object parameters. The same method can be used to parse previous parameter
     * file and place the information in getPrevParameters().
     * 
     * @param paramFileName Parameters XML file (conforming to pi_parameters.xsd) to be parsed
     * @throws Exception
     */
    public void parseParameters(final String paramFileName, final Parameters parameters) throws Exception
    {

        final ParameterHandler parameterHandler = new ParameterHandler(parameters, _logger);

        this.validateAndParseXmlFile(parameterHandler,
                                     paramFileName,
                                     OHDFewsAdapterConstants.PI_MODELPARAMETERS_XML_SCHEMA_TAG);

    }

    /**
     * Parsed a string content of parameters XML file and placed in the objects parameters
     * 
     * @param paramStringXML - a string XML file which retrieved from FEWS database
     * @throws Exception
     */
    public void parseParametersFromStringXML(final String paramStringXML, final Parameters parameters) throws Exception
    {
        _parameterHandler = new ParameterHandler(parameters, _logger);

        parseXMLString(paramStringXML, "PARAMS");
    }

    /**
     * Parse string content of a PiTimeseries or parameter XML file
     * 
     * @param xmlString - string content PiTimeseries or parameters XML file
     * @param whichHandler - PARAMS or TIMESERIES
     * @throws SAXException
     * @throws Exception
     */
    public void parseXMLString(final String xmlString, final String whichHandler) throws SAXException, Exception
    {

        BufferedInputStream bis = null;

        XMLStreamReader xsr = null;

        _logger.log(Logger.DEBUG, "parse XMLstring for " + whichHandler);

        try
        {
            bis = new BufferedInputStream(new ByteArrayInputStream(xmlString.getBytes()));

            final StreamSource source = new StreamSource(bis);

            source.setSystemId("stringXML");

            final XMLInputFactory factory = XMLInputFactory.newInstance();

            xsr = factory.createXMLStreamReader(source);

            if(whichHandler.equalsIgnoreCase("PARAMS"))
            {
                _parameterHandler.find(xsr); //depending on what parameterHandler is, calling various *Handler's find(xsr) method
            }
            else if(whichHandler.equalsIgnoreCase("TIMESERIES"))
            {
                _timeSeriesHandler.find(xsr);
            }

//            _logger.log(Logger.DEBUG, "Parsed XMLString: " + xmlString );
        }
        finally
        {//close inputStream and reader no matter what happens
            if(bis != null)
            {
                bis.close();
            }

            if(xsr != null)
            {
                xsr.close();
            }
        }
    }

    /**
     * Parse state meta File by using {@link StateMetaFileHandler}. During parsing, after obtaining the input state file
     * name, also load the state, by calling {@link State#loadState(String, Logger)}.
     * 
     * @param statesMetaFileName - Name of states XML to be parsed (conforming to pi_state.xsd)
     * @throws Exception
     */
    public void parseStatesMetaFileAndLoadState(final String statesMetaFileName, final IState state) throws Exception
    {
        _stateMetaFileHandler = new StateMetaFileHandler(state, _logger);
        this.validateAndParseXmlFile(_stateMetaFileHandler,
                                     statesMetaFileName,
                                     OHDFewsAdapterConstants.PI_STATE_XML_SCHEMA_TAG);
    }

    /**
     * Only meaningful after {@link #parseTimeSeries(List, List, boolean, long)} has been called. Otherwise returns
     * null.
     */
    public TimeSeriesHandler getTimeSeriesHandler()
    {
        return this._timeSeriesHandler;
    }

    /**
     * Only meaningful after {@link #parseStatesMetaFileAndLoadState(String, IState)} has been called. Otherwise returns
     * null.
     */
    public StateMetaFileHandler getStateMetaFileHandler()
    {
        return this._stateMetaFileHandler;
    }

    /**
     * Only meaningful after {@link #parseRunInfoFile(String, RunInfo)} has been called. Otherwise returns null.
     */
    public RunInfoHandler getRunInfoHandler()
    {
        return this._runInfoHandler;
    }

    /**
     * Science group Satish needs a Java program to parse/convert PI-XML format files containing ensemble time series
     * data to text files with data in column format. Each column is an ensemble member data. So I wrote this main
     * method and an executable JAR file("pixml_to_text_converter.jar") can be generated based on this application (see
     * the target "create_pixml_converter_jar" in ohdcommonchps/build.xml).<br>
     * For using "pixml_to_text_converter.jar":<br>
     * 1)when parsing a single PIXML file:<br>
     * java -jar pixml_to_text_converter.jar /path/to/pixml/file/pixml_to_be_parsed.xml<br>
     * Then the text file "/path/to/pixml/file/pixml_to_be_parsed.txt" will be created. 2)when parsing ALL PIXML files
     * inside a directory:<br>
     * java -jar pixml_to_text_converter.jar /path/to/pixml/file/*<br>
     * Please note: i)'*' is needed. If the argument is just the directory name, it won't work; ii)all the files inside
     * the directory must be PIXML files, because ALL the files inside this directory will be parsed. For example, if a
     * text file exists in the directory, the parsing will crash.
     * 
     * @param args - only one argument: either a specific PIXML file name or "/directory/*" and the directory must only
     *            have PIXML files in it.
     * @throws Exception
     */
    public static void main(final String[] args) throws Exception
    {
        //when no arguments, print out some help information
        if(args.length == 0)
        {
            System.out.println("**********************************************************************************************************");
            System.out.println("This Java program converts PI-XML format files containing ensemble time series data to text files with "
                + "data in column format. Each column is an ensemble member data. The program only takes one argument, "
                + "either a specific PIXML file name or '/path/to/directory/*'");
            System.out.println("1)when parsing a single PIXML file:");
            System.out.println("java -jar pixml_to_text_converter.jar /path/to/pixml/file/pixml_to_be_parsed.xml");
            System.out.println("Then the text file /path/to/pixml/file/pixml_to_be_parsed.txt will be created.\n");
            System.out.println("2)when parsing ALL PIXML files inside a directory:");
            System.out.println("java -jar pixml_to_text_converter.jar /path/to/pixml/file/*");
            System.out.println("\nPlease note: i)'*' is needed. If the argument is just a directory name, it won't work; ii)all the files "
                + "inside the directory must be PIXML files, because ALL the files inside this directory will be parsed. For example, if "
                + "a text file exists in the directory, the parsing will crash.");
            System.out.println("**********************************************************************************************************");

            return;
        }

        final FewsXMLParser xmlParser = new FewsXMLParser(OHDConstants.DUMMY_LOG);

        //all the input files must be pixml format files
        for(int i = 0; i < args.length; i++)
        {

            final String inputFileName = args[i];

            final List<String> tsFileNames = new ArrayList<String>();

            tsFileNames.add(inputFileName);//the list of file names only contain one file

            final int indexOfDot = inputFileName.lastIndexOf(".");
            final String outputFileName = inputFileName.substring(0, indexOfDot) + ".txt";

            System.out.println("I am going to parse " + inputFileName);

            final File outputFile = new File(outputFileName);
            if(outputFile.exists())
            {//delete existing file

                outputFile.delete();
            }

            outputFile.createNewFile(); //creates a brand new file

            final FileOutputStream fop = new FileOutputStream(outputFile);

            final List<RegularTimeSeries> tsList = new ArrayList<RegularTimeSeries>();

            xmlParser.parseTimeSeries(tsFileNames, tsList, false, 0);

            final long startTimeLong = tsList.get(0).getStartTime();
            final long endTimeLong = tsList.get(0).getEndTime();
            final int intervalInHours = tsList.get(0).getIntervalInHours();

            StringBuffer timeStringBuffer;

            for(final DateTime dateTime = new DateTime(startTimeLong); dateTime.getTimeInMillis() <= endTimeLong; dateTime.addHours(intervalInHours))
            {//one loop, one line

                String monthStr;
                if(dateTime.month() < 10)
                {
                    monthStr = "0" + String.valueOf(dateTime.month());
                }
                else
                {
                    monthStr = String.valueOf(dateTime.month());
                }

                final String dayStr;
                if(dateTime.dayOfMonth() < 10)
                {
                    dayStr = "0" + String.valueOf(dateTime.dayOfMonth());
                }
                else
                {
                    dayStr = String.valueOf(dateTime.dayOfMonth());
                }

                final String hourStr;
                if(dateTime.hour() < 10)
                {
                    hourStr = "0" + String.valueOf(dateTime.hour());
                }
                else
                {
                    hourStr = String.valueOf(dateTime.hour());
                }

                timeStringBuffer = new StringBuffer();
                timeStringBuffer.append(dateTime.year()).append(monthStr).append(dayStr).append(hourStr);

                final StringBuffer wholeLineBuffer = new StringBuffer();
                wholeLineBuffer.append(timeStringBuffer.toString());

                for(final RegularTimeSeries ts: tsList)
                {
                    final double mValue = ts.getMeasurementValueByTime(dateTime.getTimeInMillis(),
                                                                       ts.getMeasuringUnit());

                    String mValueStr = String.valueOf(mValue);

                    final StringBuffer mValueStringBuffer = new StringBuffer();

                    //each number has fixed width(to have nice columns in result text file): if short, pad with extra " ";if too long, take the substring
                    if(mValueStr.length() < 10)
                    {
                        final int extraSpaceNum = 10 - mValueStr.length();

                        mValueStringBuffer.append(mValueStr);

                        //pad with " "
                        for(int j = 0; j < extraSpaceNum; j++)
                        {
                            mValueStringBuffer.append(' ');
                        }

                        mValueStr = mValueStringBuffer.toString();
                    }
                    else
                    {//too long, take substring

                        mValueStr = mValueStr.substring(0, 10);
                    }

                    wholeLineBuffer.append("   " + mValueStr);
                }

//                System.out.println(wholeLineBuffer.toString());

                final byte[] lineInBytes = (wholeLineBuffer.toString() + "\n").getBytes();
                fop.write(lineInBytes);
                fop.flush();

            }//close for loop

            fop.close();

        }//close for loop

    }//close main method
}
