package ohd.hseb.util.fews;

import java.io.BufferedReader;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.util.HashMap;
import java.util.Map;
import java.util.StringTokenizer;

import ohd.hseb.measurement.MeasuringUnit;
import ohd.hseb.measurement.MeasuringUnitType;
import ohd.hseb.util.Logger;

public class NwsrfsDataTypeMappingReader
{

    private static final int MAX_LENGTH_OTHER_FIELDS = 8;
    private static final int MAX_DATATYPE_LENGTH = 12;
    private static final int NUM_FIELDS_IN_MAPPING_FILE = 6;
    public static Map<String, NwsrfsDataTypeMapping> _nwsrfsDataTypeMap = null;
    public static Logger _logger = null;
    public static final String INSTANTANEOUS_TIME_CODE = "INST";
    public static final String ACCUMULATED_TIME_CODE = "ACCM";
    public static final String MEAN_TIME_CODE = "MEAN";

    static String str[];

    /**
     * This element initializes the measuring units that models do their work in per datatype The logger set in the
     * OHDFewsadapter is passed in for logging
     */
    public synchronized static void initializeNwsrfsDataTypeMap(final Logger logger) throws Exception
    {

        _logger = logger;

        if(_nwsrfsDataTypeMap == null)
        {
            _nwsrfsDataTypeMap = new HashMap<String, NwsrfsDataTypeMapping>();

            final String dummyFile = "blank";
            final NwsrfsDataFileExtractor extractor = new NwsrfsDataFileExtractor(dummyFile);
            final InputStream input = extractor.getHandleToDataTypeMappingFile();

            final InputStreamReader isr = new InputStreamReader(input);
            final BufferedReader reader = new BufferedReader(isr);

            String line;
            while((line = reader.readLine()) != null)
            {
                if(parseLine(line) == null)
                {
                    continue;
                }

                final NwsrfsDataTypeMapping mapping = new NwsrfsDataTypeMapping();
                mapping.setDataType(str[0]);
                mapping.setNwsrfsDimension(str[1]);
                mapping.setNwsrfsUnit(str[2]);
                mapping.setIsMissingAllowed(str[3]);
                mapping.setTimeCode(str[4]);
                mapping.setFewsUnit(str[5]);

                //the following code verifies whether the dimension in the
                //NWSRFS_DATATYPE_MAPPING_FILE is correct/pre-defined
                final String nwsrfsUnitString = str[2];
                isUnitRegcognized(nwsrfsUnitString, _logger);
                isDimRecognized(nwsrfsUnitString, str[1], _logger);

                if(str[0] != null)
                {
                    _nwsrfsDataTypeMap.put(str[0], mapping);
                }
            }
            System.out.flush();
            reader.close();

            if(_nwsrfsDataTypeMap.size() == 0)
            {
                _logger.log(Logger.FATAL,
                            "The Data Type Mapping file has no uncommented lines i.e. there is no data type mapping for time series types");
                throw new Exception();
            }
        }
    }

    public static String parseLine(final String line) throws Exception
    {
        String lin = null;
        str = new String[NUM_FIELDS_IN_MAPPING_FILE];//6 columns in line
        int i;

        lin = line.trim();
        final StringTokenizer st = new StringTokenizer(lin);

        if(lin.charAt(0) == '#')
        {
            return null;
        }
        for(i = 0; i < NUM_FIELDS_IN_MAPPING_FILE; i++)
        {
            str[i] = null;
        }
        for(i = 0; i < NUM_FIELDS_IN_MAPPING_FILE; i++)
        {
            if(st.hasMoreTokens())
            {
                str[i] = st.nextToken();
            }
        }

        if(i != NUM_FIELDS_IN_MAPPING_FILE)
        {
            _logger.log(Logger.FATAL, "Incorrect number of fields in Data Type Mapping file");
            throw new Exception();
        }

        checkStringLength(str);

        return lin;
    }

    private static void checkStringLength(final String[] str) throws Exception
    {
        for(int i = 0; i < str.length; i++)
        {
            final String thisString = str[i];
            if(i == 0 && thisString.length() > MAX_DATATYPE_LENGTH)
            {
                _logger.log(Logger.FATAL, "data type name " + thisString + " is longer than " + MAX_DATATYPE_LENGTH
                    + " characters long");

                throw new Exception();
            }
            else if(i > 0 && thisString.length() > MAX_LENGTH_OTHER_FIELDS)
            {

                _logger.log(Logger.FATAL, " the field " + thisString + " is longer than " + MAX_LENGTH_OTHER_FIELDS
                    + " characters long");
                throw new Exception();
            }
        }

    }

    public static boolean isDimRecognized(final String nwsrfsUnitStr, final String nwsrfsDimStr, final Logger logger) throws Exception
    {
        final MeasuringUnit mUnit = MeasuringUnit.getMeasuringUnit(nwsrfsUnitStr);
        final MeasuringUnitType nwsrfsDim = mUnit.getType();

        if(!nwsrfsDim.toString().equalsIgnoreCase(nwsrfsDimStr))
        {
            if(logger != null)
                logger.log(Logger.FATAL, "Dimension for time series type " + nwsrfsDimStr
                    + " is incorrect in the NWSRFS_DATATYPE_MAPPING_FILE");
            throw new Exception();
        }

        return true;
    }

    public static boolean isUnitRegcognized(final String nwsrfsUnitString, final Logger logger) throws Exception
    {
        final MeasuringUnit mUnit = MeasuringUnit.getMeasuringUnit(nwsrfsUnitString);

        if(mUnit == null)
        {

            logger.log(Logger.FATAL, "Unit string " + nwsrfsUnitString
                + " is incorrect in the NWSRFS_DATATYPE_MAPPING_FILE");
            throw new Exception();
        }

        return true;

    }

    private static NwsrfsDataTypeMapping getDataTypeMappingForOneDataType(final String dataType) throws Exception
    {
        NwsrfsDataTypeMapping row = null;

        if(_nwsrfsDataTypeMap != null)
        {
            row = _nwsrfsDataTypeMap.get(dataType);

            if(row == null)
            {
                _logger.log(Logger.FATAL, "Data type: " + dataType
                    + " does not exist in the NWSRFS data type mapping file");
                throw new Exception("Data type: " + dataType + " does not exist in the NWSRFS data type mapping file");
            }
            return row;
        }

        return null;
    }

    //The following 6 methods allow a user to derive the fields from
    //the NWSRFS_DATATYPE_MAPPING_FILE, given the time series type
    //which is stored in a map
    //---------------------------------------------------------------
    public static String getDataType(final String tsType, final Logger logger) throws Exception
    {

        initializeNwsrfsDataTypeMap(logger);

        final NwsrfsDataTypeMapping oneDataTypeMapping = NwsrfsDataTypeMappingReader.getDataTypeMappingForOneDataType(tsType);

        final String dType = oneDataTypeMapping.getDataType();

        if(dType == null && logger != null)
        {
            logger.log(Logger.FATAL, "Could not find time series type: " + tsType);
            throw new Exception();
        }

        return dType;
    }

    public static String getNwsrfsDim(final String tsType, final Logger logger) throws Exception
    {

        initializeNwsrfsDataTypeMap(logger);

        final NwsrfsDataTypeMapping oneDataTypeMapping = NwsrfsDataTypeMappingReader.getDataTypeMappingForOneDataType(tsType);

        final String dim = oneDataTypeMapping.getNwsrfsDimension();

        if(dim == null && logger != null)
        {
            logger.log(Logger.FATAL, "Could not find dimension for time series type: " + tsType);
            throw new Exception();
        }

        return dim;
    }

    public static String getNwsrfsUnit(final String tsType, final Logger logger) throws Exception
    {

        initializeNwsrfsDataTypeMap(logger);

        final NwsrfsDataTypeMapping oneDataTypeMapping = NwsrfsDataTypeMappingReader.getDataTypeMappingForOneDataType(tsType);

        final String unit = oneDataTypeMapping.getNwsrfsUnit();

        if(unit == null && logger != null)
        {
            logger.log(Logger.FATAL, "Could not find nwsrfs unit for time series type: " + tsType);
            throw new Exception();
        }

        return unit;
    }

    public static String getFewsUnit(final String tsType, final Logger logger) throws Exception
    {

        initializeNwsrfsDataTypeMap(logger);

        final NwsrfsDataTypeMapping oneDataTypeMapping = NwsrfsDataTypeMappingReader.getDataTypeMappingForOneDataType(tsType);

        final String unit = oneDataTypeMapping.getFewsUnit();

        if(unit == null && logger != null)
        {
            logger.log(Logger.FATAL, "Could not find fews unit for time series type: " + tsType);
            throw new Exception();
        }

        return unit;
    }

    public static String getTimeCode(final String tsType, final Logger logger) throws Exception
    {
        initializeNwsrfsDataTypeMap(logger);

        final NwsrfsDataTypeMapping oneDataTypeMapping = NwsrfsDataTypeMappingReader.getDataTypeMappingForOneDataType(tsType);

        final String timeCode = oneDataTypeMapping.getTimeCode();

        if(timeCode == null && logger != null)
        {
            logger.log(Logger.FATAL, "Could not find time code for time series type: " + tsType);
            throw new Exception();
        }

        return timeCode;
    }

    public static boolean areMissingValuesAllowed(final String tsType, final Logger logger) throws Exception
    {

        //This method returns true if missing values are allowed and false if not.
        NwsrfsDataTypeMapping record;

        if(_nwsrfsDataTypeMap == null)
            initializeNwsrfsDataTypeMap(logger);

        for(int i = 0; i < _nwsrfsDataTypeMap.size(); i++)
        {
            record = NwsrfsDataTypeMappingReader.getDataTypeMappingForOneDataType(tsType);

            if(record.getDataType().equals(tsType))
            {
                if(record.getIsMissingAllowed().equalsIgnoreCase(OHDConstants.YES_STRING))
                {
                    return true;
                }
                else if(record.getIsMissingAllowed().equalsIgnoreCase(OHDConstants.NO_STRING))
                {
                    return false;
                }
                else
                {

                    if(logger != null)
                    {
                        logger.log(Logger.FATAL,
                                   ("missingAllowed is neither 'yes' nor 'no' for time series type " + tsType));
                        throw new Exception();
                    }

                    return false;
                }
            }
        }
        return false;
    }

    public static Map<String, NwsrfsDataTypeMapping> getDataTypeMap()
    {
        return _nwsrfsDataTypeMap;
    }

}
