package ohd.hseb.util.fews;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.PrintWriter;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.GregorianCalendar;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.TimeZone;
import java.util.Vector;

import javax.management.timer.Timer;
import javax.xml.XMLConstants;
import javax.xml.parsers.SAXParser;
import javax.xml.parsers.SAXParserFactory;
import javax.xml.transform.stream.StreamSource;
import javax.xml.validation.Schema;
import javax.xml.validation.SchemaFactory;

import ohd.hseb.measurement.RegularTimeSeries;
import ohd.hseb.util.CommandAdapter;
import ohd.hseb.util.MathHelper;

import org.xml.sax.helpers.DefaultHandler;

final public class OHDUtilities
{
    private OHDUtilities()
    {
        /*
         * This class only contains static constants, so an instance of this class should never be created. If not
         * providing this private constructor, users still can create objects of this class by using the invisible
         * default constructor provided by Java, which is wrong. Now, with this private constructor, the default
         * constructor by Java is destroyed. Now, nobdy from outside of this class can create the object of this class
         */
    }

    /**
     * Determine whether this application is running under Windows or some other platform by examining the "os.name"
     * property.
     * 
     * @return true if this application is running under a Windows OS
     */
    public static boolean isWindowsPlatform()
    {
        final String os = System.getProperty("os.name");
        return os != null && os.startsWith("Windows");
    }

    /**
     * Replace the forward slash '/'(Linux convention file separator) in the path with the back slash '\'(PC convention
     * file separator)
     */
    public static String getWindowsPath(final String pathOnLinuxConvention)
    {
        final StringBuilder strBuilder = new StringBuilder();

        final char slash = '/';
        final char backSlash = '\\';
        char currChar;

        for(int i = 0; i < pathOnLinuxConvention.length(); i++)
        {
            currChar = pathOnLinuxConvention.charAt(i);

            if(currChar == slash)
            {
                strBuilder.append(backSlash);
            }
            else
            {
                strBuilder.append(currChar);
            }
        }

        return strBuilder.toString();
    }

    /**
     * A helper method that convert a one-dimension array into a String line, ready for being inserted into xml file
     */
    public static String getStringLineFromDoubleArray(final double[] dArray)
    {
        final StringBuilder strBuilder = new StringBuilder();

        for(final double d: dArray)
        {
            strBuilder.append(d).append(" ");
        }

        return strBuilder.toString().trim();
    }

    /**
     * A helper method that convert a String line composed of double values separated by at least one space or tab into
     * a double array.
     * <p>
     * Note: assuming the parameter is a String composed of numeric value(s). If not, no exception is explicitly thrown
     * and I don't know where and how this error will be detected.
     */
    public static double[] getDoubleArrayFromString(final String str)
    {

        final String[] strArray = OHDConstants.DELIMITER_PATTERN.split(str.trim());
        final double[] doubleArray = new double[strArray.length];

        for(int i = 0; i < doubleArray.length; i++)
        {
            doubleArray[i] = Double.parseDouble(strArray[i]);
        }

        return doubleArray;

    }

    /**
     * Trim a double array to its right size
     * 
     * @param dArray
     * @param size
     * @return a double array with size of parameter
     */
    public static double[] getDoubleArrayRightSize(final double[] dArray, final int size)
    {
        final double[] newArray = new double[size];

        for(int i = 0; i < size; i++)
        {
            newArray[i] = dArray[i];
        }

        return newArray;
    }

    public static void printOutTraces()
    {

        try
        {
            throw new RuntimeException();
        }
        catch(final RuntimeException e)
        {

            System.out.println("Runtime exception traces:");
            e.printStackTrace();
        }
    }

    /**
     * check the two ints to see if numOne be equal to or multiple of numTwo. numOne should not be 0. If numOne is 0,
     * returns false.
     */
    public static boolean isEvenMultiple(final int numOne, final int numTwo)
    {

        if(numOne == 0)
        {
            return false;
        }

        if(numOne % numTwo == 0)
        {
            //good: intervalOne equals to intervalTwo OR is even multiple of intervalTwo
            return true;
        }

        //if reach here, must: numbOne != 0 && numOne is not multiple of numTwo
        return false;
    }

    /**
     * In Snow17 Fortran program, the double value precision is related to its value: the total digit number (sum of
     * digits before decimal and digits after decimal) is 7; so if the number is larger, the digits after the decimal
     * point is fewer.
     * 
     * @param val the val
     * @return the fortran precison
     */
    public static double getFortranPrecison(final double val)
    {

        if(Math.abs(val) < 1.0)
        {
            return MathHelper.roundToNDecimalPlaces(val, 8);
        }
        else if(Math.abs(val) < 10.0)
        {
            return MathHelper.roundToNDecimalPlaces(val, 7);
        }
        else if(Math.abs(val) < 100.0)
        {
            return MathHelper.roundToNDecimalPlaces(val, 7);
        }
        else if(Math.abs(val) < 1000.0)
        {
            return MathHelper.roundToNDecimalPlaces(val, 5);
        }
        else if(Math.abs(val) < 10000.0)
        {
            return MathHelper.roundToNDecimalPlaces(val, 4);
        }
        else
        { // unlikely case
            return val;
        }
    }

    /**
     * Retrieve the RTS from resultMap with time series type matching the parameter rtsType. If no such RTS in
     * resultMap, throws an Exception. Note: resultMap could have multiple RTS with the same parameterId. Returns the
     * first one. Notice the difference to {@link #removeRTSFromResultMap(Map, String)}.
     */
    public static RegularTimeSeries getRTSFromResultMap(final Map<String, RegularTimeSeries> resultMap,
                                                        final String rtsType) throws Exception
    {
        final Iterator<Map.Entry<String, RegularTimeSeries>> ite = resultMap.entrySet().iterator();

        while(ite.hasNext())
        {
            final Map.Entry<String, RegularTimeSeries> entry = ite.next();

            if(entry.getKey().matches(".*" + rtsType + ".*")) //".": any char, "*": zero or more of preceding thing
            {
                return entry.getValue();
            }
        }

        //if reach here, must be not found
        throw new Exception("Inside resultMap, no time series " + rtsType);

    }

    /**
     * Delete the RTS from resultMap with time series type matching the parameter rtsType and returns the deleted RTS.
     * If no such RTS in resultMap, throws an Exception. Note: resultMap could have multiple RTS with the same
     * parameterId. Returns the first one. Notice the difference to {@link #getRTSFromResultMap(Map, String)}.
     */
    public static RegularTimeSeries removeRTSFromResultMap(final Map<String, RegularTimeSeries> resultMap,
                                                           final String rtsType) throws Exception
    {
        final Iterator<Map.Entry<String, RegularTimeSeries>> ite = resultMap.entrySet().iterator();

        RegularTimeSeries rts;

        while(ite.hasNext())
        {
            final Map.Entry<String, RegularTimeSeries> entry = ite.next();

            if(entry.getKey().matches(".*" + rtsType + ".*")) //".": any char, "*": zero or more of preceding thing
            {
                rts = entry.getValue().clone();
                ite.remove();
                return rts;
            }
        }

        //if reach here, must be not found
        throw new Exception("Inside resultMap, no time series " + rtsType);

    }

    /**
     * Construct a string which is used as a composite key in resultMap. Right now, the policy for the composite key is:
     * locationId + TS_type + interval + qualifierId
     */
    
    public static String getCompositeKeyInResultMap(final RegularTimeSeries rts)
    {	String qualifierId = "";
        if(rts.getQualifierIds() != null && rts.getQualifierIds().size() > 0)
            qualifierId = rts.getQualifierIds().get(0);
   	
        return rts.getLocationId() + rts.getTimeSeriesType() + rts.getIntervalInHours() + qualifierId;
        //+ rts.getQualifierIds().toString();
    }

    /**
     * This method returns a TimeZone obj based on the time zone offset(double value) in fews.xml and states.xml
     * <timeZone>. For example, offset 5.5 will get TimeZone "GMT+05:30", offset -5.0 will return TZ "GMT-05:00", offset
     * 0.0 will return TZ "GMT+00:00".
     * <p>
     * See opposite method {@link #getTimeZoneRawOffsetInHours(TimeZone)}
     */
    public static TimeZone getTimeZoneFromOffSet(final double tzOffset)
    {
        final StringBuilder timeZoneStr = new StringBuilder(OHDConstants.TIMEZONE_GMT_STRING);

        final int hours = (int)tzOffset;
        if(hours > 0)
        {
            timeZoneStr.append("+").append(hours); //add "+" between GMT and number
        }
        else if(hours < 0)
        { // < 0
            timeZoneStr.append(hours); //already has "-" sign
        }

        //convert the decimal part into minutes if it exists
        if(tzOffset != hours)
        {
            final int minutes = (int)(Math.abs(Math.abs(tzOffset) - Math.abs(hours)) * OHDConstants.MINUTES_PER_HOUR);

            if(minutes < 10)
            {
                timeZoneStr.append(":0").append(minutes); //padding with "0" to make it two digits
            }
            else
            {
                timeZoneStr.append(":").append(minutes); //naturally two digits
            }
        }

        return TimeZone.getTimeZone(timeZoneStr.toString());
    }

    /**
     * Returns the number of hours(could be positive or negative) to add to GMT hour to get the local hour.
     * <p>
     * Note: the returned number is integer, even though time zone related hour shift could be partial, like 1.5 hours.
     * But this scenario may not exist in U.S. and we keep it simple.
     * <p>
     * See the opposite method {@link #getTimeZoneFromOffSet(double)}.
     */
    public static int getTimeZoneRawOffsetInHours(final TimeZone localTZ)
    {
        return (int)(localTZ.getRawOffset() / Timer.ONE_HOUR);
    }

    /**
     * Returns the local time current hour when the GMT time is 12 o'clock(12Z). For example, it returns 7 for EST/EDT:
     * it is 7 o'clock in EST at 12Z; it returns 4 for PST/PDT: it is 4 o'clock in PST at 12Z.
     * 
     * @param localTZ
     */
    public static int getLocalTimeAt12Z(final TimeZone localTZ)
    {
        return 12 + getTimeZoneRawOffsetInHours(localTZ);
    }

    public static void validateXmlFileAgainstSchema(final String xmlFileName, final String schemaFileName) throws Exception
    {
        final SchemaFactory factory = SchemaFactory.newInstance(XMLConstants.W3C_XML_SCHEMA_NS_URI);

        final Schema schema = factory.newSchema(new StreamSource(new File(schemaFileName)));

        final SAXParserFactory spf = SAXParserFactory.newInstance();

        spf.setSchema(schema);

        final SAXParser saxParser = spf.newSAXParser();

        saxParser.parse(new File(xmlFileName), new DefaultHandler());
    }

    /**
     * Returns a String like 1988-01-01, based on time zone
     * 
     * @param tz
     */
    @SuppressWarnings("boxing")
    public static String getCurrentDateString(final TimeZone tz)
    {
        final Calendar cal = new GregorianCalendar(tz);

        final String curDateStr = String.format("%4d-%02d-%02d",
                                                cal.get(Calendar.YEAR),
                                                cal.get(Calendar.MONTH) + 1,
                                                cal.get(Calendar.DAY_OF_MONTH));

        return curDateStr;
    }

    /**
     * Returns a String like 06:00:00, based on time zone
     * 
     * @param tz
     */
    @SuppressWarnings("boxing")
    public static String getCurrentTimeString(final TimeZone tz)
    {
        final Calendar cal = new GregorianCalendar(tz);

        final String curTimeStr = String.format("%02d:%02d:%02d",
                                                cal.get(Calendar.HOUR_OF_DAY),
                                                cal.get(Calendar.MINUTE),
                                                cal.get(Calendar.SECOND));

        return curTimeStr;
    }

    /**
     * Return the minimal int from the int array
     * 
     * @param intArr
     * @return
     */
    public static int getMinimalInt(final int[] intArr)
    {
        int min = intArr[0]; //assuming 1st ele. is the smallest

        for(int i = 0; i < intArr.length; i++)
        {
            if(min > intArr[i])
            {
                min = intArr[i];
            }
        }

        return min;
    }

    /**
     * Return the minimal long from the long array
     * 
     * @param intArr
     * @return
     */
    public static long getMinimalLong(final long[] intArr)
    {
        long min = intArr[0]; //assuming 1st ele. is the smallest

        for(int i = 0; i < intArr.length; i++)
        {
            if(min > intArr[i])
            {
                min = intArr[i];
            }
        }

        return min;
    }

    /**
     * Returns a String with fixed length. The int value will be aligned to the left. Padding on the right side with
     * " ".
     * 
     * @param iValue - the value to be inside the returned String, left aligned
     * @param strLength - the total length of the returned String
     */
    public static String getFormatString(final int iValue, final int strLength)
    {

        final StringBuilder result = new StringBuilder(String.valueOf(iValue));

        final int padNum = strLength - result.length();

        for(int i = 0; i < padNum; i++)
        {
            result.append(" ");
        }

        return result.toString();
    }

    /**
     * Returns a String with fixed length. The double value will be aligned to the left. Padding on the right side with
     * " ".
     * <p>
     * Note: a special case: if the parameter dValue is NaN(e.g. 0.0/0.0), the returned String is "0.0     "
     * 
     * @param dValue - the value to be inside the returned String, left aligned
     * @param decimalPlaces - the number of digits after the decimal point
     * @param strLength: the total length of the returned String
     */
    public static String getFormatString(final double dValue, final int decimalPlaces, final int strLength)
    {

        final StringBuilder result = new StringBuilder(String.valueOf(MathHelper.roundToNDecimalPlaces(dValue,
                                                                                                       decimalPlaces)));

        final int padNum = strLength - result.length();

        for(int i = 0; i < padNum; i++)
        {
            result.append(" ");
        }

        return result.toString();
    }

    /**
     * Returns a String with fixed length. Aligned to the left. Assuming the input String length is shorter than the
     * parameter strLength. Otherwise, the method will just return the passed in string, with no change.
     * 
     * @param str
     * @param strLength
     * @return
     */
    public static String getFormatString(final String str, final int strLength)
    {
        final StringBuilder result = new StringBuilder(str);
        result.append(getBlankString(strLength - str.length()));

        return result.toString();
    }

    /**
     * Returns a blank string with length of the parameter strLength.
     * 
     * @param strLength - the length of the returned blank string.
     */
    public static String getBlankString(final int strLength)
    {
        final StringBuilder result = new StringBuilder();

        for(int i = 0; i < strLength; i++)
        {
            result.append(" ");
        }

        return result.toString();

    }

    /**
     * Returns a String line containing the values inside the RTS, separated by a space. Unit information is ignored.
     */
    public static String getStringOfValuesFromRTS(final RegularTimeSeries rts)
    {
        final StringBuilder result = new StringBuilder();

        for(int i = 0; i < rts.getMeasurementCount(); i++)
        {
            //result.append(" " + getFortranPrecison(rts.getMeasurementByIndex(i).getValue()));
            result.append(" ").append(getFortranPrecison(rts.getMeasurementByIndex(i).getValue()));
        }

        return result.toString().trim(); //get rid of the leading " "
    }

    public static boolean isLeapYear(final int year)
    {
        boolean isLeap = false;

        if((year % 4) == 0)
        {
            if((year % 100) == 0)
            {
                if((year % 400) == 0) //2000 is a leap year, 1900 and 2100 are not
                {
                    isLeap = true;
                }
                else
                {
                    isLeap = false;
                }
            }
            else
            {
                isLeap = true;
            }
        }
        else
        {
            isLeap = false;
        }

        return isLeap;
    } //end isLeapYear()

    /**
     * Returns a static object of SimpleDateFormat in GMT timezone. Its pattern is the passed in argument, e.g.
     * "MM/dd/yyyy HHmm", "MM/dd/yyyy HH", or {@link OHDConstants#TIME_FORMAT_STR}
     */
    public static SimpleDateFormat getSdfGMT(final String dateTimePattern)
    {
        final SimpleDateFormat sdf = new SimpleDateFormat(dateTimePattern);
        sdf.setTimeZone(OHDConstants.GMT_TIMEZONE);

        return sdf;
    }

    /**
     * @return the current JAR location with path. Note: this is not the current working directory. For example: if you
     *         are at /fs/home/fews, running this command: java -jar ./installed/abc.jar, then this method returns the
     *         string "/fs/home/fews/installed/abc.jar". The current working directory is "/fs/home/fews/." in this
     *         example.
     */
    public static String getCurrJarWithPath()
    {
        final String currJarLocation = OHDUtilities.class.getProtectionDomain()
                                                         .getCodeSource()
                                                         .getLocation()
                                                         .toString();
        //return something like "file:/fs/pda/users/joe/installed/abc.jar"

        return currJarLocation.substring(5);//trimmed "file:" in the beginning, just return "/fs/pda/users/joe/installed/abc.jar"
    }

    /**
     * Please see the comments in {@link #getCurrJarWithPath()}. Following the example there, this method returns
     * "/fs/home/fews/installed/" (note: there is a slash at the end of the string)
     */
    public static String getCurrJarContainingDir()
    {
        final String jarWithPath = getCurrJarWithPath();

        final int index = jarWithPath.lastIndexOf(File.separator);

        return jarWithPath.substring(0, index + 1);
    }

    /**
     * If the file name has extension of ".xml", returns true; otherwise, returns false;
     */
    public static boolean isXmlFile(final String fileName)
    {
        return (fileName.trim().endsWith(".xml"));
    }

    /**
     * If the file name has extension of ".nc", returns true; otherwise, returns false;
     */
    public static boolean isNetcdfFile(final String fileName)
    {
        return (fileName.trim().endsWith(".nc"));
    }

    /**
     * A state consists of a number of locations. from which the adapter reads and writes the model state data. A such
     * reading is reading by the adapter and writing is writing by the adapter.
     * 
     * @author camachof
     */
    public static class StateLocation
    {

        protected String readLocation;
        protected String writeLocation;
        protected String type;

        /**
         * Gets the value of the readLocation property.
         * 
         * @return possible object is {@link String }
         */
        public String getReadLocation()
        {
            return readLocation;
        }

        /**
         * Sets the value of the readLocation property.
         * 
         * @param value allowed object is {@link String }
         */
        public void setReadLocation(final String value)
        {
            this.readLocation = value;
        }

        /**
         * Gets the value of the writeLocation property.
         * 
         * @return possible object is {@link String }
         */
        public String getWriteLocation()
        {
            return writeLocation;
        }

        /**
         * Sets the value of the writeLocation property.
         * 
         * @param value allowed object is {@link String }
         */
        public void setWriteLocation(final String value)
        {
            this.writeLocation = value;
        }

        /**
         * Gets the value of the type property.
         * 
         * @return possible object is {@link String }
         */
        public String getType()
        {
            return type;
        }

        /**
         * Sets the value of the type property.
         * 
         * @param value allowed object is {@link String }
         */
        public void setType(final String value)
        {
            this.type = value;
        }

        @Override
        public String toString()
        {
            final StringBuffer resultStr = new StringBuffer();
            if(this.getType() != null)
            {
                //resultStr.append("Type = " + this.getType()).append(OHDConstants.NEW_LINE);
                resultStr.append("Type = ").append(this.getType()).append(OHDConstants.NEW_LINE);
            }
            if(this.getReadLocation() != null)
            {
                //resultStr.append("Read Location = " + this.getReadLocation()).append(OHDConstants.NEW_LINE);
                resultStr.append("Read Location = ").append(this.getReadLocation()).append(OHDConstants.NEW_LINE);
            }
            if(this.getWriteLocation() != null)
            {
                //resultStr.append("Write Location = " + this.getWriteLocation()).append(OHDConstants.NEW_LINE);
                resultStr.append("Write Location = ").append(this.getWriteLocation()).append(OHDConstants.NEW_LINE);
            }
            return resultStr.toString();
        }

    } //close inner class StateLocation

    /**
     * Grep the specific marker in the specific file. If not found any, return null; if find one or multiple lines,
     * return the list of lines(strings).
     * 
     * @param fileName
     * @param markerStr
     */
    public static List<String> findLineMarkerInFile(final String fileName, final String markerStr) throws Exception
    {

        final String[] command = new String[]{"grep", markerStr, fileName};

        final CommandAdapter cmdAdapter = new CommandAdapter();

        cmdAdapter.invoke(command, true);

        final List<String> commandOutputLines = cmdAdapter.getCommandOutput();

        if(commandOutputLines.isEmpty())
        {
            return null;
        }
        else
        {
            return commandOutputLines;
        }
    }

    /**
     * The line containing the String lineMarker is replaced by the new line.
     * <p>
     * If the lineMarker is not found in the file, an Exception is thrown.
     */
    public static void replaceLineInFile(final String lineMarker, final String newLine, final File file) throws Exception
    {
        final Vector<String> linesVec = new Vector<String>();
    
        boolean beReplaced = false;
    
        //read line by line, replace the old string with new one, store all lines
        final BufferedReader reader = new BufferedReader(new FileReader(file));
        String origLine = null;
        while((origLine = reader.readLine()) != null)
        {
            if(origLine.contains(lineMarker) == false)
            {
                linesVec.add(origLine);
            }
            else
            { //if contains the marker, replace with the new line
                linesVec.add(newLine);
    
                beReplaced = true;
            }
        }
        reader.close();
    
        if(beReplaced == false)
        {
            System.out.println("There is no line containing " + lineMarker + " in " + file.getAbsolutePath());
        }
    
        //now, print out all the lines to the same file
        final PrintWriter writer = new PrintWriter(new FileWriter(file));
        for(int i = 0; i < linesVec.size(); i++)
        {
            writer.println(linesVec.get(i));
        }
    
        writer.close();
    
    }

    /**
     * Append line(s) to the end of the file.
     * 
     * @param fileName - the file name with path. At the end of this method, this file will be modified.
     * @param lines - the list of lines: each line will be appended to the end of the file
     */
    public static void appendLinesToFile(final String fileName, final List<String> lines)
    {
        try
        {
            final FileWriter fileWriter = new FileWriter(fileName, true);
    
            final BufferedWriter bufferedWriter = new BufferedWriter(fileWriter);
    
            for(final String str: lines)
            {
                bufferedWriter.write(str + "\n"); //so all these new lines are individual lines, not jammed into one line
            }
    
            bufferedWriter.close();
        }
        catch(final Exception e)
        {
            System.out.println(e.getMessage());
        }
    }

    /**
     * If the directory does not exist, create it; if it exists, delete all the files and sub-directories (and
     * sub-sub-directories if they exist) in it. Unlike FileUtils.cleanDirectory, which will throw an IOException if the
     * directory does not exist. FileUtiles.cleanDirectory(..) is also painful to use because it deletes dot file in the
     * directory and sometimes that dot file can not be deleted and an IOException is thrown.
     * 
     * @param dirName
     */
    public static void prepareDir(final String dirName)
    {
    
        final File directory = new File(dirName);
    
        //first, check if this dir exists or not: if not(fresh case), create it and returns true, end.
        if(directory.exists() == false)
        {
            //the directory does not exist, so create it
    
            System.out.println("The dir[" + dirName + "] does not exist yet. Created it.");
            directory.mkdirs();
    
            return;
        }
    
        //if reaching here, the dir does exist, now delete all files in directory
        final File[] dirContentAsArray = directory.listFiles();
        for(final File fileOrDir: dirContentAsArray)
        {
    
            if(fileOrDir.isDirectory())
            {//it's a sub-directory
    
                //calling it self to delete the content within the sub-directory
                prepareDir(fileOrDir.getAbsolutePath());
            }
    
            boolean isDirectory = false;
            if(fileOrDir.isDirectory())
            {
                isDirectory = true;
            }
    
            final boolean deletionStatus = fileOrDir.delete(); //now, this could be a file or an emptied directory
    
            //check the deletion status
            if(!deletionStatus)
            {
                // Failed to delete file or the empty sub-dir
                System.out.println("Failed to delete " + fileOrDir);
            }
            else if(isDirectory)
            {//if the deletion was successful and this round is deleting a directory
                System.out.println("Successfully deleted the empty sub-dir " + fileOrDir);
            }
        }//close for loop
    
        return;
    }

} //close class
