package ohd.hseb.util.fews;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.TimeZone;

import ohd.hseb.util.Logger;
import ohd.hseb.util.fews.GroupOfParameters.DateTime;

public abstract class Parameters implements IParameters
{

    protected final Map<GroupOfParameters.ValidPeriod, GroupOfParameters> _validPeriodParametersMap;
    protected final Map<String, GroupOfParameters> _groupParametersMap;

    private String _version = null;
    protected Logger _logger; //set by OHDFewsAdapter

    /**
     * This class assumes the _logger is set using setLogger() for writing out diagnostics
     */
    public Parameters()
    {
        // We need to keep the order of insertion to be able to get the last inserted ValidPeriod object when repeated.
        _validPeriodParametersMap = new LinkedHashMap<GroupOfParameters.ValidPeriod, GroupOfParameters>(); //sorted in natural order of key(String)
        _groupParametersMap = new LinkedHashMap<String, GroupOfParameters>(); //sorted in natural order of key(String)

    }

    /**
     * Gets the value of the version property.
     * 
     * @return possible object is {@link String }
     */
    public String getVersion()
    {
        if(_version == null)
        {
            return "1.5";
        }
        return _version;
    }

    /**
     * Sets the value of the version property.
     * 
     * @param value allowed object is {@link String }
     */
    public void setVersion(final String value)
    {
        this._version = value;
    }

    /**
     * This method is called in OHDFewsAdapter.java, near the end finishing parsing params.xml (inside
     * ParameterHandler.java endDocument()).
     * <p>
     * _xmlParser.parseParameters(_parameterFileName); Or:
     * <p>
     * SAXParser parser = factory.newSAXParser();<br>
     * parser.parse(inputXMLFileName, paramHandler); This method should be overwrite by the classes extending this
     * class. As the List should be transformed to a Map
     * 
     * @throws Exception
     */
    public abstract void setParametersFromList(final List<GroupOfParameters> paramsList) throws Exception;

    /**
     * For Java models(snow17, sacsma, sacsmaHT and UHG), this method can be implemented to extract values from
     * _paramsMap to instance variables to avoid frequently access {@link #_validPeriodParametersMap}. For legacy
     * models, this method remains empty and do nothing.
     * <p>
     * This method is called in {@link #setParametersFromList(Map)}.
     */
    protected abstract void extractValuesFromMap() throws Exception;

    /**
     * For Java models(snow17, sacsma, sacsmaHT and UHG), this method can be implemented to set values to _paramsMap
     * from instance variables to allow changes on instance variables to be saved into the Map.
     * {@link #_validPeriodParametersMap}. For legacy models, this method remains empty and do nothing.
     * <p>
     * This method is called in {@link #setParametersFromList(Map)}.
     */
    protected abstract void setValuesToMap() throws Exception;

    protected long asLong(final DateTime dateTime) throws ParseException
    {
        final SimpleDateFormat dateFmt = new SimpleDateFormat(OHDConstants.DATE_TIME_FORMAT_STR);
        final TimeZone timeZone = OHDConstants.GMT_TIMEZONE;
        dateFmt.setTimeZone(timeZone);

        return (dateFmt.parse(dateTime.getDate() + " " + dateTime.getTime()).getTime());
    }

    public void setLogger(final Logger logger)
    {
        _logger = logger;

    }

    public abstract List<GroupOfParameters> getParamsList();

    public abstract void setNumberOfValidPeriods(final int validPeriodNumber);

    public abstract int getNumberOfValidPeriods();

    @Override
    public abstract boolean equals(final Object other);

}
