package ohd.hseb.util.fews;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URL;
import java.util.Enumeration;
import java.util.jar.JarEntry;
import java.util.jar.JarFile;

import org.jfree.io.FileUtilities;

public class ResourceTools
{
    /**
     * Copy one file to another file. The source file must be in Java's classpath in order for Java to find it. See
     * below. The special advantage of this method is that the source file can be inside a JAR file. A scenario like
     * this: copying a html inside a JAR file to a temporary file and let firefox open it.<br>
     * If the resource file location is not on the Java's classpath, you may use the method
     * {@link #copyfile(String, String)}.
     * 
     * @param resourceFile - the source file which *must* be in Eclipse's or Ant's Java classpath.For example, src/,
     *            test/, nonsrc/ and build/ are in Eclipse's classpath. Suppose the resource file is
     *            nonsrc/aaa/bbb/hello.txt, then this parameter value should be "aaa/bbb/hello.txt"
     *            ("/aaa/bbb/hello.txt" won't work in Eclipse and the command line, "./aaa/bbb/hello.txt" works in
     *            Eclipse only) -- a relative path from Java's classpath, since inside the method,
     *            ClassLoader.getSystemResourceAsStream(String fileName) is used.<br>
     *            When the resource file is bundled inside a JAR file, its path inside the JAR file must be equal to the
     *            parameter value. For the example above, hello.txt must be located inside the JAR file at
     *            [top]aaa/bbb/hello.txt for the same code to find it so that the same code works in Eclipse and in JAR
     *            mode.<br>
     * @param dtFile - the destination file. If it exists beforehand, it will be washed/updated. This value no need to
     *            be on Java's class path. It can be absolute or relative value. It only needs to be an existing path.
     */
    public static void copyResourcefile(final String resourceFile, final String dtFile) throws Exception
    {
        final InputStream in = ResourceTools.class.getClassLoader().getResourceAsStream(resourceFile);

        if(in == null)
        {//resourceFile cannot be located

            throw new Exception("The resource file["
                + resourceFile
                + "] cannot be located by Java. It must be a relative path starting from any Java's classpath, not using \"/\" or \".\". "
                + "For example, if the resource file is \"nonsrc/aaa/bbb/hello.txt\", then the resource parameter should just be "
                + "\"aaa/bbb/hello.txt\"  ");
        }

        final OutputStream out = new FileOutputStream(new File(dtFile));

        final byte[] buf = new byte[1024];
        int len;
        while((len = in.read(buf)) > 0)
        {
            out.write(buf, 0, len);
        }
        in.close();
        out.close();

//        System.out.println("Copied the file " + resourceFile + " to the file " + dtFile);
    }

    /**
     * Copy one file, not directory, to another file. Please also see {@link #copyResourcefile(String, String)}.
     * 
     * @param srFile - the source file. It can be absolute path, or relative path.
     * @param dtFile - the destination file. If it exists beforehand, it will be washed/updated.
     * @throws Exception - when the source file does not exist or the copying failed
     */
    public static void copyfile(final String srFile, final String dtFile) throws Exception
    {
        final File f1 = new File(srFile);
        final File f2 = new File(dtFile);
        final InputStream in = new FileInputStream(f1);

        //For Overwrite the file.
        final OutputStream out = new FileOutputStream(f2);

        final byte[] buf = new byte[1024];
        int len;
        while((len = in.read(buf)) > 0)
        {
            out.write(buf, 0, len);
        }
        in.close();
        out.close();

//        System.out.println("Copied the file " + srFile + " to the file " + dtFile);
    }

    /**
     * This Java method does like Linux command "cp -Rf dir1 dir2". Copy a directory including its files,
     * sub-directories(and sub-sub-directories) to the destination directory. The important thing is dir1 could be
     * inside a JAR file.
     * 
     * @param srcDirName - it can be a directory inside a JAR file or a directory on Java's classpath, e.g.
     *            nonsrc/aaa/bbb(since nonsrc/ is on the Eclipse project classpath, here the parameter value should be
     *            "/aaa/bbb"). For more information on "being on Java's classpath", please see comments of
     *            {@link #copyResourcefile(String, String)}.<br>
     *            Note: the src directory cannot be an absolute path because "Java's classpath + the absolute path" is a
     *            strange non-existing path.
     * @param destDirName - the destination directory. It can be absolute or relative path.
     * @throws Exception
     */
    public static void copyFolder(final String srcDirName, final String destDirName) throws Exception
    {

        final URL srcURL = ResourceTools.class.getResource(srcDirName);

        final String resourcePath = srcURL.toString();

        if(resourcePath.startsWith("jar:"))
        {//the resource is inside a JAR file, e.g. "jar:file:/fs/pda/users/lix/devl/whole_tree/HEFS/trunk/libs/installed/EVS.jar!/statsexplained/"

            //extract the jar file: get rid of "jar:file:" at the beginning, get the string till "!"
            final String jarFile = resourcePath.substring(9, resourcePath.indexOf("!"));

            final String parentDir = new File(destDirName).getParent();

            copyDirInJarFile(jarFile, srcDirName, parentDir);

            return;
        }

        //if reaching here, must be in Eclipse mode: srcDirName is not inside a JAR file, it must be on the classpath of the Eclipse project
        copyDirInClasspath(srcDirName, destDirName);

    }

    /**
     * Copy a directory inside the JAR file to be a sub directory in the file system.
     * 
     * @param jarName - the JAR file name with full path
     * @param dirNameInsideJar - the directory name inside the JAR file
     * @param destParentDirName - the directory inside the JAR file will become a sub directory here
     * @throws Exception
     */
    public static void copyDirInJarFile(final String jarName, String dirNameInsideJar, String destParentDirName) throws Exception
    {

        //if starts with "/"(linux case) or "\"(pc case), trim it since the code below(entry.getName()) returns top level directory without the leading file separator
        if(dirNameInsideJar.startsWith("/") || dirNameInsideJar.startsWith("\\"))
        {
            dirNameInsideJar = dirNameInsideJar.substring(1);
        }

        //In both Linux and PC, if destDirName not ending with separator, append one
        if(!destParentDirName.endsWith(File.separator))
        {
            destParentDirName = destParentDirName + File.separator;
        }

        final JarFile jarFile = new JarFile(jarName);

        final Enumeration<JarEntry> enume = jarFile.entries();

        while(enume.hasMoreElements())
        {
            final JarEntry entry = enume.nextElement();

            final String entryNameInJar = entry.getName();

            if(entryNameInJar.contains(dirNameInsideJar))
            {
                final String destName = destParentDirName + entryNameInJar; //destName could be a dir or file

                if(entry.isDirectory())
                {
                    new File(destName).mkdir();
                }
                else
                {//this case is a file inside the JAR
                    copyResourcefile(entryNameInJar, destName);
                }
            }

        } //close while block

        jarFile.close();

    } //close the method

    /**
     * Copy a directory including its files, sub-directories(and sub-sub-directories) to the destination directory.(This
     * method is invoked recursively). The source directory can not be inside a JAR file.
     * 
     * @param srcDir - the src directory name. Note: it MUST be on Java's classpath. e.g. nonsrc/aaa/bbb(since nonsrc/
     *            is on the Eclipse project classpath, here the parameter value should be "/aaa/bbb").For more
     *            information on "being on Java's classpath", please see comments of
     *            {@link #copyResourcefile(String, String)}.<br>
     *            Note: the src directory cannot be an absolute path because "Java's classpath + the absolute path" is a
     *            strange non-existing path.
     * @param destDir - the destination directory. It can be absolute or relative path.
     * @throws Exception
     */
    public static void copyDirInClasspath(final String srcDir, final String destDir) throws Exception
    {
        final File srcFile = FileUtilities.findFileOnClassPath(srcDir); //srcDir starts with "/", see comments above

        final File[] files = srcFile.listFiles();

        for(final File file: files)
        {
            if(file.getPath().contains(".svn"))
            {//do not bother with anything related to ".svn"
                continue;
            }

            if(file.isDirectory())
            {
                final String newDirName = srcDir + File.separator + file.getName();
                final String newDestDirName = destDir + File.separator + file.getName();

                copyDirInClasspath(newDirName, newDestDirName);

            }
            else
            {//if file, then copy it

                final File destDirFile = new File(destDir);
                //if directory not exists, create it
                if(!destDirFile.exists())
                {
                    final boolean result = destDirFile.mkdirs(); //comparing to mkdir(), mkdirs() can create nested directory

                    if(!result)
                    {
                        throw new Exception("Failed to create " + destDir);
                    }
                }

                /*
                 * srcDir starts with "/" and copyResourcefile(src, dest) src MUST not start with "/" -- what a
                 * confusion! Here rips the beginning "/" first
                 */
                final String srcFileNameInClassPath = srcDir + File.separator + file.getName();
                final String srcFileNameInClassPathNew = srcFileNameInClassPath.substring(1);

                final String destFileName = destDir + File.separator + file.getName();

                copyResourcefile(srcFileNameInClassPathNew, destFileName);
            }
        }//close for loop
    }

    /**
     * Delete the directory, including files and its sub directories inside.
     * 
     * @param dirName - must be a directory name, otherwise an Exception will be thrown.
     */
    public static void deleteDir(final String dirName) throws Exception
    {//this method is called recursively, see below

        final File directory = new File(dirName);

        if(!directory.exists())
        {//the directory does not exist, do nothing. Without this check, when the dir does not exist, it hangs

            return;
        }

        if(!directory.isDirectory())
        {
            throw new Exception(dirName + " is a file, not a directory");
        }

        final File[] files = directory.listFiles();

        //check if the directory is empty directory, since file.delete() only works on single file or empty directory
        if(files.length == 0)
        {//empty directory, now directory.delete() works

            if(!directory.delete())
            {
                // Failed to delete file
                System.out.println("Failed to delete " + directory);
            }

            return;
        }

        //if reach here, must be a non-empty directory
        for(final File file: files)
        {

            if(file.isDirectory())
            {//is a sub directory, recursively call this method

                deleteDir(file.getAbsolutePath());
            }
            else
            {//is a file, now file.delete() works

                if(!file.delete())
                {
                    // Failed to delete file
                    System.out.println("Failed to delete " + file);
                }

            }
        }//close for loop

        //if this directory "dirName" had no sub directories, then when reaching here, it should be an empty dir now. 
        deleteDir(dirName);
    }
}
