package ohd.hseb.util.fews;

import java.io.File;
import java.util.ArrayList;
import java.util.List;
import java.util.Properties;
import java.util.TimeZone;

import ohd.hseb.time.DateTime;
import ohd.hseb.util.Logger;

public class RunInfo implements Cloneable
{
    private String _daylightSavingObservatingTimeZone;
    private TimeZone _timeZone;
    private String _diagnosticFile;
    private String _workDir;
    private String _inputStateDescriptionFile;
    private String _inputParameterFile;
    private String _inputNetCdfFile = null;
    private String _outputNetCdfFile = null;
    private String _inputDataSetDir;
    private String _outputStateDescriptionFile;
    private String _outputTimeSeriesFile;
    private long _runStartTimeLong;
    private long _runEndTimeLong;
    private long _runLastObservationTimeLong;
    private long _time0Long; // start of a forecast for a forecast run
    private Properties _properties;
    private final List<String> _inputTimeSeriesFileList;
    private final List<String> _inputParameterFileList;
    private final List<String> _inputStateDescriptionFileList;
    private int _timeZoneRawOffsetInHours = (int)OHDConstants.MISSING_DATA;
    private String _version = null;

    private final Logger _logger;

    public RunInfo(final Logger logger)
    {
        _logger = logger;
        _runLastObservationTimeLong = DateTime.DEFAULT_DATE_TIME_LONG;
        _runStartTimeLong = DateTime.DEFAULT_DATE_TIME_LONG;
        _runEndTimeLong = DateTime.DEFAULT_DATE_TIME_LONG;
        _time0Long = DateTime.DEFAULT_DATE_TIME_LONG;
        _inputTimeSeriesFileList = new ArrayList<String>();
        _inputParameterFileList = new ArrayList<String>();
        _inputStateDescriptionFileList = new ArrayList<String>();
    }

    /**
     * checks for optional properties 1) legacyLocation 2) irregularTsModFileName 3) regularTsModFileName if defined,
     * make sure file(s) exist;
     * 
     * @param properties the properties that were loaded from the input properties file
     * @throws Exception
     */
    public void validateOptionalRunInfo() throws Exception
    {
        for(String tsFileName: this.getInputTimeSeriesFileList())
        {
            if(tsFileName != null)
            {
                final File tsFile = new File(tsFileName);

                if(tsFile.exists())
                {
                    _logger.log(Logger.DEBUG, "TsFile used:" + tsFileName);
                }
                else
                {
                    tsFileName = null;
                    _logger.log(Logger.DEBUG, "TsFile used:" + tsFileName);
                }
            }
        }

        /*
         * for defining legacy executables
         */

        final String legacyLocationProperty = this.getProperties()
                                                  .getProperty(ohd.hseb.util.fews.OHDConstants.LEGACY_LOCATION_DIR);

        if(legacyLocationProperty != null)
        {
            final String legacyExecutableDir = legacyLocationProperty + "/";

            _logger.log(Logger.DEBUG, "LegacyExecutable Dir used:" + legacyExecutableDir);
            if(!new File(legacyExecutableDir).exists())
            {
                throw new Exception("The directory for the executable file " + legacyExecutableDir + " does not exist");
            }

            this.getProperties().setProperty(OHDConstants.LEGACY_LOCATION_DIR, legacyExecutableDir);
        }

    }

    private boolean validateFileExists(final String fileName)
    {
        boolean result = false;
        final File file = new File(fileName);

        if(file.exists())
        {
            result = true;
        }

        return result;
    }

    /**
     * First, check all the required elements are present in run_info.xml. Then validate that the end time
     * {@link #_runEndTimeLong} is after the initial state time {@link #_runStartTimeLong}; LSTCMPDY(
     * {@link #_runLastObservationTimeLong} and {@link #_time0Long} are after the initial state time
     * {@link #_runStartTimeLong} and before or equal to the end time {@link #_runEndTimeLong}; If any check is false, a
     * Exception is thrown.
     */
    public void validatRunInfo() throws Exception
    {
        // check all required elements
        if(this.getInputParametersFileList() != null && this.getInputParametersFileList().size() > 0)
        {
            for(final String inputParameterFile: this.getInputParametersFileList())
                if(inputParameterFile != null && !validateFileExists(inputParameterFile))
                {
                    throw new Exception("Please provide correct information from run info in the model parameter file name: "
                        + inputParameterFile + " does not exist!");
                }
        }

        /*------------------some util programs don't have states.xml specified in run_info.xml ------------------*/
        if(this.getInputStateDescriptionFileList() != null && this.getInputStateDescriptionFileList().size() > 0)
            for(final String inputStateDescriptionFile: this.getInputStateDescriptionFileList())
                if(inputStateDescriptionFile != null && !validateFileExists(inputStateDescriptionFile))
                {
                    throw new Exception(inputStateDescriptionFile + " does not exist!");
                }

        if(this.getInputTimeSeriesFileList() != null && this.getInputTimeSeriesFileList().size() > 0)
        {
            for(final String inputTimeSeriesFile: this.getInputTimeSeriesFileList())
            {
                if(inputTimeSeriesFile != null && !validateFileExists(inputTimeSeriesFile))
                {
                    throw new Exception("Please provide correct information from run info in the model Time Series file name "
                        + inputTimeSeriesFile);
                }
            }
        }
        // Get output directory without the file name
        String fewsDataOutputLocation = null;
        if(this.getOutputTimeSeriesFile() != null)
        {
            final File dirCheckTemp = new File(this.getOutputTimeSeriesFile());
            fewsDataOutputLocation = dirCheckTemp.getParent();

            final File dirCheck = new File(fewsDataOutputLocation);
            if(!dirCheck.canWrite())
            {
                throw new Exception("Output directory, " + dirCheck.getAbsolutePath()
                    + ", does not exist or cannot be written to.");
            }
        }

        if(this.getDiagnosticFile() != null)
        {
            final File dirCheckTemp = new File(this.getDiagnosticFile());
            fewsDataOutputLocation = dirCheckTemp.getParent();

            final File dirCheck = new File(fewsDataOutputLocation);
            if(!dirCheck.canWrite())
            {
                throw new Exception("Output directory, " + dirCheck.getAbsolutePath()
                    + ", does not exist or cannot be written to.");
            }
        }
        else
        {
            throw new Exception("ERROR: Diagnostic file information could not be found in run info. Output will be send to standard out");
        }

        final String _fewsDataWorkLocation = this.getWorkDir();

        final File dirCheck = new File(_fewsDataWorkLocation);
        if(!dirCheck.canWrite())
        {
            throw new Exception("Work directory, " + dirCheck.getAbsolutePath()
                + ", does not exist or cannot be written to.");
        }

        String errorMessage;

        final String startStr = DateTime.getDateTimeStringFromLong(_runStartTimeLong, _timeZone);
        final String endStr = DateTime.getDateTimeStringFromLong(_runEndTimeLong, _timeZone);
        final String lastObsStr = DateTime.getDateTimeStringFromLong(_runLastObservationTimeLong, _timeZone);
        final String time0Str = DateTime.getDateTimeStringFromLong(_time0Long, _timeZone);

        if(_runEndTimeLong < _runStartTimeLong)
        {
            errorMessage = "Error: the model run end time " + endStr + " is earlier than the initial state time "
                + startStr;

            throw new Exception(errorMessage);
        }
        else if(_runLastObservationTimeLong != DateTime.DEFAULT_DATE_TIME_LONG)
        {//_runLastObservationTimeLong is not required by schema, so it may not be used; but now it is used

            if(_runLastObservationTimeLong < _runStartTimeLong || _runLastObservationTimeLong > _runEndTimeLong)
            {
                errorMessage = "Error: the last observation time " + lastObsStr + " is NOT between initial state time "
                    + startStr + " and the end time " + endStr;

                throw new Exception(errorMessage);
            }
        }
        else if(_time0Long < _runStartTimeLong || _time0Long > _runEndTimeLong)
        {
            errorMessage = "Error: the time0 " + time0Str + " is NOT between initial state time " + startStr
                + " and the end time " + endStr;

            throw new Exception(errorMessage);
        }

        //if reach here, no Exception has been thrown, the validation passes
        _logger.log(Logger.DEBUG, "RunInfo validation passes. start time: " + startStr + "; end time: " + endStr
            + "; last observation time: " + lastObsStr + "; time0: " + time0Str);

    }

    public Properties getProperties()
    {
        return _properties;
    }

    public void setProperties(final Properties props)
    {
        this._properties = props;
    }

    public TimeZone getTimeZone()
    {
        return _timeZone;
    }

    public void setTimeZone(final TimeZone zone)
    {
        _timeZone = zone;
    }

    public String getDaylightSavingObservatingTimeZone()
    {
        return _daylightSavingObservatingTimeZone;
    }

    public void setDaylightSavingObservatingTimeZone(final String daylightSavingObservatingTimeZone)
    {
        _daylightSavingObservatingTimeZone = daylightSavingObservatingTimeZone;
    }

    /**
     * Returns the number of hours(could be positive or negative) to add to GMT hour to get the local hour.
     * <p>
     * Note: the returned number is integer, even though time zone related hour shift could be partial, like 1.5 hours.
     * But this scenario may not exist in U.S. and we keep it simple.
     */
    public int getTimeZoneRawOffsetInHours()
    {
        if(_timeZoneRawOffsetInHours == (int)OHDConstants.MISSING_DATA)
        {
            _timeZoneRawOffsetInHours = OHDUtilities.getTimeZoneRawOffsetInHours(getTimeZone());
        }
        return _timeZoneRawOffsetInHours;
    }

    public long getTime0Long()
    {
        return _time0Long;
    }

    public void setTime0Long(final DateTime dateTime)
    {
        try
        {
            _time0Long = dateTime.getTimeInMillis();
        }
        catch(final Exception e)
        {
            _logger.log(Logger.WARNING, "Unable to parse time0 date in run file");
            _time0Long = DateTime.DEFAULT_DATE_TIME_LONG;
        }
    }

    public String getDiagnosticFile()
    {
        return _diagnosticFile;
    }

    public void setDiagnosticFile(final String file)
    {
        _diagnosticFile = file;
    }

    public String getWorkDir()
    {
        return _workDir;
    }

    public void setWorkDir(final String dir)
    {
        _workDir = dir;
    }

    public String getInputStateDescriptionFile()
    {
        return _inputStateDescriptionFile;
    }

    public void setInputStateDescriptionFile(final String stateDescriptionFile)
    {
        _inputStateDescriptionFile = stateDescriptionFile;
    }

    public List<String> getInputTimeSeriesFileList()
    {
        return _inputTimeSeriesFileList;
    }

    public void setInputTimeSeriesFileList(final String inputTimeSeriesFile)
    {
        _inputTimeSeriesFileList.add(inputTimeSeriesFile);
    }

    public List<String> getInputParametersFileList()
    {
        return _inputParameterFileList;
    }

    public void setInputParametersFileList(final String inputStatesFile)
    {
        _inputParameterFileList.add(inputStatesFile);
    }

    public List<String> getInputStateDescriptionFileList()
    {
        return _inputStateDescriptionFileList;
    }

    public void setInputStateDescriptionFileListList(final String inputStatesFile)
    {
        _inputStateDescriptionFileList.add(inputStatesFile);
    }

    public void setInputNetCdfFile(final String inputNetCdfFile)
    {
        _inputNetCdfFile = inputNetCdfFile;
    }

    public void setOutputNetCdfFile(final String outputNetCdfFile)
    {
        _outputNetCdfFile = outputNetCdfFile;
    }

    public String getInputParameterFile()
    {
        return _inputParameterFile;
    }

    public void setInputParameterFile(final String inputParameterFile)
    {
        _inputParameterFile = inputParameterFile;
    }

    public long getRunStartTimeLong()
    {
        return _runStartTimeLong;
    }

    public void setRunStartTimeLong(final DateTime startDateTime)
    {
        try
        {
            _runStartTimeLong = startDateTime.getTimeInMillis();
        }
        catch(final Exception e)
        {
            _logger.log(Logger.WARNING, "Unable to parse start date in run file");
        }
    }

    public long getRunEndTimeLong()
    {
        return _runEndTimeLong;
    }

    public void setRunEndTimeLong(final DateTime endDateTime)
    {
        try
        {
            _runEndTimeLong = endDateTime.getTimeInMillis();
        }
        catch(final Exception e)
        {
            _logger.log(Logger.WARNING, "Unable to parse end date in run file");
        }
    }

    public void setRunEndTime(final long endTime)
    {
        _runEndTimeLong = endTime;
    }

    public String getInputDataSetDir()
    {
        return _inputDataSetDir;
    }

    public void setInputDataSetDir(final String dataSetDir)
    {
        _inputDataSetDir = dataSetDir;
    }

    public String getOutputStateDescriptionFile()
    {
        return _outputStateDescriptionFile;
    }

    public void setOutputStateDescriptionFile(final String stateDescriptionFile)
    {
        _outputStateDescriptionFile = stateDescriptionFile;
    }

    public String getOutputTimeSeriesFile()
    {
        return _outputTimeSeriesFile;
    }

    public String getInputNetCdfFile()
    {
        return _inputNetCdfFile;
    }

    public String getOutputNetCdfFile()
    {
        return _outputNetCdfFile;
    }

    public void setOutputTimeSeriesFile(final String outputTimeSeriesFile)
    {
        _outputTimeSeriesFile = outputTimeSeriesFile;
    }

    /** return LSTCMPDY. Since it is optional, if it is not available, return time0 */
    public long getRunLastObservationTimeLong()
    {
        if(_runLastObservationTimeLong == DateTime.DEFAULT_DATE_TIME_LONG)
        {//lastObservationDateTime is not present,  set it to time0
            _runLastObservationTimeLong = getTime0Long();
        }
        else if(_runLastObservationTimeLong > getTime0Long())
        {//lastObservationDateTime is present, check if it is beyond time0. It can not be beyond time0, if so, reset it to time0 too
            _logger.log(Logger.WARNING,
                        "lastObservationDateTime["
                            + DateTime.getDateTimeStringFromLong(_runLastObservationTimeLong, _timeZone)
                            + "] is later than time0[" + DateTime.getDateTimeStringFromLong(_time0Long, _timeZone)
                            + "]. Reset it to time0");

            _runLastObservationTimeLong = getTime0Long();
        }

        return _runLastObservationTimeLong;
    }

    /** set LSTCMPDY */
    public void setRunLastObservationTimeLong(final DateTime dateTime)
    {
        try
        {
            _runLastObservationTimeLong = dateTime.getTimeInMillis();
        }
        catch(final Exception e)
        {
            _logger.log(Logger.WARNING, "Unable to parse last observation date in run file");
        }
    }

    public boolean equals(final RunInfo runInfo)
    {
        boolean result = true;
        if((!(runInfo._diagnosticFile.equals(this._diagnosticFile))) || (!(runInfo._workDir.equals(this._workDir)))
            || (!(runInfo._inputDataSetDir.equals(this._inputDataSetDir)))
            || (!(runInfo._outputStateDescriptionFile.equals(this._outputStateDescriptionFile)))
            || (!(runInfo._inputStateDescriptionFile.equals(this._inputStateDescriptionFile)))
            || (!(runInfo._timeZone.getID().equals(this._timeZone.getID())))
            || (!(runInfo._runEndTimeLong == this._runEndTimeLong))
            || (!(runInfo._runLastObservationTimeLong == this._runLastObservationTimeLong))
            || (!(runInfo._runStartTimeLong == this._runStartTimeLong)) || (!(runInfo._time0Long == this._time0Long)))
        {
            result = false;
        }

        return result;
    }

    /**
     * Put run info into the log.
     * 
     * @param debugLevel
     */
    public void logRunInfo(final int debugLevel)
    {
        StringBuilder resultStr = new StringBuilder();

        _logger.log(debugLevel, "Run Info:");
        _logger.log(debugLevel, "Time Zone = " + this.getTimeZone().getID());
        _logger.log(debugLevel,
                    "Start Date Time  = "
                        + DateTime.getDateTimeStringFromLong(this.getRunStartTimeLong(), getTimeZone()) + " "
                        + this.getTimeZone().getID());
        _logger.log(debugLevel,
                    "End Date Time = " + DateTime.getDateTimeStringFromLong(getRunEndTimeLong(), getTimeZone()) + " "
                        + this.getTimeZone().getID());
        _logger.log(debugLevel, "Time 0 = " + DateTime.getDateTimeStringFromLong(this.getTime0Long(), getTimeZone())
            + " " + this.getTimeZone().getID());
        _logger.log(debugLevel,
                    "LATCMPDY = "
                        + DateTime.getDateTimeStringFromLong(this.getRunLastObservationTimeLong(), getTimeZone()) + " "
                        + this.getTimeZone().getID());
        _logger.log(debugLevel, "Work Dir = " + this.getWorkDir());
        _logger.log(debugLevel, "Output Diagnostic File = " + this.getDiagnosticFile());
        for(final String parameters: this.getInputParametersFileList())
        {
            resultStr.append(parameters).append(" ");
        }
        _logger.log(debugLevel, "Input Parameter File = " + resultStr);

        resultStr = new StringBuilder();
        for(final String states: this.getInputStateDescriptionFileList())
        {
            resultStr.append(states).append(" ");
        }
        _logger.log(debugLevel, "Input States File = " + resultStr);

        resultStr = new StringBuilder();
        for(final String timeSeries: this.getInputTimeSeriesFileList())
        {
            resultStr.append(timeSeries).append(OHDConstants.NEW_LINE);
        }
        _logger.log(debugLevel, "Input TimeSeries File = " + resultStr);
        _logger.log(debugLevel, "Input NetCdf File = " + this.getInputNetCdfFile());
        _logger.log(debugLevel, "Output NetCdf File = " + this.getOutputNetCdfFile());
        if(_properties != null)
        {
            _logger.log(debugLevel, "properties = " + _properties.toString());
        }

    }

    /**
     * @return the version
     */
    public String getVersion()
    {
        return _version;
    }

    /**
     * @param version the version to set
     */
    public void setVersion(final String version)
    {
        this._version = version;
    }

    @Override
    public String toString()
    {
        final StringBuilder resultStr = new StringBuilder();

        resultStr.append("Run Info:").append(OHDConstants.NEW_LINE);
        resultStr.append("Time Zone = ").append(this.getTimeZone().getID()).append(OHDConstants.NEW_LINE);
        resultStr.append("Start Date Time  = ")
                 .append(DateTime.getDateTimeStringFromLong(this.getRunStartTimeLong(), getTimeZone()))
                 .append(" ")
                 .append(this.getTimeZone().getID())
                 .append(OHDConstants.NEW_LINE);
        resultStr.append("End Date Time = ")
                 .append(DateTime.getDateTimeStringFromLong(getRunEndTimeLong(), getTimeZone()))
                 .append(" ")
                 .append(this.getTimeZone().getID())
                 .append(OHDConstants.NEW_LINE);
        resultStr.append("Time 0 = ")
                 .append(DateTime.getDateTimeStringFromLong(this.getTime0Long(), getTimeZone()))
                 .append(" ")
                 .append(this.getTimeZone().getID())
                 .append(OHDConstants.NEW_LINE);
        resultStr.append("LATCMPDY = ")
                 .append(DateTime.getDateTimeStringFromLong(this.getRunLastObservationTimeLong(), getTimeZone()))
                 .append(" ")
                 .append(this.getTimeZone().getID())
                 .append(OHDConstants.NEW_LINE);
        resultStr.append("Work Dir = ").append(this.getWorkDir()).append(OHDConstants.NEW_LINE);
        resultStr.append("Output Diagnostic File = ").append(this.getDiagnosticFile()).append(OHDConstants.NEW_LINE);
        resultStr.append("Input Parameter File = ").append(this.getInputParameterFile()).append(OHDConstants.NEW_LINE);

        for(final String parameters: this.getInputParametersFileList())
        {
            resultStr.append(parameters).append(OHDConstants.NEW_LINE);
        }

        resultStr.append("Input States File = ");
        for(final String states: this.getInputStateDescriptionFileList())
        {
            resultStr.append(states).append(OHDConstants.NEW_LINE);
        }
        resultStr.append("Input TimeSeries File = ");
        for(final String timeSeries: this.getInputTimeSeriesFileList())
        {
            resultStr.append(timeSeries).append(OHDConstants.NEW_LINE);
        }
        resultStr.append("Input NetCdf File = ").append(this.getInputNetCdfFile()).append(OHDConstants.NEW_LINE);
        resultStr.append("Output NetCdf File = ").append(this.getOutputNetCdfFile()).append(OHDConstants.NEW_LINE);
        // resultStr.append("Time Zone = " + this.getOutputStateDescriptionFile()).append(NEW_LINE);
        if(_properties != null)
        {
            resultStr.append("properties = ").append(_properties.toString()).append(OHDConstants.NEW_LINE);
        }

        return resultStr.toString();
    }

    @Override
    public Object clone()
    {
        try
        {
            final RunInfo cloned = (RunInfo)super.clone();
            return cloned;
        }
        catch(final CloneNotSupportedException e)
        {
            return null;
        }
    }

}
