package ohd.hseb.util.fews;

import java.util.Properties;
import java.util.TimeZone;

import javax.xml.stream.XMLStreamReader;

import org.xml.sax.SAXException;

import ohd.hseb.time.DateTime;
import ohd.hseb.util.Logger;

final public class RunInfoHandler extends FewsXmlHandler
{
    private RunInfo _runInfo;

    private boolean _readingInProperties = false;
    private String _key = null;
    private String _value = null;

    private final Properties _properties = new Properties();

    public RunInfoHandler(final RunInfo runInfo, final Logger logger)
    {
        _runInfo = runInfo;
        _logger = logger;
    }

    public void setModelRunInfo(final RunInfo runInfo)
    {
        _runInfo = runInfo;
    }

    public RunInfo getModelRunInfo()
    {
        return _runInfo;
    }

    enum runInfoElements
    {
        RUN, LOGLEVEL, TIMEZONE, DAYLIGHTSAVINGOBSERVINGTIMEZONE, STARTDATETIME, ENDDATETIME, TIME0, LASTOBSERVATIONDATETIME, WORKDIR, INPUTPARAMETERFILE, INPUTSTATEDESCRIPTIONFILE, INPUTTIMESERIESFILE, INPUTNETCDFFILE, OUTPUTDIAGNOSTICFILE, OUTPUTSTATEDESCRIPTIONFILE, OUTPUTTIMESERIESFILE, OUTPUTNETCDFFILE, PROPERTIES, STRING, INT, DOUBLE
    };

    @Override
    public void startElement(final XMLStreamReader reader) throws SAXException
    {
        final String elementName = reader.getLocalName().trim().toUpperCase();

        try
        {
            switch(runInfoElements.valueOf(elementName))
            {
                case STARTDATETIME:
                {
                    final DateTime dateTime = getDateTime(reader);
                    _runInfo.setRunStartTimeLong(dateTime);
                    break;
                }
                case ENDDATETIME:
                {
                    final DateTime dateTime = getDateTime(reader);
                    _runInfo.setRunEndTimeLong(dateTime);

                    break;
                }
                case TIME0:
                {
                    final DateTime dateTime = getDateTime(reader);
                    _runInfo.setTime0Long(dateTime);
                    break;
                }
                case LASTOBSERVATIONDATETIME:
                {
                    final DateTime dateTime = getDateTime(reader);
                    _runInfo.setRunLastObservationTimeLong(dateTime);
                    break;
                }
                case PROPERTIES:
                {
                    _readingInProperties = true;
                    break;
                }
                case STRING: //must be within <properties>
                {
                    setPropertiesValues(reader);
                    break;
                }
                case INT: //must be within <properties>
                {
                    setPropertiesValues(reader);
                    break;
                }
                case DOUBLE: //must be within <properties>
                {
                    setPropertiesValues(reader);
                    break;
                }
                case RUN:
                    final int attributes = reader.getAttributeCount();
                    for(int i = 0; i < attributes; i++)
                    {
                        String attributeName = reader.getAttributeName(i).toString();

                        if(attributeName != null)
                        {
                            attributeName = attributeName.trim();
                        }
                        if("version".equalsIgnoreCase(attributeName))
                        {
                            _runInfo.setVersion(reader.getAttributeValue(i).toString().trim());
                        }
                    } // end for

                    break;

                default:
                    break;
            }
        }
        catch(final IllegalArgumentException iae)
        {
            _logger.log(Logger.WARNING, "'" + reader.getLocalName().trim()
                + "' has not been set in the run info object.");
        }

    }

    private void setPropertiesValues(final XMLStreamReader reader)
    {
        if(this._readingInProperties)
        {

            final int attributes = reader.getAttributeCount();
            for(int i = 0; i < attributes; i++)
            {
                String attributeName = reader.getAttributeName(i).toString();
                if(attributeName != null)
                {
                    attributeName = attributeName.trim();
                }

                if("key".equalsIgnoreCase(attributeName))
                {
                    _key = reader.getAttributeValue(i).toString().trim();
                }
                else if("value".equalsIgnoreCase(attributeName))
                {
                    _value = reader.getAttributeValue(i).toString().trim();
                }
            } // end for
        } // end if 
    }

    @Override
    public void endElement(final XMLStreamReader reader) throws SAXException
    {
        final String elementName = reader.getLocalName().trim().toUpperCase();
        try
        {
            switch(runInfoElements.valueOf(elementName))
            {

                case TIMEZONE:
                {
                    final TimeZone timeZone = OHDUtilities.getTimeZoneFromOffSet(Double.valueOf(_curValue.trim()));
                    _runInfo.setTimeZone(timeZone);
                    _logger.log(Logger.DEBUG,
                                "The time zone offset value in the run info xml file is " + _curValue.trim()
                                    + ". Accordingly, Time Zone is: " + timeZone.getID());
                    break;
                }

                case DAYLIGHTSAVINGOBSERVINGTIMEZONE:
                {

                    TimeZone timeZone = null;
                    switch(_curValue.trim())
                    {
                        case "AST":
                            timeZone = OHDUtilities.getTimeZoneFromOffSet(-9.0);
                            // Note getting the timezone in a format different to GMT cause chps to fail as the daylight saving will switch the dates by one hour and CHPS can not handle it.
                            //timeZone = TimeZone.getTimeZone("US/Alaska");
                            break;
                        case "PST":
                            timeZone = OHDUtilities.getTimeZoneFromOffSet(-8.0);
                            //timeZone = TimeZone.getTimeZone("America/Los_Angeles"); //  US/Pacific
                            break;
                        case "MST":
                            timeZone = OHDUtilities.getTimeZoneFromOffSet(-7.0);
                            //timeZone = TimeZone.getTimeZone("US/Mountain");
                            break;
                        case "CST":
                            timeZone = OHDUtilities.getTimeZoneFromOffSet(-6.0);
                            //timeZone = TimeZone.getTimeZone("America/Chicago"); // "US/Central"
                            break;
                        case "IET":
                            timeZone = OHDUtilities.getTimeZoneFromOffSet(-5.0);
                            //timeZone = TimeZone.getTimeZone("America/Indiana/Indianapolis");
                            break;
                        case "CNT":
                            timeZone = OHDUtilities.getTimeZoneFromOffSet(-3.5);
                            //timeZone = TimeZone.getTimeZone("America/St_Johns");
                            break;
                        case "AGT":
                            timeZone = OHDUtilities.getTimeZoneFromOffSet(-3.0);
                            //timeZone = TimeZone.getTimeZone("America/Buenos");
                            break;
                        case "BET":
                            timeZone = OHDUtilities.getTimeZoneFromOffSet(-3.0);
                            break;
                        case "WET":
                            timeZone = OHDUtilities.getTimeZoneFromOffSet(0.0);
                            break;
                        case "CET":
                            timeZone = OHDUtilities.getTimeZoneFromOffSet(1.0);
                            break;
                        case "MET":
                            timeZone = OHDUtilities.getTimeZoneFromOffSet(1.0);
                            break;
                        case "EET":
                            timeZone = OHDUtilities.getTimeZoneFromOffSet(2.0);
                            break;
                        case "AZT":
                            timeZone = OHDUtilities.getTimeZoneFromOffSet(4.0);
                            break;
                        case "NET":
                            timeZone = OHDUtilities.getTimeZoneFromOffSet(4.0);
                            break;
                        case "AET":
                            timeZone = OHDUtilities.getTimeZoneFromOffSet(10.0);
                            break;
                        case "AWT":
                            timeZone = OHDUtilities.getTimeZoneFromOffSet(8.0);
                            break;
                        case "NST":
                            timeZone = OHDUtilities.getTimeZoneFromOffSet(12.0);
                            break;
                        default:
                            _logger.log(Logger.ERROR,
                                        "Value '"
                                            + _curValue.trim()
                                            + "' is not facet-valid with respect to enumeration '[AST, PST, MST, CST, IET, CNT, AGT, BET, WET, CET, MET, EET, AZT, NET, AET, AWT, NST]'. It must be a value from the enumeration.");
                            timeZone = OHDUtilities.getTimeZoneFromOffSet(0.0);
                            break;
                    }

                    _runInfo.setTimeZone(timeZone);
                    _runInfo.setDaylightSavingObservatingTimeZone(_curValue.trim());
                    _logger.log(Logger.DEBUG,
                                "The time zone offset value in the run info xml file is " + _curValue.trim()
                                    + ". Accordingly, Time Zone is: " + timeZone.getID());
                }
                case OUTPUTDIAGNOSTICFILE:
                {
                    _runInfo.setDiagnosticFile(_curValue);
                    break;
                }
                case WORKDIR:
                {
                    _runInfo.setWorkDir(_curValue);
                    break;
                }
                case INPUTSTATEDESCRIPTIONFILE:
                {
                    _runInfo.setInputStateDescriptionFile(_curValue);
                    _runInfo.setInputStateDescriptionFileListList(_curValue);
                    break;
                }

                case OUTPUTSTATEDESCRIPTIONFILE:
                {
                    _runInfo.setOutputStateDescriptionFile(_curValue);
                    break;
                }
                case INPUTTIMESERIESFILE:
                {
                    _runInfo.setInputTimeSeriesFileList(_curValue);
                    break;
                }
                case OUTPUTTIMESERIESFILE:
                {
                    _runInfo.setOutputTimeSeriesFile(_curValue);
                    break;
                }
                case INPUTNETCDFFILE:
                {
                    _runInfo.setInputNetCdfFile(_curValue);
                    break;
                }

                case OUTPUTNETCDFFILE:
                {
                    _runInfo.setOutputNetCdfFile(_curValue);
                    break;
                }

                case INPUTPARAMETERFILE:
                {
                    _runInfo.setInputParameterFile(_curValue);
                    _runInfo.setInputParametersFileList(_curValue);
                    break;
                }
                case PROPERTIES:
                {
                    _readingInProperties = false;
                    _runInfo.setProperties(this._properties);
                    break;
                }
                case STRING:
                {
                    _properties.setProperty(_key, _value);
                    break;
                }
                case INT:
                {
                    _properties.setProperty(_key, _value);
                    break;
                }
                case DOUBLE:
                {
                    _properties.setProperty(_key, _value);
                    break;
                }
                default:
                    break;
            }
            _curValue = "";

        }
        catch(final IllegalArgumentException iae)
        {
            _curValue = "";
            _logger.log(Logger.WARNING, "'" + reader.getLocalName().trim()
                + "' has not been set in the run info object.");
        }
    }

    private DateTime getDateTime(final XMLStreamReader reader) throws SAXException
    {
        String date = null;
        String time = null;
        final int attributes = reader.getAttributeCount();
        //get the String representing date and time in attributes, format "yyyy-MM-dd HH:mm:ss"
        for(int i = 0; i < attributes; i++)
        {
            String attributeName = reader.getAttributeName(i).toString();
            if(attributeName != null)
            {
                attributeName = attributeName.trim();
            }

            if("date".equalsIgnoreCase(attributeName))
            {
                date = reader.getAttributeValue(i).toString().trim();
            }
            else if("time".equalsIgnoreCase(attributeName))
            {
                time = reader.getAttributeValue(i).toString().trim();
            }
        }

        DateTime dateTime;
        try
        {
            dateTime = new DateTime(date, time, _runInfo.getTimeZone());
        }
        catch(final Exception e)
        {
            throw new SAXException(e);
        }

        return dateTime;
    }
}
