package ohd.hseb.util.fews;

import java.io.BufferedWriter;
import java.io.FileInputStream;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStream;
import java.util.HashMap;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import java.util.TimeZone;
import java.util.TreeMap;

import ohd.hseb.util.Logger;
import ohd.hseb.util.fews.OHDUtilities.StateLocation;

public abstract class State implements IState
{
    private Map<String, Object> _statesMap;
    private String _name = null;
    private String _id = null;
    private TimeZone _timeZone;
    private long _dateTime;
    private String _comment;
    private String _version;
    private String _daylightSavingObservatingTimeZone;

    private Map<String, StateLocation> _stateLocation;

    public final static int HASHMAP = 0;
    public final static int TREEMAP = 1;
    public final static int LINKEDHASHMAP = 2;
    public final static int IDENTITYHASHMAP = 3;
    public final static int WEAKHASHMAP = 4;

    protected Logger _logger; //set by OHDFewsAdapter

    public State()
    {
        _statesMap = new HashMap<String, Object>(10);
    }

    /**
     * Parse the property format file and load the entries into {@link #_statesMap}. All *ModelState.java constructors
     * do not fill their states' values. It is this method that does from the input state file.<br>
     * 
     * @param stateFileName - the initial state file name, e.g. statesI.txt; 1)if it is a xml file(when doing carryover
     *            transfer, there are two "readLocation" in states.xml and one is previous params.xml), do not load the
     *            file; 2)if it is a NetCDF file(in GriddedFfg version one, the first grid states.xml "readLocation" is
     *            the RFC NetCDF template file name), do not load the state.
     */
    public void loadState(final String stateFileName, final Logger logger) throws IOException, Exception
    {
        //do not load a xml file
        if(OHDUtilities.isXmlFile(stateFileName))
        {
            logger.log(Logger.DEBUG, "The file[" + stateFileName
                + "] is a xml file, not inintial state text format file. Do not load.");

            return;
        }

        //do not load a NetCDF file
        if(OHDUtilities.isNetcdfFile(stateFileName))
        {
            logger.log(Logger.DEBUG, "The file[" + stateFileName
                + "] is a NetCDF file, not inintial state text format file. Do not load.");

            return;
        }

        //if reach here, must be statesI.txt 
        final Properties statesProp = new Properties();

        final InputStream readStateStream = new FileInputStream(stateFileName);
        statesProp.load(readStateStream);
        readStateStream.close();

        logger.log(Logger.DEBUG, "Loading the state from the file: " + stateFileName);

        //copy entries from states file to the states map
        for(final Map.Entry<Object, Object> value: statesProp.entrySet())
        {

            this.setStateValue(value.getKey().toString(), (value.getValue()));

            logger.log(Logger.DEBUG, value.toString());
        }

        extractValuesFromMap();

        logger.log(Logger.DEBUG, "Loaded the state from the file: " + stateFileName);

    }

    /**
     * Print out states based on the strings from {@link #getStringsFromState()} which can be overridden by individual
     * model state.
     */
    public void writeState(final String outputStateFileName, final Logger logger) throws Exception
    {
        final BufferedWriter outFile = new BufferedWriter(new FileWriter(outputStateFileName));

        try
        {
            //print out states in alphabetic order
            outFile.write(getStringsFromState());
        }
        finally
        {
            outFile.close();
        }

        logger.log(Logger.DEBUG, "Output the state to the file: " + outputStateFileName);
    }

    public Map<String, Object> getStateMap()
    {
        return _statesMap;
    }

    public void setStatesMap(final Map<String, Object> map)
    {
        _statesMap = map;
    }

    /**
     * This method is much more primitive: comparing to {@link #setStateValue(String, double)}, it does not round the
     * value if the string represents a double. This method can be used to insert a String line representing an array of
     * doubles or used in *OptFileConvert when parsing the deck cards.
     */
    public void insertStateAsString(final String tag, final String valueStr)
    {
        _statesMap.put(tag, valueStr);
    }

    /**
     * Set the state value to the new value, without numeric rounding.
     * 
     * @param stateName - the state String name
     * @param dValue - the state double value
     */
    protected void setStateValue(final String stateName, final Object value)
    {
        _statesMap.put(stateName, value);
    }

    /**
     * Set the state value to the new value, without numeric rounding.
     * 
     * @param stateName - the state String name
     * @param dValue - the state double value
     */
    protected void setStateValue(final String stateName, final double dValue)
    {
        this.insertStateAsString(stateName, String.valueOf(dValue));
    }

    /**
     * Set the state value to the int
     */
    protected void setStateValue(final String stateName, final int iValue)
    {
        this.insertStateAsString(stateName, String.valueOf(iValue));
    }

    /**
     * Get the state value as a String. This method is more primitive, compared to {@link #getDoubleDataState(String)}
     * and {@link #getIntDataState(String)}. Throws an Exception if the state does not exist.
     */
    public String getStateAsString(final String tag) throws Exception
    {
        if(isStateExisting(tag) == false)
        {
            throw new Exception("The state " + tag + " is not present.");
        }

        return (String)_statesMap.get(tag);
    }

    /**
     * returns the state value as double; if the state does not exist, throws an Exception
     */
    public double getDoubleDataState(final String tag) throws Exception
    {
        if(isStateExisting(tag) == false)
        {
            throw new Exception("The state " + tag + " is not present.");
        }

        return Double.parseDouble(_statesMap.get(tag).toString());
    }

    /**
     * The double array is stored as one line in {@link #_statesMap}, separated by one space. Convert the one line of
     * string into a double array and returns it.
     */
    protected double[] getDoubleArrayDataState(final String tag) throws Exception
    {
        return OHDUtilities.getDoubleArrayFromString(getStateAsString(tag));
    }

    /**
     * Returns the state value as int. Note: if the string stored in the map has a decimal point, this method will throw
     * run time IllegalFormatException, e.g. "0.0" will cause the problem, while "0" is correct.
     */
    protected int getIntDataState(final String tag) throws Exception
    {
        if(isStateExisting(tag) == false)
        {
            throw new Exception("The state " + tag + " is not present.");
        }

        return Integer.parseInt(_statesMap.get(tag).toString());
    }

    /**
     * returns true if the stateTag exists in _statesMap; otherwise, returns false.
     * 
     * @param stateTag - state tag
     */
    protected boolean isStateExisting(final String stateTag)
    {
        if(_statesMap.containsKey(stateTag))
        {
            return true;
        }

        return false;
    }

    protected void deleteState(final String tag)
    {
        _statesMap.remove(tag);
    }

    public String getName()
    {
        return _name;
    }

    public void setName(final String name)
    {
        _name = name;
    }

    public String getId()
    {
        return _id;
    }

    public void setId(final String id)
    {
        _id = id;
    }

    public void setTimeZone(final TimeZone timeZone)
    {
        _timeZone = timeZone;
    }

    public TimeZone getTimeZone()
    {
        return _timeZone;
    }

    public void setDaylightSavingObservatingTimeZone(final String daylightSavingObservatingTimeZone)
    {
        _daylightSavingObservatingTimeZone = daylightSavingObservatingTimeZone;
    }

    public String getDaylightSavingObservatingTimeZone()
    {
        return _daylightSavingObservatingTimeZone;
    }

    public void setDateTime(final long dateTime)
    {
        _dateTime = dateTime;
    }

    public long getDateTime()
    {
        return _dateTime;
    }

    public void setComment(final String comment)
    {
        _comment = comment;
    }

    public String getComment()
    {
        return _comment;
    }

    /**
     * Gets the value of the version property.
     * 
     * @return possible object is {@link String }
     */
    public String getVersion()
    {
        if(_version == null)
        {
            return "1.2";
        }
        return _version;
    }

    /**
     * Sets the value of the version property.
     * 
     * @param value allowed object is {@link String }
     */
    public void setVersion(final String value)
    {
        this._version = value;
    }

    /**
     * This method confirm that "UNIT=METRIC" is explicitly present in the statesI.txt. If not, or something else, like
     * "UNIT=ENGLISH", throws an Exception.
     */
    protected void confirmUnitMetric() throws Exception
    {
        this.confirmUnit(OHDConstants.UNIT_METRIC);
    }

    /**
     * This method confirm the states is in ENGLISH unit. If found in METRIC unit, throws an Exception.
     */
    protected void confirmUnitEnglish() throws Exception
    {
        this.confirmUnit(OHDConstants.UNIT_ENGLISH);
    }

    private void confirmUnit(final String unitStr) throws Exception
    {
        if(isStateExisting(OHDConstants.UNIT_TAG) == false)
        {
            throw new Exception("Error: states unit is not present.");
        }

        if(getStateAsString(OHDConstants.UNIT_TAG).equals(unitStr))
        {//in correct unit

            _logger.log(Logger.DEBUG, "The states are in unit " + unitStr);

            return;
        }

        //if reach here, wrong or un-recognized unit
        throw new Exception("Error: states is not allowed in unit " + getStateAsString(OHDConstants.UNIT_TAG));
    }

    /**
     * Gets the value of the StateLocation property.
     * <p>
     * This accessor method returns a reference to the live map, not a snapshot. Therefore any modification you make to
     * the returned map will be present inside the State object.
     * <p>
     * Objects of the following type(s) are allowed in the list {@link StateLocation }
     */
    public Map<String, StateLocation> getStateLocation()
    {
        if(_stateLocation == null)
        {
            _stateLocation = new HashMap<String, StateLocation>();
        }
        return this._stateLocation;
    }

    public void setStateLocation(final StateLocation value)
    {
        if(_stateLocation == null)
        {
            _stateLocation = new HashMap<String, StateLocation>();
        }
        if(value != null)
        {
            final String pathFileName = value.getReadLocation();
            final int fileSeparatorIndex = pathFileName.lastIndexOf(System.getProperty("file.separator"));
            final String fileName = pathFileName.substring(fileSeparatorIndex + 1);
            this._stateLocation.put(fileName, value);
        }
    }

    /**
     * Sorting a string integer format separated by a prompt sign '#' in alphabet order
     * 
     * @param comparator - string integer comparator
     * @param stateMap - Map<String, String> for state model
     * @return A string integer format separated by a prompt sign '#' in alphabet order
     */
    public String getStringsFromState(final StringIntegerComparator comparator, final Map<String, Object> stateMap)
    {
        final StringBuilder resultStr = new StringBuilder();

        // print out states in alphabetic order 
        final TreeMap<String, Object> treeMap = new TreeMap<String, Object>(comparator);

        treeMap.putAll(stateMap);

        final Set<Map.Entry<String, Object>> mySet = treeMap.entrySet();

        for(final Map.Entry<String, Object> curEntry: mySet)
        {
            resultStr.append(curEntry.getKey()).append("=").append(curEntry.getValue()).append(OHDConstants.NEW_LINE);
        }

        return resultStr.toString();
    } // end method

    public void setLogger(final Logger logger)
    {
        _logger = logger;
    }

    /**
     * The result depends on the method {@link #getStringsFromState()}.
     */
    @Override
    public String toString()
    {
        final StringBuilder resultStr = new StringBuilder();

        resultStr.append("States:").append(OHDConstants.NEW_LINE);
        resultStr.append(getStringsFromState());

        //the following lines are not outputed in writeState()
        resultStr.append("name= ").append(_name).append(OHDConstants.NEW_LINE);
        resultStr.append("id= ").append(_id).append(OHDConstants.NEW_LINE);

        return resultStr.toString();
    }

    /**
     * Print out state variables in alphabetic order in the property format, e.g. "state_name= value". The values are
     * from {@link #_statesMap}. So it reflects the original values from statesI.txt, not the current values which have
     * been modified. This method could be overridden by individual subclasses to reflect the current values.
     */
    protected String getStringsFromState()
    {
        final StringBuilder resultStr = new StringBuilder();

        // print out states in alphabetic order and the int order 
        final TreeMap<String, Object> treeMap = new TreeMap<String, Object>(new StringIntegerComparator(true));
        treeMap.putAll(getStateMap());

        final Set<Map.Entry<String, Object>> mySet = treeMap.entrySet();

        for(final Map.Entry<String, Object> curEntry: mySet)
        {
            resultStr.append(curEntry.getKey()).append("=").append(curEntry.getValue()).append(OHDConstants.NEW_LINE);
        }

        return resultStr.toString();
    }

    /**
     * For Java models(snow17, sacsma, and sacsmaHT), this method can be implemented to extract values from _statesMap
     * to instance variables to avoid frequently access the {@link #_statesMap}. In the meantime, it is important to
     * overwrite {@link #getStringsFromState()} to have correct statesO.txt since the values in _statesMap are not
     * updated.
     * <p>
     * This method is called in {@link #loadState(String, Logger)}.
     */
    public void extractValuesFromMap() throws Exception
    {
        //empty
    }

}
