package ohd.hseb.util.fews;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.TimeZone;

import javax.xml.stream.XMLStreamReader;

import ohd.hseb.time.DateTime;
import ohd.hseb.util.Logger;
import ohd.hseb.util.fews.OHDUtilities.StateLocation;

import org.xml.sax.SAXException;

/**
 * Used in {@link FewsXMLParser} to parse the state meta file(states.xml):
 * <p>
 * 1)get the input state file name, then call {@link State#loadState(String, Logger)} to load the state {@link #_state}
 * <br>
 * 2)get dateTime(long), stateId(String) and stateName(String) for {@link #_state}. dateTime value(long in GMT time) is
 * very important. stateId and stateName do not matter at all.
 * <p>
 * <timeZone> element is optional in states.xml. If absent, the time zone by default is GMT. If present, the time has to
 * be converted to GMT to be stored in dateTime in {@link #_state}.
 * 
 * @author FewsPilot Team
 */
final public class StateMetaFileHandler extends FewsXmlHandler
{

    private final IState _state;

    private final SimpleDateFormat _dateFmt;
    private StateLocation _stateLocation = null;
    private int _countLocations;
    private boolean _stateLocationWithXML = false; //for carryover transfer case
    private String _previousType;

    private String _writeLocation;

    /**
     * OHDFewsadapter use this constructor
     */
    public StateMetaFileHandler(final IState state, final Logger logger)
    {
        _state = state;
        _dateFmt = new SimpleDateFormat(OHDConstants.DATE_TIME_FORMAT_STR);
        _dateFmt.setTimeZone(OHDConstants.GMT_TIMEZONE);
        //default is GMT tz; but will reset to other tz if <timeZone> element exists in the states.xml file

        _logger = logger;
    }

    /**
     * @param current
     * @param reader
     */
    public void charElement(final String current, final XMLStreamReader reader)
    {
        _curValue = reader.getText();

    } //close method    

    /**
     * @throws SAXException
     */
    @Override
    protected void endDocument() throws SAXException
    {

        if(_countLocations > 2)
        {
            final SAXException spe = new SAXException("The number of 'State Location' is bigger than Two, only a maximum of Two State Locations is allowed");
            throw spe;
        }
        else
        {
            if(_countLocations > 1 && !_stateLocationWithXML)
            {
                final SAXException spe = new SAXException("The State Location must contain one readLocation element with an XML file reference");
                throw spe;
            }
        }
    }

    /**
     * @param reader
     * @throws Exception
     */
    @Override
    public void startElement(final XMLStreamReader reader) throws Exception
    {
        final int numAttributes = reader.getAttributeCount();
        final String elementName = reader.getLocalName().trim();
        if("State".equals(elementName))
        {
            _countLocations = 0;

            //going through each attributes of "State" element, find the attribute with name of "version", retrieve its value and set _state's version
            for(int i = 0; i < numAttributes; i++)
            {
                if("version".equals(reader.getAttributeName(i).toString()))
                {
                    _state.setVersion(reader.getAttributeValue(i).toString().trim());

                    break;
                }
            }

        }
        else if("dateTime".equals(elementName))
        {
            String dateStr = "", timeStr = "";
            for(int i = 0; i < numAttributes; i++)
            {
                final String attributeName = reader.getAttributeName(i).toString();
                if("date".equals(attributeName))
                {
                    dateStr = reader.getAttributeValue(i).toString().trim();
                }
                if("time".equalsIgnoreCase(attributeName))
                {
                    timeStr = reader.getAttributeValue(i).toString().trim();
                }
            }

            try
            {
                //<timeZone> element shows up before <dateTime> element, so _dateFmt TZ might have been reset already               
                _state.setDateTime((_dateFmt.parse(dateStr + " " + timeStr)).getTime());
            }
            catch(final ParseException e)
            {
                _logger.log(Logger.ERROR, "Couldn't parse states xml dateTime element. date=" + dateStr + " time="
                    + timeStr);
                throw new Exception();
            }

            _logger.log(Logger.DEBUG,
                        "Initial State Time is "
                            + DateTime.getDateTimeStringFromLong(_state.getDateTime(), _dateFmt.getTimeZone()));

        }
        else if("stateLoc".equals(elementName))
        {
            _countLocations++;
            _stateLocation = new StateLocation();
            if("type".equals(reader.getAttributeName(0).toString()))
            {
                _stateLocation.setType(reader.getAttributeValue(0).toString().trim());
            }
        }
    }

    /**
     * @param reader
     */
    @Override
    public void endElement(final XMLStreamReader reader) throws SAXException
    {
        final String s = _curValue.toString();
        final String elementName = reader.getLocalName().trim();

        if("stateId".equals(elementName))
        {
            _state.setId(s);
        }
        else if("stateName".equals(elementName))
        {
            _state.setName(s);
        }
        else if("readLocation".equals(elementName))
        {

            if(_countLocations == 1)
            {
                _stateLocation.setReadLocation(s);
                _previousType = s;
            }
            else
            {
                final String type1 = s.toLowerCase().contains("xml") ? "xml" : "txt";
                final String type2 = _previousType.toLowerCase().contains("xml") ? "xml" : "txt";

                if(!type1.equals(type2))
                {
                    _previousType = s;
                    _stateLocation.setReadLocation(s);
                }
                else
                {
                    String message = "The State Location does not contain a XML type file, only: ";
                    if(type1.equals("xml"))
                    {
                        message = "The State Location contains two readLocation elements with the same type file: ";
                    }
                    final SAXException spe = new SAXException(message + type1);
                    throw spe;
                }
            }

            if(s.toLowerCase().contains("xml"))
            {
                _stateLocationWithXML = true;
            }

            if(_state != null)
            {
                try
                {
                    _state.loadState(_stateLocation.getReadLocation(), _logger);
                }
                catch(final Exception e)
                {
                    throw new SAXException(e.getMessage());
                }
            }
        }
        else if("writeLocation".equals(elementName))
        {
            _stateLocation.setWriteLocation(s);

            if(!s.toLowerCase().contains("xml"))
            {//if entering this block, assuming it is .txt file, since it is not .xml file(assuming the writeLocation file can be only .txt, e.g. statesO.txt or parameter xml file)

                if(_writeLocation != null)
                {//both _writeLocation files are not .xml file, strange
                    _logger.log(Logger.ERROR, "The output state file has already been set to " + _writeLocation
                        + ". Now it will be re-set to " + s);
                }

                _writeLocation = s;
            }
        }
        else if("timeZone".equals(elementName))
        { // reset _dateFmt time zone if there is <timeZone> element; otherwise, by default it is GMT

            final TimeZone timeZone = OHDUtilities.getTimeZoneFromOffSet(Double.parseDouble(_curValue.toString()));

            _dateFmt.setTimeZone(timeZone); // reset _dateFmt time zone based on the <timeZone> value

            _state.setTimeZone(timeZone);

            _logger.log(Logger.DEBUG, "Time Zone in states.xml is: " + timeZone.getID());
        }
        else if("daylightSavingObservingTimeZone".equals(elementName))
        {
            TimeZone timeZone = null;
            switch(_curValue.trim())
            {
                case "AST":
                    timeZone = OHDUtilities.getTimeZoneFromOffSet(-9.0);
                    break;
                case "PST":
                    timeZone = OHDUtilities.getTimeZoneFromOffSet(-8.0);
                    break;
                case "MST":
                    timeZone = OHDUtilities.getTimeZoneFromOffSet(-7.0);
                    break;
                case "CST":
                    timeZone = OHDUtilities.getTimeZoneFromOffSet(-6.0);
                    break;
                case "IET":
                    timeZone = OHDUtilities.getTimeZoneFromOffSet(-5.0);
                    break;
                case "CNT":
                    timeZone = OHDUtilities.getTimeZoneFromOffSet(-3.5);
                    break;
                case "AGT":
                    timeZone = OHDUtilities.getTimeZoneFromOffSet(-3.0);
                    break;
                case "BET":
                    timeZone = OHDUtilities.getTimeZoneFromOffSet(-3.0);
                    break;
                case "WET":
                    timeZone = OHDUtilities.getTimeZoneFromOffSet(0.0);
                    break;
                case "CET":
                    timeZone = OHDUtilities.getTimeZoneFromOffSet(1.0);
                    break;
                case "MET":
                    timeZone = OHDUtilities.getTimeZoneFromOffSet(1.0);
                    break;
                case "EET":
                    timeZone = OHDUtilities.getTimeZoneFromOffSet(2.0);
                    break;
                case "AZT":
                    timeZone = OHDUtilities.getTimeZoneFromOffSet(4.0);
                    break;
                case "NET":
                    timeZone = OHDUtilities.getTimeZoneFromOffSet(4.0);
                    break;
                case "AET":
                    timeZone = OHDUtilities.getTimeZoneFromOffSet(10.0);
                    break;
                case "AWT":
                    timeZone = OHDUtilities.getTimeZoneFromOffSet(8.0);
                    break;
                case "NST":
                    timeZone = OHDUtilities.getTimeZoneFromOffSet(12.0);
                    break;
                default:
                    _logger.log(Logger.ERROR,
                                "Value '"
                                    + _curValue.trim()
                                    + "' is not facet-valid with respect to enumeration '[AST, PST, MST, CST, IET, CNT, AGT, BET, WET, CET, MET, EET, AZT, NET, AET, AWT, NST]'. It must be a value from the enumeration.");
                    timeZone = OHDUtilities.getTimeZoneFromOffSet(0.0);
                    break;
            }
            _dateFmt.setTimeZone(timeZone); // reset _dateFmt time zone based on the <timeZone> value
            //_logger.log(Logger.DEBUG, "Time Zone in states.xml is: " + timeZone.getID());
            _state.setTimeZone(timeZone);
            _logger.log(Logger.DEBUG, "The time zone offset value in the state xml file is " + _curValue.trim()
                + ". Accordingly, Time Zone is: " + timeZone.getID());
        }
        else if("stateLoc".equals(elementName))
        {
            _state.setStateLocation(_stateLocation);
        }

        // reset buffer for next element.
        _curValue = "";

    }

    /**
     * This methods work, because only one state file name (name.txt) is saved in the states.xml file. If more than one
     * state filename is going to exist this method will need to be replaced and the logic will change as we will need
     * to find a way to get the correct state name.
     */
    public String getOutputStateFileName()
    {
        return _writeLocation;
    }

    /**
     * Right now, the only usage is for Gridded FFG: <input> is the rfc template netcdf file.
     */
    public String getInputStateFileName()
    {
        return _stateLocation.getReadLocation();
    }

}
