package ohd.hseb.util.fews;

import java.io.BufferedInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.TimeZone;
import java.util.TreeMap;

import javax.xml.stream.XMLStreamReader;

import ohd.hseb.measurement.Measurement;
import ohd.hseb.measurement.MeasuringUnit;
import ohd.hseb.measurement.RegularTimeSeries;
import ohd.hseb.time.DateTime;
import ohd.hseb.util.Logger;
import ohd.hseb.util.fews.ohdmodels.ModelDriver;
import ohd.hseb.util.io.EndianConvertingInputStream;

import org.xml.sax.SAXException;

/**
 * This file is used to extract RegularTimeSeries(s) from the fews xml files which follow pi_timeseries.xsd schema. All
 * the extracted RTS are located in _tsList. Later, this class also parse the input MOD fews xml file, which put the
 * result RTS in {@link ModelDriver#getInputModsList()}. Fews xml file may be in non-GMT time zone, but the resulted RTS
 * are in GMT time. The resulted RTS(s) are actually FewsRegularTimeSeries.
 * 
 * @author FewsPilot Team
 */
final public class TimeSeriesHandler extends FewsXmlHandler
{

    //units defined by FEWS
    private final static String SECOND = "second";
    private final static String MINUTE = "minute";
    private final static String HOUR = "hour";
    private final static String DAY = "day";
    private final static String WEEK = "week";
    private final static String YEAR = "year";
    private final static String NONEQUIDISTANT = "nonequidistant";

    private boolean _areMissingValuesAlwaysAllowed = false; //this value is set by user and this overrides the true/false from nwsrfs_datatype_mapping_file.txt
    private boolean _paramMisssingValueAllowed = false; //from nwsrfs_datatype_mapping_file.txt and based on parameterId type

    /**
     * _fewsToLocalUnitMap represents a mapping of units used by FEWS to those used by OHD's code
     */
    private final static Map<String, MeasuringUnit> _fewsToLocalUnitMap = new TreeMap<String, MeasuringUnit>(String.CASE_INSENSITIVE_ORDER);
    //key is case-insensitive, e.g. mm, MM, mM all are mm unit

    private final List<RegularTimeSeries> _tsList;
    private FewsRegularTimeSeries _fewsRTS;

    /* ----------The information from <header> portion of fews xml file ----------- */
    //For RTS:
    private FEWS_RTS_TYPE _type;
    private String _locationId;
    private List<String> _qualifierIdList = null;
    private String _parameterId; //corresponds to RTS _timeSeriesType
    private String _ensembleId = RegularTimeSeries.NO_ENSEMBLE_ID;
    private int _ensembleMemberIndex = 0;
    private int _interval;
    private long _startDateAndTime, _endDateAndTime;
    private double _missingValInHeader; //stores the missing value in the header of xml file, but the result RegularTimeSeries obj's _missingValue is always -999.0(OHDConstants.MISSING_DATA)
    private String _stationName; //corresponds to RTS _name
    private MeasuringUnit _eventUnit; //used to set RTS _measuringUnit and each Measurement obj unit

    //For FewsRTS:
    private String _longName = "";
    private String _sourceOrganization = "";
    private String _sourceSystem = "";
    private String _fileDescription = "";
    private String _creationDate = null; //optional, may not be present
    private String _creationTime = null; //optional, may not be present
    private List<TimeSeriesThreshold> _hlThresholdsList = null;
    private Double _x = null; //optional, may not be present
    private Double _y = null; //optional, may not be present
    private Double _z = null; //optional, may not be present
    private Float _latitude = null; //optional, may not be present
    private Float _longitude = null; //optional, may not be present
    private String _version = null;
    private String _timeStepTimes = null;

    //hold everything separately rather than providing default constructors in other classes such as measurement.
    private double _eventValue;
    private final boolean _eventFlag = true; //we don't use it at all

    private int _eventNum = 0;
    private long _eventDateTime;

    private final SimpleDateFormat _dateFmt;
    private String _dateStr = "", _timeStr = "";
    private String _eventComment = "";

    /**
     * Used for reading from a binary file. If it is null, either binary reading was not set by the caller or a binary
     * file does not exist.
     */
    private EndianConvertingInputStream _binaryInputStream = null;

    /**
     * Stores the initial state time, which is used to determine what values to check for missing.
     */
    private Calendar _initStateTimeCal = null;

    /**
     * Stores the time zone read in xml file
     */
    private TimeZone _timeZone;

    static
    {
        populateFewsToLocalUnitMap();
    }

    /**
     * By calling this constructor, it is assumed binary file reading is to be enabled if a .bin file exists.
     * 
     * @param tsList The time series list to populate.
     * @param xmlFileName The name of the XML file that is going to be read in.
     * @param logger The logger to use.
     */
    public TimeSeriesHandler(final List<RegularTimeSeries> tsList, final String xmlFileName, final Logger logger)
    {
        this(tsList, logger);
        initiateForBinaryFileReadingIfAvailable(new File(xmlFileName));
    }

    /**
     * Initializes the TimeSeriesHandler, without binary reading. Call initiate method at a later time if binary reading
     * is to be enabled.
     * 
     * @param tsList The time series list to populate.
     * @param logger The logger to use.
     */
    public TimeSeriesHandler(final List<RegularTimeSeries> tsList, final Logger logger)
    {
        _tsList = tsList; //alias, _tsList will be filled, then tsList pass-by-reference back 
        _logger = logger;

        _dateFmt = new SimpleDateFormat(OHDConstants.DATE_TIME_FORMAT_STR);

        /*
         * GMT is the default TZ; During parsing, if there is <timeZone>..</timeZone> in the fews RTS xml file, re-set
         * to that TZ.
         */
        _dateFmt.setTimeZone(OHDConstants.GMT_TIMEZONE);
    }

    /**
     * If the input 'xmlFile' is a human readable file with extension '.xml', this method only sets _binaryInputStream
     * as null and doesn't do anything else. If the input 'xmlFile' is just a short human readble xml file only
     * containing the header(the real content of TSs is in the corresponding binary file). There are two kinds of binary
     * files: FASTINFO and another kind non-FASTINFO binary file. This method works fine with both binary cases:
     * _binaryInputStream will be set.
     * 
     * @param xmlFile
     */
    public void initiateForBinaryFileReadingIfAvailable(final File xmlFile)
    {
        _binaryInputStream = null;

        String binFileName = xmlFile.getAbsolutePath();
        if(binFileName.endsWith(".xml"))
        {
            binFileName = binFileName.substring(0, binFileName.length() - 4) + ".bin";
        }
        else if(binFileName.endsWith(".fi"))
        {
            binFileName = binFileName.substring(0, binFileName.length() - 3) + ".bin";
        }
        else
        {
            throw new RuntimeException("The time series file[" + xmlFile + "] is neither xml file nor FASTINFO file!");
        }

        final File binFile = new File(binFileName);//this file with extension '.bin' may or may not exist. It's a guess. See below

        try
        {
            _binaryInputStream = new EndianConvertingInputStream(new BufferedInputStream(new FileInputStream(binFile)));
            _binaryInputStream.setEndianFlag(EndianConvertingInputStream.LITTLE_ENDIAN_DATA_FLAG);
        }
        catch(final FileNotFoundException e)
        {
            /*
             * Apparently, no file exists with the guessed file name 'xxx.bin' -- in this case we think 'xxx.xml' (the
             * original input parameter file name) is the file containing time sereis data here, leaving the method with
             * _binaryInputStream as null
             */
            return;
        }
    }

    private void readTimeSeriesDataFromBinary() throws SAXException
    {
        if(_binaryInputStream == null)
        {
            return;
        }

        try
        {
            for(int i = 0; i < _fewsRTS.getMeasurementCount(); i++)
            {
                double value = _binaryInputStream.readFloatSwap();
                value = getNewEventValueByCheckingMissingValue(_fewsRTS.getMeasurementTimeByIndex(i), value);
                _fewsRTS.setMeasurementByIndex(value, i);
            }
        }
        catch(final IOException ioe)
        {
            throw new SAXException("IOException occurred before reading enough floats from binary file: "
                + ioe.getMessage());
        }
    }

    /**
     * Close {@link #_binaryInputStream} if it is open.
     */
    @Override
    protected void endDocument() throws IOException
    {
        if(_binaryInputStream != null)
        {
            _binaryInputStream.close();
        }
        if(_version != null)
        {
            _fewsRTS.setVersion(_version);
        }
    }

    @Override
    public void startElement(final XMLStreamReader reader) throws Exception
    {
        _curValue = "";

        final String elementName = reader.getLocalName().trim();

        if("TimeSeries".equals(elementName))
        {
            final int attributes = reader.getAttributeCount();
            //get the String representing version
            for(int i = 0; i < attributes; i++)
            {
                String attributeName = reader.getAttributeName(i).toString();
                if(attributeName != null)
                {
                    attributeName = attributeName.trim();
                }

                if("version".equalsIgnoreCase(attributeName))
                {
                    _version = reader.getAttributeValue(i).toString().trim();
                }
            }
        }

        if("series".equals(elementName))
        {
            _qualifierIdList = new ArrayList<String>();
        }

        if("startDate".equals(elementName) || "endDate".equals(elementName))
        {

            final int attributes = reader.getAttributeCount();
            //get the String representing date and time in attributes, format "yyyy-MM-dd HH:mm:ss"
            for(int i = 0; i < attributes; i++)
            {
                String attributeName = reader.getAttributeName(i).toString();
                if(attributeName != null)
                {
                    attributeName = attributeName.trim();
                }

                if("date".equalsIgnoreCase(attributeName))
                {
                    _dateStr = reader.getAttributeValue(i).toString().trim();
                }
                if("time".equalsIgnoreCase(attributeName))
                {
                    _timeStr = reader.getAttributeValue(i).toString().trim();
                }
            }

            /* --set _fewsRTS start time or end time -- */
            try
            {
                final long timeLong = _dateFmt.parse(_dateStr + " " + _timeStr).getTime();
                //even though fews.xml could be in non-GMT timezone, _dateFmt is in that tz, here, timeLong is GMT

                if("startDate".equalsIgnoreCase(elementName))
                {
                    _startDateAndTime = timeLong; //GMT time
                }
                else if("endDate".equalsIgnoreCase(elementName))
                {
                    _endDateAndTime = timeLong; //GMT time
                }
            }
            catch(final ParseException e)
            {
                _logger.log(Logger.ERROR, "Couldn't parse time series startDate or endDate elements.");

                throw new Exception();
            }

        }
        else if("event".equals(elementName))
        {
            final int attributes = reader.getAttributeCount();
            for(int i = 0; i < attributes; i++)
            {
                String attributeName = reader.getAttributeName(i).toString();
                if(attributeName != null)
                {
                    attributeName = attributeName.trim();
                }
                if("value".equals(attributeName))
                {
                    _eventValue = Double.parseDouble(reader.getAttributeValue(i).toString().trim());
                }
                else if("time".equals(attributeName))
                {
                    _timeStr = reader.getAttributeValue(i).toString().trim();
                }
                else if("date".equals(attributeName))
                {
                    _dateStr = reader.getAttributeValue(i).toString().trim();
                }
                else if("comment".equals(attributeName))
                {
                    _eventComment = reader.getAttributeValue(i).toString().trim();
                }
            }

            /* --------------get event time in long ------------- */
            try
            {
                _eventDateTime = _dateFmt.parse(_dateStr + " " + _timeStr).getTime();
                //even though fews.xml could be in non-GMT timezone, _dateFmt is in that tz, here, timeLong is GMT
            }
            catch(final ParseException e)
            {
                _logger.log(Logger.ERROR, "Couldn't parse time series event date or time. date=" + _dateStr + " time="
                    + _timeStr);

                throw new Exception();
            }
        }
        else if("timeStep".equals(elementName))
        {//unlike other variables, _interval is obtained in startElement()

            String timeStepUnit = "";
            int timeStepDivider = 1;
            int timeStepMultiplier = 0;
            String timeStepTimes = "";

            final int attributes = reader.getAttributeCount();
            //conflict in schema vs. example, check and code for..
            for(int i = 0; i < attributes; i++)
            {
                String attributeName = reader.getAttributeName(i).toString();
                if(attributeName != null)
                {
                    attributeName = attributeName.trim();
                }
                if("unit".equals(attributeName))
                {
                    timeStepUnit = reader.getAttributeValue(i).toString().trim();
                }
                else if("divider".equals(attributeName))
                {
                    timeStepDivider = Integer.parseInt(reader.getAttributeValue(i).toString().trim());
                }
                else if("multiplier".equals(attributeName))
                {
                    timeStepMultiplier = Integer.parseInt(reader.getAttributeValue(i).toString().trim());
                }
                else if("times".equals(attributeName))
                {
                    timeStepTimes = reader.getAttributeValue(i).toString().trim();
                    _timeStepTimes = timeStepTimes;

                }

            } //close for loop

            _interval = getTimeStepInHours(timeStepUnit, timeStepMultiplier, timeStepDivider, timeStepTimes);
            //           System.out.println("End Get Time Step: " + _interval);
        } //close else if 

        if("thresholds".equals(elementName))
        {
            _hlThresholdsList = new ArrayList<TimeSeriesThreshold>();
        }

        if("highLevelThreshold".equals(elementName))
        {
            final TimeSeriesThreshold threshold = new TimeSeriesThreshold(_eventUnit);
            final int attributes = reader.getAttributeCount();
            for(int i = 0; i < attributes; i++)
            {
                String attributeName = reader.getAttributeName(i).toString();
                if(attributeName != null)
                {
                    attributeName = attributeName.trim();
                }
                if("id".equals(attributeName))
                {
                    threshold.setId(reader.getAttributeValue(i).toString().trim());
                }
                else if("name".equals(attributeName))
                {
                    threshold.setName(reader.getAttributeValue(i).toString().trim());
                }
                else if("value".equals(attributeName))
                {
                    threshold.setValue(Float.parseFloat(reader.getAttributeValue(i).toString().trim()));
                }
            }
            _hlThresholdsList.add(threshold);
        }

    } //close method

    @Override
    public void endElement(final XMLStreamReader reader) throws SAXException
    {
        final String elementName = reader.getLocalName().trim();
        if("header".equals(elementName))
        {
            //       	System.out.println("Header End Element: " + elementName);
            _fewsRTS = new FewsRegularTimeSeries(_startDateAndTime, _endDateAndTime, _interval, _eventUnit);
            //fill in other values

            _fewsRTS.setLocationId(_locationId);
            //we may not always be sent the qualifier Id
//            if(_qualifierId == null)
//            {
//                _fewsRTS.setQualifierId(_locationId);
//            }
//            else
//            {
//                _fewsRTS.setQualifierId(_qualifierId);                
//            }

            //dataType = this following field
            _fewsRTS.setTimeSeriesType(_parameterId);

            // make sure timeseriesType in FEWS xml file matches our (NWSRFS) time code

            FEWS_RTS_TYPE expectedType;
            try
            {
                expectedType = FewsRegularTimeSeries.getFewsTimeCode(NwsrfsDataTypeMappingReader.getTimeCode(_parameterId,
                                                                                                             _logger));
            }
            catch(final Exception e)
            {
                throw new SAXException(e.getMessage());
            }

            final FEWS_RTS_TYPE actualType = _type;

            if(expectedType != actualType)
            {
                throw new SAXException("The time series  for location id: " + _locationId + " and data type "
                    + _parameterId + " has a time code of " + actualType
                    + " this does not match the NWSRFS time code of " + expectedType + ".");
            }

            _fewsRTS.setType(_type);
            _fewsRTS.setMissingValue(OHDConstants.MISSING_DATA); //always use -999.0(OHDConstants.MISSING_DATA)
            _fewsRTS.setName(_stationName);

            _fewsRTS.setLongName(_longName);
            _fewsRTS.setSourceOrganization(_sourceOrganization);
            _fewsRTS.setSourceSystem(_sourceSystem);
            _fewsRTS.setFileDescription(_fileDescription);

            _fewsRTS.setIntervalTimeStepTimes(_timeStepTimes);

            if(_x != null)
            {
                _fewsRTS.setX(_x);
            }
            if(_y != null)
            {
                _fewsRTS.setY(_y);
            }
            if(_z != null)
            {
                _fewsRTS.setZ(_z);
            }
            if(_latitude != null)
            {
                _fewsRTS.setLatitude(_latitude);
            }
            if(_longitude != null)
            {
                _fewsRTS.setLongitude(_longitude);
            }

            if(_creationDate != null && _creationTime != null)
            { //the input time series does have <creationDate> and <creationTime>
                try
                {
                    _fewsRTS.setCreationDateTime(new DateTime(_creationDate, _creationTime));
                }
                catch(final Exception e)
                {
                    throw new SAXException(e.getMessage());
                }
            }

            _fewsRTS.setEnsembleId(_ensembleId);
            _fewsRTS.setEnsembleMemberIndex(_ensembleMemberIndex);

            // Get true or false if missing value from parameter Id is in Mapping file.
            try
            {
                _paramMisssingValueAllowed = NwsrfsDataTypeMappingReader.areMissingValuesAllowed(_parameterId, _logger);
            }
            catch(final Exception e)
            {
                throw new SAXException(e.getMessage());
            }

            //Read in the data from binary if it exists.
            if(_binaryInputStream != null)
            {
                this.readTimeSeriesDataFromBinary();
            }

            _fewsRTS.setTimeSeriesThresholdsList(_hlThresholdsList);

        }
        else if("event".equals(elementName))
        {

            //check if this is missing and missing not allowed for this datatype (timeseriestype above)
            _eventValue = getNewEventValueByCheckingMissingValue(_fewsRTS.getMeasurementTimeByIndex(_eventNum),
                                                                 _eventValue);

            // Use MeasurementByTime as we don't know if we will have equidistant input time series.
            _fewsRTS.setMeasurementByTime(new Measurement(_eventValue, _eventUnit, _eventFlag, _eventComment),
                                          _eventDateTime);
            _eventNum++;
        }
        else if("series".equals(elementName))
        {//push series on to list

            if(_qualifierIdList != null & _qualifierIdList.size() > 0)
            {
                _fewsRTS.setQualifierIds(_qualifierIdList);
            }
            else
            {
                //we may not always be sent the qualifier Id
                //_fewsRTS.setQualifierId(_locationId);    		                
                _qualifierIdList.add(_locationId);
                _fewsRTS.setQualifierIds(_qualifierIdList);
            }

            _tsList.add(_fewsRTS);
            _logger.log(Logger.DEBUG, "finished extracting time series " + _fewsRTS.getTimeSeriesType());

            _eventNum = 0; //reset to 0 for next RTS
        }
        else if("type".equals(elementName))
        {
            try
            {
                _type = FEWS_RTS_TYPE.convertFromString(_curValue);
            }
            catch(final Exception e)
            {
                throw new SAXException(e.getMessage());
            }
        }
        else if("parameterId".equals(elementName))
        {
            _parameterId = _curValue;
        }

        else if("qualifierId".equals(elementName))
        {
            //even though FEWS' schema allows for multiple qualifier ids - we are  only going to hold a single one
            //so we'll read the first one and ignore subsequent ones
//            if(_qualifierId == null)
//            {
//                _qualifierId = _curValue;                
//            }
            _qualifierIdList.add(_curValue);
        }
        //If ensembleId is present, the member index must be present.  That requirement should be caught
        //when the time series XML is validated.
        else if("ensembleId".equals(elementName))
        {
            _ensembleId = _curValue;
        }
        else if("ensembleMemberIndex".equals(elementName))
        {
            try
            {
                _ensembleMemberIndex = Integer.parseInt(_curValue);
            }
            catch(final NumberFormatException nfe)
            {
                throw new SAXException("ensmebleMemberIndex is not a number: '" + _curValue + "'.");
            }
        }
        else if("locationId".equals(elementName))
        {
            _locationId = _curValue;
        }
        else if("longName".equals(elementName))
        {
            _longName = _curValue;
        }
        else if("stationName".equals(elementName))
        {
            _stationName = _curValue;
        }
        else if("units".equals(elementName))
        {
            _eventUnit = TimeSeriesHandler.retrieveMeasuringUnitForFEWSUnit(_curValue);

            if(_eventUnit == null)
            {
                throw new SAXException("The unit from the input file " + _curValue + " is not recognized.");
            }

            // make sure units in FEWS xml are of correct type (L, L/T, L3, etc)
            String expectedUnitType = "";
            try
            {
                expectedUnitType = NwsrfsDataTypeMappingReader.getNwsrfsDim(_parameterId, _logger);//getTimeCode(_parameterId,_logger));
            }
            catch(final Exception e)
            {
                throw new SAXException(e.getMessage());
            }

            final String actualUnitType = _eventUnit.getType().getName();

            if(!actualUnitType.equals(expectedUnitType))
            {
                throw new SAXException("The time series  for location id: " + _locationId + " and data type "
                    + _parameterId + " has a unit type of " + actualUnitType
                    + " this does not match the NWSRFS unit type of " + expectedUnitType + ".");
            }

            //if the unit string in input fews.xml is not recognized, _eventUnit is faking unit
            if(_eventUnit == null)
            {
                throw new SAXException("The unit from the input file " + _curValue + " is not recognized.");
            }

        }
        else if("sourceOrganisation".equals(elementName))
        {
            _sourceOrganization = _curValue;
        }
        else if("sourceSystem".equals(elementName))
        {
            _sourceSystem = _curValue;
        }
        else if("fileDescription".equals(elementName))
        {
            _fileDescription = _curValue;
        }
        else if("creationDate".equals(elementName))
        {
            _creationDate = _curValue;
        }
        else if("creationTime".equals(elementName))
        {
            _creationTime = _curValue;
        }
        else if("timeZone".equals(elementName))
        {

            _timeZone = OHDUtilities.getTimeZoneFromOffSet(Double.parseDouble(_curValue.trim()));

            _dateFmt.setTimeZone(_timeZone);

            _logger.log(Logger.DEBUG, "The time zone offset value in the input xml file is " + _curValue.trim()
                + ". Accordingly, Time Zone is: " + _timeZone.getID());

        }
        else if("daylightSavingObservingTimeZone".equals(elementName))
        {
            switch(_curValue.trim())
            {
                case "AST":
                    _timeZone = OHDUtilities.getTimeZoneFromOffSet(-9.0);
                    break;
                case "PST":
                    _timeZone = OHDUtilities.getTimeZoneFromOffSet(-8.0);
                    break;
                case "MST":
                    _timeZone = OHDUtilities.getTimeZoneFromOffSet(-7.0);
                    break;
                case "CST":
                    _timeZone = OHDUtilities.getTimeZoneFromOffSet(-6.0);
                    break;
                case "IET":
                    _timeZone = OHDUtilities.getTimeZoneFromOffSet(-5.0);
                    break;
                case "CNT":
                    _timeZone = OHDUtilities.getTimeZoneFromOffSet(-3.5);
                    break;
                case "AGT":
                    _timeZone = OHDUtilities.getTimeZoneFromOffSet(-3.0);
                    break;
                case "BET":
                    _timeZone = OHDUtilities.getTimeZoneFromOffSet(-3.0);
                    break;
                case "WET":
                    _timeZone = OHDUtilities.getTimeZoneFromOffSet(0.0);
                    break;
                case "CET":
                    _timeZone = OHDUtilities.getTimeZoneFromOffSet(1.0);
                    break;
                case "MET":
                    _timeZone = OHDUtilities.getTimeZoneFromOffSet(1.0);
                    break;
                case "EET":
                    _timeZone = OHDUtilities.getTimeZoneFromOffSet(2.0);
                    break;
                case "AZT":
                    _timeZone = OHDUtilities.getTimeZoneFromOffSet(4.0);
                    break;
                case "NET":
                    _timeZone = OHDUtilities.getTimeZoneFromOffSet(4.0);
                    break;
                case "AET":
                    _timeZone = OHDUtilities.getTimeZoneFromOffSet(10.0);
                    break;
                case "AWT":
                    _timeZone = OHDUtilities.getTimeZoneFromOffSet(8.0);
                    break;
                case "NST":
                    _timeZone = OHDUtilities.getTimeZoneFromOffSet(12.0);
                    break;
                default:
                    _logger.log(Logger.ERROR,
                                "Value '"
                                    + _curValue.trim()
                                    + "' is not facet-valid with respect to enumeration '[AST, PST, MST, CST, IET, CNT, AGT, BET, WET, CET, MET, EET, AZT, NET, AET, AWT, NST]'. It must be a value from the enumeration.");
                    _timeZone = OHDUtilities.getTimeZoneFromOffSet(0.0);
                    break;
            }
            _dateFmt.setTimeZone(_timeZone);

            _logger.log(Logger.DEBUG, "The time zone offset value in the input xml file is " + _curValue.trim()
                + ". Accordingly, Time Zone is: " + _timeZone.getID());
        }

        else if("missVal".equals(elementName))
        {
            if(_curValue.length() > 0)
            {
                _missingValInHeader = Double.parseDouble(_curValue.trim());
            }
            else
            { //set to default value, e.g. -999.0
                _missingValInHeader = OHDConstants.MISSING_DATA;
            }
        }
        else if("x".equals(elementName))
        {
            _x = Double.parseDouble(_curValue);
        }
        else if("y".equals(elementName))
        {
            _y = Double.parseDouble(_curValue);
        }
        else if("z".equals(elementName))
        {
            _z = Double.parseDouble(_curValue);
        }
        else if("lat".equals(elementName))
        {
            _latitude = Float.parseFloat(_curValue);
        }
        else if("lon".equals(elementName))
        {
            _longitude = Float.parseFloat(_curValue);
        } //close else if

    } //close method

    /**
     * Check this event value is missing value or not, according to the missing value defined in the xml header. Throws
     * an Exception if conditions meet. Otherwise, returns an appropriate event value. Regardless what "missingValue"
     * defined in the xml file, we always use -999.0(OHDConstants.MISSING_DATA)
     * <p>
     * Note: sometimes, the missing value defined in the header of the xml file is "NaN" and event values are "NaN".
     * This method handles this case correctly.
     * 
     * @param eventTime
     * @param eventVal
     * @return
     * @throws Exception
     */
    private double getNewEventValueByCheckingMissingValue(final long eventTime, final double eventVal) throws SAXException
    {

        /*----------------determine this event value is missing value or not -----------------*/
        boolean isEventValueMissingValue = false; //initialized to false

        if(Double.isNaN(eventVal) || eventVal == OHDConstants.MISSING_DATA)
        {//event value is "NaN" or "-999.0"
            isEventValueMissingValue = true;
        }
        else if(Double.isNaN(_missingValInHeader) == false && _missingValInHeader == eventVal)
        {//missing value in header is not "NaN" and it equals to the event value
            isEventValueMissingValue = true;
        }

        /*----------------throws Exception if meets the conditions -----------------*/
        if((!_areMissingValuesAlwaysAllowed)
            && ((_initStateTimeCal == null) || (eventTime > _initStateTimeCal.getTimeInMillis())))
        {
            if(!_paramMisssingValueAllowed && isEventValueMissingValue)
            {
                throw new SAXException("Error extracting time series - missing values not allowed but found in file for TS Type: "
                    + _parameterId + " at " + DateTime.getDateTimeStringFromLong(eventTime, _timeZone));
            }
        }

        /*----------------if reach here, returns appropriate value -----------------*/
        if(isEventValueMissingValue)
        {
            return OHDConstants.MISSING_DATA; //regardless what "missingValue" defined in the xml file, we always use -999.0(OHDConstants.MISSING_DATA)
        }
        else
        {
            return eventVal;
        }
    }

    public void characters(final char[] chars, final int startIndex, final int endIndex)
    {
        _curValue += new String(chars, startIndex, endIndex);
    }

    public boolean willTimeSeriesDataBeReadInFromBinaryFile()
    {
        return (_binaryInputStream != null);
    }

    private int getTimeStepInHours(final String timeStepUnit,
                                   final int timeStepMultiplier,
                                   final int timeStepDivider,
                                   final String timeStepTimes)
    {
        int retVal = 0;

        /*
         * FB 1899 Add the times option into the timestep element. Times define the time step by a list of times without
         * dates. ie "10:00 23:00"
         */
        if(!timeStepTimes.isEmpty())
        {
            // Get a array of the times.
            final String[] timeStepTimesArray = OHDConstants.DELIMITER_PATTERN.split(timeStepTimes.trim());

            try
            {
                // Add the a date and the seconds to the time to use existing library DateTime,
                final String dummyDate = "2000-10-10";
                String seconds = "";
                if(timeStepTimesArray[0].length() == 5)
                {
                    seconds = ":00";
                }
                if(timeStepTimesArray.length > 1)
                {
                    final DateTime time1 = new DateTime(dummyDate, timeStepTimesArray[0] + seconds);
                    final DateTime time2 = new DateTime(dummyDate, timeStepTimesArray[1] + seconds);
                    // compute the time step interval in hours, we assume that the times are equals.

                    retVal = time2.getNwsrfsHour() - time1.getNwsrfsHour();
                }
                else
                {
                    retVal = 24;
                }
            }
            catch(final Exception e)
            {
                // TODO Auto-generated catch block
                e.printStackTrace();
            }
        }
        else

        if(timeStepUnit.equalsIgnoreCase(TimeSeriesHandler.SECOND))
        {
            retVal = (timeStepMultiplier / timeStepDivider) / 3600;
        }
        else if(timeStepUnit.equalsIgnoreCase(TimeSeriesHandler.MINUTE))
        {
            retVal = timeStepMultiplier / timeStepDivider / 60;
        }
        else if(timeStepUnit.equalsIgnoreCase(TimeSeriesHandler.HOUR))
        {
            retVal = timeStepMultiplier / timeStepDivider;
        }
        else if(timeStepUnit.equalsIgnoreCase(TimeSeriesHandler.DAY))
        {
            retVal = (timeStepMultiplier / timeStepDivider) * 24;
        }
        else if(timeStepUnit.equalsIgnoreCase(TimeSeriesHandler.WEEK))
        {
            retVal = (timeStepMultiplier / timeStepDivider) * 24 * 7;
        }
        else if(timeStepUnit.equalsIgnoreCase(TimeSeriesHandler.YEAR))
        {
            //could be 364 if leap year - check
            retVal = (timeStepMultiplier / timeStepDivider) * 365 * 24 * 7;
        }
        else if(timeStepUnit.equalsIgnoreCase(TimeSeriesHandler.NONEQUIDISTANT))
        {
            retVal = 1;
        }

        return retVal;
    }

    public void setAreMissingValuesAlwaysAllowed(final boolean b)
    {
        this._areMissingValuesAlwaysAllowed = b;
    }

    public void setInitialStateTime(final long millis)
    {
        _initStateTimeCal = Calendar.getInstance(OHDConstants.GMT_TIMEZONE);
        _initStateTimeCal.setTime(new Date(millis));
    }

    public static MeasuringUnit retrieveMeasuringUnitForFEWSUnit(final String unitStr)
    {
        return TimeSeriesHandler._fewsToLocalUnitMap.get(unitStr);
    }

    public static void repopulateFewsToLocalUnitMap()
    {
        TimeSeriesHandler._fewsToLocalUnitMap.clear();
        populateFewsToLocalUnitMap();
    }

    public static void populateFewsToLocalUnitMap()
    {
        // Read these mappings from a file, not hardcoded like this.
        // Get a complete list from Delft. 
        // Anything not currently handled by MeasuringUnit will need to be handled.

        //read the new map file we are introducing... - maps time series type to known unit string

        if(TimeSeriesHandler._fewsToLocalUnitMap.size() == 0)
        {
            TimeSeriesHandler._fewsToLocalUnitMap.put("mm", MeasuringUnit.mm);
            TimeSeriesHandler._fewsToLocalUnitMap.put("cm", MeasuringUnit.cm);
            TimeSeriesHandler._fewsToLocalUnitMap.put("cfs", MeasuringUnit.cfs);
            TimeSeriesHandler._fewsToLocalUnitMap.put("cms", OHDConstants.DISCHARGE_UNIT);
            TimeSeriesHandler._fewsToLocalUnitMap.put("m3", MeasuringUnit.m3);
            TimeSeriesHandler._fewsToLocalUnitMap.put("cmsd", MeasuringUnit.cmsd);
            TimeSeriesHandler._fewsToLocalUnitMap.put("cfsd", MeasuringUnit.cfsd);
            TimeSeriesHandler._fewsToLocalUnitMap.put("acft", MeasuringUnit.acft);
            TimeSeriesHandler._fewsToLocalUnitMap.put("tcum", MeasuringUnit.tcum);
            TimeSeriesHandler._fewsToLocalUnitMap.put("m3/s", OHDConstants.DISCHARGE_UNIT);
            TimeSeriesHandler._fewsToLocalUnitMap.put("feet", MeasuringUnit.feet);
            TimeSeriesHandler._fewsToLocalUnitMap.put("hours", MeasuringUnit.hours);
            TimeSeriesHandler._fewsToLocalUnitMap.put("days", MeasuringUnit.days);
            TimeSeriesHandler._fewsToLocalUnitMap.put("inches", MeasuringUnit.inches);
            TimeSeriesHandler._fewsToLocalUnitMap.put("IN", MeasuringUnit.inches);
            TimeSeriesHandler._fewsToLocalUnitMap.put("kcfs", MeasuringUnit.kcfs);
            TimeSeriesHandler._fewsToLocalUnitMap.put("meters", MeasuringUnit.meters);
            TimeSeriesHandler._fewsToLocalUnitMap.put("m", MeasuringUnit.meters);
            TimeSeriesHandler._fewsToLocalUnitMap.put("ft", MeasuringUnit.feet);
            TimeSeriesHandler._fewsToLocalUnitMap.put("degrees F", MeasuringUnit.degreesFahrenheit);
            TimeSeriesHandler._fewsToLocalUnitMap.put("degrees C", MeasuringUnit.degreesCelsius);
            TimeSeriesHandler._fewsToLocalUnitMap.put("oF", MeasuringUnit.degreesFahrenheit);
            TimeSeriesHandler._fewsToLocalUnitMap.put("DEGF", MeasuringUnit.degreesFahrenheit);
            TimeSeriesHandler._fewsToLocalUnitMap.put("oC", MeasuringUnit.degreesCelsius);
            TimeSeriesHandler._fewsToLocalUnitMap.put("DEGC", MeasuringUnit.degreesCelsius);
            TimeSeriesHandler._fewsToLocalUnitMap.put("%", MeasuringUnit.percentDecimal);
            TimeSeriesHandler._fewsToLocalUnitMap.put("", MeasuringUnit.unitlessReal);
            TimeSeriesHandler._fewsToLocalUnitMap.put("-", MeasuringUnit.unitlessInt);
            TimeSeriesHandler._fewsToLocalUnitMap.put("unitless", MeasuringUnit.unitless);

            // the following are NWSRFS units, this allows FEWS to give us files in our units
            TimeSeriesHandler._fewsToLocalUnitMap.put("PCTD", MeasuringUnit.percentDecimal);
            TimeSeriesHandler._fewsToLocalUnitMap.put("INT", MeasuringUnit.unitlessInt);
            TimeSeriesHandler._fewsToLocalUnitMap.put("REAL", MeasuringUnit.unitlessReal);
        }

    }
}
