package ohd.hseb.util.fews.ohdmodels;

import java.io.BufferedReader;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.List;
import java.util.Properties;

import javax.management.timer.Timer;

import ohd.hseb.measurement.MeasuringUnit;
import ohd.hseb.measurement.RegularTimeSeries;
import ohd.hseb.time.DateTime;
import ohd.hseb.time.JulianDayAndHour;
import ohd.hseb.util.Logger;
import ohd.hseb.util.SortedProperties;
import ohd.hseb.util.fews.LegacyAdapter;
import ohd.hseb.util.fews.NwsrfsDataFileExtractor;
import ohd.hseb.util.fews.NwsrfsDataTypeMappingReader;
import ohd.hseb.util.fews.OHDConstants;

/**
 * This class is responsible for writing out the time series objects, parameters and states in the Model Driver to the
 * format expected by the Legacy (FORTRAN, C, C++) version of the various models and for reading in the results and
 * putting them back in to appropriate objects so that XML that FEWS expects can be written out.
 * 
 * @author FewsPilot team
 */
public class LegacyModelAdapter extends LegacyAdapter
{

    private static final String OUTPUT_STATES_FILENAME = "statesO.txt";

    private static final String OUTPUT_TS_FILENAME = "outputs.txt";

    private static final String INPUT_STATES_FILENAME = "statesI.txt";

    private static final String PARAMS_FILENAME = "params.txt";

    private static final String PARAMS_SAVED_FILENAME = "params_saved.txt";

    private static final String PARAMS_FILE_UNCHANGED = "PARAMS_UNCHANGED";

    private static final String INPUT_TS_FILENAME = "ts.txt";

    private static final String ARGUMENTS_FILENAME = "arguments.txt";

    public static final String MODS_TXT_FILE = "mods.txt";

    //property names in arguments.txt

    private static final String DATA_TYPE_FILE_NAME = "dataTypeFileName";
    private static final String DEBUG_FLAG = "debugFlag";
    private static final String DIAG_FILE_NAME = "diagFileName";
    private static final String JULIAN_DAY_FOR_END_OF_RUN = "julianDayForEndOfRun";
    private static final String JULIAN_DAY_FOR_LAST_OBS_DATE = "julianDayForLastObsDate";
    private static final String JULIAN_DAY_FOR_START_OF_DRIVER_TS_DATA = "julianDayForStartOfDriverTsData";
    private static final String JULIAN_DAY_FOR_START_OF_RUN = "julianDayForStartOfRun";
    private static final String JULIAN_HOUR_FOR_END_OF_RUN = "julianHourForOfEndOfRun";
    private static final String JULIAN_HOUR_FOR_LAST_OBS_DATE = "julianHourForLastObsDate";
    private static final String JULIAN_HOUR_FOR_START_OF_DRIVER_TS_DATA = "julianHourForStartOfDriverTsData";
    private static final String JULIAN_HOUR_FOR_START_OF_RUN = "julianHourForStartOfRun";
    private static final String LOCAL_HOUR_OFFSET = "localHourOffset";
    private static final String OUTPUT_STATE_FILE_NAME = "outputStateFileName";
    private static final String OUTPUT_TS_FILE_NAME = "outputTsFileName";
    private static final String TS_FILE_NAME = "tsFileName";
    private static final String PARAM_FILE_NAME = "paramFileName";
    private static final String STATE_FILE_NAME = "stateFileName";
    private static final String PARAMS_STATUS_INFO = "prevParamsInfo";
    public static final String DATA_UNIT_FILE_NAME = "dataUnitFileName";
    public static final String MOD_FILE_NAME = "modFileName";
    public static final String START_OF_DAY = "startOfDay";

    //protected String _workDir = null;
    //protected final Logger _logger;
    protected String _tsFileName;
    protected String _paramFileName;
    protected String _stateFileName;
    protected String _outputTsFileName;
    protected String _outputStateFileName;
    //protected String _diagFileName;
    protected ModelDriver _modelDriver;
    //protected String _printDebugInfo;

    private long _initialStateJulianDay;
    private long _initialStateJulianHour;
    private long _initialStateTime;

    private long _computationStartJulianDay;
    private long _computationStartJulianHour;

    private long _computationEndJulianDay;
    private long _computationEndJulianHour;
    private long _computationEndTime;

    private long _modelSwitchFromObservedToForecastJulianDay;
    private long _modelSwitchFromObservedToForecastJulianHour;

    private String _paramsStatusInfoString;

    private String _justDriverName;

    //private NwsrfsDataFileExtractor _extractor = null;

    public LegacyModelAdapter(final ModelDriver callingDriver, final String workDir, final Logger logger) throws Exception
    {
        _workDir = workDir;
        _logger = logger;
        _tsFileName = _workDir + "/" + INPUT_TS_FILENAME;
        _paramFileName = _workDir + "/" + PARAMS_FILENAME;
        _stateFileName = _workDir + "/" + INPUT_STATES_FILENAME;
        _outputTsFileName = _workDir + "/" + OUTPUT_TS_FILENAME;
        _outputStateFileName = _workDir + "/" + OUTPUT_STATES_FILENAME;
        _diagFileName = _workDir + "/" + OHDConstants.DIAGNOSTICS_FILENAME;
        //just store the handle
        _modelDriver = callingDriver;
        _printDebugInfo = "" + logger.getPrintDebugInfo();
        _paramsStatusInfoString = PARAMS_FILE_UNCHANGED;

        _modelDriver.runModelDriverValidation();

        // get just driver name for logging
        final String driverName = _modelDriver.getClass().getName();
        _justDriverName = driverName.substring(driverName.lastIndexOf(".") + 1); //only need the driver name

        _extractor = new NwsrfsDataFileExtractor(_workDir);

        _extractor.copyNwsrfsFilesFromClasspath();
    }

    // LegacyModelAdapter used for testing
    public LegacyModelAdapter(final String workDir, final Logger logger) throws Exception
    {
        _workDir = workDir;
        _logger = logger;
        _tsFileName = _workDir + "/" + INPUT_TS_FILENAME;
        _paramFileName = _workDir + "/" + PARAMS_FILENAME;
        _stateFileName = _workDir + "/" + INPUT_STATES_FILENAME;
        _outputTsFileName = _workDir + "/" + OUTPUT_TS_FILENAME;
        _outputStateFileName = _workDir + "/" + OUTPUT_STATES_FILENAME;
        _diagFileName = _workDir + "/" + OHDConstants.DIAGNOSTICS_FILENAME;

        // extract datatype and dataunit file from jar file
        _extractor = new NwsrfsDataFileExtractor(_workDir);

        _extractor.copyNwsrfsFilesFromClasspath();
    }

    /**
     * Instantiates a CommandAdapter and runs the external executable
     * 
     * @param args Arguments to be passed to external executable
     * @throws Exception
     */
    public void executeModel(final String executableName, SortedProperties properties) throws Exception
    {

        if(properties == null)
        {
            properties = new SortedProperties();
        }

        createModelProperties(properties);

        final String[] arguments = {_workDir + "/" + ARGUMENTS_FILENAME};

        // write out properties to arguments.txt file
        final FileOutputStream argsFile = new FileOutputStream(_workDir + "/" + ARGUMENTS_FILENAME);
        properties.store(argsFile, null);
        argsFile.close();

        // stop to take a reading
        _modelDriver.getStopWatch().stop();
        final double timeOnePeriod = _modelDriver.getStopWatch().getElapsedTimeInSeconds();

        // restart to continue reading
        _modelDriver.getStopWatch().restart();
        //Execute the legacy model
        executeLegacy(executableName, arguments);

        // stop to take a reading
        _modelDriver.getStopWatch().stop();
        final double timeTwoPeriod = _modelDriver.getStopWatch().getElapsedTimeInSeconds() - timeOnePeriod;

        _logger.log(Logger.DEBUG, _justDriverName + " time spent RUNNING legacy model: " + timeTwoPeriod + " sec."); // timeTwoPeriod

        // restart to continue reading
        _modelDriver.getStopWatch().restart();

    }

    /**
     * Convert timing info from Fews into julian day and hour for passing to legacy models. The run start is the date in
     * the states.xml at the start of the run, the date start date is + 1 driver time step, and the end of the run is
     * the end of the driver time series
     */

    private void calculateDateArguments() throws Exception
    {
        _initialStateTime = _modelDriver.getInitialStateTime();
        final DateTime initialStateDate = new DateTime(_initialStateTime, _modelDriver.getStartHourOfDay());
        final JulianDayAndHour initialStateAsJulian = initialStateDate.convertDateTimeToJulianDayAndHour();
        _initialStateJulianDay = initialStateAsJulian.getJulianDay();
        _initialStateJulianHour = initialStateAsJulian.getJulianHourOfDay();

        final DateTime computationStartDate = new DateTime(_modelDriver.getComputationStartTime(),
                                                           _modelDriver.getStartHourOfDay());
        final JulianDayAndHour computationStartAsJulian = computationStartDate.convertDateTimeToJulianDayAndHour();
        _computationStartJulianDay = computationStartAsJulian.getJulianDay();
        _computationStartJulianHour = computationStartAsJulian.getJulianHourOfDay();

        _computationEndTime = _modelDriver.getComputationEndTime();
        final DateTime computationEndDateAndTime = new DateTime(_computationEndTime, _modelDriver.getStartHourOfDay());
        final JulianDayAndHour computationEndAsJulian = computationEndDateAndTime.convertDateTimeToJulianDayAndHour();
        _computationEndJulianDay = computationEndAsJulian.getJulianDay();
        _computationEndJulianHour = computationEndAsJulian.getJulianHourOfDay();

        final DateTime modelRunSwitchFromObservedToForecastDate = new DateTime(_modelDriver.getSwitchFromObservedToForecastTime(),
                                                                               _modelDriver.getStartHourOfDay());
        final JulianDayAndHour switchFromObservedToForecastAsJulian = modelRunSwitchFromObservedToForecastDate.convertDateTimeToJulianDayAndHour();
        _modelSwitchFromObservedToForecastJulianDay = switchFromObservedToForecastAsJulian.getJulianDay();
        _modelSwitchFromObservedToForecastJulianHour = switchFromObservedToForecastAsJulian.getJulianHourOfDay();

    }

    /**
     * wrapper method for converting two forms of output (time series and diagnostics) from txt format to internal
     * lists; later converted to xml; the output time series list is returned
     * 
     * @return output regular ts list
     * @throws Exception
     */

    public List<RegularTimeSeries> readOutputFlatFilesToLists() throws Exception
    {
        super.readFromFlatFileToDiagnosticsList();

        List<RegularTimeSeries> retTsList = new ArrayList<RegularTimeSeries>();
        retTsList = readFromFlatFileToReguarTSList();
        return retTsList;
    }

    /**
     * Loads the 2 column time series written out in a file by the legacy program in to an existing time series object
     * 
     * @param existTS
     * @param fileName
     * @return
     * @throws Exception
     */

    private List<RegularTimeSeries> readFromFlatFileToReguarTSList() throws Exception
    {
        // stop to take a reading
        _modelDriver.getStopWatch().stop();
        final double timeOnePeriod = _modelDriver.getStopWatch().getElapsedTimeInSeconds();

        // restart to continue reading
        _modelDriver.getStopWatch().restart();

        float value = 0.0f;
        //String[] aTSLine = null;
        String[] headerLine = null;
        String qualifierId, timeSeriesType;
        int timeStep, count;
        String lineStr;
        RegularTimeSeries ts = null;
        final List<RegularTimeSeries> retTsList = new ArrayList<RegularTimeSeries>();
        final String fullyQualifiedOutputFileName = _outputTsFileName;
        final BufferedReader TSReader = new BufferedReader(new FileReader(fullyQualifiedOutputFileName));

        final SimpleDateFormat dateFormat = new SimpleDateFormat("yyyyMMddHH");
        dateFormat.setTimeZone(OHDConstants.GMT_TIMEZONE);

        try
        {

            lineStr = TSReader.readLine();

            while(lineStr != null)
            {

                headerLine = OHDConstants.DELIMITER_PATTERN.split(lineStr.trim());

                if(headerLine.length != 5)
                {
                    _logger.log(Logger.FATAL, "Header line in output TS file " + fullyQualifiedOutputFileName
                        + " from FORTRAN was not correct");

                    throw new Exception();
                }
                //This is NWSRFS timeseriesid
                qualifierId = headerLine[0];
                timeSeriesType = headerLine[1];
                timeStep = Integer.parseInt(headerLine[2]);

                final long dataStartTime = dateFormat.parse(headerLine[3]).getTime();
                count = Integer.parseInt(headerLine[4]);
                final long endTime = dataStartTime + (count - 1) * timeStep * Timer.ONE_HOUR;

                final String nwsUnit = NwsrfsDataTypeMappingReader.getNwsrfsUnit(timeSeriesType, _logger);
                if(nwsUnit == null)
                {
                    _logger.log(Logger.FATAL, "nwsrfsUnit null for time series type" + timeSeriesType);

                    throw new Exception();
                }

                ts = new RegularTimeSeries(dataStartTime, endTime, timeStep, MeasuringUnit.getMeasuringUnit(nwsUnit));

                retTsList.add(ts);
                final List<String> qualifierIds = new ArrayList<String>();
                qualifierIds.add(qualifierId);
                ts.setQualifierIds(qualifierIds);
                //setQualifierId(qualifierId);
                // initially ts location id is set == to driving ts loc id
                // if qualifier id == location id, qualifier id not written out

                if(_modelDriver.getDrivingRTS() != null)
                {
                    ts.setLocationId(_modelDriver.getDrivingRTS().getLocationId());
                }
                else
                {
                    ts.setLocationId(qualifierId);
                }

                ts.setTimeSeriesType(timeSeriesType);
                lineStr = TSReader.readLine();
                String[] aTSLine;
                for(int i = 0; i < count; i++)
                {
                    if(lineStr != null)
                    {
                        aTSLine = lineStr.trim().split(" ");
                        if(aTSLine.length == 2)
                        {
                            value = Float.parseFloat(aTSLine[0]);

                            ts.setMeasurementByIndex(value, i);
                            lineStr = TSReader.readLine();
                        }
                        else
                        {
                            _logger.log(Logger.FATAL, "Invalid Data in file from Legacy Model" + lineStr);
                            throw new Exception();
                        }
                    }
                    else
                    {
                        _logger.log(Logger.FATAL, "Invalid Data in file from Legacy Model for " + timeSeriesType
                            + " Data");
                        throw new Exception();
                    }

                }
            }
        }
        finally
        {
            TSReader.close();
        }

        // stop to take a reading
        _modelDriver.getStopWatch().stop();
        final double timeTwoPeriod = _modelDriver.getStopWatch().getElapsedTimeInSeconds() - timeOnePeriod;

        _logger.log(Logger.DEBUG, _justDriverName + " time spent READING text files: " + timeTwoPeriod + " sec."); // timeTwoPeriod

        // restart to continue reading
        _modelDriver.getStopWatch().restart();
        return retTsList;

    }

    /**
     * Writes the input time series out to a file in a 2-column format (date/time and value).
     * 
     * @param ts - time series to be written
     * @throws IOException
     */
    public void writeRegularTimeSeriesDataToFlatFile(final List<RegularTimeSeries> tsList) throws Exception
    {

        final FileWriter outFile = new FileWriter(_workDir + "/" + INPUT_TS_FILENAME);
        final StringBuilder outputLines = new StringBuilder();

        RegularTimeSeries rts;

        final String dateFormatString = "yyyyMMddHH";
        final SimpleDateFormat simpleDateFormat = new SimpleDateFormat(dateFormatString);
        simpleDateFormat.setTimeZone(OHDConstants.GMT_TIMEZONE);

        String timeString;
        String timeSeriesType = null;
        StringBuilder headerLine = null;
        String nwsrfsUnitStr = null;
        String nwsrfsDim = null;
        String nwsrfsUnit = null;

        try
        {
            outputLines.append(tsList.size()).append(OHDConstants.NEW_LINE);

            for(int i = 0; i < tsList.size(); i++)
            {
                timeSeriesType = tsList.get(i).getTimeSeriesType();

                rts = tsList.get(i);

                if(timeSeriesType == null)
                {
                    _logger.log(Logger.FATAL,
                                "time series type null from tsList.get(i).getTimeSeriesType() in LegacyModelAdapter.java");

                    throw new Exception();
                }
                nwsrfsUnitStr = NwsrfsDataTypeMappingReader.getNwsrfsUnit(timeSeriesType, _logger);
                if(nwsrfsUnitStr == null)
                {
                    _logger.log(Logger.FATAL, "nwsrfsUnit null for time series type" + timeSeriesType);

                    throw new Exception();
                }
                final MeasuringUnit getTSUnitType = MeasuringUnit.getMeasuringUnit(nwsrfsUnitStr);

                if(getTSUnitType == null)
                {
                    _logger.log(Logger.DEBUG, "Time series type " + timeSeriesType
                        + " does not exist in dataTypeToMeasuringUnitMap");
                }

                nwsrfsDim = NwsrfsDataTypeMappingReader.getNwsrfsDim(timeSeriesType, _logger);
                if(nwsrfsDim == null)
                {
                    _logger.log(Logger.FATAL, "nwsrfsDimension null for time series type" + timeSeriesType);

                    throw new Exception();
                }

                //write it out in the unit for the NWSRFS datatype
                nwsrfsUnit = NwsrfsDataTypeMappingReader.getNwsrfsUnit(timeSeriesType, _logger);
                if(nwsrfsUnit == null)
                {
                    _logger.log(Logger.FATAL, "nwsrfsUnit null for time series type" + timeSeriesType);

                    throw new Exception();
                }

                final long tsIntervalInMillis = rts.getIntervalInMillis();
                rts.trimTimeSeriesAtStart(_initialStateTime + tsIntervalInMillis);
                rts.trimTimeSeriesAtEnd(_computationEndTime);
                final int numMeasurement = rts.getMeasurementCount();
                final MeasuringUnit tsUnit = MeasuringUnit.getMeasuringUnit(nwsrfsUnit);
                headerLine = new StringBuilder(rts.getQualifierIds().get(0));
                headerLine.append(" ")
                          .append(rts.getTimeSeriesType())
                          .append(" ")
                          .append(rts.getIntervalInHours())
                          .append(" ")
                          .append(numMeasurement)
                          .append(" ")
                          .append(nwsrfsDim)
                          .append(" ")
                          .append(getTSUnitType);
                outputLines.append(headerLine).append(OHDConstants.NEW_LINE);

                _logger.log(Logger.DEBUG, "Header line for time series = " + headerLine);

                long eventTimeLong = rts.getStartTime();

                final double missVal = rts.getMissingValue();

                for(int j = 0; j < numMeasurement; j++)
                {
                    timeString = simpleDateFormat.format(eventTimeLong);

                    double tsVal = rts.getMeasurementValueByIndex(j, tsUnit);

                    if(missVal != OHDConstants.MISSING_DATA
                        && Double.toString(tsVal).equalsIgnoreCase(Double.toString(missVal)))
                    {
                        tsVal = OHDConstants.MISSING_DATA;
                    }

                    outputLines.append(tsVal).append(" ").append(timeString).append(OHDConstants.NEW_LINE);

                    eventTimeLong += tsIntervalInMillis;
                }

            }
            outFile.write(outputLines.toString());
        }
        finally
        {
            outFile.close();
        }
    }

    private void createModelProperties(final Properties properties)
    {
        properties.put(TS_FILE_NAME, _tsFileName);
        properties.put(PARAM_FILE_NAME, _paramFileName);
        properties.put(STATE_FILE_NAME, _stateFileName);
        properties.put(OUTPUT_TS_FILE_NAME, _outputTsFileName);
        properties.put(OUTPUT_STATE_FILE_NAME, _outputStateFileName);
        properties.put(DIAG_FILE_NAME, _diagFileName);
        properties.put(DEBUG_FLAG, _printDebugInfo);

        properties.put(JULIAN_DAY_FOR_START_OF_RUN, String.valueOf(_initialStateJulianDay));
        properties.put(JULIAN_HOUR_FOR_START_OF_RUN, String.valueOf(_initialStateJulianHour));
        properties.put(JULIAN_DAY_FOR_START_OF_DRIVER_TS_DATA, String.valueOf(_computationStartJulianDay));
        properties.put(JULIAN_HOUR_FOR_START_OF_DRIVER_TS_DATA, String.valueOf(_computationStartJulianHour));
        properties.put(JULIAN_DAY_FOR_END_OF_RUN, String.valueOf(_computationEndJulianDay));
        properties.put(JULIAN_HOUR_FOR_END_OF_RUN, String.valueOf(_computationEndJulianHour));
        properties.put(JULIAN_DAY_FOR_LAST_OBS_DATE, String.valueOf(_modelSwitchFromObservedToForecastJulianDay));
        properties.put(JULIAN_HOUR_FOR_LAST_OBS_DATE, String.valueOf(_modelSwitchFromObservedToForecastJulianHour));

        if(_modelDriver.needCarryoverTransfer())
        {
            _paramsStatusInfoString = _workDir + "/" + PARAMS_SAVED_FILENAME; // the string will contain the params_saved.txt file with path
        }
        else
        {
            _paramsStatusInfoString = PARAMS_FILE_UNCHANGED;
        }

        properties.put(PARAMS_STATUS_INFO, _paramsStatusInfoString);
        properties.put(LOCAL_HOUR_OFFSET, String.valueOf(_modelDriver.getRunInfo().getTimeZoneRawOffsetInHours()));
        properties.put(START_OF_DAY, String.valueOf(_modelDriver.getStartHourOfDay()));
        properties.put(DATA_TYPE_FILE_NAME, _extractor.getDataTypeMappingFullPathForLegacyModels());

    }

    /**
     * This methods converts all the FEWS xml files to simple text files and places them in the "work" directory. The
     * files created include: <br>
     * 1. arguments.txt (most of the info in runInfo.xml) <br>
     * 2. ts.txt (the contents of inputs.xml)<br>
     * 3. params.txt (the contents of the element "OPERATION_CONTENTS" in params.xml <br>
     * 4. statesI.txt (the input state file defined in states.xml) <br>
     * 5. nwsrfs_datatype_mapping_file.txt (info specific to nwsrfs operations)
     */
    public void writeStandAloneFiles() throws Exception
    {
        // stop to take a reading
        _modelDriver.getStopWatch().stop();
        final double timeOnePeriod = _modelDriver.getStopWatch().getElapsedTimeInSeconds();

        // restart to continue reading
        _modelDriver.getStopWatch().restart();

        this.writeParamsToFile((LegacyModelParameters)_modelDriver.getParameters(), PARAMS_FILENAME);

        calculateDateArguments();

        if(_modelDriver.getTsList() != null && _modelDriver.getTsList().size() > 0)
        {
            this.writeRegularTimeSeriesDataToFlatFile(_modelDriver.getTsList());
        }

        _modelDriver.getState().writeState(this.getStateFileName(), _logger);

        final LegacyModelParameters paramsSaved = (LegacyModelParameters)_modelDriver.getPreviousParameters();

        if(_modelDriver.needCarryoverTransfer() && paramsSaved.getParamsMap() != null
            && paramsSaved.getParamsMap().size() > 0)
        {
            this.writeParamsToFile(paramsSaved, PARAMS_SAVED_FILENAME);
        }

        // stop to take a reading
        _modelDriver.getStopWatch().stop();
        final double timeTwoPeriod = _modelDriver.getStopWatch().getElapsedTimeInSeconds() - timeOnePeriod;

        _logger.log(Logger.DEBUG, _justDriverName + " time spent WRITING text files: " + timeTwoPeriod + " sec."); // timeTwoPeriod

        // restart to continue reading
        _modelDriver.getStopWatch().restart();
    }

    public void writeParamsToFile(final LegacyModelParameters params, final String fileName) throws Exception
    {
        final FileWriter outFile = new FileWriter(_workDir + "/" + fileName);

        String operationContents;

//        if(false)
//        {
//            //operationContents = this._modelDriver._converter.getParameterAndStatesFromXml((ModelParameters)this._modelDriver.getParameters(),
//            //                                                                              (ModelState)this._modelDriver.getState());
//        }
//        else
//        {
        operationContents = params.getOptFile();

//        }

        outFile.write(operationContents);
        outFile.close();

    }

    public void writeModsToFile(final List<RegularTimeSeries> inputModsList) throws Exception
    {
        final FileWriter outFile = new FileWriter(_workDir + "/" + MODS_TXT_FILE);
        if(inputModsList.isEmpty())
        {
            // don't write anything to file
        }
        else
        {
            outFile.write(_modelDriver.getModsStringFromTS(inputModsList));
        }

        outFile.close();
    }

    public void writeModsToFileUsingModValues(final List<ModValues> inputMods) throws Exception
    {
        final FileWriter outFile = new FileWriter(_workDir + "/" + MODS_TXT_FILE);
        if(inputMods.isEmpty())
        {
            // don't write anything to file
        }
        else
        {
            outFile.write(_modelDriver.getModsStringFromModValues(inputMods));
        }

        outFile.close();
    }

    public void setComputationEndTime(final long time)
    {
        _computationEndTime = time;
    }

    public void setInitialStateTime(final long time)
    {
        _initialStateTime = time;
    }

    public String getStateFileName()
    {
        return _stateFileName;
    }

    public void setStateFileName(final String stateFileName)
    {
        _stateFileName = stateFileName;
    }

    public String getOutputStateFileName()
    {
        return _outputStateFileName;
    }

    public void setOutputStateFileName(final String stateFileName)
    {
        _outputStateFileName = stateFileName;
    }

}
