package ohd.hseb.util.fews.ohdmodels;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.List;

import ohd.hseb.db.DbTimeHelper;
import ohd.hseb.measurement.RegularTimeSeries;
import ohd.hseb.time.DateTime;
import ohd.hseb.util.Logger;
import ohd.hseb.util.fews.Driver;
import ohd.hseb.util.fews.FewsXMLParser;
import ohd.hseb.util.fews.NwsrfsDataTypeMappingReader;
import ohd.hseb.util.fews.OHDConstants;

/**
 * ModelDriver is the common abstract class used by all OHD modules that interface to FEWS through the OHDFEWSAdapter
 * The following features are provided by using the OHDFewsadapter and extending the ModelDriver class <br>
 * 1. parsing and storing of elements and properties in run_info.xml file (standard file provided by FEWS) <br>
 * 2. parsing and storing of input time series in inputs.xml file (standard file provided by FEWS) <br>
 * 3. parsing and storing of parameters in params.xml file (standard file provided by FEWS) <br>
 * 4. storing and writing of output time series to outputs.xml file (standard file read in by FEWS) <br>
 * <br>
 * Other features: <br>
 * 1. input time series provided to module are validated for consistency and sufficiency (see
 * runModelDriverValidation()) <br>
 * 2. parameter and states classes for storing and accessing parameters and states in memory <br>
 * 3. a time series class (see RegularTimeSeries) for working with time series data in memory (many helper methods)
 * 
 * @author FewsPilot Team <br>
 */
public abstract class ModelDriver extends Driver
{
    protected ModelParameters _savedParameters;

    /**
     * RTS that is moving forward the time steps. For Snow17ModelDriver: MAT TS; For SacSmaModelDriver(and
     * SacSmaHTModelDriver): RAIM TS; For UnitHGModelDriver: TCI TS; etc
     */
    protected RegularTimeSeries _drivingTs = null;

    private boolean _needCarryoverTransfer = false; //default: assuming no carryover transfer needed

    /**
     * Run model calculation by calling {@link #execute()}, then output results by calling {@link #outputResults()}.
     */
    public void executeDriver() throws Exception
    {

        // get just driver name for logging
        final String driverName = this.getClass().getName();
        final String justDriverName = driverName.substring(driverName.lastIndexOf(".") + 1); //only need the driver name

        // Check if states.xml have more than one stateLoc value.
        // If parameter values are set in the states.xml file they need to
        // be read and compare them with the existing parameters values
        // already set in the model.
        final String savedParameterFileName = this.getParamFileNameFromState("read");

        if(savedParameterFileName != null)
        {
            final FewsXMLParser xmlParser = new FewsXMLParser(_logger);

            xmlParser.parseParameters(savedParameterFileName, getPreviousParameters());
            setNeedCarryoverTransfer(!((ModelParameters)getParameters()).equalsDefaultParams(getPreviousParameters()));

        }

        // stop to take a reading
        _stopWatch.stop();
        final double timeOnePeriod = _stopWatch.getElapsedTimeInSeconds();

        // restart to continue reading
        _stopWatch.restart();
        /** --------------------------run model -------------------------- */
        execute();

        // stop to take a reading
        _stopWatch.stop();
        final double timeTwoPeriod = _stopWatch.getElapsedTimeInSeconds() - timeOnePeriod;

        _logger.log(Logger.DEBUG, justDriverName + " time spent executing model: " + timeTwoPeriod + " sec."); // timeTwoPeriod

        // restart to continue reading
        _stopWatch.restart();

        this.outputResults();

        // stop to take a reading
        _stopWatch.stop();
        final double timeThreePeriod = _stopWatch.getElapsedTimeInSeconds() - timeOnePeriod - timeTwoPeriod;

        _logger.log(Logger.DEBUG, justDriverName
            + " time spent writing output xml files (output time series and output states): " + timeThreePeriod
            + " sec."); // timeThreePeriod

        // restart to continue reading
        _stopWatch.restart();
    }

    /**
     * This private method produces output time series file, over-write states.xml, save current parameter, and output
     * statesO.txt. For now, it seems that only models need to output results, while utility programs output nothing.
     */
    private void outputResults() throws Exception
    {
        // write the resultTime series to Fews timeseries xml files
        writeOutputTimeseries();

        super.writeStatesMetaFile();

        // write the parameters to state directory
        final String outputDir = this.getParamFileNameFromState("write");
        if(outputDir != null)
        {
            // Copies src file to dst file.
            // If the dst file does not exist, it is created
            final File src = new File(_runInfo.getInputParameterFile());
            final File dst = new File(outputDir);

            final InputStream in = new FileInputStream(src);
            final OutputStream out = new FileOutputStream(dst);

            // Transfer bytes from in to out
            final byte[] buf = new byte[1024];
            int len;
            while((len = in.read(buf)) > 0)
            {
                out.write(buf, 0, len);
            }
            in.close();
            out.close();
        }

        // output the state at the end of the run to a file
        _state.writeState(getOutputStateFile(), _logger);
    }

    /**
     * Validate {@link #_runInfo}. Trim {@link #_drivingTs} to the exact computation start time and end time, based on
     * run_info.xml. Go through each RegularTimeSeries in _tsList to see if it has enough data and it is in sync with
     * {@link #_drivingTs}. If check fails, throws an Exception.
     * <p>
     * Note: 1)this method assumes {@link #_drivingTs} has been set already; 2)For any models which wants more
     * validation, it needs to have its own {@link #runModelDriverValidation()};
     */
    protected void runModelDriverValidation() throws Exception
    {
        super.runDriverValidation();

        setOutputLocationId();

        /*
         * Set _startHourOfDay according to the property StartHourOfDay in run_info.xml
         */
        if(getDriverProperties().containsKey(OHDConstants.START_HOUR_OF_DAY))
        {
            _startHourOfDay = Integer.parseInt(getDriverProperties().getProperty(OHDConstants.START_HOUR_OF_DAY));

            _logger.log(Logger.DEBUG, OHDConstants.START_HOUR_OF_DAY + " exists in run info. Set start hour of day to "
                + _startHourOfDay + " GMT.");
        }
        else
        {
            _logger.log(Logger.DEBUG, OHDConstants.START_HOUR_OF_DAY + " does not exist in run info.");
        }

        /*
         * Adjust _startHourOfDay if the property StartLocalHourOfDay exists in run_info.xml
         */
        if(getDriverProperties().containsKey(OHDConstants.START_LOCAL_HOUR_OF_DAY))
        {
            _startHourOfDay = Integer.parseInt(getDriverProperties().getProperty(OHDConstants.START_LOCAL_HOUR_OF_DAY))
                - _runInfo.getTimeZoneRawOffsetInHours();

            _logger.log(Logger.DEBUG, OHDConstants.START_LOCAL_HOUR_OF_DAY + " in run info="
                + getDriverProperties().getProperty(OHDConstants.START_LOCAL_HOUR_OF_DAY)
                + "  Time Zone Offset(Hour) =" + _runInfo.getTimeZoneRawOffsetInHours()
                + "  Set start hour of the day to " + _startHourOfDay + " GMT.");
        }
        else
        {
            _logger.log(Logger.DEBUG, OHDConstants.START_LOCAL_HOUR_OF_DAY + " does not exist in run info.");
        }

        _logger.log(Logger.DEBUG, "Start hour of the day is at " + _startHourOfDay + " o'clock GMT.");

        runInputTsValidation();

        //if reach here, no Exception has been thrown, all checks must have passed
        _logger.log(Logger.DEBUG, "ModelDriver validation has passed");

        return;
    }

    /**
     * Set {@link #_outputLocationId}. BaseFlow needs to override this method, since it may have none input TS and the
     * properties of "outputLocationId" is not required.
     */
    protected void setOutputLocationId() throws Exception
    {
        _outputLocationId = _drivingTs.getLocationId();

        if(getDriverProperties().containsKey(OHDConstants.OUTPUT_LOCATION_ID))
        {
            _outputLocationId = getDriverProperties().getProperty(OHDConstants.OUTPUT_LOCATION_ID);
        }
    }

    /**
     * Make sure {@link #_drivingTs} is not NULL; Then check all the TSs inside {@link #_tsList} covers the time period
     * defined in run_info.xml; finally check all TSs inside _tsList is in sync with _drivingTs. Note: this method does
     * not check MOD TSs in sync with _drivingTs. The related subclass ModelDriver has its modsTsList and need to call
     * {@link #checkInputTsInSync(List)} itself. Called in {@link #runModelDriverValidation()}<br>
     * Note: added a feature to check _startHourOfDay is a time step in sync with _drivingTs.
     */
    protected void runInputTsValidation() throws Exception
    {

        if(_drivingTs == null)
        {
            throw new Exception("Cannot run ModelDriver.runDriverValidation(), because _drivingTs is null.");
        }

        if(_drivingTs.isInTimeSteps(_startHourOfDay) == false)
        {
            throw new Exception("Input time series is out of sync with the start of day(" + _startHourOfDay + "GMT)!");
        }

        /* -------- trim _drivingTs to the exact start time and end time based on run_info.xml -------- */
        _drivingTs.trimTimeSeriesAtStartWithCheck(getComputationStartTime());
        _drivingTs.trimTimeSeriesAtEndWithCheck(getComputationEndTime());

        /* ----------------validate the RTS in inputs.xml has enough data------------ */
        for(final RegularTimeSeries rts: _tsList)
        {
            //FB27113 - Extend some input TS (i.e., PELV, RQOT data type) for WaterCoach, if tsType is allowed missing value.
            if (rts.getEndTime() < getComputationEndTime())
            {
               boolean missingAllowed = NwsrfsDataTypeMappingReader.areMissingValuesAllowed(rts.getTimeSeriesType(), _logger);
            
               if (missingAllowed == true)
               {
                  _logger.log(Logger.WARNING, rts.getTimeSeriesType() + " input data endDate (" + DateTime.getDateTimeStringFromLong(rts.getEndTime(),OHDConstants.GMT_TIMEZONE)+
            	              ") does not stop at the model endRun (" + DateTime.getDateTimeStringFromLong(getComputationEndTime(),OHDConstants.GMT_TIMEZONE) + 
            	              "). It is extended beyond T0.");
            	
            	  rts.extendTimeSeries(getComputationEndTime(), OHDConstants.MISSING_DATA);
               }   
            }
        	
            //check if has enough data
            rts.checkHasEnoughData(getInitialStateTime(), getComputationEndTime());
            
            _logger.log(Logger.DEBUG, "Input time series " + rts.getTimeSeriesType() + " has enough data.");
        }

        checkInputTsInSync(_tsList);

        return;

    }

    /**
     * Check every TSs in the parameter list is in sync with {@link #_drivingTs}. Otherwise, throw an Exception.
     * 
     * @param list
     * @throws Exception
     */
    protected void checkInputTsInSync(final List<RegularTimeSeries> list) throws Exception
    {
        for(final RegularTimeSeries rts: list)
        {
            //check if in sync with _drivingTs
            rts.checkSyncness(_drivingTs);
            _logger.log(Logger.DEBUG,
                        "Input time series " + rts.getTimeSeriesType() + " is in sync with "
                            + _drivingTs.getTimeSeriesType());

        } //close for loop
    }

    public ModelParameters getPreviousParameters()
    {
        return _savedParameters;
    }

    public void setPreviousParameters(final ModelParameters params)
    {
        _savedParameters = params;
    }

    public int getDrivingTsInterval()
    {
        return _drivingTs.getIntervalInHours();
    }

    public long getDrivingTsIntervalInMillis()
    {
        return _drivingTs.getIntervalInMillis();
    }

    public RegularTimeSeries getDrivingRTS()
    {
        return _drivingTs;
    }

    public void setDrivingRTS(final RegularTimeSeries drivtingTs)
    {
        _drivingTs = drivtingTs;
    }

    /**
     * Returns the time that computation starts, which is run start time(initial state time) plus
     * getDrivingTsIntervalInMillis().
     * <p>
     * Some models may not have their drivingTs yet, calling this method at wrong timing could cause
     * NullPointerException.
     */
    public long getComputationStartTime()
    {
        return _runInfo.getRunStartTimeLong() + getDrivingTsIntervalInMillis();
    }

    /**
     * Return the run start time, specified in run_info.xml
     */
    public long getRunStartTimeLong()
    {
        return _runInfo.getRunStartTimeLong();
    }

    /**
     * Return the time that the model computation ends, specified in run_info.xml
     */
    @Override
    public long getComputationEndTime()
    {
        return _runInfo.getRunEndTimeLong();
    }

    /*
     * this method is used for non java models when that need to convert a mod in xml format to ascii text
     */
    public String getModsStringFromTS(final List<RegularTimeSeries> inputModsList) throws Exception
    {
        return "This method should be overriden by non java models";
    }

    /**
     * @param inputModsList
     * @return
     * @throws Exception
     */
    public String getModsStringFromModValues(final List<ModValues> inputModsList) throws Exception
    {
        return "This method should be overriden by non java models";
    }

    /**
     * @param needCox is set true if the Params is different from the previous Params when the state was saved,
     *            otherwise set to false(also default value), no carryover transfer needed.
     */
    public void setNeedCarryoverTransfer(final boolean needCox)
    {
        _needCarryoverTransfer = needCox;
    }

    /**
     */
    public boolean needCarryoverTransfer()
    {
        return _needCarryoverTransfer;
    }

    /**
     * Get the parameter name from the model state object.
     * 
     * @return
     */
    private String getParamFileNameFromState(final String location)
    {
        String fileName = null;
        String key = null;
        for(final String keyFileName: getState().getStateLocation().keySet())
        {
            if(keyFileName.toLowerCase().contains("xml"))
            {
                key = keyFileName;
                break;
            }
        }

        if(key != null)
        {
            if(location.equals("write"))
            {
                fileName = getState().getStateLocation().get(key).getWriteLocation();
            }
            else
            {
                // location = "read"
                fileName = getState().getStateLocation().get(key).getReadLocation();
            }
        }

        return fileName;
    }

} //close class
