package ohd.hseb.util.fews.ohdmodels;

import java.text.ParseException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import ohd.hseb.util.Logger;
import ohd.hseb.util.fews.GroupOfParameters;
import ohd.hseb.util.fews.GroupOfParameters.ValidPeriod;
import ohd.hseb.util.fews.OHDConstants;
import ohd.hseb.util.fews.ParameterType;
import ohd.hseb.util.fews.ParameterType.Row;
import ohd.hseb.util.fews.ParameterType.Table;
import ohd.hseb.util.fews.Parameters;

import org.xml.sax.SAXException;

public class ModelParameters extends Parameters
{

    private int _validPeriodCount = 0;

    /**
     * This class assumes the _logger is set using setLogger() for writing out diagnostics
     */
    public ModelParameters()
    {
        // We need to keep the order of insertion to be able to get the last inserted ValidPeriod object when repeated.
    }

    public double getDoubleDataParameter(final String paramName) throws Exception
    {
        return this.getParamWithValidation(paramName, ParameterType.doubleType).getDblValue();
    }

    /**
     * Extract a table from parameter xml file and return them as an array of double.
     */
    public double[] getDoubleArrayParameter(final String paramTag) throws Exception
    {

        final Table table = this.getTableDataParameter(paramTag);//the array of double is a table in params.xml

        final double[] doubleArray = new double[table.getRow().size()];
        int index = 0;

        for(final Row row: table.getRow())
        {
            doubleArray[index] = Double.parseDouble(row.getA());

            index++;
        }

        return doubleArray;
    }
    /**
     * Extract a table with more than one column from parameter xml file and return them as an array of double.
     */
    public double[] getDoubleArrayFromTableParameter(final String paramTag) throws Exception
    {

        final Table table = this.getTableDataParameter(paramTag);//the array of double is a table in params.xml

        final ArrayList<String> listvalues = new ArrayList<String>();
     
        for(final Row row: table.getRow())
        {
        	String[] strsplit = row.toString().split(OHDConstants.NEW_LINE);
        	
        	for(int i = 0; i < strsplit.length; i++)
        	{
        		char alph = strsplit[i].charAt(0);
        		
        		listvalues.add(row.get(alph));
        	}
        }
        
        final double[] doubleArray = new double[listvalues.size()];
        
        for(int i = 0; i < listvalues.size(); i++)
        {
        	doubleArray[i] = Double.valueOf(listvalues.get(i));
        }
        
        return doubleArray;
    }

    /**
     * Extract tables from different group in parameter xml file. Then convert each table to an array of double. Then
     * returns a map with key of ValidPeriod objects and with value of double array object.
     * 
     * @throws Exception
     */
    public Map<ValidPeriod, double[]> getDoubleArrayParameterMap(final String paramTag) throws Exception
    {
        double[] newArray = null;
        int tableSize = 0;
        final Map<ValidPeriod, double[]> paramMap = new HashMap<ValidPeriod, double[]>();

        for(final GroupOfParameters groupParams: super._validPeriodParametersMap.values())
        {
            if(groupParams.getParameterMap().containsKey(paramTag) == false
                || groupParams.getParameterMap().get(paramTag).getType() != ParameterType.tableType)
            {
                throw new Exception("The parameter " + paramTag + " in the group " + groupParams.getId()
                    + " does not exist or is not Table.");
            }

            final Table table = groupParams.getParameterMap().get(paramTag).getTable();

            // convert the table result to double[]
            if(table != null && table.getRow() != null && (tableSize = table.getRow().size()) > 0)
            {
                newArray = new double[tableSize];
                for(int i = 0; i < tableSize; i++)
                {
                    final Row row = table.getRow().get(i);
                    newArray[i] = Double.parseDouble(row.getA());
                }
            }

            paramMap.put(groupParams.getValidPeriod(), newArray);
        }//close for loop

        return paramMap;
    }

    public int getIntDataParameter(final String paramName) throws Exception
    {
        return this.getParamWithValidation(paramName, ParameterType.integerType).getIntValue();
    }

    /**
     * Return String value of a parameter. If the parameter does not exist, throw an Exception.
     * <p>
     * Note: the returned String may contain leading or trailing spaces. It is not trimmed.
     */
    public String getStringDataParameter(final String tag) throws Exception
    {
        if(isParamExisting(tag) == false)
        {
            throw new Exception("The parameter with id " + tag + " was not found.");
        }

        return super._validPeriodParametersMap.get(OHDConstants.DEFAULT_VALID_PERIOD)
                                              .getParameterMap()
                                              .get(tag)
                                              .getStringValue();
    }

    /**
     * Check the parameter exists or not in the default GroupOfParameters and if it is the correct parameter type. If
     * not, throws an Exception; if passing the two checks, returns the parameter.
     */
    private ParameterType getParamWithValidation(final String paramName, final int paramType) throws Exception
    {
        final Map<String, ParameterType> defaultParamMap = super._validPeriodParametersMap.get(OHDConstants.DEFAULT_VALID_PERIOD)
                                                                                          .getParameterMap();

        if(defaultParamMap.containsKey(paramName) == false)
        {
            throw new Exception("The parameter " + paramName + " does not exist.");
        }

        final ParameterType param = defaultParamMap.get(paramName);

        if(param.getType() != paramType)
        {
            throw new Exception("The parameter " + paramName + " is not the type " + ParameterType.typeName[paramType]);
        }

        return param;
    }

    /**
     * Returns the parameter value(true or false). The parameter is supposed to have "boolValue" tag and value of "true"
     * or "false" in params.xml file. But to be backward compatible, for now, it can also have "stringValue" as the tag
     * and the value be "YES" or "NO".<br>
     * Note: after RFCs have run the migration script getting rid of the old parameter xml files, "YES"/"NO" as String
     * type parameter will not exist anymore. The code can be simplified at that time.
     * 
     * @throws Exception
     */
    public boolean getBooleanDataParameter(final String paramName) throws Exception
    {
        final Map<String, ParameterType> defaultParamMap = super._validPeriodParametersMap.get(OHDConstants.DEFAULT_VALID_PERIOD)
                                                                                          .getParameterMap();

        if(defaultParamMap.containsKey(paramName) == false)
        {
            throw new Exception("The parameter " + paramName + " does not exist.");
        }

        final ParameterType param = defaultParamMap.get(paramName);

        if(param.getType() == ParameterType.stringType)
        {
            _logger.log(Logger.WARNING, "The parameter " + paramName
                + " is String type. It can be accepted for now. But it is better to change to boolean type.");

            final String result = param.getStringValue();
            if(OHDConstants.YES_STRING.equalsIgnoreCase(result))
            {
                return true;
            }
            else if(OHDConstants.NO_STRING.equalsIgnoreCase(result))
            {
                return false;
            }
            else
            {
                throw new Exception("The parameter " + paramName
                    + " as a String can only be YES or NO. Or it can be changed to boolValue with true or false.");
            }
        }
        else if(param.getType() == ParameterType.booleanType)
        {
            return param.getBoolValue();
        }
        else
        {
            throw new Exception("The parameter " + paramName + " is neither String type or Boolean type.");
        }

    }

    public Table getTableDataParameter(final String tag) throws Exception
    {

        if(isParamExisting(tag) == false)
        {
            throw new Exception("The parameter with id " + tag + " was not found.");
        }

        return super._validPeriodParametersMap.get(OHDConstants.DEFAULT_VALID_PERIOD)
                                              .getParameterMap()
                                              .get(tag)
                                              .getTable();
    }

    /**
     * Unclear return type, compared to {@link #getIntDataParameter(String)}, {@link #getDoubleDataParameter(String)}
     * and {@link #getStringDataParameter(String)}. Get the parameter Object that will be cast by the called method. If
     * id is DEFAILT_GROUP it assume the default valid period (null valid period) was selected and get the "default"
     * valid period from the Map _validPeriodParametersMap if a valid period object is passed it will use to retrieve
     * data from the map. if Id is a string, this represent the group Id and it will be used to retrieve data from
     * _groupParametersMap
     * 
     * @throws Exception
     */
    private Object getParameter(final GroupOfParameters.ValidPeriod vPeriod, final String tag, final Object type) throws Exception
    {

        Object returnObject = null;
        GroupOfParameters.ValidPeriod validPeriod;

        if(vPeriod != null)
        {
            validPeriod = vPeriod;
        }
        else
        {
            validPeriod = OHDConstants.DEFAULT_VALID_PERIOD; //actually OHDConstants.DEFAULT_VALID_PERIOD is null too
        }

        if(super._validPeriodParametersMap.get(validPeriod).getParameterMap().containsKey(tag) == false)
        {
            throw new Exception("The parameter with id " + tag + " was not found.");
        }

        /** end log */

        if(type.equals("Integer"))
        {
            returnObject = super._validPeriodParametersMap.get(validPeriod).getParameterMap().get(tag).getIntValue();
        }
        else if(type.equals("String"))
        {
            returnObject = super._validPeriodParametersMap.get(validPeriod).getParameterMap().get(tag).getStringValue();
        }
        else if(type.equals("Double"))
        {
            returnObject = super._validPeriodParametersMap.get(validPeriod).getParameterMap().get(tag).getDblValue();
        }
        else if(type.equals("Boolean"))
        {
            returnObject = super._validPeriodParametersMap.get(validPeriod).getParameterMap().get(tag).getBoolValue();
        }
        else if(type.equals("Table"))
        {
            returnObject = super._validPeriodParametersMap.get(validPeriod).getParameterMap().get(tag).getTable();
        }
        else if(type.equals("Object"))
        {
            if(super._validPeriodParametersMap.get(validPeriod).getParameterMap().get(tag).getIntValue() != null)
            {
                returnObject = super._validPeriodParametersMap.get(validPeriod)
                                                              .getParameterMap()
                                                              .get(tag)
                                                              .getIntValue();
            }
            else if(super._validPeriodParametersMap.get(validPeriod).getParameterMap().get(tag).getStringValue() != null)
            {
                returnObject = super._validPeriodParametersMap.get(validPeriod)
                                                              .getParameterMap()
                                                              .get(tag)
                                                              .getStringValue();
            }
            else if(super._validPeriodParametersMap.get(validPeriod).getParameterMap().get(tag).getDblValue() != null)
            {
                returnObject = super._validPeriodParametersMap.get(validPeriod)
                                                              .getParameterMap()
                                                              .get(tag)
                                                              .getDblValue();
            }
            else if(super._validPeriodParametersMap.get(validPeriod).getParameterMap().get(tag).getBoolValue() != null)
            {
                returnObject = super._validPeriodParametersMap.get(validPeriod)
                                                              .getParameterMap()
                                                              .get(tag)
                                                              .getBoolValue();
            }
            else if(super._validPeriodParametersMap.get(validPeriod).getParameterMap().get(tag).getTable() != null)
            {
                returnObject = super._validPeriodParametersMap.get(validPeriod).getParameterMap().get(tag).getTable();
            }
        }

        if(returnObject == null)
        {
            throw new Exception("The parameter with id " + tag + " is present, but is wrong type.");
        }

        return returnObject;

    }

    /**
     * Gets all parameters for a giving group.
     * 
     * @param groupId The id of the group of parameters to retrieve.
     * @return a ModelGroupOfParameters object containing all parameters values for the giving group id.
     */
    public GroupOfParameters getParameters(final String groupId)
    {
        GroupOfParameters params = null;
        for(final GroupOfParameters groupParam: super._validPeriodParametersMap.values())
        {
            if(groupParam != null && groupParam.getId().equalsIgnoreCase(groupId))
            {
                params = groupParam;
                break;
            }
        }
        return params;
    }

    /**
     * Check if the parameter existing or not
     */
    public boolean isParamExisting(final String tag)
    {
        return super._validPeriodParametersMap.get(OHDConstants.DEFAULT_VALID_PERIOD).isParameterExisting(tag);
    }

    public Map<GroupOfParameters.ValidPeriod, GroupOfParameters> getParamsMap()
    {
        return Collections.unmodifiableMap(super._validPeriodParametersMap);
    }

    /**
     * @param validPeriod
     * @param paramsMap
     */
    public void setParametersForValidPeriodFromMap(final GroupOfParameters.ValidPeriod validPeriod,
                                                   final Map<String, ParameterType> paramsMap)
    {
        if(super._validPeriodParametersMap.get(validPeriod) == null)
        {
            final GroupOfParameters group = new GroupOfParameters();
            group.setId(null);
            super._validPeriodParametersMap.put(validPeriod, group);
        }
        super._validPeriodParametersMap.get(validPeriod).getParameterMap().putAll(paramsMap);
    }

    @Override
    public void setNumberOfValidPeriods(final int validPeriodNumber)
    {
        _validPeriodCount = validPeriodNumber;
    }

    @Override
    public int getNumberOfValidPeriods()
    {
        return _validPeriodCount;
    }

    /**
     * This method is called in OHDFewsAdapter.java, near the end finishing parsing params.xml (inside
     * ParameterHandler.java endDocument()).
     * <p>
     * _xmlParser.parseParameters(_parameterFileName); Or:
     * <p>
     * SAXParser parser = factory.newSAXParser();<br>
     * parser.parse(inputXMLFileName, paramHandler);
     */
    @Override
    public void setParametersFromList(final List<GroupOfParameters> paramsList) throws Exception
    {

        // Check if valid period already exist in map, we need to add/replace the new values to these default otherwise it will overwrite all existing values.
        // need to iterate in all keys.

        if(_validPeriodCount > 1 || _validPeriodCount == 0)
        {
            final Exception spe = new SAXException("The number of default valid periods is Zero or bigger than One");
            throw spe;
        }

        for(final GroupOfParameters groupParams: paramsList)
        {
            super._validPeriodParametersMap.put(groupParams.getValidPeriod(), groupParams);
            //super.addGroupOfParameters(groupParams);
        }

        extractValuesFromMap(); //retrieve all the values from _paramsMap to subclass instance variables if the method has been overwriten
    }

    @Override
    public List<GroupOfParameters> getParamsList()
    {
        return Collections.unmodifiableList(new ArrayList<GroupOfParameters>(super._validPeriodParametersMap.values()));
    }

    /**
     * overloaded method for simplicity: param as <doubleData> in params.xml
     * 
     * @param tag
     * @param value - double
     */
    public void insertParameter(final String tag, final double value)
    {
        this.insertParameter(tag, Double.valueOf(value));
    }

    public void insertParameter(final String tag, final double[] values)
    {
        final Table table = new Table();
        final Row columnTypes = new Row();
        Row row = null;
        for(int i = 0; i < values.length; i++)
        {
            row = new Row();
            row.setA(Double.toString(values[i]));
            table.setRow(row);
        }
        columnTypes.setA("double");
        table.setColumnTypes(columnTypes);

        this.insertParameter(tag, table);
    }

    /**
     * overloaded method for simplicity: param as <intData> in params.xml
     * 
     * @param tag
     * @param value - int
     */
    public void insertParameter(final String tag, final int value)
    {
        this.insertParameter(tag, Integer.valueOf(value));
    }

    /**
     * overloaded method for simplicity: param as <boolValue> in params.xml
     * 
     * @param tag
     * @param value - boolean
     */
    public void insertParameter(final String tag, final boolean value)
    {
        this.insertParameter(tag, Boolean.valueOf(value));
    }

    public void insertParameter(final String tag, final ParameterType value)
    {
        insertParameter(tag, value, OHDConstants.DEFAULT_VALID_PERIOD);
    }

    public void insertParameter(final String tag, final Object value)
    {
        GroupOfParameters group = null;
        final ParameterType param = new ParameterType();
        param.setId(tag);
        if(value instanceof Double)
        {
            param.setDblValue((Double)value);
        }
        else if(value instanceof Integer)
        {
            param.setIntValue((Integer)value);
        }
        else if(value instanceof Boolean)
        {
            param.setBoolValue((Boolean)value);
        }
        else if(value instanceof String)
        {
            param.setStringValue((String)value);
        }
        else if(value instanceof Table)
        {
            param.setTable((Table)value);
        }
        //check if the group exist, if so add the parameter to the this group, otherwise we need to create the group
        group = super._validPeriodParametersMap.get(OHDConstants.DEFAULT_VALID_PERIOD);
        if(group != null)
        {
            group.setParameter(param);
        }
        else
        {
            group = new GroupOfParameters();
            group.setId(null);
            group.setParameter(param);
            group.setValidPeriod(OHDConstants.DEFAULT_VALID_PERIOD);
            super._validPeriodParametersMap.put(OHDConstants.DEFAULT_VALID_PERIOD, group);
        }
    }

    /**
     * overloaded method for simplicity: param as
     * <table>
     * in params.xml
     * 
     * @param tag
     * @param value - Table
     */
    public void insertParameter(final String tag, final Table value)
    {
        GroupOfParameters group = null;
        final ParameterType param = new ParameterType();
        param.setId(tag);
        param.setTable(value);
        //check if the group exist, if so add the parameter to the this group, otherwise we need to create the group
        group = super._validPeriodParametersMap.get(OHDConstants.DEFAULT_VALID_PERIOD);
        if(group != null)
        {
            group.setParameter(param);
        }
        else
        {
            group = new GroupOfParameters();
            group.setId(null);
            group.setParameter(param);
            group.setValidPeriod(OHDConstants.DEFAULT_VALID_PERIOD);
            super._validPeriodParametersMap.put(OHDConstants.DEFAULT_VALID_PERIOD, group);
        }
    }

    /**
     * Insert a parameter in the giving validPeriod.
     * 
     * @param tag
     * @param value
     * @param validPeriod
     */
    public void insertParameter(final String tag, final Object value, final GroupOfParameters.ValidPeriod validPeriod)
    {
        GroupOfParameters group = null;
        final ParameterType param = new ParameterType();
        param.setId(tag);
        if(value instanceof Double)
        {
            param.setDblValue((Double)value);
        }
        else if(value instanceof Integer)
        {
            param.setIntValue((Integer)value);
        }
        else if(value instanceof Boolean)
        {
            param.setBoolValue((Boolean)value);
        }
        else if(value instanceof String)
        {
            param.setStringValue((String)value);
        }
        else if(value instanceof Table)
        {
            param.setTable((Table)value);
        }
        //check if the group exist, if so add the parameter to the this group, otherwise we need to create the group
        group = super._validPeriodParametersMap.get(validPeriod);
        if(group != null)
        {
            group.setParameter(param);
        }
        else
        {
            group = new GroupOfParameters();
            group.setId(null);
            group.setParameter(param);
            group.setValidPeriod(validPeriod);
            super._validPeriodParametersMap.put(validPeriod, group);
        }
    }

    /**
     * Warning: the parameter tag is never used in the method.
     * 
     * @param tag
     * @param parameter
     * @param validPeriod
     */
    public void insertParameter(final String tag,
                                final ParameterType parameter,
                                final GroupOfParameters.ValidPeriod validPeriod)
    {

        GroupOfParameters group = null;

        group = super._validPeriodParametersMap.get(validPeriod);
        if(group != null)
        {
            group.setParameter(parameter);
        }
        else
        {
            group = new GroupOfParameters();
            group.setId(null);
            group.setParameter(parameter);
            group.setValidPeriod(validPeriod);
            super._validPeriodParametersMap.put(validPeriod, group);
        }
    }

    /**
     * For Java models(snow17, sacsma, sacsmaHT and UHG), this method can be implemented to extract values from
     * _paramsMap to instance variables to avoid frequently access {@link #_validPeriodParametersMap}. For legacy
     * models, this method remains empty and do nothing.
     */
    @Override
    protected void extractValuesFromMap() throws Exception
    {
        //empty
    }

    @Override
    protected void setValuesToMap() throws Exception
    {

    }

    //In alphabetic order, each line is "parameter_name= value"
    @Override
    public String toString()
    {
        final StringBuilder resultStr = new StringBuilder();

        resultStr.append("Parameters:").append(OHDConstants.NEW_LINE);
        resultStr.append("Version = " + this.getVersion()).append(OHDConstants.NEW_LINE);

        for(final GroupOfParameters params: this._validPeriodParametersMap.values())
        {
            resultStr.append(params.toString()).append(OHDConstants.NEW_LINE);
        }

        return resultStr.toString();
    }

    /**
     * Similar to equals method the only difference is that is does not check the whole object only look for the default
     * parameter. The Default parameters is the one with Group Id = "Default" and does not have a valid period (null in
     * this case).
     */
    public boolean equalsDefaultParams(final Object other)
    {
        boolean isParamEqual = false;

        if(other == this)
        {
            isParamEqual = true;
        }
        else if(other == null)
        {
            isParamEqual = false;
        }
        else if(!getClass().equals(other.getClass()))
        {
            isParamEqual = false;
        }
        else if(other instanceof Parameters)
        {
            final ModelParameters otherModelParams = (ModelParameters)other;

            final GroupOfParameters paramsGroupParams = this.getParamsMap().get(OHDConstants.DEFAULT_VALID_PERIOD);
            final GroupOfParameters otherParamsGroupParams = otherModelParams.getParamsMap()
                                                                             .get(OHDConstants.DEFAULT_VALID_PERIOD);

            isParamEqual = paramsGroupParams.equals(otherParamsGroupParams);

        }

        if(isParamEqual)
        {
            _logger.log(Logger.DEBUG, "parameters are equal");
        }
        else
        {
            _logger.log(Logger.DEBUG, "parameters are not equal");
        }

        return isParamEqual;
    }

    @Override
    public boolean equals(final Object other)
    {
        boolean returnVal = false;

        if(other == this)
        {
            returnVal = true;
        }
        else if(other == null)
        {
            returnVal = false;
        }
        else if(!getClass().equals(other.getClass()))
        {
            returnVal = false;
        }
        else if(other instanceof ModelParameters)
        {
            final ModelParameters other2 = (ModelParameters)other;
            returnVal = testEqualsWithNull(this.getVersion(), other2.getVersion());

            final Iterator<GroupOfParameters.ValidPeriod> it = other2.getParamsMap().keySet().iterator();
            // first check the size() is equal
            if(this.getParamsMap().size() != other2.getParamsMap().size())
            {
                _logger.log(Logger.ERROR, "Error: the two ModelParameters objects have different number of parameters");
                return false;
            }

            for(final GroupOfParameters.ValidPeriod validPeriod: this.getParamsMap().keySet())
            {
                final GroupOfParameters.ValidPeriod validPeriodMods = it.next();
                final GroupOfParameters params = this.getParamsMap().get(validPeriod);
                final GroupOfParameters paramsMods = other2.getParamsMap().get(validPeriodMods);

                if(params.equals(paramsMods))
                {
                    continue;
                }
                else
                {
                    _logger.log(Logger.DEBUG, "params are not equal");

                    return false;
                }

            }

        }
        _logger.log(Logger.DEBUG, "params are equal");
        return returnVal;

    }

    private static boolean testEqualsWithNull(final Object a, final Object b)
    {
        return (a == null && b == null) || (a != null && b != null) && (a.equals(b));
    }

    /**
     * Get the parameter for a valid period passed as Long object that represent the date of the long period.
     * 
     * @param date
     * @param tag
     * @param type
     * @return
     * @throws Exception
     */
    public Object getParameter(final Long date, final String tag, final Object type) throws Exception
    {
        GroupOfParameters.ValidPeriod validPeriod = OHDConstants.DEFAULT_VALID_PERIOD;
        // If date is not null and it is found we will use the validPeriod object otherwise the "Default" valid period will be used.
        // "Default" validPeriod is an null valid period. It could be change later to any other value defined.
        if(date != null)
        {
            // used for log only
            //_parameterSearchDate = DbTimeHelper.getDateTimeStringFromLongTime(date);

            try
            {
                for(final GroupOfParameters params: super._validPeriodParametersMap.values())
                {
                    final ValidPeriod period = params.getValidPeriod();
                    if(period != OHDConstants.DEFAULT_VALID_PERIOD)
                    {
                        if(period.getStartDate() != null && period.getEndDate() != null
                            && date >= this.asLong(period.getStartDate()) && date <= this.asLong(period.getEndDate()))
                        {
                            validPeriod = period;
                            //break;
                        }
                        else if(period.getValidBeforeDate() != null && this.asLong(period.getValidBeforeDate()) >= date)
                        {
                            validPeriod = period;
                            //break;
                        }
                        else if(period.getValidAfterDate() != null && this.asLong(period.getValidAfterDate()) <= date)
                        {
                            validPeriod = period;
                            //break;
                        }
                        else if(period.getStartMonthDay() != null && period.getEndMonthDay() != null)
                        {
                            validPeriod = period;
                            //break;
                        }
                        else if(period.getMonthDay() != null && period.getMonthDay().size() > 0)
                        {
                            validPeriod = period;
                            //break;
                        }
                        else if(period.getMonth() != null && period.getMonth().size() > 0)
                        {
                            validPeriod = period;
                            //break;
                        }
                        else if(period.getDay() != null && period.getDay().size() > 0)
                        {
                            validPeriod = period;
                            //break;
                        }
                    }
                } // end for loop
            }
            catch(final ParseException pe)
            {
                final String message = "Error when getting parameter values for " + tag;

                throw new Exception(message);
            }
        }
        return this.getParameter(validPeriod, tag, type);

    }

    /**
     * Unclear return type, compared to {@link #getIntDataParameter(String)}, {@link #getDoubleDataParameter(String)}
     * and {@link #getStringDataParameter(String)}.
     * 
     * @throws Exception
     * @deprecated use specific method for each data type instead, for instance use getDoubleDataParameter() to retrieve
     *             double values.
     */
    @Deprecated
    public Object getParameter(final String tag) throws Exception
    {
        return this.getParameter(OHDConstants.DEFAULT_VALID_PERIOD, tag, "Object");
    }
}
