package ohd.hseb.charter;

import java.awt.Rectangle;
import java.awt.Shape;
import java.awt.geom.AffineTransform;

import org.jfree.chart.JFreeChart;
import org.jfree.chart.plot.DefaultDrawingSupplier;
import org.jfree.ui.Align;
import org.jfree.ui.HorizontalAlignment;
import org.jfree.ui.RectangleEdge;
import org.jfree.util.ShapeUtilities;

import ohd.hseb.hefs.utils.tools.ArrayTools;

/**
 * Constants associated with {@link ChartEngine}, and possibly {@link JFreeChart} in general, along with tools working
 * with those constants. Note that many of these should likely be changed to use enumerators instead of ints. That can
 * be done at a later date.
 * 
 * @author hankherr
 */
public abstract class ChartConstants
{

    public static String NO_SHAPE = "none";
    public static Shape[] SHAPES = ArrayTools.concat(DefaultDrawingSupplier.createStandardSeriesShapes(),
                                                              new Shape[]{ShapeUtilities.createDiagonalCross(3, 0.5f),
                                                                  ShapeUtilities.createRegularCross(3, 0.5f),});
    public static String[] SHAPE_NAMES = {"square", "circle", "up triangle", "diamond", "horizontal rectangle",
        "down triangle", "horizontal ellipse", "right triangle", "vertical rectangle", "left triangle", "x", "cross"};
    public static final Double DEFAULT_SHAPE_SIZE = 1.0;

    public static Integer LEFT_YAXIS_INDEX = 0;
    public static Integer RIGHT_YAXIS_INDEX = 1;
    public static String[] YAXIS_XML_STRINGS = {"left", "right"};
    public static String DOMAIN_AXIS_STRING = "domain";
    /**
     * The value is set to be different than both {@link #LEFT_YAXIS_INDEX} and {@link #RIGHT_YAXIS_INDEX}.
     */
    public static Integer DOMAIN_AXIS = -1;

    public static final int AXIS_IS_TIME = 0;
    public static final int AXIS_IS_NUMERICAL = 1;
    public static final int AXIS_IS_LOGARITHMIC = 2;
    public static final int AXIS_IS_PROBABILITY = 3;
    public static final int AXIS_IS_NORMALIZED_PROBABILITY = 4;
    public static final int AXIS_IS_TRANSLATED = 5;
    public static final int AXIS_IS_CATEGORICAL = 6; //for WRES: This is hidden from GraphGen for now.
    public static final String[] AXIS_CHOICES = {"Time", "Numerical", "Logarithmic", "Probability",
        "Normalized Probability", "Translated"};

    public static String[] POSITION_STRINGS = {"TOP", "RIGHT", "LEFT", "BOTTOM"};
    public static RectangleEdge[] POSITION_EDGE = {RectangleEdge.TOP, RectangleEdge.RIGHT, RectangleEdge.LEFT,
        RectangleEdge.BOTTOM};

    public static String[] ALIGNMENT_STRINGS = {"LEFT", "CENTER", "RIGHT"};
    public static Integer[] ALIGNMENT_INTEGERS = {Align.LEFT, Align.CENTER, Align.RIGHT};
    public static HorizontalAlignment[] ALIGNMENT_HORIZONTAL_ALIGNMENTS = {HorizontalAlignment.LEFT,
        HorizontalAlignment.CENTER, HorizontalAlignment.RIGHT};

    public static String[] IMAGE_ALIGNMENT_STRINGS = {"LEFT", "CENTER", "RIGHT", "FIT"};
    public static Integer[] IMAGE_ALIGNMENT_INTEGERS = {Align.LEFT, Align.CENTER, Align.RIGHT, Align.FIT};

    public static String[] IMAGE_TRANSPARENCY_STRINGS = {"0.0", "0.1", "0.2", "0.3", "0.4", "0.5", "0.6", "0.7", "0.8",
        "0.9", "1.0"};
    public static Double[] IMAGE_TRANSPARENCY_DOUBLES = {0.0, 0.1, 0.2, 0.3, 0.4, 0.5, 0.6, 0.7, 0.8, 0.9, 1.0};

    public static String[] SIGNIFICANT_DIGITS_STRINGS = {"1", "2", "3", "4", "5", "6", "7", "8", "9", "10"};
    public static Integer[] SIGNIFICANT_DIGITS_INTEGERS = {1, 2, 3, 4, 5, 6, 7, 8, 9, 10};

    public static String DATA_SOURCE_RANGE_AXIS_PARAMETER_ID_PROBABILITY = "PROBABILITY";
    public static String DATA_SOURCE_RANGE_AXIS_UNITS_PROBABILITY = "PROB";

    /**
     * @param shapeName Name of shape to acquire.
     * @return The shape.
     */
    public static Shape getShape(final String shapeName)
    {
        return getShape(shapeName, 1.0d);
    }

    /**
     * @param shapeName Name of shape to acquire.
     * @param scale size of shape to transform.
     * @return The shape.
     */
    public static Shape getShape(final String shapeName, final Double scale)
    {
        if(scale == null)
        {
            return getShape(shapeName);
        }

        if(isNoShape(shapeName))
        {
            return null;
        }

        final AffineTransform transformer = new AffineTransform();
        transformer.scale(scale, scale);

        for(int i = 0; i < SHAPE_NAMES.length; i++)
        {
            if(shapeName.equalsIgnoreCase(SHAPE_NAMES[i]))
            {
                //Transform shape before returning!
                return transformer.createTransformedShape(SHAPES[i]);
            }
        }

        //Transform shape before returning!
        return transformer.createTransformedShape(SHAPES[1]); //Circle default.
    }

    /**
     * @return A {@link Shape} that represents no shape. It is a rectangle of size 0.
     */
    public static Shape getNoShape()
    {
        return new Rectangle(0, 0);
    }

    /**
     * @return True if the shape matches what is returned by {@link #getNoShape()}.
     */
    public static boolean isNoShape(final Shape shape)
    {
        if(!(shape instanceof Rectangle))
        {
            return false;
        }
        if((((Rectangle)shape).getWidth() == 0) && (((Rectangle)shape).getHeight() == 0))
        {
            return true;
        }
        return false;
    }

    /**
     * @param shapeName Name of the shape to check.
     * @return True if the shape specifies no shape identifier.
     */
    public static boolean isNoShape(final String shapeName)
    {
        return shapeName.equalsIgnoreCase(NO_SHAPE);
    }

    /**
     * General method used to search an array of strings for a string and return a corresponding object.
     * 
     * @param identityString String to search for.
     * @param strings String array to search. Match ignores case.
     * @param objects Objects to return.
     * @return Object found or null if no match made.
     */
    public static Object determineObjectBasedOnIdentityString(final String identityString,
                                                              final String[] strings,
                                                              final Object[] objects)
    {
        for(int i = 0; i < strings.length; i++)
        {
            if(strings[i].equalsIgnoreCase(identityString))
            {
                return objects[i];
            }
        }
        return null;
    }

    /**
     * General method used to search an array of objects for an object (using equals method) and return the matching
     * identifying string.
     * 
     * @param obj Object to look for.
     * @param strings Objects to search.
     * @param objects Identifying strings.
     * @return Identifying string for the object or null if no match made.
     */
    public static String determineIdentitifyingStringFromObject(final Object obj,
                                                                final String[] strings,
                                                                final Object[] objects)
    {
        for(int i = 0; i < objects.length; i++)
        {
            if(objects[i].equals(obj))
            {
                return strings[i];
            }
        }
        return null;
    }

    /**
     * @param positionString A string within {@link #POSITION_STRINGS}.
     * @return {@link RectangleEdge} corresponding to the position string.
     */
    public static RectangleEdge determineRectangleEdgeForPosition(final String positionString)
    {
        return (RectangleEdge)determineObjectBasedOnIdentityString(positionString, POSITION_STRINGS, POSITION_EDGE);
    }

    /**
     * @param edge {@link RectangleEdge} within JFreeChart for which we need an identifying string.
     * @return An identifying string that can be used in XML.
     */
    public static String determinePositionStringForRectangleEdge(final RectangleEdge edge)
    {
        return determineIdentitifyingStringFromObject(edge, POSITION_STRINGS, POSITION_EDGE);
    }

    /**
     * @param alignString A {@link String} within {@link #ALIGNMENT_STRINGS}.
     * @return JFreeChart {@link Integer} constant corresponding to the alignment string or null if none.
     */
    public static Integer determineAlignmentIntegerForAlignmentString(final String alignString)
    {
        return (Integer)determineObjectBasedOnIdentityString(alignString, ALIGNMENT_STRINGS, ALIGNMENT_INTEGERS);
    }

    /**
     * @param alignment Integer identifying the alignment within JFreeChart.
     * @return An identifying string that can be used in XML.
     */
    public static String determineAlignmentStringForAlignmentInteger(final Integer alignment)
    {
        return determineIdentitifyingStringFromObject(alignment, ALIGNMENT_STRINGS, ALIGNMENT_INTEGERS);
    }

    /**
     * @param alignString {@link String} identifier for the alignment.
     * @return The {@link HorizontalAlignment} defined for it.
     */
    public static HorizontalAlignment determineHorizontalAlignmentFromAlignmentString(final String alignString)
    {
        return (HorizontalAlignment)determineObjectBasedOnIdentityString(alignString,
                                                                         ALIGNMENT_STRINGS,
                                                                         ALIGNMENT_HORIZONTAL_ALIGNMENTS);
    }

    /**
     * @param alignString A string within {@link #IMAGE_ALIGNMENT_STRINGS}.
     * @return JFreeChart {@link Integer} constant corresponding to the alignment string or null if none.
     */
    public static Integer determineImageAlignmentForAlignmentString(final String alignString)
    {
        return (Integer)determineObjectBasedOnIdentityString(alignString,
                                                             IMAGE_ALIGNMENT_STRINGS,
                                                             IMAGE_ALIGNMENT_INTEGERS);
    }

    /**
     * @param alignment {@link Integer} identifying the alignment within JFreeChart.
     * @return An identifying string that can be used in XML.
     */
    public static String determineImageAlignmentStringForAlignment(final Integer alignment)
    {
        return determineIdentitifyingStringFromObject(alignment, IMAGE_ALIGNMENT_STRINGS, IMAGE_ALIGNMENT_INTEGERS);
    }

    /**
     * @param alignString A string within {@link #IMAGE_TRANSPARENCY_STRINGS}.
     * @return {@link Double} constant corresponding to the transparency or null if none.
     */
    public static Double determineImageTransparencyForTransparencyString(final String transparencyString)
    {
        return (Double)determineObjectBasedOnIdentityString(transparencyString,
                                                            IMAGE_TRANSPARENCY_STRINGS,
                                                            IMAGE_TRANSPARENCY_DOUBLES);
    }

    /**
     * @param alignment {@link Integer} identifying the alignment within JFreeChart.
     * @return An identifying string that can be used in XML.
     */
    public static String determineImageTransparencyStringForTransparency(final Double alignment)
    {
        return determineIdentitifyingStringFromObject(alignment,
                                                      IMAGE_TRANSPARENCY_STRINGS,
                                                      IMAGE_TRANSPARENCY_DOUBLES);
    }

    /**
     * @param alignString A string within {@link #SIGNIFICANT_DIGITS_STRINGS}.
     * @return JFreeChart {@link Integer} constant corresponding to the alignment string or null if none.
     */
    public static Integer determineSignificantDigitsForSignificantDigitsString(final String transparencyString)
    {
        return (Integer)determineObjectBasedOnIdentityString(transparencyString,
                                                             SIGNIFICANT_DIGITS_STRINGS,
                                                             SIGNIFICANT_DIGITS_INTEGERS);
    }

    /**
     * @param alignment Integer identifying the alignment within JFreeChart.
     * @return An identifying string that can be used in XML.
     */
    public static String determineSignificantDigitsStringForSignificantDigits(final Integer alignment)
    {
        return determineIdentitifyingStringFromObject(alignment,
                                                      SIGNIFICANT_DIGITS_STRINGS,
                                                      SIGNIFICANT_DIGITS_INTEGERS);
    }

    /**
     * @param axisString One of {@link #DOMAIN_AXIS_STRING} or an element in {@link #YAXIS_XML_STRINGS}.
     * @return {@link #DOMAIN_AXIS} or one of {@link #LEFT_YAXIS_INDEX} or {@link #RIGHT_YAXIS_INDEX}. Null is returned
     *         if string is not recognized.
     */
    public static Integer determineAxisIndexFromString(final String axisString)
    {
        if(axisString.equalsIgnoreCase(DOMAIN_AXIS_STRING))
        {
            return DOMAIN_AXIS;
        }
        for(int i = 0; i < YAXIS_XML_STRINGS.length; i++)
        {
            if(YAXIS_XML_STRINGS[i].equalsIgnoreCase(axisString))
            {
                return i;
            }
        }
        return null;
    }

    /**
     * @param typeString One of {@link #AXIS_CHOICES}.
     * @return One of the AXIS_IS* static variables specifying the axis type.
     */
    public static int determineAxisTypeIntFromAxisTypeString(final String typeString)
    {
        for(int i = 0; i < AXIS_CHOICES.length; i++)
        {
            if(AXIS_CHOICES[i].equalsIgnoreCase(typeString))
            {
                return i;
            }
        }
        return -1;
    }
    
    /**
     * @param type
     * @return True if the provided type is {@link #AXIS_IS_CATEGORICAL}.
     */
    public static boolean isAxisTypeCategorical(final int type)
    {
        return (type == AXIS_IS_CATEGORICAL);
    }

    /**
     * @param type Type int to check
     * @return True if the axis type is translated; false otherwise.
     */
    public static boolean isAxisTypeTranslated(final int type)
    {
        return (type == AXIS_IS_TRANSLATED);
    }

    /**
     * @return True if the provided axis type int denotes a type that is compatible with {@link #AXIS_IS_TIME}.
     */
    public static boolean isAxisTypeTime(final int type)
    {
        try
        {
            ChartTools.checkCompatibilityOfAxis(AXIS_IS_TIME, type);
            return true;
        }
        catch(final Exception e)
        {
            return false;
        }
    }

}
