package ohd.hseb.charter;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.TimeZone;

import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.LogManager;
import org.jfree.chart.JFreeChart;
import org.jfree.chart.axis.LogarithmicAxis;
import org.jfree.chart.axis.ValueAxis;
import org.jfree.chart.plot.CombinedDomainXYPlot;
import org.jfree.chart.plot.XYPlot;
import org.jfree.data.time.TimeSeriesCollection;
import org.jfree.data.xy.XYDataset;

import com.google.common.collect.Maps;

import ohd.hseb.charter.datasource.XYChartDataSource;
import ohd.hseb.charter.datasource.XYChartDataSourceException;
import ohd.hseb.charter.datasource.instances.CategoricalXYChartDataSource;
import ohd.hseb.charter.jfreechartoverride.ExtendedCombinedDomainXYPlot;
import ohd.hseb.charter.jfreechartoverride.ExtendedXYPlot;
import ohd.hseb.charter.parameters.AxisParameters;
import ohd.hseb.charter.parameters.ChartDrawingParameters;
import ohd.hseb.charter.parameters.ChartParametersException;
import ohd.hseb.charter.parameters.DataSourceDrawingParameters;
import ohd.hseb.charter.parameters.SeriesDrawingParameters;
import ohd.hseb.charter.parameters.SubPlotParameters;
import ohd.hseb.charter.parameters.ThresholdListParameters;
import ohd.hseb.charter.parameters.ThresholdParameters;
import ohd.hseb.charter.plotter.XYChartPlotter;
import ohd.hseb.charter.plotter.XYChartPlotterException;
import ohd.hseb.charter.plotter.XYChartPlotterFactory;
import ohd.hseb.charter.tools.DateAxisPlus;
import ohd.hseb.charter.tools.NormalizedProbabilityAxis;
import ohd.hseb.charter.tools.NumberAxisOverride;
import ohd.hseb.charter.tools.ProbabilityAxis;
import ohd.hseb.charter.tools.TranslatedAxis;
import ohd.hseb.charter.translator.AxisTranslatorException;
import ohd.hseb.charter.translator.AxisTranslatorParameters;
import ohd.hseb.hefs.utils.arguments.ArgumentsProcessor;
import ohd.hseb.hefs.utils.arguments.DefaultArgumentsProcessor;
import ohd.hseb.util.misc.HStopWatch;

/**
 * This is the main entry-point to the OHD charting tool. It can be constructed in four ways:<br>
 * <br>
 * 1. Pass in a list of data sources, in which case the _chartParameters are initialized so that every source and every
 * chart series and every subplot has parameters, and all parameters are initialized to a default value.<br>
 * 2. Pass in a list of data sources and a set of user-specified override ChartDrawingParameters. In this case, the
 * default parameters are built as above, and then the override parameters applied on top of them.<br>
 * 3. Pass in another ChartEngine. This will be a copy of that ChartEngine, but will share the underlying data source
 * data.<br>
 * 4. Pass in a list of ChartEngine objects. This result in a chart that is a combination of the passed in charts,
 * combined in the order specified by the list (later charts overriding earlier charts). The default parameters are
 * computed after the charts are combined, and the override parameters are a combination of all of the override
 * parameters for the passed in ChartEngines (again, combined in list order). The data source indices are increased for
 * the engines after the first so that they can be put into a single list.<br>
 * <br>
 * Each of the constructors includes an ArgumentsProcessor object, which may be used within the ChartParameters object.
 * For example, arguments can be used to specify flexible labels. Whenever an ArgumentsProcessor is included in the
 * constructor argument list, pass in null if you don't want to use it. <br>
 * <br>
 * After the constructor is called, getChartParametes() can be called to acquire user modifiable parameters; it is a
 * combination of the computed default parameters and the provided override parameters. The method
 * overrideParameters(...) can be called to override the current parameters, which may have already been overridden. The
 * return of buildChart() can be displayed in a ChartEngineChartPanel instance.<br>
 * <br>
 * Every time ChartEngine builds a chart, it remembers many aspects of the build in order to be able to build the chart
 * more quickly if it needs to build it again. To clear that memory and force a complete rebuild, which may be necessary
 * if changes are made to the underlying data somehow, call clearMemory().
 * 
 * @author herrhd
 */
public class ChartEngine
{

    private static final Logger LOG = LogManager.getLogger(ChartEngine.class);
    /**
     * The data sources used to build the plot.
     */
    private final List<XYChartDataSource> _dataSources = new ArrayList<XYChartDataSource>();

    /**
     * Maps the data set to the data source that provided the data set. For use by external tools.
     */
    private final HashMap<XYDataset, XYChartDataSource> _dataSetToDataSourceMap = Maps.newHashMap();

    /**
     * Map of the sub plot indices to data sources indices plotted on that plot. The indices correspond to the
     * _dataSources list.
     */
    private final HashMap<Integer, List<Integer>> _subPlotIndexToDataSourceIndex = Maps.newHashMap();

    /**
     * Maps the data source index to the index of the subplot on which the corresponding data source is plotted.
     */
    private final HashMap<Integer, Integer> _dataSourceIndexToSubPlotIndex = Maps.newHashMap();

    /**
     * Map of the sub plot index to the XYPlot constructed for the subplot.
     */
    private final HashMap<Integer, XYPlot> _subPlotIndexToXYPlot = Maps.newHashMap();

    /**
     * Built as a byproduct of the _subPlotIndexToDataSourceIndex map, this specifies the calculated axis types
     * specified by each of the data sources.
     */
    private final HashMap<Integer, int[]> _subPlotIndexToComputedRangeAxisTypes = Maps.newHashMap();

    /**
     * Built as a byproduct of the _subPlotIndexToDataSourceIndex map, this specifies the units corresponding to an axis
     * within a subplot, as dictated by the data sources.
     */
    private final HashMap<Integer, HashMap<Integer, List<String>>> _subPlotIndexToRangeAxisToUnitsString =
                                                                                                         Maps.newHashMap();

    /**
     * After the constructor is called, this object can be accessed to change the chart drawing parameters
     * (getChartParameters) prior to building the chart.
     */
    private ChartDrawingParameters _chartParameters = null;

    /**
     * A copy of the original override parameters, if anything needs access to it.
     */
    private ChartDrawingParameters _originalOverrideParameters = null;

    /**
     * Records the parameters used the last time this engine built the chart.
     */
    private ChartDrawingParameters _parametersLastUsedForBuild = null;

    /**
     * The last built JFreeChart
     */
    private JFreeChart _chartResultingFromLastBuild = null;

    /**
     * Arguments that can be used by the parameters.
     */
    private ArgumentsProcessor _arguments = DefaultArgumentsProcessor.createEmpty();

    /**
     * @param arguments The ArgumentsProcessor specifying arguments that all parameters can use. Cannot be null.
     * @param dataSources Computed data sources, complete with initial parameters.
     */
    public ChartEngine(final ArgumentsProcessor arguments,
                       final List<XYChartDataSource> dataSources) throws ChartEngineException
    {
        if(arguments != null)
        {
            _arguments = arguments;
        }
        _dataSources.addAll(dataSources);

        final ChartDrawingParameters defaultParameters = new ChartDrawingParameters(_dataSources);
        defaultParameters.setArguments(_arguments);
        defaultParameters.synchronizeSubPlotParametersListWithDataSourceParameters();
        defaultParameters.setupDefaultParameters();

        try
        {
            defaultParameters.haveAllParametersBeenSet();
        }
        catch(final ChartParametersException e)
        {
            throw new ChartEngineException("A default set of parameters is not fully specified: " + e.getMessage());
        }

        _chartParameters = defaultParameters;
    }

    /**
     * @param arguments The ArgumentsProcessor specifying arguments that all parameters can use. Can be null.
     * @param dataSources Computed data sources, complete with initial parameters.
     * @param overrideParameters User overridden parameters, possibly read in from an XML file, for example. After this
     *            is done, the overrideParameters will be ensured to have one set of parameters per data source and one
     *            set of parameters per subplot.
     */
    public ChartEngine(final ArgumentsProcessor arguments,
                       final List<XYChartDataSource> dataSources,
                       final ChartDrawingParameters overrideParameters) throws ChartEngineException
    {
        if(arguments != null)
        {
            _arguments = arguments;
        }
        initializeFromDataSourcesAndOverrideParameters(dataSources, overrideParameters);
    }

    /**
     * This is, for the most part, a copy constructor. However, the underlying data sources will share the same raw
     * data, which is used by JFreeChart. Hence, if you change the data set in the basis, you should reset the pointer
     * all together; do not change the data set itself.
     * 
     * @param basis That from which to make a copy.
     * @throws XYChartDataSourceException
     */
    public ChartEngine(final ChartEngine basis) throws ChartEngineException
    {
        _arguments = basis._arguments;
        for(int i = 0; i < basis.getDataSources().size(); i++)
        {
            try
            {
                _dataSources.add(basis.getDataSources().get(i).returnNewInstanceWithCopyOfInitialParameters());
            }
            catch(final XYChartDataSourceException e)
            {
                throw new ChartEngineException("Cannot add basis data sources: " + e.getMessage());
            }
        }

        _chartParameters = (ChartDrawingParameters)basis.getChartParameters().clone();
        try
        {
            _chartParameters.haveAllParametersBeenSet();
        }
        catch(final ChartParametersException e)
        {
            throw new ChartEngineException("A default set of parameters is not fully specified: " + e.getMessage());
        }
    }

    /**
     * Constructs the data sources list and override parameters based on a list of engines. The engines passed in are
     * not changed; the data sources list inside this will be constructed as copies (sharing the same underlying data)
     * and have their indices changed to match the index in the new list.
     * 
     * @param arguments The ArgumentsProcessor specifying arguments that all parameters can use. Can be null.
     * @param engines List of engines to copy, in order.
     * @param passInNullForMe Anything. In order for this constructor to be seen as different from the other List based
     *            constructor, it requires another arg.
     * @throws ChartEngineException If a problem occurs.
     */
    public ChartEngine(final ArgumentsProcessor arguments,
                       final List<ChartEngine> engines,
                       final Object passInNullForMe) throws ChartEngineException
    {
        if(arguments != null)
        {
            _arguments = arguments;
        }

        //Create a unified list of data sources, adding each source drawing parameters to the _chartParameters
        //as we go, adjusting the source index as needed.
        final List<XYChartDataSource> dataSources = new ArrayList<XYChartDataSource>();

        //Records the combined data source parameters in a ChartDrawingParameters object.
        final ChartDrawingParameters combinedParms = new ChartDrawingParameters();

        //The master threshold list
        final ThresholdListParameters combinedDefaultThresholdList = new ThresholdListParameters();

        //Loop through each chart engine.
        for(int engineIndex = 0; engineIndex < engines.size(); engineIndex++)
        {
            //Adjustment factor to use on the source indes.
            final int sourceIndexAdjust = dataSources.size();

            //For each data source for this engine, add the source as a copy with a new index to dataSources.
            for(int i = 0; i < engines.get(engineIndex).getDataSources().size(); i++)
            {
                //Get a modifiable copy of the source; one that includes the same data objects, but a copy of the initial parameters.
                try
                {
                    final XYChartDataSource source = engines.get(engineIndex)
                                                            .getDataSources()
                                                            .get(i)
                                                            .returnNewInstanceWithCopyOfInitialParameters();
                    source.getDefaultFullySpecifiedDataSourceDrawingParameters()
                          .setDataSourceOrderIndex(dataSources.size());
                    dataSources.add(source);

                }
                catch(final XYChartDataSourceException e)
                {
                    throw new ChartEngineException("Cannot combine data sources: " + e.getMessage());
                }
            }

            //If there are original overrides for the chart engine, then use them to override the combined
            //data source parameters.  The effect is that the overrides are combined into one large override
            //set of parameters.
            if(engines.get(engineIndex).getOriginalOverrideParameters() != null)
            {
                final ChartDrawingParameters overrideParams = engines.get(engineIndex).getOriginalOverrideParameters();
                for(int j = 0; j < overrideParams.getDataSourceParametersCount(); j++)
                {
                    final DataSourceDrawingParameters newParms =
                                                               (DataSourceDrawingParameters)overrideParams.getDataSourceParameters()
                                                                                                          .get(j)
                                                                                                          .clone();
                    newParms.setDataSourceOrderIndex(newParms.getDataSourceOrderIndex() + sourceIndexAdjust);
                    combinedParms.addDataSourceParameters(newParms);
                }

                combinedParms.copyOverriddenParametersExceptDataSource(overrideParams, false, true);
            }

            //Update the combinedDefaultThresholdList to contain those within the reference engine's ChartParameters.
            //These thresholds must have passed the haveAllParametersBeenSet test, otherwise the reference would have
            //caused an error, so I know they are complete.
            final ThresholdListParameters toAdd = engines.get(engineIndex).getChartParameters().getThresholdList();
            for(int i = 0; i < toAdd.getThresholdParametersList().size(); i++)
            {
                final String evaluatedIdentifier = toAdd.getThresholdParametersList().get(i).getEvaluatedIdentifier();
                final ThresholdParameters newParms = (ThresholdParameters)toAdd.getThresholdParametersList()
                                                                               .get(i)
                                                                               .clone();
                newParms.setIdentifier(evaluatedIdentifier);
                combinedDefaultThresholdList.addThreshold(newParms);
            }
        }

        //For the threshold parameters, I allowed the nested loop within the loop above to accumulate the thresholds
        //in the combinedDefaultThresholdList.  All of the thresholds from all references should have been accounted
        //for and the identifiers have been translated to the evaluated one, making the list appropriate for default
        //thresholds.  Hence, I need to clear out the combinedParams threshold list and use the default list.
        combinedParms.getThresholdList().clearParameters();
        combinedParms.getThresholdList().addThresholds(combinedDefaultThresholdList);

        initializeFromDataSourcesAndOverrideParameters(dataSources, combinedParms);
    }

    /**
     * Assumes no arguments.
     * 
     * @param dataSources Computed data sources, complete with initial parameters.
     */
    public ChartEngine(final List<XYChartDataSource> dataSources) throws ChartEngineException
    {
        this(null, dataSources);
    }

    /**
     * Call to initialize {@link #_chartParameters} based on a list of data sources and override parameters.  
     * Note that the color palettes and default defining series parameters provided within the override data source
     * drawing parameters will be applied within this method BEFORE the override data source parameters (series
     * parameters) are applied.  Hence, those define the defaults over which the overrides are applied!
     */
    private void initializeFromDataSourcesAndOverrideParameters(final List<XYChartDataSource> dataSources,
                                                                final ChartDrawingParameters overrideParameters) throws ChartEngineException
    {
        _dataSources.addAll(dataSources);

        final ChartDrawingParameters defaultParameters = new ChartDrawingParameters(_dataSources);

        //This for loop ensures that every series in override has a corresponding series in defaultParameters.
        //If a series does not, one is created with its setupDefaultParameters() method called.  
        //This avoid triggering the haveAllParametersBeenSet check below.
        for(int i = 0; i < overrideParameters.getDataSourceParametersCount(); i++)
        {
            final DataSourceDrawingParameters parms =
                                                    defaultParameters.getDataSourceParameters(overrideParameters.getDataSourceParameters()
                                                                                                                .get(i)
                                                                                                                .getDataSourceOrderIndex());
            for(int j = 0; j < overrideParameters.getDataSourceParameters().get(i).getSeriesParametersCount(); j++)
            {
                if(!parms.doSeriesDrawingParametersExistForSeriesIndex(j))
                {
                    final SeriesDrawingParameters newParms = new SeriesDrawingParameters(j);
                    newParms.setupDefaultParameters();
                    parms.addSeriesDrawingParameters(newParms);
                }
            }
        }

        _chartParameters = defaultParameters;
        _chartParameters.setupDefaultParameters();
        
        try
        {
            defaultParameters.haveAllParametersBeenSet();
        }
        catch(final ChartParametersException e)
        {
            throw new ChartEngineException("A default set of parameters is not fully specified: " + e.getMessage());
        }

        //Apply the color palettes and default defining chart series parameters now before the rest of the overrides are applied.
        for(int i = 0; i < _chartParameters.getDataSourceParametersCount(); i++)
        {
            final DataSourceDrawingParameters parms = _chartParameters.getDataSourceParameters(i);
            if (overrideParameters.getDataSourceParameters(i) != null)
            {
                parms.copyOverriddenDefaultSeriesSpecifyParametersOnly(overrideParameters.getDataSourceParameters(i));
            }
            parms.applyColorPalette();
            parms.applyDefaultDefiningSeriesParameters();
        }

        overrideParameters(overrideParameters);
        _chartParameters.setArguments(_arguments);
    }

    /**
     * Overrides the current _chartParameters based on the passed in override. The override will be copied and stored as
     * _originalOverrideParameters. These steps are performed (1) copy the overridden data source parameters; (2)
     * synchronize the subplot parameters so that there is one per needed plot, also initializing axis visibility; (3)
     * copy all the other overridden parameters; and (4) make a clone of the override and store as
     * _originalOverrideParameters.
     * 
     * @param override Parameters overridding defaults.
     */
    public void overrideParameters(final ChartDrawingParameters override)
    {
        //_chartParameters.copyOverriddenParameters(override);
        _chartParameters.copyOverriddenDataSourceParametersOnly(override);
        _chartParameters.synchronizeSubPlotParametersListWithDataSourceParameters(true);
        _chartParameters.copyOverriddenParametersExceptDataSource(override, true, true);
        _originalOverrideParameters = (ChartDrawingParameters)override.clone();
        _chartParameters.setArguments(_arguments);
    }

    /**
     * Verifies consistent axis types and that at least one data source was provided. Also calls checkForValidity for
     * each data source (see DefaultXYChartDataSource).
     * 
     * @throws ChartEngineException If a problem is encountered.
     * @throws XYChartDataSourceException If a datasource validity check fails.
     */
    private void checkForValidityOfDataSources() throws ChartEngineException, XYChartDataSourceException
    {
        //Need at least one data source.
        if(_dataSources.size() == 0)
        {
            throw new ChartEngineException("No charting data sources specified.");
        }

        //Need equal x-axis types for all data source.
        for(int i = 1; i < _dataSources.size(); i++)
        {
            if(_dataSources.get(i).getXAxisType() != _dataSources.get(i - 1).getXAxisType())
            {
                throw new ChartEngineException("Charting data sources " + (i) + " and " + (i - 1)
                    + " do not have identical x-axis types.");
            }
        }

        //Need one data source parameters object per source and its parameters must be valid relative to the 
        //data source.
        for(int i = 0; i < _dataSources.size(); i++)
        {
            if(this._chartParameters.getDataSourceParameters(i) == null)
            {
                throw new ChartEngineException("Data sources parameters appear to contain indices that are out of "
                    + "sequence.  The data source index " + i + " has no specified parameters.");
            }
            _dataSources.get(i).checkForValidity(this._chartParameters.getDataSourceParameters(i));
        }
    }

    private JFreeChart constructChartBasedOnDataSources() throws ChartEngineException, XYChartDataSourceException
    {
        final HStopWatch watch = new HStopWatch();
        watch.start();
        //HStopWatch subWatch = new HStopWatch();

        //If no change has occurred since last build, return the last built chart.
        if((_parametersLastUsedForBuild != null)
            && (this.getChartParameters().equals(this._parametersLastUsedForBuild)))
        {
            return this._chartResultingFromLastBuild;
        }

        //The _defaultParameters must be fully specified!
        //subWatch.start();
        try
        {
            _chartParameters.haveAllParametersBeenSet();
        }
        catch(final ChartParametersException e)
        {
            //e.printStackTrace();
            throw new ChartEngineException("The chart parameters are not fully set: " + e.getMessage());
        }
        //subWatch.stop();
        //System.out.println("####>>   elapsed time to check for parms = " + subWatch.getElapsedMillis());

        //Prepare the data for all data sources and validate the parameters.
        //subWatch.start();
        for(int i = 0; i < _dataSources.size(); i++)
        {
            _dataSources.get(i).prepareXYDataset(_chartParameters, i);
        }
        checkForValidityOfDataSources();
        //subWatch.stop();
        //System.out.println("####>>   elapsed time to prepare data and validate = " + subWatch.getElapsedMillis());

        //Get the sorted list of subplot indices and build the map.
        final List<Integer> subPlotIndices = buildSubPlotIndexToDataSourceAndComputedRangeAxisTypeMaps();

        //Initialize the combined plot.
        final CombinedDomainXYPlot combinedPlot = setupDomainAxisAndCombinedPlot();

        //subWatch.start();
        //Add the subplots.
        for(int i = 0; i < subPlotIndices.size(); i++)
        {
            final SubPlotParameters subPlotParms = _chartParameters.getSubPlot(subPlotIndices.get(i));
            try
            {
                final XYPlot subPlot = this.createOrAddToSubPlot(subPlotIndices.get(i));
                combinedPlot.add(subPlot, subPlotParms.getPlotWeight());
            }
            catch(final ChartEngineException e)
            {
                LOG.error("Unable to create subplot with index " + i + ": " + e.getMessage());
                throw e;
            }
        }
        //subWatch.stop();
        //System.out.println("####>>   elapsed time to create subplots = " + subWatch.getElapsedMillis());

        //Create the chart 
        //subWatch.start();
        final JFreeChart chart = new JFreeChart(null, null, combinedPlot, true);
        //subWatch.stop();
        //System.out.println("####>>   elapsed time to constuct JFreeChart = " + subWatch.getElapsedMillis());

        try
        {
            applyChartDrawingParametersToChart(chart);
        }
        catch(final ChartParametersException e)
        {
//            e.printStackTrace();
            final ChartEngineException newE = new ChartEngineException("Unable to apply chart parameters: "
                + e.getMessage());
            newE.setStackTrace(e.getStackTrace());
            throw newE;
        }

        //Recompute the axis limits

        watch.stop();
        LOG.debug("Elapsed time to build chart = " + watch.getElapsedMillis());

        this._parametersLastUsedForBuild = (ChartDrawingParameters)this._chartParameters.clone();
        this._chartResultingFromLastBuild = chart;

        return chart;
    }

    /**
     * Sets up the domain axis for the CombinedDomainXYPlot by using the axis type implied by the data sources and
     * overridden by the domain axis in _chartParameters.
     * 
     * @return A CombinedDomainXYPlot with the domain axis setup.
     * @throws ChartEngineException If axis type is invalid in some way.
     */
    private CombinedDomainXYPlot setupDomainAxisAndCombinedPlot() throws ChartEngineException
    {
        ExtendedCombinedDomainXYPlot combinedPlot = null;

        //Get the override setting from _chartParameters.  If not specified, use the calculated one found in the
        //data sources.  If it is specified, then use it.
        final int domainOverride = this._chartParameters.getDomainAxis().getAxisTypeInt();
        int axisTypeInt = _dataSources.get(0).getXAxisType();
        if(domainOverride >= 0)
        {
            axisTypeInt = domainOverride;
        }

        ChartTools.checkCompatibilityOfAxis(_dataSources.get(0).getXAxisType(), axisTypeInt);

        //Create the CombinedDomainXYPlot.
        if(this._dataSources.get(0).getXAxisType() == ChartConstants.AXIS_IS_TIME)
        {
            final DateAxisPlus dateAxis = new DateAxisPlus();
            dateAxis.setTimeZone(TimeZone.getTimeZone("GMT"));
            combinedPlot = new ExtendedCombinedDomainXYPlot(dateAxis);
        }
        else if (axisTypeInt == ChartConstants.AXIS_IS_CATEGORICAL)
        {
            final NumberAxisOverride axis = new NumberAxisOverride("Domain Axis");
            axis.setCategoricalLabels(((CategoricalXYChartDataSource)_dataSources.get(0)).getXCategories());
            combinedPlot = new ExtendedCombinedDomainXYPlot(axis);
        }
        else if(axisTypeInt == ChartConstants.AXIS_IS_NUMERICAL)
        {
            combinedPlot = new ExtendedCombinedDomainXYPlot(new NumberAxisOverride("Domain Axis"));
        }
        else if(axisTypeInt == ChartConstants.AXIS_IS_LOGARITHMIC)
        {
            combinedPlot = new ExtendedCombinedDomainXYPlot(new LogarithmicAxis("Domain Axis"));
        }
        else if(axisTypeInt == ChartConstants.AXIS_IS_NORMALIZED_PROBABILITY)
        {
            combinedPlot = new ExtendedCombinedDomainXYPlot(new NormalizedProbabilityAxis("Domain Axis"));
        }
        else if(axisTypeInt == ChartConstants.AXIS_IS_PROBABILITY)
        {
            combinedPlot = new ExtendedCombinedDomainXYPlot(new ProbabilityAxis("Domain Axis"));
        }
        else if(axisTypeInt == ChartConstants.AXIS_IS_TRANSLATED)
        {
            throw new ChartEngineException("Domain axis cannot be translated.");
        }
        else
        {
            throw new ChartEngineException("X-axis type constant of " + _dataSources.get(0).getXAxisType()
                + " is not recognized.");
        }
        return combinedPlot;
    }

    /**
     * @param axisTypeInt The integer specifying the axis type. One of ChartToolsAndConstants.AXIS_IS* constants.
     * @param otherAxis The other axis, useful when this is a translated axis.
     * @param initialLabel The label to display.
     * @param translatorParameters The translator parameters used to build the axis, if used. Null is allowed.
     * @return An axis to use for the range axis.
     * @throws ChartEngineException If the axis type is positive and unrecognized or it incompatible with the default
     *             axis type. Generally, the default axis types are date, numerical, or translated. Compatibility is
     *             checked via ChartToolsAndConstants.checkCompatibilityOfAxis.
     */
    private ValueAxis setupRangeAxis(final int axisTypeInt,
                                     final ValueAxis otherAxis,
                                     final String initialLabel,
                                     final AxisTranslatorParameters translatorParameters) throws ChartEngineException
    {
        if(axisTypeInt == ChartConstants.AXIS_IS_TIME)
        {
            final DateAxisPlus axis = new DateAxisPlus();
            axis.setTimeZone(TimeZone.getTimeZone("GMT"));
            axis.setLabel(initialLabel);
            return axis;
        }
        else if((axisTypeInt == ChartConstants.AXIS_IS_NUMERICAL) || (axisTypeInt < 0))
        {
            return new NumberAxisOverride(initialLabel);
        }
        else if(axisTypeInt == ChartConstants.AXIS_IS_LOGARITHMIC)
        {
            return new LogarithmicAxis(initialLabel);
        }
        else if(axisTypeInt == ChartConstants.AXIS_IS_NORMALIZED_PROBABILITY)
        {
            return new NormalizedProbabilityAxis(initialLabel);
        }
        else if(axisTypeInt == ChartConstants.AXIS_IS_PROBABILITY)
        {
            return new ProbabilityAxis(initialLabel);
        }
        else if(axisTypeInt == ChartConstants.AXIS_IS_TRANSLATED)
        {
            if(!(otherAxis instanceof NumberAxisOverride))
            {
                throw new ChartEngineException("Range axis is translated, but other axis is not numerical.");
            }
            try
            {
                return new TranslatedAxis(initialLabel, (NumberAxisOverride)otherAxis, translatorParameters);
            }
            catch(final AxisTranslatorException e)
            {
                LOG.warn("Error loading axis translator " + translatorParameters.getTranslatorName() + ": "
                    + e.getMessage());
                return TranslatedAxis.buildDefaultErrorTranslatedAxis("Error Loading Translator!",
                                                                      (NumberAxisOverride)otherAxis,
                                                                      translatorParameters);
            }
        }
        throw new ChartEngineException("Range axis type constant of " + axisTypeInt + " is not recognized.");
    }

    /**
     * @param calculatedAxisType The default (calculated) axis type for the axis determined by the data sources. If
     *            negative, then no data was plotted against that axis, and the returned axis is a NumericalAxis,
     *            assuming the user does not override. This must never by TRANSLATED, since a data source cannot be
     *            plotted against a translated axis.
     * @param axisParms The axis parameters specifying the override axis type and translator function if needed.
     * @return An integer providing the axis type, which is either the calculated type or, if positive, the int provided
     *         in the axisParms.
     * @throws ChartEngineException If the parameter implied axis type is not compatible with the calculated axis type.
     */
    private int determineAxisTypeInt(final int calculatedAxisType,
                                     final AxisParameters axisParms) throws ChartEngineException
    {
        int axisTypeInt = calculatedAxisType;
        if(axisParms.getAxisTypeInt() >= 0)
        {
            axisTypeInt = axisParms.getAxisTypeInt();
            ChartTools.checkCompatibilityOfAxis(calculatedAxisType, axisTypeInt);
        }
        return axisTypeInt;
    }

    /**
     * @param subPlotIndex The subplot index for which to setup the range axes and initialize the subplot.
     * @return XYPlot with range axes setup.
     * @throws ChartEngineException
     */
    private XYPlot setupRangeAxesAndXYPlotForSubPlot(final int subPlotIndex) throws ChartEngineException
    {
        XYPlot subPlot = this._subPlotIndexToXYPlot.get(subPlotIndex);
        if(subPlot == null)
        {
            subPlot = new ExtendedXYPlot();

            final int leftRangeAxisType =
                                        determineAxisTypeInt(_subPlotIndexToComputedRangeAxisTypes.get(subPlotIndex)[ChartConstants.LEFT_YAXIS_INDEX],
                                                             _chartParameters.getSubPlot(subPlotIndex).getLeftAxis());
            final int rightRangeAxisType =
                                         determineAxisTypeInt(_subPlotIndexToComputedRangeAxisTypes.get(subPlotIndex)[ChartConstants.RIGHT_YAXIS_INDEX],
                                                              _chartParameters.getSubPlot(subPlotIndex).getRightAxis());

            //For use with translated axes, setup the other axis parameter identifiers and units.
            if(_chartParameters.getSubPlot(subPlotIndex).getLeftAxis().getAxisTranslator() != null)
            {
                _chartParameters.getSubPlot(subPlotIndex)
                                .getLeftAxis()
                                .getAxisTranslator()
                                .setOtherAxisUnits(this._subPlotIndexToRangeAxisToUnitsString.get(subPlotIndex)
                                                                                             .get(ChartConstants.RIGHT_YAXIS_INDEX));
            }
            if(_chartParameters.getSubPlot(subPlotIndex).getRightAxis().getAxisTranslator() != null)
            {
                _chartParameters.getSubPlot(subPlotIndex)
                                .getRightAxis()
                                .getAxisTranslator()
                                .setOtherAxisUnits(this._subPlotIndexToRangeAxisToUnitsString.get(subPlotIndex)
                                                                                             .get(ChartConstants.LEFT_YAXIS_INDEX));
            }

            //Initialize both y-axes.  We initialize the left-hand axis first if (1) the right hand axis is translated
            //or (2) the left hand axis is not translated.  Otherwise, the right hand is first.
            ValueAxis leftRangeAxis = null;
            ValueAxis rightRangeAxis = null;
            if((ChartConstants.isAxisTypeTranslated(rightRangeAxisType))
                || (!ChartConstants.isAxisTypeTranslated(leftRangeAxisType)))
            {

                leftRangeAxis = this.setupRangeAxis(leftRangeAxisType,
                                                    null,
                                                    "Left Range Axis",
                                                    _chartParameters.getSubPlot(subPlotIndex)
                                                                    .getLeftAxis()
                                                                    .getAxisTranslator());
                rightRangeAxis = this.setupRangeAxis(rightRangeAxisType,
                                                     leftRangeAxis,
                                                     "Right Range Axis",
                                                     _chartParameters.getSubPlot(subPlotIndex)
                                                                     .getRightAxis()
                                                                     .getAxisTranslator());
            }
            else
            {
                rightRangeAxis = this.setupRangeAxis(rightRangeAxisType,
                                                     null,
                                                     "Right Range Axis",
                                                     _chartParameters.getSubPlot(subPlotIndex)
                                                                     .getRightAxis()
                                                                     .getAxisTranslator());
                leftRangeAxis = this.setupRangeAxis(leftRangeAxisType,
                                                    rightRangeAxis,
                                                    "Left Range Axis",
                                                    _chartParameters.getSubPlot(subPlotIndex)
                                                                    .getLeftAxis()
                                                                    .getAxisTranslator());
            }

            subPlot.setRangeAxis(ChartConstants.LEFT_YAXIS_INDEX, leftRangeAxis);
            subPlot.setRangeAxis(ChartConstants.RIGHT_YAXIS_INDEX, rightRangeAxis);
        }
        return subPlot;
    }

    /**
     * Creates the subplot corresponding to subPlotIndex using setupRangeAxesAndXYPlotForSubPlot and adding all data
     * sources to the returned XYPlot.
     * 
     * @param subPlotIndex The index of the subplot to create.
     * @return An XYPlot with range axes specified and sources added.
     * @throws ChartEngineException If the axis type is rejected by setupRangeAxesAndXYPlotForSubPlot or if the plotter
     *             has a problem drawing a data source on the plot.
     */
    private XYPlot createOrAddToSubPlot(final int subPlotIndex) throws ChartEngineException
    {
        //Get the source indices for the subplot and build the range axis..
        final List<Integer> sourcesIndices = this._subPlotIndexToDataSourceIndex.get(subPlotIndex);
        final XYPlot subPlot = setupRangeAxesAndXYPlotForSubPlot(subPlotIndex);

        //This variable is used to track the number of data sets in the plot after applying a plotter.
        //This allows for a plotter to add data sets, itself, if necessary.  However, the tricky part is that
        //when created the subplot comes with a single data set pre-added, so that getDatasetCount returns 1,
        //even though this algorithms hasn't added any.  To account for that, the variable is initialized to 0
        //and is not set within the loop until the end after the plotter is applied (rather than the beginning).
        int numberOfDataSets = 0;

        //For each source index...
        for(int i = 0; i < sourcesIndices.size(); i++)
        {
            //Get the parameters and dataset.
            final DataSourceDrawingParameters sourceParms =
                                                          this._chartParameters.getDataSourceParameters(sourcesIndices.get(i));
            final XYDataset dataset = this._dataSources.get(sourcesIndices.get(i)).getXYDataSet();

            //Set the dataset for the sub plot.
            subPlot.setDataset(numberOfDataSets, dataset);

            //Map the dataset to the appropriate y-axis.
            if(sourceParms.getYAxisIndex() != null)
            {
                subPlot.mapDatasetToRangeAxis(numberOfDataSets, sourceParms.getYAxisIndex());
            }
            else
            {
                subPlot.mapDatasetToRangeAxis(numberOfDataSets, 0);
            }

            //Apply drawing parameters to the sub plot.
            applyPlotterSettings(subPlot, numberOfDataSets, sourceParms);

            //TODO May need to reactivate this line in the future.  Not sure...
            //Trim off the NaNs at the ends of time series.
            //trimPlottedTimeSeries(subPlot);

            numberOfDataSets = subPlot.getDatasetCount();

            _dataSetToDataSourceMap.put(dataset, _dataSources.get(sourcesIndices.get(i)));
        }

        _subPlotIndexToXYPlot.put(subPlotIndex, subPlot);

        return subPlot;
    }

    /**
     * Currently, this is not called.
     * 
     * @param subPlot Trim missing/NaN values for the time series to plot, if any. Right now, this is not used.
     */
    @SuppressWarnings("unused")
    private void trimPlottedTimeSeries(final XYPlot subPlot)
    {
        //Remove any extraneous NaNs at the beginning or end of time series.
        for(int i = 0; i < subPlot.getDatasetCount(); i++)
        {
            if(subPlot.getDataset(i) instanceof TimeSeriesCollection)
            {
                final TimeSeriesCollection workingSet = (TimeSeriesCollection)subPlot.getDataset(i);
                for(int j = 0; j < workingSet.getSeriesCount(); j++)
                {
                    int startPoint = 0;
                    while((startPoint < workingSet.getSeries(j).getItemCount())
                        && (Float.isNaN(workingSet.getSeries(j).getDataItem(startPoint).getValue().floatValue())))
                    {
                        startPoint++;
                    }
                    if(startPoint > 0)
                    {
                        workingSet.getSeries(j).delete(0, startPoint - 1);
                    }

                    int endPoint = workingSet.getSeries(j).getItemCount() - 1;
                    if(endPoint >= 0)
                    {
                        while((Float.isNaN(workingSet.getSeries(j).getDataItem(endPoint).getValue().floatValue()))
                            && (endPoint >= 0))
                        {
                            endPoint--;
                        }
                        if(endPoint < workingSet.getSeries(j).getItemCount() - 1)
                        {
                            workingSet.getSeries(j).delete(endPoint + 1, workingSet.getSeries(j).getItemCount() - 1);
                        }
                    }
                }
            }
        }
    }

    /**
     * @param plot Plot to which the dataset belongs.
     * @param dataSetIndexWithinPlot Index of the dataset within the plot.
     * @param parms Parameters for the data source that yielded the data set.
     * @throws ChartEngineException
     */
    private void applyPlotterSettings(final XYPlot plot,
                                      final Integer dataSetIndexWithinPlot,
                                      final DataSourceDrawingParameters parms) throws ChartEngineException
    {
        try
        {
            final XYChartPlotter plotter = XYChartPlotterFactory.loadXYChartPlotter(parms.getPlotterName());
            plotter.applyPlotterSettings(plot, dataSetIndexWithinPlot, parms);
        }
        catch(final XYChartPlotterException e)
        {
            throw new ChartEngineException(e.getMessage());
        }
    }

    /**
     * This builds the mapping of subplot index to datasources plotted on it, and the mapping of subplot index to range
     * axis types. It will throw an exception if computed axis types for data sources are not compatible with each
     * other. It will also throw an exception if a computed axis type is translated, because you cannot plot data
     * against a translated axis. Errors based on the axis type parameters will be determined in the setupRangeAxis
     * methods.
     * 
     * @return List of subplot indices found.
     * @throws ChartEngineException If an axis is set to translated but data is plotted against it, or if an axis type
     *             is not compatible with data source computed data type.
     */
    private List<Integer> buildSubPlotIndexToDataSourceAndComputedRangeAxisTypeMaps() throws ChartEngineException
    {
        final List<Integer> indicesFound = new ArrayList<Integer>();

        for(int dataSourceIndex = 0; dataSourceIndex < this._dataSources.size(); dataSourceIndex++)
        {
            //Determine the parameters and subplot index.  If undefined, subplot index is assumed to be 0.
            final DataSourceDrawingParameters parameters = _chartParameters.getDataSourceParameters(dataSourceIndex);
            Integer subPlotIndex = parameters.getSubPlotIndex();
            if(subPlotIndex == null)
            {
                subPlotIndex = 0;
            }

            //Add to the listing of data sources for the subplot.
            List<Integer> listing = this._subPlotIndexToDataSourceIndex.get(subPlotIndex);
            if(listing == null)
            {
                listing = new ArrayList<Integer>();
                _subPlotIndexToDataSourceIndex.put(subPlotIndex, listing);
            }
            listing.add(dataSourceIndex);
            _dataSourceIndexToSubPlotIndex.put(dataSourceIndex, subPlotIndex);

            //A computed data type can NEVER be translated, because no data can be plotted against a translated
            //axis!!!
            if(_dataSources.get(dataSourceIndex).getComputedDataType() == ChartConstants.AXIS_IS_TRANSLATED)
            {
                throw new ChartEngineException("Data source with index " + dataSourceIndex
                    + " specifies a range axis type of translated, "
                    + "but data cannot be plotted against a translated axis.");
            }

            //Add to the range axis types mapping ===============================================
            int[] rangeAxisTypes = _subPlotIndexToComputedRangeAxisTypes.get(subPlotIndex);
            if(rangeAxisTypes == null)
            {
                rangeAxisTypes = new int[ChartConstants.YAXIS_XML_STRINGS.length];
                Arrays.fill(rangeAxisTypes, -1);
                rangeAxisTypes[parameters.getYAxisIndex()] = _dataSources.get(dataSourceIndex).getComputedDataType();
                _subPlotIndexToComputedRangeAxisTypes.put(subPlotIndex, rangeAxisTypes);
            }
            else
            {
                if(rangeAxisTypes[parameters.getYAxisIndex()] < 0)
                {
                    rangeAxisTypes[parameters.getYAxisIndex()] =
                                                               _dataSources.get(dataSourceIndex).getComputedDataType();
                }
                else
                {
                    //If an array already exists, compare its value for the axis specified by parameters.getYAxisIndex()
                    //with the value implied by the data sources.  If they are not equal, that is bad.
                    try
                    {
                        ChartTools.checkCompatibilityOfAxis(rangeAxisTypes[parameters.getYAxisIndex()],
                                                            _dataSources.get(dataSourceIndex).getComputedDataType());
                    }
                    catch(final ChartEngineException e)
                    {
                        throw new ChartEngineException("Problem for subplot " + subPlotIndex + " and y-axis "
                            + ChartConstants.YAXIS_XML_STRINGS[parameters.getYAxisIndex()] + ": " + e.getMessage());
                    }
                }
            }

            //Add to the suplot to range axis to unit strings ====================================
            HashMap<Integer, List<String>> unitsMapping = _subPlotIndexToRangeAxisToUnitsString.get(subPlotIndex);
            if(unitsMapping == null)
            {
                unitsMapping = new HashMap<Integer, List<String>>();
                _subPlotIndexToRangeAxisToUnitsString.put(subPlotIndex, unitsMapping);
            }
            List<String> units = unitsMapping.get(parameters.getYAxisIndex());
            if(units == null)
            {
                units = new ArrayList<String>();
                unitsMapping.put(parameters.getYAxisIndex(), units);
            }
            if((_dataSources.get(dataSourceIndex).getUnitsString() != null)
                && (!_dataSources.get(dataSourceIndex).getUnitsString().isEmpty()))
            {
                units.add(_dataSources.get(dataSourceIndex).getUnitsString());
            }

            //Add to returned list of indices found.
            if(indicesFound.indexOf(subPlotIndex) < 0)
            {
                indicesFound.add(subPlotIndex);
            }

        }

        Collections.sort(indicesFound);
        return indicesFound;
    }

    private void applyChartDrawingParametersToChart(final JFreeChart chart) throws ChartParametersException
    {
        _chartParameters.applyAllSettings(chart,
                                          _subPlotIndexToXYPlot,
                                          _dataSources.get(0).getXAxisType(),
                                          _subPlotIndexToComputedRangeAxisTypes);
    }

    /**
     * Builds the chart, passing into the other version the already define _arguments. This method will return the
     * results of a previous build if the parameters have not changed.
     * 
     * @return A JFreeChart object displaying the data source with appearance dictated by the _defaultParameters and
     *         _overrideParameters.
     * @throws ChartEngineException If non-DataSourceDrawingParameters or those that involve interaction between data
     *             sources are not valid, including verifying that all x-axis types are identical.
     * @throws XYChartDataSourceException If DataSourceDrawingParameters are not valid relative to the data source.
     */
    public JFreeChart buildChart() throws ChartEngineException, XYChartDataSourceException
    {
        final JFreeChart chart = constructChartBasedOnDataSources();
        return chart;
    }

    /**
     * Clears out memory of the previous build before building the chart, ensuring a new chart is built. This should be
     * called if the underlying data sources were updated in some way.
     * 
     * @return A JFreeChart object displaying the data source with appearance dictated by the _defaultParameters and
     *         _overrideParameters.
     * @throws ChartEngineException If non-DataSourceDrawingParameters or those that involve interaction between data
     *             sources are not valid, including verifying that all x-axis types are identical.
     * @throws XYChartDataSourceException If DataSourceDrawingParameters are not valid relative to the data source.
     */
    public JFreeChart rebuildChart() throws ChartEngineException, XYChartDataSourceException
    {
        clearMemory();
        return buildChart();
    }

    /**
     * ChartEngine remembers information about the previous build in order to save time on later chart builds. That
     * memory includes the parameters used, resulting chart, the data source index to subplot index map, and all subplot
     * index to something maps. This method clears all of that memory forcing a complete rebuild.
     */
    public void clearMemory()
    {
        this._parametersLastUsedForBuild = null;
        this._chartResultingFromLastBuild = null;
        this._dataSourceIndexToSubPlotIndex.clear();
        this._subPlotIndexToComputedRangeAxisTypes.clear();
        this._subPlotIndexToDataSourceIndex.clear();
        this._subPlotIndexToRangeAxisToUnitsString.clear();
        this._subPlotIndexToXYPlot.clear();
    }

    /**
     * @return Returns true if all of the data sources are empty.
     */
    public boolean isChartEmpty()
    {
        for(int i = 0; i < _dataSources.size(); i++)
        {
            if(!_dataSources.get(i).isEmpty())
            {
                return false;
            }
        }
        return true;
    }

    public void setArguments(final ArgumentsProcessor arguments)
    {
        _arguments = arguments;
        this._chartParameters.setArguments(arguments);
    }

    public List<XYChartDataSource> getDataSources()
    {
        return this._dataSources;
    }

    public ChartDrawingParameters getChartParameters()
    {
        return _chartParameters;
    }

    /**
     * Modifying these parameters has no affect on the chart image. Use getChartParameters() to do that.
     * 
     * @return A copy of the original override parameters that were used for drawing. The original override parameters
     *         are useful when combining several charts into one.
     */
    public ChartDrawingParameters getOriginalOverrideParameters()
    {
        return this._originalOverrideParameters;
    }

    public void setOriginalOverrideParameters(final ChartDrawingParameters originalOverrideParameters)
    {
        this._originalOverrideParameters = originalOverrideParameters;
    }

    public HashMap<Integer, int[]> getSubPlotIndexToRangeAxisTypes()
    {
        return _subPlotIndexToComputedRangeAxisTypes;
    }

    public XYPlot getSubPlot(final int subPlotIndex)
    {
        return this._subPlotIndexToXYPlot.get(subPlotIndex);
    }

    /**
     * @param subPlotIndex Subplot index of the unit strings to retrieve.
     * @param axis The axis index (-1 domain, 0 left, 1 right) of the axis.
     * @return A list of the units for data plotted againt the axis in the subplot.
     */
    public List<String> getAxisUnitStrings(final int subPlotIndex, final int axis)
    {
        return _subPlotIndexToRangeAxisToUnitsString.get(subPlotIndex).get(axis);
    }

    public XYChartDataSource getDataSourceThatProducedDataSet(final XYDataset set)
    {
        return _dataSetToDataSourceMap.get(set);
    }
}