package ohd.hseb.charter;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.Point;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import javax.swing.JFrame;
import javax.swing.JPanel;

import org.jfree.chart.ChartPanel;
import org.jfree.chart.JFreeChart;
import org.jfree.chart.axis.ValueAxis;
import org.jfree.chart.plot.CombinedDomainXYPlot;
import org.jfree.chart.plot.XYPlot;

import com.google.common.base.Supplier;

import nl.wldelft.util.timeseries.TimeSeriesArrays;
import ohd.hseb.charter.datasource.XYChartDataSource;
import ohd.hseb.charter.datasource.instances.CategoricalXYChartDataSource;
import ohd.hseb.charter.panel.AbstractChartEngineTableModel;
import ohd.hseb.charter.panel.ChartEngineChartAndTablePanel;
import ohd.hseb.charter.panel.ChartEngineTableModel;
import ohd.hseb.hefs.utils.gui.tools.HSwingFactory;
import ohd.hseb.hefs.utils.tsarrays.TimeSeriesArraysTools;
import ohd.hseb.util.fews.IDriver;

/**
 * Tools related to building a component to display a chart and interacting with those components.
 * 
 * @author Hank.Herr
 */
public abstract class ChartPanelTools
{

    /**
     * Calls {@link ChartPanelTools#createPanelForChartAndEngine(JFreeChart, ChartEngine, boolean)} within a try
     * command. If an exception occurs, the returned panel contains an error message. The {@link JFreeChart} instance is
     * constructed by called {@link ChartEngine#buildChart()} for the provided chart engine.
     * 
     * @param engine Engine fully specified so that {@link ChartEngine#buildChart()} can be called.
     * @param showTableInitially Passed through to the
     *            {@link ChartPanelTools#createPanelForChartAndEngine(JFreeChart, ChartEngine, boolean)} method.
     * @return {@link ChartEngineChartAndTablePanel} instance or a {@link JPanel} error message pane if an error occurs.
     */
    public static JPanel buildPanelFromChartEngine(final ChartEngine engine, final boolean showTableInitially)
    {

        try
        {
            final JFreeChart builtChart = engine.buildChart();
            final JPanel panel = ChartPanelTools.createPanelForChartAndEngine(builtChart, engine, showTableInitially);
            return panel;
        }
        catch(final Exception e)
        {
            e.printStackTrace();

            final JPanel errorPanel = new JPanel(new BorderLayout());
            errorPanel.add(HSwingFactory.createErrorMessagePane("Error building chart panel for chart engine: "
                + e.getMessage(), Arrays.toString(e.getStackTrace())), BorderLayout.CENTER);
            return errorPanel;
        }
    }

    /**
     * Calls {@link ChartPanelTools#createPanelForChartAndEngine(JFreeChart, ChartEngine, boolean, Class)} method.
     * within a try command. If an exception occurs, the returned panel contains an error message. The
     * {@link JFreeChart} instance is constructed by called {@link ChartEngine#buildChart()} for the provided chart
     * engine.
     * 
     * @param engine Engine fully specified so that {@link ChartEngine#buildChart()} can be called.
     * @param showTableInitially Passed through to the
     *            {@link ChartPanelTools#createPanelForChartAndEngine(JFreeChart, ChartEngine, boolean, Class)} method.
     * @param chartTableModelClass Passed through to the
     *            {@link ChartPanelTools#createPanelForChartAndEngine(JFreeChart, ChartEngine, boolean, Class)} method.
     * @return {@link ChartEngineChartAndTablePanel} instance or a {@link JPanel} error message pane if an error occurs.
     */
    public static JPanel buildPanelFromChartEngine(final ChartEngine engine,
                                                   final boolean showTableInitially,
                                                   final Class<? extends AbstractChartEngineTableModel> chartTableModelClass)
    {

        try
        {
            final JFreeChart builtChart = engine.buildChart();
            final JPanel panel = ChartPanelTools.createPanelForChartAndEngine(builtChart,
                                                                              engine,
                                                                              showTableInitially,
                                                                              chartTableModelClass);
            return panel;
        }
        catch(final Exception e)
        {
            e.printStackTrace();

            final JPanel errorPanel = new JPanel(new BorderLayout());
            errorPanel.add(HSwingFactory.createErrorMessagePane("Error building chart panel for chart engine: "
                + e.getMessage(), Arrays.toString(e.getStackTrace())), BorderLayout.CENTER);
            return errorPanel;
        }
    }

    /**
     * Calls {@link ChartPanelTools#createPanelForChartAndEngine(JFreeChart, ChartEngine, boolean)} within a try
     * command. If an exception occurs, the returned panel contains an error message.
     * 
     * @param engine Engine fully specified so that buildChart() can be called.
     * @param builtChart A {@link JFreeChart} to display that corresponds to the engine.
     * @param showTableInitially Passed through to the
     *            {@link ChartPanelTools#createPanelForChartAndEngine(JFreeChart, ChartEngine, boolean)} method.
     * @return {@link ChartEngineChartAndTablePanel} instance or a {@link JPanel} error message pane if an error occurs.
     */
    public static JPanel buildPanelFromChartEngine(final ChartEngine engine,
                                                   final JFreeChart builtChart,
                                                   final boolean showTableInitially)
    {

        try
        {
            final JPanel panel = ChartPanelTools.createPanelForChartAndEngine(builtChart, engine, showTableInitially);
            return panel;
        }
        catch(final Exception e)
        {
            //e.printStackTrace();

            final JPanel errorPanel = new JPanel(new BorderLayout());
            errorPanel.add(HSwingFactory.createErrorMessagePane(e.getMessage()), BorderLayout.CENTER);
            return errorPanel;
        }
    }

    /**
     * Calls {@link ChartPanelTools#createPanelForChartAndEngine(JFreeChart, ChartEngine, boolean, Class)} within a try
     * command. If an exception occurs, the returned panel contains an error message.
     * 
     * @param engine Engine fully specified so that buildChart() can be called.
     * @param builtChart A {@link JFreeChart} to display that corresponds to the engine.
     * @param showTableInitially Passed through to the
     *            {@link ChartPanelTools#createPanelForChartAndEngine(JFreeChart, ChartEngine, boolean, Class)} method.
     * @param chartTableModelClass Passed through to the
     *            {@link ChartPanelTools#createPanelForChartAndEngine(JFreeChart, ChartEngine, boolean, Class)} method.
     * @return {@link ChartEngineChartAndTablePanel} instance or a {@link JPanel} error message pane if an error occurs.
     */
    public static JPanel buildPanelFromChartEngine(final ChartEngine engine,
                                                   final JFreeChart builtChart,
                                                   final boolean showTableInitially,
                                                   final Class<? extends AbstractChartEngineTableModel> chartTableModelClass)
    {

        try
        {
            final JPanel panel = ChartPanelTools.createPanelForChartAndEngine(builtChart,
                                                                              engine,
                                                                              showTableInitially,
                                                                              chartTableModelClass);
            return panel;
        }
        catch(final Exception e)
        {
            //            e.printStackTrace();

            final JPanel errorPanel = new JPanel(new BorderLayout());
            errorPanel.add(HSwingFactory.createErrorMessagePane(e.getMessage(), Arrays.toString(e.getStackTrace())),
                           BorderLayout.CENTER);
            return errorPanel;
        }
    }

    /**
     * @return Calls constructor
     *         {@link ChartEngineChartAndTablePanel#ChartEngineChartAndTablePanel(ChartEngine, JFreeChart, Supplier, boolean, boolean, boolean, boolean, boolean, boolean)}
     *         and returns it.
     */
    public static ChartEngineChartAndTablePanel createPanelForChartAndEngine(final JFreeChart chart,
                                                                             final ChartEngine engine,
                                                                             final boolean showTable,
                                                                             final Supplier<ChartEngineTableModel> tableModelSupplier)
    {
        return new ChartEngineChartAndTablePanel(engine,
                                                 chart,
                                                 tableModelSupplier,
                                                 showTable,
                                                 false,
                                                 true,
                                                 true,
                                                 true,
                                                 true);
    }

    /**
     * Calls {@link createPanelForChartAndEngine} where the {@link Supplier} provided returns an instance of the
     * provided class with its empty constructor called.
     */
    public static ChartEngineChartAndTablePanel createPanelForChartAndEngine(final JFreeChart chart,
                                                                             final ChartEngine engine,
                                                                             final boolean showTable,
                                                                             final Class<? extends AbstractChartEngineTableModel> chartTableModelClass)
    {
        return createPanelForChartAndEngine(chart, engine, showTable, new Supplier<ChartEngineTableModel>()
        {
            @Override
            public ChartEngineTableModel get()
            {
                try
                {
                    final AbstractChartEngineTableModel tableModel = (AbstractChartEngineTableModel)chartTableModelClass.getDeclaredConstructor().newInstance();                                       
                    return tableModel;
                }
                catch(final Exception e)
                {
                    e.printStackTrace();
                    throw new IllegalArgumentException("Unable to instantiate class " + chartTableModelClass + ": "
                        + e.getMessage());
                }
            }
        });
    }

    /**
     * Calls
     * {@link ChartEngineChartAndTablePanel#ChartEngineChartAndTablePanel(ChartEngine, JFreeChart, boolean, boolean, boolean, boolean, boolean, boolean)}
     * and returns it. The default {@link ChartEngineTableModel} {@link Supplier} defined in
     * {@link ChartEngineChartAndTablePanel} will be used.
     */
    public static ChartEngineChartAndTablePanel createPanelForChartAndEngine(final JFreeChart chart,
                                                                             final ChartEngine engine,
                                                                             final boolean showTable)
    {
        return new ChartEngineChartAndTablePanel(engine, chart, showTable, false, true, true, true, true);
    }

    /**
     * Computes a pixel point from the given data point accounting for axis scaling due to maximum chart size being
     * exceeded.
     * 
     * @param panel The {@link ChartPanel} that is displaying the chart. This is used to acquire rendering information;
     *            {@link ChartPanel#getChartRenderingInfo()}.
     * @param engine The {@link ChartEngine} that yielded the chart.
     * @param dataSourceIndex The index of the data source corresponding to the data point. This is used to determine
     *            the subplot and axis against which it is plotted.
     * @param dataPoint The data point to convert.
     * @return A pixel {@link Point} that corresponds to the data point.
     */
    public static Point convertDataPointToPixels(final ChartPanel panel,
                                                 final ChartEngine engine,
                                                 final int dataSourceIndex,
                                                 final Point.Double dataPoint)
    {
        final int trueSubPlotIndex = ChartTools.detetermineTrueSubplotIndex(engine, panel.getChart(), dataSourceIndex);
        final CombinedDomainXYPlot plot = (CombinedDomainXYPlot)panel.getChart().getXYPlot();
        final XYPlot subPlot = (XYPlot)plot.getSubplots().get(trueSubPlotIndex);

        //Get the domain axis.
        final ValueAxis domainAxis = panel.getChart().getXYPlot().getDomainAxis();

        //Get the range axis
        final int axisIndex = engine.getChartParameters().getDataSourceParameters(dataSourceIndex).getYAxisIndex();
        final ValueAxis rangeAxis = subPlot.getRangeAxis(axisIndex);

        //get the value
        final double xPoint = domainAxis.valueToJava2D(dataPoint.getX(),
                                                       panel.getChartRenderingInfo()
                                                            .getPlotInfo()
                                                            .getSubplotInfo(trueSubPlotIndex)
                                                            .getDataArea(),
                                                       subPlot.getDomainAxisEdge()); //TODO Does this edge need to be overall????????
        final double yPoint = rangeAxis.valueToJava2D(dataPoint.getY(),
                                                      panel.getChartRenderingInfo()
                                                           .getPlotInfo()
                                                           .getSubplotInfo(trueSubPlotIndex)
                                                           .getDataArea(),
                                                      subPlot.getRangeAxisEdge(axisIndex));

        //Need to account for transformations due to exceeding maximum size
        //        panel.translateJava2DToScreen(new Point2D.Double(xPoint, yPoint));

        return new Point((int)(panel.getScaleX() * xPoint), (int)(panel.getScaleY() * yPoint));
    }

    /**
     * @param ts Time series of which the first one provides the unit.
     * @return Returns the string 'Value [unit]', unless the ts is empty. If it is, it returns null. This is useful for
     *         labels within {@link ChartEngine} charts.
     */
    public static String buildDefaultLabelFromTimeSeriesArrays(final TimeSeriesArrays ts, final String unitOverride)
    {
        String unit = "";
        if(ts.size() > 0)
        {
            unit = ts.get(0).getHeader().getUnit();
        }
        if(unitOverride != null)
        {
            unit = unitOverride;
        }
        if(ts.size() > 0)
        {
            return TimeSeriesArraysTools.returnOneWordDataTypeSummary(ts.get(0).getHeader().getParameterId()) + " ["
                + unit + "]";
        }
        return null;
    }
    
    /**
     * General test main for checking out appearance of plots and basic functionality.
     * @param args
     */
    public static void main(final String[] args)
    {
        final List<double[]> xValues = new ArrayList<>();
        final float barWidth = 0.5f;
       
        xValues.add(new double[]{1, 2, 3, 4, 5});
        xValues.add(new double[]{1, 2, 3, 4, 5});
        xValues.add(new double[]{1, 2, 3, 4, 5});

        final List<double[]> yValues = new ArrayList<>();
        yValues.add(new double[]{10, 9, 8,11, 9});
        yValues.add(new double[]{8, 10, 9,5, 12});
        yValues.add(new double[]{3, 12, 6,7, 12});
        
        final String[] xCats =new String[]{"Apples", "Oranges", "Pears", "Grapes", "Peaches"};
        XYChartDataSource source;
        try
        {
            
//            source = new NumericalXYChartDataSource(null, 0, xValues, yValues);
            source = new CategoricalXYChartDataSource(null, 0, xCats, yValues);
            final ChartEngine engine = ChartTools.buildChartEngineFromXYChartDataSource(source);
            
            engine.getChartParameters().getDomainAxis().getNumericAxisParameters().setCategoryLabels("Apples", "Oranges", "Pears", "Hanks", "Peaches");
            engine.getChartParameters().getDomainAxis().getNumericAxisParameters().setIntOnlyTickSpacing(true);
            engine.getChartParameters().getSubPlot(0).getLeftAxis().getNumericAxisParameters().setRangeIncludesZero(true);
            
            engine.getChartParameters().getDataSourceParameters(0).setPlotterName("Bar");
            engine.getChartParameters().getDataSourceParameters(0).getSeriesDrawingParametersForSeriesIndex(0).setFillColor(Color.red);
            engine.getChartParameters().getDataSourceParameters(0).getSeriesDrawingParametersForSeriesIndex(0).setBarWidth(0.5f);
            engine.getChartParameters().getDataSourceParameters(0).getSeriesDrawingParametersForSeriesIndex(0).setBoxWidth(0.8d);

            engine.getChartParameters().getDataSourceParameters(0).getSeriesDrawingParametersForSeriesIndex(1).setFillColor(Color.blue);
            engine.getChartParameters().getDataSourceParameters(0).getSeriesDrawingParametersForSeriesIndex(1).setFillColor(Color.blue);
            engine.getChartParameters().getDataSourceParameters(0).getSeriesDrawingParametersForSeriesIndex(1).setBarWidth(barWidth);
            
            engine.getChartParameters().getDataSourceParameters(0).getSeriesDrawingParametersForSeriesIndex(2).setFillColor(Color.green);
            engine.getChartParameters().getDataSourceParameters(0).getSeriesDrawingParametersForSeriesIndex(2).setFillColor(Color.green);
            engine.getChartParameters().getDataSourceParameters(0).getSeriesDrawingParametersForSeriesIndex(2).setBarWidth(barWidth);
            final JPanel panel = ChartPanelTools.buildPanelFromChartEngine(engine, false);

            //Put panel in a frame.
            final JFrame frame = new JFrame();
            frame.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
            frame.setSize(new Dimension(1000, 750));
            frame.setContentPane(panel);
            frame.setVisible(true);
        }
        catch(final Exception e)
        {
            // TODO Auto-generated catch block
            e.printStackTrace();
        }
    }

}
