package ohd.hseb.charter;

import java.awt.Color;
import java.awt.Stroke;
import java.awt.image.BufferedImage;
import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

import javax.imageio.ImageIO;

import org.jfree.chart.ChartPanel;
import org.jfree.chart.ChartRenderingInfo;
import org.jfree.chart.JFreeChart;
import org.jfree.chart.annotations.XYLineAnnotation;
import org.jfree.chart.axis.DateAxis;
import org.jfree.chart.axis.NumberAxis;
import org.jfree.chart.axis.ValueAxis;
import org.jfree.chart.plot.CombinedDomainXYPlot;
import org.jfree.chart.plot.Marker;
import org.jfree.chart.plot.XYPlot;
import org.jfree.ui.Layer;

import com.google.common.collect.Lists;

import nl.wldelft.util.timeseries.TimeSeriesArrays;
import ohd.hseb.charter.datasource.XYChartDataSource;
import ohd.hseb.charter.datasource.instances.TimeSeriesArraysXYChartDataSource;
import ohd.hseb.charter.parameters.ChartDrawingParameters;
import ohd.hseb.charter.parameters.DataSourceDrawingParameters;
import ohd.hseb.charter.tools.NumberAxisOverride;
import ohd.hseb.hefs.utils.arguments.ArgumentsProcessor;
import ohd.hseb.hefs.utils.gui.tools.ColorTools;
import ohd.hseb.hefs.utils.tools.FileTools;
import ohd.hseb.hefs.utils.tools.ListTools;
import ohd.hseb.hefs.utils.xml.XMLTools;
import ohd.hseb.util.data.DataPoint;

/**
 * General tools available related to {@link JFreeChart} and {@link ChartEngine}. For tools related to GUI interaction
 * or to construct a panel, see {@link ChartPanelTools}. For tools that are directly related to constants used by
 * {@link ChartEngine}, see {@link ChartConstants}.
 * 
 * @author Hank.Herr
 */
public abstract class ChartTools
{

    /**
     * Shortcut to building a ChartEngine from a set of time series. The arguments are set to null.
     * 
     * @param timeSeries TimeSeriesArrays, each of which is turned into a TimeSeriesArraysXYChartDataSource object.
     * @return A ChartEngine ready to build the chart.
     */
    public static ChartEngine buildChartEngineFromTimeSeriesArrays(final TimeSeriesArrays timeSeries)
    {
        TimeSeriesArraysXYChartDataSource source;
        try
        {
            source = new TimeSeriesArraysXYChartDataSource(null, 0, timeSeries);
            source.getDefaultFullySpecifiedDataSourceDrawingParameters()
                  .setDefaultRangeAxisTitle(ChartPanelTools.buildDefaultLabelFromTimeSeriesArrays(timeSeries, null));
            source.getDefaultFullySpecifiedDataSourceDrawingParameters().setDefaultDomainAxisTitle("Time");

            final ArrayList<XYChartDataSource> sources = new ArrayList<XYChartDataSource>();
            sources.add(source);
            final ChartEngine engine = new ChartEngine(null, sources);
            return engine;
        }
        catch(final Exception e)
        {
            //Should never happen!
            e.printStackTrace();
        }
        return null;
    }

    /**
     * A short cut to building a chart engine that can build a chart that displays a single data source. The arguments
     * are set to null.
     * 
     * @param source
     * @return
     */
    public static ChartEngine buildChartEngineFromXYChartDataSource(final XYChartDataSource source)
    {
        final ArrayList<XYChartDataSource> sources = new ArrayList<XYChartDataSource>();
        sources.add(source);
        return buildChartEngineFromXYChartDataSources(Lists.newArrayList(sources));
    }

    /**
     * Buids a chart engine from a listof {@link XYChartDataSource} instances.
     * 
     * @return The ChartEngine built or null if a problem occurs (it will also dump a stack trace to stdout in that
     *         case).
     */
    public static ChartEngine buildChartEngineFromXYChartDataSources(final List<XYChartDataSource> sources)
    {
        ChartEngine engine;
        try
        {
            engine = new ChartEngine(null, sources);
            return engine;
        }
        catch(final Exception e)
        {
            //Should never happen!
            e.printStackTrace();
        }
        return null;
    }

    /**
     * The returned {@link ChartEngine} will contains the data sources provided and have {@link ChartDrawingParameters}
     * parameters built by combining the following:<br>
     * <br>
     * 1. The defaults determined from the sources and standard defaults.<br>
     * 2. Overriding them with the default template loaded from the supplied resource or file name.<br>
     * 3. Overriding them with the overrides specified in the supplied {@link ChartDrawingParameters} likely read in
     * from configuration of some kind.
     * 
     * @param sources The sources for which to construct the {@link ChartEngine}.
     * @param arguments The arguments to use. Null is allowed.
     * @param defaultTemplateXMLFileOrResourceName The name of the resource or file that specifies the default
     *            {@link ChartDrawingParameters}. The file will be processed first as a system resource and, if it fails
     *            (either to find or read the resource), then it will process it as a file. If that also fails, this
     *            will throw an exception.
     * @param overrideParameters The override {@link ChartDrawingParameters} to copy overtop of the defaults.
     * @return The {@link ChartEngine} ready to either build an image or put it in a component.
     * @throws ChartEngineException If at any point in the process something breaks down, either while attempting to
     *             load the default template or while attempting to build the {@link ChartEngine}.
     */
    public static ChartEngine buildChartEngine(final List<XYChartDataSource> sources,
                                               final ArgumentsProcessor arguments,
                                               final String defaultTemplateXMLFileOrResourceName,
                                               final ChartDrawingParameters overrideParameters) throws ChartEngineException
    {
        //Load the default ChartParameters
        final ChartDrawingParameters defaultParameters = new ChartDrawingParameters();
        try
        {
            XMLTools.readXMLFromResource(defaultTemplateXMLFileOrResourceName, defaultParameters);
        }
        catch(final Throwable t)
        {
            try
            {
                XMLTools.readXMLFromFile(new File(defaultTemplateXMLFileOrResourceName), defaultParameters);
            }
            catch(final Throwable t2)
            {
                throw new ChartEngineException("Unable to load default chart drawing parameters from resource or file with name '"
                    + defaultTemplateXMLFileOrResourceName + "': " + t2.getMessage());
            }
        }
        return buildChartEngine(sources, arguments, defaultParameters, overrideParameters);
    }

    /**
     * The returned {@link ChartEngine} will contains the data sources provided and have {@link ChartDrawingParameters}
     * parameters built by combining the following:<br>
     * <br>
     * 1. The defaults determined from the sources and standard defaults.<br>
     * 2. Overriding them with the default parameters specified in the supplied {@link ChartDrawingParameters} likely
     * read in from configuration of some kind.<br>
     * 3. Overriding them with the overrides specified in the supplied {@link ChartDrawingParameters} likely read in
     * from configuration of some kind.
     * 
     * @param sources The sources for which to construct the {@link ChartEngine}.
     * @param arguments The arguments to use. Null is allowed.
     * @param defaultParameters The default {@link ChartDrawingParameters}.
     * @param overrideParameters The override {@link ChartDrawingParameters} to copy overtop of the defaults.
     * @return The {@link ChartEngine} ready to either build an image or put it in a component.
     * @throws ChartEngineException If at any point in the process something breaks down, either while attempting to
     *             load the default template or while attempting to build the {@link ChartEngine}.
     */
    public static ChartEngine buildChartEngine(final List<XYChartDataSource> sources,
                                               final ArgumentsProcessor arguments,
                                               final ChartDrawingParameters defaultParameters,
                                               final ChartDrawingParameters overrideParameters) throws ChartEngineException
    {
        //Create the engine using the default parameters as the override.
        //Then override those parameters.  
        final ChartEngine engine = new ChartEngine(arguments, sources, defaultParameters);
        if(overrideParameters != null)
        {
            engine.overrideParameters(overrideParameters);
        }
        return engine;
    }

    /**
     * @param calculatedAxisType The data source computed axis type.
     * @param overrideAxisType The axis type to be checked against the calculated type. If negative, this method does no
     *            checking.
     * @throws ChartEngineException If the calculated is a time axis, but the override is not; if the calculated is
     *             numerical but override is time or translated; if the calculated is translated and the override is
     *             anything positive; or if the calculated is negative (implying no data plotted against axis) and the
     *             override is not translated.
     */
    public static void checkCompatibilityOfAxis(final int calculatedAxisType,
                                                final int overrideAxisType) throws ChartEngineException
    {
        if(overrideAxisType < 0)
        {
            return;
        }
        if(calculatedAxisType == ChartConstants.AXIS_IS_TIME)
        {
            if((overrideAxisType == ChartConstants.AXIS_IS_NORMALIZED_PROBABILITY)
                || (overrideAxisType == ChartConstants.AXIS_IS_NUMERICAL)
                || (overrideAxisType == ChartConstants.AXIS_IS_PROBABILITY)
                || (overrideAxisType == ChartConstants.AXIS_IS_LOGARITHMIC))
            {
                throw new ChartEngineException("Data to plot against axis is time based, but the overridden axis type is numerical.");
            }
            if(overrideAxisType == ChartConstants.AXIS_IS_CATEGORICAL)
            {
                throw new ChartEngineException("Data to plot against axis is time based, but the overridden axis type is categorical.");
            }
            if(overrideAxisType == ChartConstants.AXIS_IS_TRANSLATED)
            {
                throw new ChartEngineException("Data to plot against axis is time, but the overridden axis type is translated.");
            }
        }
        else if(calculatedAxisType == ChartConstants.AXIS_IS_CATEGORICAL)
        {
            if(overrideAxisType != ChartConstants.AXIS_IS_CATEGORICAL)
            {
                throw new ChartEngineException("Data to plot against axis is categorical, but the overridden axis type is not.");
            }
        }
        else if((calculatedAxisType == ChartConstants.AXIS_IS_NUMERICAL)
            || (calculatedAxisType == ChartConstants.AXIS_IS_PROBABILITY)
            || (calculatedAxisType == ChartConstants.AXIS_IS_NORMALIZED_PROBABILITY))
        {
            if(overrideAxisType == ChartConstants.AXIS_IS_TIME)
            {
                throw new ChartEngineException("Data to plot against axis is numerical, but the overridden axis type is time-based.");
            }
            if(overrideAxisType == ChartConstants.AXIS_IS_CATEGORICAL)
            {
                throw new ChartEngineException("Data to plot against axis is numerical, but the overridden axis type is categorical.");
            }
            if(overrideAxisType == ChartConstants.AXIS_IS_TRANSLATED)
            {
                throw new ChartEngineException("Data to plot against axis is numerical, but the overridden axis type is translated.");
            }
        }
        else if(calculatedAxisType == ChartConstants.AXIS_IS_TRANSLATED)
        {
            if(overrideAxisType >= 0)
            {
                throw new ChartEngineException("Axis is translated, so it cannot be defined multiple times or overridden by a user.");
            }
        }
        else if(calculatedAxisType < 0)
        {
            if((overrideAxisType >= 0) && (overrideAxisType != ChartConstants.AXIS_IS_TRANSLATED))
            {
                throw new ChartEngineException("No data is plotted against axis, but the parameters are overriding the type.");
            }
        }
    }

    /**
     * This calls {@link #checkCompatibilityOfAxis(int, int)}.
     * 
     * @param axisType int representing the axis type; see AXIS_IS* constants within {@link ChartConstants}. For
     *            example, {@link ChartConstants#AXIS_IS_TIME}.
     * @return {@link List} of compatible axis type integers. Note that if the list is empty, the axis type is not
     *         compatible with any other axis. For example, if an axis is defined as translated, the user is not allowed
     *         to change it to another type and no other attempt to define the axis as translated are allowed.
     */
    public static List<Integer> obtainListOfCompatibleAxesForAxisType(final int axisType)
    {
        final List<Integer> results = new ArrayList<Integer>();
        for(int i = 0; i < ChartConstants.AXIS_CHOICES.length; i++)
        {
            try
            {
                checkCompatibilityOfAxis(axisType, i);
                results.add(i);
            }
            catch(final ChartEngineException e)
            {
                //Do nothing
            }
        }

        return results;
    }

    /**
     * Builds an array of colors of the specified size that moves from blue to cyan to green to yellow to red. Comes
     * from old ESPADP spaghetti plot coloring code, though it is not an exact match any more. Now it calls
     * {@link ColorTools#buildColorPalette(int, Color...)}, which cannot guarantee that the five colors in the
     * algorithm, blue, cyan, green, yellow, and red, will each be represented (see the javadoc).
     * 
     * @param numberOfColors Number of colors to generate.
     * @return An array of colors.
     */
    public static Color[] buildESPADPColorPalette(final int numberOfColors)
    {
        return ColorTools.buildColorPalette(numberOfColors,
                                            Color.BLUE,
                                            Color.CYAN,
                                            Color.GREEN,
                                            Color.YELLOW,
                                            Color.RED);
    }

    /**
     * Take the given shape names and apply them to the series in the provided {@link DataSourceDrawingParameters} so
     * that it follows a rotation.
     * 
     * @param shapes See {@value ChartConstants#SHAPE_NAMES} for valid shapes (or just pass it in to use default
     *            shapes).
     */
    public static void applyRotatingShapeSchemeToSeries(final String[] shapes, final DataSourceDrawingParameters parms)
    {
        for(int seriesIndex = 0; seriesIndex < parms.getSeriesParametersCount(); seriesIndex++)
        {
            parms.getSeriesDrawingParametersForSeriesIndex(seriesIndex)
                 .setShapeName(shapes[seriesIndex % shapes.length]);
        }
    }

    /**
     * Take the given colors and applies them to the series in the provided {@link DataSourceDrawingParameters} so that
     * it follows a rotation.
     */
    public static void applyRotatingColorSchemeToSeries(final Color[] colors, final DataSourceDrawingParameters parms)
    {
        for(int seriesIndex = 0; seriesIndex < parms.getSeriesParametersCount(); seriesIndex++)
        {
            parms.getSeriesDrawingParametersForSeriesIndex(seriesIndex)
                 .setLineColor(colors[seriesIndex % colors.length]);
            parms.getSeriesDrawingParametersForSeriesIndex(seriesIndex)
                 .setFillColor(parms.getSeriesDrawingParametersForSeriesIndex(seriesIndex).getLineColor());
        }
    }

    /**
     * Take the given colors and applies them to the series in the provided {@link DataSourceDrawingParameters} so that
     * it follows a rotation.
     */
    public static void applyRotatingColorSchemeToSeries(final Color[] lineColors,
                                                        final Color[] fillColors,
                                                        final DataSourceDrawingParameters parms)
    {
        for(int seriesIndex = 0; seriesIndex < parms.getSeriesParametersCount(); seriesIndex++)
        {
            if(lineColors != null)
            {
                parms.getSeriesDrawingParametersForSeriesIndex(seriesIndex)
                     .setLineColor(lineColors[seriesIndex % lineColors.length]);
            }
            if(fillColors != null)
            {
                parms.getSeriesDrawingParametersForSeriesIndex(seriesIndex)
                     .setFillColor(fillColors[seriesIndex % fillColors.length]);
            }
        }
    }

    /**
     * In the passed in combined domain plot, return a list of all of the markers defined. This was written for use in
     * {@link #clearAllMarkerLabelsInCombinedDomainXYPlot(CombinedDomainXYPlot)}.
     * 
     * @param plot The plot to search.
     * @return List of all markers for any subplot.
     */
    @SuppressWarnings("unchecked")
    public static List<Marker> returnAllMarkersInCombinedDomainXYPlot(final CombinedDomainXYPlot plot)
    {
        final List<Marker> results = new ArrayList<Marker>();
        ListTools.allAllItemsToListIfNotNull(results, plot.getDomainMarkers(Layer.BACKGROUND));
        ListTools.allAllItemsToListIfNotNull(results, plot.getDomainMarkers(Layer.FOREGROUND));
        for(int i = 0; i < plot.getSubplots().size(); i++)
        {
            final XYPlot subPlot = (XYPlot)plot.getSubplots().get(i);
            for(int j = 0; j < subPlot.getDatasetCount(); j++)
            {
                ListTools.allAllItemsToListIfNotNull(results, subPlot.getRangeMarkers(j, Layer.BACKGROUND));
                ListTools.allAllItemsToListIfNotNull(results, subPlot.getRangeMarkers(j, Layer.FOREGROUND));
            }
        }
        return results;
    }

    /**
     * Clears the label for all markers defined in the combined domain plot passed in. This tool is useful when creating
     * an icon version of a chart, such as within the thumbnails for GraphGen, since it removes some image clutter.
     */
    public static void clearAllMarkerLabelsInCombinedDomainXYPlot(final CombinedDomainXYPlot plot)
    {
        final List<Marker> marks = returnAllMarkersInCombinedDomainXYPlot(plot);
        for(int i = 0; i < marks.size(); i++)
        {
            marks.get(i).setLabel("");
        }
    }

    /**
     * Generates an output image file from a {@link JFreeChart} chart.
     * 
     * @param outputFile File to generate. The extension indicates the image type. It is recommended that JPEG is
     *            avoided due to potential coloring issues.
     * @param chart The chart.
     * @param width The width of the image to create.
     * @param height The height of the image to create.
     * @throws IOException If {@link ImageIO#write(java.awt.image.RenderedImage, String, File)} has an issue. The
     *             existing and writability of the file's parent directory are checked before generating the image.
     */
    public static void generateOutputImageFile(final File outputFile,
                                               final JFreeChart chart,
                                               final int width,
                                               final int height) throws IOException
    {
        //Checks for existing of output parent dir to avoid creating a null pointer exception.
        if((outputFile.getParentFile() != null) && (!outputFile.getParentFile().exists()))
        {
            throw new IOException("The parent of the file '" + outputFile.getAbsolutePath() + "' does not exist.");
        }
        if((outputFile.getParentFile() != null) && (!outputFile.getParentFile().canWrite()))
        {
            throw new IOException("The parent of the file '" + outputFile.getAbsolutePath()
                + "' cannot be written to.");
        }

        // formatName must be a String image name of the image type... "png", "bmp", "jpg", "jpeg", "gif"... usually the file extension.
        final String formatName = FileTools.getExtension(outputFile).toUpperCase();
        BufferedImage image = null;
        if((formatName.equalsIgnoreCase("jpg")) || (formatName.equalsIgnoreCase("jpeg"))
            || (formatName.equalsIgnoreCase("bmp")))
        {
            image = chart.createBufferedImage(width, height, BufferedImage.TYPE_INT_RGB, new ChartRenderingInfo());
        }
        else
        {
            image = chart.createBufferedImage(width, height);
        }
        ImageIO.write(image, formatName, outputFile);
    }

    /**
     * After a {@link JFreeChart} has been constructed via {@link ChartEngine#buildChart()}, this will force the axes to
     * be square (domain and range having the same limits). The squaring is applied over all subplots, but the same axis
     * for each subplot.<br>
     * <br>
     * Squaring is done after the plot is created by ChartEngine, so that axis limits therein will not be altered (and
     * thus the squaring will be lost after the chart is redrawn).
     * 
     * @param chart The chart to square. Note that {@link JFreeChart#getPlot()} must return a
     *            {@link CombinedDomainXYPlot}
     * @param rangeAxisString Identifies which range axis to square with the domain axis. Must be one of the
     *            {@link ChartConstants#YAXIS_XML_STRINGS} values.
     */
    public static void squareAxes(final JFreeChart chart, final String rangeAxisString)
    {
        final CombinedDomainXYPlot plot = (CombinedDomainXYPlot)chart.getPlot();

        double lb = plot.getDomainAxis().getLowerBound();
        double ub = plot.getDomainAxis().getUpperBound();

        for(int i = 0; i < plot.getSubplots().size(); i++)
        {
            final XYPlot subPlot = (XYPlot)plot.getSubplots().get(i);
            if(subPlot.getRangeAxis(ChartConstants.determineAxisIndexFromString(rangeAxisString)).getLowerBound() < lb)
            {
                lb = subPlot.getRangeAxis(ChartConstants.determineAxisIndexFromString(rangeAxisString)).getLowerBound();
            }
            if(subPlot.getRangeAxis(ChartConstants.determineAxisIndexFromString(rangeAxisString)).getUpperBound() > ub)
            {
                ub = subPlot.getRangeAxis(ChartConstants.determineAxisIndexFromString(rangeAxisString)).getUpperBound();
            }

        }

        plot.getDomainAxis().setLowerBound(lb);
        plot.getDomainAxis().setUpperBound(ub);
        for(int i = 0; i < plot.getSubplots().size(); i++)
        {
            final XYPlot subPlot = (XYPlot)plot.getSubplots().get(i);
            subPlot.getRangeAxis(ChartConstants.determineAxisIndexFromString(rangeAxisString)).setLowerBound(lb);
            subPlot.getRangeAxis(ChartConstants.determineAxisIndexFromString(rangeAxisString)).setUpperBound(ub);
        }

    }

    /**
     * Adds a diagonal line centered at the origin to the subplot pointed to by the provided index. Note that it is
     * assumed for the provided chart {@link JFreeChart#getPlot()} returns a {@link CombinedDomainXYPlot} and that the
     * index provides the subplot within that combined domain plot.<br>
     * <br>
     * The plot is edited after being generated by {@link ChartEngine} so that the parameter and data sources of the
     * chart are not modified (and, thus, the diagonal line will be lost if the chart is redrawn).
     * 
     * @param chart Chart to modify, for which {@link JFreeChart#getPlot()} returns a {@link CombinedDomainXYPlot}.
     * @param subPlotIndex The subplot index.
     * @param lineLimits The line to be created must start and end somewhere. This number defines that somewhere, with
     *            -1 multiplied by it defining the lower bound and it directly defining the upper bound.
     * @param stroke {@link Stroke} to use to draw the annotation.
     * @param color {@link Color} to use.
     */
    public static void addDiagonalLine(final JFreeChart chart,
                                       final int subPlotIndex,
                                       final double lineLimits,
                                       final Stroke stroke,
                                       final Color color)
    {
        final CombinedDomainXYPlot plot = (CombinedDomainXYPlot)chart.getPlot();
        final XYLineAnnotation diagonal = new XYLineAnnotation(-1 * lineLimits,
                                                               -1 * lineLimits,
                                                               lineLimits,
                                                               lineLimits,
                                                               stroke,
                                                               color);
        ((XYPlot)plot.getSubplots().get(subPlotIndex)).clearAnnotations();
        ((XYPlot)plot.getSubplots().get(subPlotIndex)).addAnnotation(diagonal);
    }

    /**
     * Adds a diagonal line centered at the origin to the subplot pointed to by the provided index. The diagonal will go
     * from one edge of the plot to the other. Note that it is assumed for the provided chart
     * {@link JFreeChart#getPlot()} returns a {@link CombinedDomainXYPlot} and that the index provides the subplot
     * within that combined domain plot.<br>
     * <br>
     * The plot is edited after being generated by {@link ChartEngine} so that the parameter and data sources of the
     * chart are not modified (and, thus, the diagonal line will be lost if the chart is redrawn).
     * 
     * @param chart Chart to modify, for which {@link JFreeChart#getPlot()} returns a {@link CombinedDomainXYPlot}.
     * @param subPlotIndex The subplot index.
     * @param stroke {@link Stroke} to use to draw the annotation.
     * @param color {@link Color} to use.
     */
    public static void addDiagonalLineToEdge(final JFreeChart chart,
                                             final int subPlotIndex,
                                             final Stroke stroke,
                                             final Color color)
    {
        final CombinedDomainXYPlot plot = (CombinedDomainXYPlot)chart.getPlot();
        final XYPlot subPlot = (XYPlot)plot.getSubplots().get(subPlotIndex);
        final double lowValue = Math.min(subPlot.getDomainAxis().getRange().getLowerBound(),
                                         subPlot.getRangeAxis().getRange().getLowerBound());
        final double highValue = Math.max(subPlot.getDomainAxis().getRange().getUpperBound(),
                                          subPlot.getRangeAxis().getRange().getUpperBound());
        final XYLineAnnotation diagonal = new XYLineAnnotation(lowValue, lowValue, highValue, highValue, stroke, color);
        ((XYPlot)plot.getSubplots().get(subPlotIndex)).clearAnnotations();
        ((XYPlot)plot.getSubplots().get(subPlotIndex)).addAnnotation(diagonal);
    }

    /**
     * @return The number rounded to an appropriate format based on the axis, so that it contains a appropriate number
     *         of decimal places (for a number axis) or is rounded to a long (for date axis).
     */
    public static double roundToAxisFormat(final ValueAxis axis, final double number)
    {
        if(axis instanceof NumberAxis)
        {
            final NumberAxis numAxis = (NumberAxis)axis;
            String numStr;
            if(numAxis.getNumberFormatOverride() != null)
            {
                numStr = numAxis.getNumberFormatOverride().format(number);
            }
            else
            {
                numStr = numAxis.getTickUnit().valueToString(number);
            }
            return Double.parseDouble(numStr);
        }
        else if(axis instanceof DateAxis)
        {
            return (long)number;
        }
        else
        {
            throw new IllegalArgumentException("Provided axis is neither a NumberAxis nor a DateAxis");
        }
    }

    /**
     * This is compatible with {@link ChartEngine}.
     * 
     * @param chart Must be a chart display a {@link CombinedDomainXYPlot}.
     * @return A {@link DataPoint} containing the lower bounds of the axes in data scale displayed within subplot 0
     *         (only) of the {@link CombinedDomainXYPlot}.
     */
    public static DataPoint constructLowerBoundsDataPoint(final JFreeChart chart, final int subPlotIndex)
    {
        final CombinedDomainXYPlot plot = (CombinedDomainXYPlot)chart.getXYPlot();
        final XYPlot checkPlot = (XYPlot)plot.getSubplots().get(subPlotIndex);
        final DataPoint lowerBounds = new DataPoint(checkPlot.getRangeAxisCount());

        lowerBounds.setX(checkPlot.getDomainAxis().getLowerBound());
        lowerBounds.setY(0, checkPlot.getRangeAxis(0).getLowerBound());
        if(checkPlot.getRangeAxisCount() == 2)
        {
            lowerBounds.setY(1, checkPlot.getRangeAxis(1).getLowerBound());
        }
        return lowerBounds;
    }

    /**
     * This is compatible with {@link ChartEngine}.
     * 
     * @param chart Must be a chart display a {@link CombinedDomainXYPlot}.
     * @return A {@link DataPoint} containing the upper bounds of the axes in data scale displayed within subplot 0
     *         (only) of the {@link CombinedDomainXYPlot}.
     */
    public static DataPoint constructUpperBoundsDataPoint(final JFreeChart chart, final int subPlotIndex)
    {
        final CombinedDomainXYPlot plot = (CombinedDomainXYPlot)chart.getXYPlot();
        final XYPlot checkPlot = (XYPlot)plot.getSubplots().get(subPlotIndex);
        final DataPoint upperBounds = new DataPoint(checkPlot.getRangeAxisCount());

        upperBounds.setX(checkPlot.getDomainAxis().getUpperBound());
        upperBounds.setY(0, checkPlot.getRangeAxis(0).getUpperBound());
        if(checkPlot.getRangeAxisCount() == 2)
        {
            upperBounds.setY(1, checkPlot.getRangeAxis(1).getUpperBound());
        }
        return upperBounds;
    }

    /**
     * This is compatible with {@link ChartEngine}.
     * 
     * @return True if the chart already has limits that equal the provided {@link DataPoint} upper bound and lower
     *         bounds.
     */
    public static boolean areAxisLimitsIdentical(final JFreeChart chart,
                                                 final int subPlotIndex,
                                                 final DataPoint lowerBounds,
                                                 final DataPoint upperBounds)
    {
        final DataPoint checkLowerBounds = constructLowerBoundsDataPoint(chart, subPlotIndex);
        final DataPoint checkUpperBounds = constructUpperBoundsDataPoint(chart, subPlotIndex);

        return (lowerBounds.equals(checkLowerBounds) && upperBounds.equals(checkUpperBounds));
    }

    /**
     * The method is useful for identifying the true index of the subplot within the {@link CombinedDomainXYPlot}
     * contained within a {@link ChartEngine} built {@link JFreeChart}. User's can specify subplot indices within the
     * {@link ChartEngine} framework that do not include 0. However, the true subplot indices must ALWAYS start with 0.
     * This method can be used to find that true subplot index if needed for some reason (see
     * {@link ChartPanelTools#convertDataPointToPixels(ChartPanel, ChartEngine, int, java.awt.geom.Point2D.Double)}).
     * 
     * @param chartEngine The engine that generated the chart.
     * @param chart The chart.
     * @param dataSourceIndex The index of the data source for which to determine the true subplot index.
     * @return The true index of the subplot corresponding to the data source within the generated chart.
     */
    public static int detetermineTrueSubplotIndex(final ChartEngine chartEngine,
                                                  final JFreeChart chart,
                                                  final int dataSourceIndex)
    {

        final XYPlot subPlot = chartEngine.getSubPlot(chartEngine.getChartParameters()
                                                                 .getDataSourceParameters()
                                                                 .get(dataSourceIndex)
                                                                 .getSubPlotIndex());
        return ((CombinedDomainXYPlot)chart.getXYPlot()).getSubplots().indexOf(subPlot);
    }

    /**
     * @param chart
     * @return True if the domain axis in the provided chart, which is assumed to be {@link CombinedDomainXYPlot}
     *         consistent with {@link ChartEngine}, is a categorical.
     */
    public static boolean isDomainAxisCategorical(final JFreeChart chart)
    {
        if(!(((CombinedDomainXYPlot)chart.getPlot()).getDomainAxis() instanceof NumberAxisOverride))
        {
            return false;
        }
        final NumberAxisOverride axis = (NumberAxisOverride)((CombinedDomainXYPlot)chart.getPlot()).getDomainAxis();
        return axis.isCategorical();
    }

    /**
     * @param chart The chart.
     * @param category The category for whicht compute its numerical value.
     * @return The numerical value corresponding to a category.
     */
    public static double computeNumericalValueForCategory(final JFreeChart chart, final String category)
    {
        if(!(((CombinedDomainXYPlot)chart.getPlot()).getDomainAxis() instanceof NumberAxisOverride))
        {
            throw new IllegalStateException("Cannot compute numerical value for a category with a domain axis that is a "
                + ((CombinedDomainXYPlot)chart.getPlot()).getDomainAxis().getClass());
        }
        final NumberAxisOverride axis = (NumberAxisOverride)((CombinedDomainXYPlot)chart.getPlot()).getDomainAxis();
        return axis.getXValueForCategory(category);
    }

    /**
     * @param chart The chart.
     * @param number The numerical value.
     * @return The category corresponding to the numerical value.
     */
    public static String computeCategoryForNumericalValue(final JFreeChart chart, final double number)
    {
        if(!(((CombinedDomainXYPlot)chart.getPlot()).getDomainAxis() instanceof NumberAxisOverride))
        {
            throw new IllegalStateException("Cannot compute numerical value for a category with a domain axis that is a "
                + ((CombinedDomainXYPlot)chart.getPlot()).getDomainAxis().getClass());
        }
        final NumberAxisOverride axis = (NumberAxisOverride)((CombinedDomainXYPlot)chart.getPlot()).getDomainAxis();
        return axis.getCategoryForXValue(number);
    }

}
