package ohd.hseb.charter.datasource;

import java.util.Map;

import org.jfree.data.xy.XYDataset;

import com.google.common.collect.Maps;

import ohd.hseb.charter.ChartConstants;
import ohd.hseb.charter.ChartParameters;
import ohd.hseb.charter.panel.ChartEngineChartAndTablePanel;
import ohd.hseb.charter.parameters.ChartDrawingParameters;
import ohd.hseb.charter.parameters.DataSourceDrawingParameters;
import ohd.hseb.charter.parameters.SeriesDrawingParameters;

/**
 * Default implementation of {@link XYChartDataSource}. <br>
 * <br>
 * The constructor of each data source must initialize the data source parameters object by calling
 * {@link #getDefaultFullySpecifiedDataSourceDrawingParameters()} and setting the default parameters and series
 * parameters for each series. It should begin by calling {@link #constructAllSeriesDrawingParameters(int)} to setup all
 * initialized parameters. It must ensure that there exists one {@link SeriesDrawingParameters} object per series being
 * drawn. <br>
 * <br>
 * The method {@link #buildXYDataset(ChartDrawingParameters, int)} builds the dataset for plotting, and is called within
 * {@link #prepareXYDataset(ChartDrawingParameters, int)} which is called when the engine builds the chart. Any subclass
 * must override the build method. It must also override {@link #getNumberOfSeries()} and
 * {@link #returnNewInstanceWithCopyOfInitialParameters()} which returns a copy of the parameters that shares the data
 * sources but uses a copy of the initial parameters. Lastly, be sure to initialize the {@link #_xAxisType} and
 * {@link #_computedDataType} attributes in the constructor of the subclass using the set methods.<br>
 * <br>
 * Note that the {@link #buildXYDataset(DataSourceDrawingParameters)} and
 * {@link #prepareXYDataset(DataSourceDrawingParameters)} methods are usually called directly only for unit testing, but
 * it some cases may be called by other build and prepare methods to avoid unnecessary code.
 * 
 * @author herrhd
 */
public abstract class DefaultXYChartDataSource implements XYChartDataSource
{

    /**
     * The initial set of drawing parameters. These form the basis of the drawing parameters used as the default drawing
     * parameters. When initialized, the data source index will be -1.
     */
    private DataSourceDrawingParameters _initialParameters = new DataSourceDrawingParameters(-1);

    /**
     * The most recent build data set, set through calls to either {@link #setXYDataset(XYDataset)} or, most likely,
     * {@link #prepareXYDataset(ChartDrawingParameters, int)} or {@link #prepareXYDataset(DataSourceDrawingParameters)}.
     */
    private XYDataset _xyDataset = null;

    /**
     * Stores the domain type for the data, which must be either {@link ChartConstants#AXIS_IS_TIME} or
     * {@link ChartConstants#AXIS_IS_NUMERICAL}.
     */
    private int _xAxisType = -1;

    /**
     * Stores the range axis type for the data, which I think must be {@link ChartConstants#AXIS_IS_NUMERICAL}, but I
     * could be wrong.
     */
    private int _computedDataType = -1;

    /**
     * A {@link String} defining the units of the data.
     */
    private String _unitsString = null;

    /**
     * The object that generated the data source.
     */
    private DataSourceGenerator _generator = null;

    /**
     * The name for the source to display in the data table.
     */
    private String _sourceNameInTable = null;

    /**
     * A map to store the column header to display for a series index. In other words, a name for each series.
     */
    private final Map<Integer, String> _seriesIndexToColumnHeaderMap = Maps.newHashMap();

    /**
     * A name to associate with the domain column within a data table.
     */
    private String _domainHeader = null;

    /**
     * Tool provided to help subclasses override {@link #returnNewInstanceWithCopyOfInitialParameters()}. It copies
     * {@link #_xAxisType}, {@link #_computedDataType}, {@link #_initialParameters}, {@link #_unitsString}, and
     * {@link #_generator} (by link) into the passed in source.
     * 
     * @param sourceToCopyTo
     */
    protected void copyTheseParametersIntoDataSource(final DefaultXYChartDataSource sourceToCopyTo)
    {
        sourceToCopyTo.setComputedDataType(_computedDataType);
        sourceToCopyTo.setXAxisType(_xAxisType);
        sourceToCopyTo.getDefaultFullySpecifiedDataSourceDrawingParameters()
                      .copyFrom(this.getDefaultFullySpecifiedDataSourceDrawingParameters());
        sourceToCopyTo.setUnitsString(_unitsString);
        sourceToCopyTo.setGenerator(_generator);
    }

    /**
     * @param s The name to use for this source if the data is displayed in a {@link ChartEngineChartAndTablePanel}'s
     *            table. It is null be default, so that a default must be constructed.
     */
    public void setSourceNameInTable(final String s)
    {
        _sourceNameInTable = s;
    }

    /**
     * Associates a table column header {@link String} with a series index.
     * 
     * @param seriesIndex Index of the series with the header.
     * @param header The header to display for the series column specified by the index. A null return should be taken
     *            to mean that a default must be constructed.
     */
    public void addChartTableColumnHeader(final int seriesIndex, final String header)
    {
        _seriesIndexToColumnHeaderMap.put(seriesIndex, header);
    }

    /**
     * This assumes all series have the same general domain variable, even if its different values.
     * 
     * @param header The header to display in a table of data for the domain variable of a series. It is null by default
     *            so that a default must be constructed.
     */
    public void setDomainHeader(final String header)
    {
        _domainHeader = header;
    }

    /**
     * Sets the {@link DataSourceDrawingParameters} to be stored in {@link #_initialParameters}.
     */
    public void setInitialDatasetDrawingParameters(final DataSourceDrawingParameters initialParameters)
    {
        this._initialParameters = initialParameters;
    }

    public void setXYDataset(final XYDataset dataset)
    {
        _xyDataset = dataset;
    }

    /**
     * Should be one of ChartToolsAndConstants.AXIS_IS*.
     * 
     * @param type
     */
    public void setXAxisType(final int type)
    {
        this._xAxisType = type;
    }

    @Override
    public int getXAxisType()
    {
        return this._xAxisType;
    }

    public void setComputedDataType(final int type)
    {
        this._computedDataType = type;
    }

    @Override
    public int getComputedDataType()
    {
        return this._computedDataType;
    }

    public void setUnitsString(final String units)
    {
        _unitsString = units;
    }

    @Override
    public String getUnitsString()
    {
        return this._unitsString;
    }

    @Override
    public DataSourceGenerator getGenerator()
    {
        return this._generator;
    }

    public void setGenerator(final DataSourceGenerator generator)
    {
        this._generator = generator;
    }

    /**
     * Sets up the initial parameters for use by a subclass.
     */
    protected void constructAllSeriesDrawingParameters(final int numberOfSeries)
    {
        _initialParameters.synchronizeSeriesParametersListWithNumberOfSeriesPlotted(numberOfSeries);
    }

    /**
     * This is called by {@link #prepareXYDataset(DataSourceDrawingParameters)}, which is called directly in unit
     * testing. However, it may be called by the other {@link #buildXYDataset(ChartDrawingParameters, int)} method, so
     * long as what is needed to build the {@link XYDataset} is contained completely within the
     * {@link DataSourceDrawingParameters}.<br>
     * <br>
     * This is the build method employed during unit testing, typically.
     * 
     * @param parameters The {@link DataSourceDrawingParameters} used to build the returned {@link XYDataset}.
     * @return Never return null!!! This is the {@link XYDataset} specified by the instance of
     *         {@link DefaultXYChartDataSource}.
     * @throws An {@link XYChartDataSourceException} if a problem occurs.
     */
    protected abstract XYDataset buildXYDataset(DataSourceDrawingParameters parameters) throws XYChartDataSourceException;

    /**
     * This relays the call to {@link #buildXYDataset(DataSourceDrawingParameters)}, unless a special build algorithm is
     * necessary that makes use of other information in the {@link ChartDrawingParameters} that cannot be relayed
     * through {@link DataSourceDrawingParameters}. In that case, override this method.<br>
     * <br>
     * This is the build method used outside of testing.
     * 
     * @param chartParameters The parameters from which the {@link DataSourceDrawingParameters} necessary for building
     *            the {@link XYDataset} along with other information is drawn.
     * @param dataSourceIndex The index of the data source withing the provided parameters.
     * @return Never return null!!! This is the {@link XYDataset} specified by the instance of DefaultXYChartDataSource.
     * @throws An {@link XYChartDataSourceException} if a problem occurs.
     */
    protected XYDataset buildXYDataset(final ChartDrawingParameters chartParameters,
                                       final int dataSourceIndex) throws XYChartDataSourceException
    {
        return buildXYDataset(chartParameters.getDataSourceParameters(dataSourceIndex));
    }

    @Override
    public void prepareXYDataset(final ChartDrawingParameters chartParameters,
                                 final int dataSourceIndex) throws XYChartDataSourceException
    {
        if(this._xyDataset == null)
        {
            setXYDataset(buildXYDataset(chartParameters, dataSourceIndex));
        }
    }

    /**
     * ONLY USED IN TESTING! Its a version that allows for executing without needing a full set of
     * {@link ChartParameters}.
     */
    @Override
    public void prepareXYDataset(final DataSourceDrawingParameters parameters) throws XYChartDataSourceException
    {
        if(this._xyDataset == null)
        {
            setXYDataset(buildXYDataset(parameters));
        }
    }

    /**
     * These checks are made relative to the {@link #_xyDataset} and passed in parameters to validate. It returns an
     * error if (1) the parameters are null; (2) at least one series does not have corresponding series parameters
     * 
     * @throws XYChartDataSourceException If a check fails.
     */
    @Override
    public void checkForValidity(final DataSourceDrawingParameters parametersToValidate) throws XYChartDataSourceException
    {
        if(parametersToValidate == null)
        {
            throw new XYChartDataSourceException("Parameters for data source are not defined.");
        }

        for(int i = 0; i < this.getXYDataSet().getSeriesCount(); i++)
        {
            final SeriesDrawingParameters seriesParms =
                                                      parametersToValidate.getSeriesDrawingParametersForSeriesIndex(i);
            if(seriesParms == null)
            {
                throw new XYChartDataSourceException("Sseries " + i + " does not have a corresponding"
                    + "set of seriesDrawingParameters.");
            }
        }

        if(parametersToValidate.getYAxisIndex() == null)
        {
            throw new XYChartDataSourceException("Y axis index not specified for data source.");
        }
        if(parametersToValidate.getSubPlotIndex() == null)
        {
            throw new XYChartDataSourceException("Subplot index not specified for data source.");
        }
    }

    @Override
    public DataSourceDrawingParameters getDefaultFullySpecifiedDataSourceDrawingParameters()
    {
        return this._initialParameters;
    }

    @Override
    public XYDataset getXYDataSet()
    {
        return this._xyDataset;
    }

    @Override
    public int getNumberOfSeries()
    {
        return _xyDataset.getSeriesCount();
    }

    @Override
    public Object getSeriesValue(final int seriesIndex, final int valueIndex, final boolean xValue)
    {
        if(xValue)
        {
            return _xyDataset.getXValue(seriesIndex, valueIndex);
        }
        return _xyDataset.getYValue(seriesIndex, valueIndex);
    }

    @Override
    public int getSeriesValueCount(final int seriesIndex)
    {
        return _xyDataset.getItemCount(seriesIndex);

    }

    @Override
    public int getDataSourceOrderIndex()
    {
        return this._initialParameters.getDataSourceOrderIndex();
    }

    @Override
    public boolean isEmpty()
    {
        return (this.getNumberOfSeries() == 0);
    }

    /**
     * @return By default, this returns null so that the legend entry is used as the column header.
     */
    @Override
    public String getChartTableColumnHeader(final int seriesIndex)
    {
        return _seriesIndexToColumnHeaderMap.get(seriesIndex);
    }

    @Override
    public String getChartTableSourceName()
    {
        return _sourceNameInTable;
    }

    @Override
    public String getChartTableDomainHeader()
    {
        return _domainHeader;
    }

}
