package ohd.hseb.charter.datasource;

import ohd.hseb.charter.panel.AbstractChartEngineTableModel;
import ohd.hseb.charter.panel.ChartEngineChartAndTablePanel;
import ohd.hseb.charter.parameters.ChartDrawingParameters;
import ohd.hseb.charter.parameters.DataSourceDrawingParameters;

import org.jfree.data.xy.XYDataset;

/**
 * Interface for an XYChartDataSource. You should subclass DefaultXYChartDataSource instead of implementing this
 * interface.
 * 
 * @author herrhd
 */
public interface XYChartDataSource
{
    /**
     * @return A fully initialized set of parameters that can be used for displaying the source.
     */
    public DataSourceDrawingParameters getDefaultFullySpecifiedDataSourceDrawingParameters();

    /**
     * @return A JFreeChart XYDataset for display on an XYPlot.
     */
    public XYDataset getXYDataSet();

    /**
     * The x-axis type; see ChartToolsAndConstants.
     * 
     * @return
     */
    public int getXAxisType();

    /**
     * The type of data plotted on the y-axis, which must be one of the data types in the ChartToolsAndConstants (time,
     * numerical, etc.).
     */
    public int getComputedDataType();

    /**
     * @return A String identifying the units of the compute ddata (i.e. CFS or cms), if any.
     */
    public String getUnitsString();

    /**
     * This can only be run AFTER the XYDataset is prepared.
     * 
     * @param parametersToValidate DataSourceDrawingParameters parameters to validate based on this data source.
     * @throws XYChartDataSourceException
     */
    public void checkForValidity(DataSourceDrawingParameters parametersToValidate) throws XYChartDataSourceException;

    /**
     * Prepares an XYDataset to display the data in JFreeChart.
     * 
     * @param chartParameters The entire set of parameters for the chart, in case something other than
     *            {@link DataSourceDrawingParameters} is needed. For example, time series drawing requires the chart's
     *            time zone.
     * @param dataSourceIndex Index of the {@link DataSourceDrawingParameters} corresponding to this.
     * @throws XYChartDataSourceException
     */
    public void prepareXYDataset(ChartDrawingParameters chartParameters, int dataSourceIndex) throws XYChartDataSourceException;

    /**
     * Prepares an XYDataset to display the data in JFreeChart given only a {@link DataSourceDrawingParameters}. If
     * preparation must include other items from {@link ChartDrawingParameters}, defaults must be assumed. This method
     * is typically only used for testing.
     * 
     * @param parameters The {@link DataSourceDrawingParameters} to use. In this case, any other parameters must be
     *            defaulted to something.
     * @throws XYChartDataSourceException
     */
    public void prepareXYDataset(DataSourceDrawingParameters parameters) throws XYChartDataSourceException;

    /**
     * @return The order of the data source. This must be consistent with parameters that refer to this data source.
     */
    public int getDataSourceOrderIndex();

    /**
     * @return The number of series calculated.
     */
    public int getNumberOfSeries();

    /**
     * @return A copy of the data source in which the underlying data objects are shared, but the initial parameters are
     *         a clone. The XYDataset will be left empty.
     * @throws XYChartDataSourceException
     */
    public XYChartDataSource returnNewInstanceWithCopyOfInitialParameters() throws XYChartDataSourceException;

    /**
     * @return True if no series are calculated.
     */
    public boolean isEmpty();

    /**
     * @param seriesIndex Index of series.
     * @param valueIndex Index of value in series.
     * @param xValue True if for x-axis value or false if for range or y-axis value.
     * @return An object containing the value, typically Date or a number.
     */
    public Object getSeriesValue(int seriesIndex, int valueIndex, boolean xValue);

    /**
     * @param seriesIndex Index of series.
     * @return number of data values in the series.
     */
    public int getSeriesValueCount(int seriesIndex);

    /**
     * @return Returns the DataSourceGenerator that generated this data source, or the original data source if this is a
     *         copy. This may be null if no generator is specified.
     */
    public DataSourceGenerator getGenerator();

    /**
     * @param seriesIndex Index of series
     * @return A string to display as a column header for the series in a {@link ChartEngineChartAndTablePanel}. If null
     *         is returned, then the legend entry for the series pulled from chart parameters will be displayed.
     */
    public String getChartTableColumnHeader(int seriesIndex);

    /**
     * @return A name that will be displayed in a {@link ChartEngineChartAndTablePanel} source selection choice box if
     *         displayed as a table within the panel. Return null to use a name constructed in
     *         {@link AbstractChartEngineTableModel}.
     */
    public String getChartTableSourceName();

    /**
     * @return A name to display in a {@link ChartEngineChartAndTablePanel} for the domain variable of all series in the
     *         source.
     */
    public String getChartTableDomainHeader();
}
