package ohd.hseb.charter.datasource.instances;

import org.jfree.data.xy.DefaultXYDataset;
import org.jfree.data.xy.XYBarDataset;
import org.jfree.data.xy.XYDataset;

/**
 * This data set is necessary due to the nature of the bar plotter. Specifically, it takes an original data set, maps it
 * to a special data set so that bars can be non-overlapping, and then puts that data set into the bar dataset. In order
 * for tool tips to work, we need to be able to map the final, bar data set to the original data set, which this allows
 * (the bar data set, itself, records an underlying data set, which is the middle man in the scenario above; we need the
 * original).
 * 
 * @author Hank.Herr
 */
public class CategoricalXYBarDataset extends XYBarDataset
{

    private final XYDataset _originalDataset;

    public CategoricalXYBarDataset(final XYDataset originalDataset,
                                   final double overlapProportion,
                                   final float barWidth)
    {
        super(constructAdjustedOriginalDataset(originalDataset, overlapProportion, barWidth), barWidth);
        _originalDataset = originalDataset;
    }

    public XYDataset getOriginalDataset()
    {
        return _originalDataset;

    }

    private static XYDataset constructAdjustedOriginalDataset(final XYDataset rootSet,
                                                              double overlapProportion,
                                                              float barWidth)
    {
        final DefaultXYDataset dataSet = new DefaultXYDataset();

        //Default to full width.
        if(barWidth <= 0)
        {
            barWidth = 1.0f;
        }

        //Default to completely overlapping.
        if((0 > overlapProportion) || (1 < overlapProportion))
        {
            overlapProportion = 1.0;
        }

        //Compute adjustment factors. 
        final double totalRangeOfCenters = (rootSet.getSeriesCount() - 1) * ((1.0 - overlapProportion) * barWidth);
        final double firstCenter = -1.0 * totalRangeOfCenters / rootSet.getSeriesCount();
        double adjustFactor = firstCenter;

        //For each series.
        for(int i = 0; i < rootSet.getSeriesCount(); i++)
        {
            final double[][] seriesData = new double[2][rootSet.getItemCount(i)];
            for(int j = 0; j < rootSet.getItemCount(i); j++)
            {
                seriesData[0][j] = rootSet.getXValue(i, j) + adjustFactor;
                seriesData[1][j] = rootSet.getYValue(i, j);
            }

            dataSet.addSeries(rootSet.getSeriesKey(i), seriesData);

            //Shift the adjustFactor up.
            adjustFactor += (barWidth * (1.0 - overlapProportion));
        }
        return dataSet;
    }
}
