package ohd.hseb.charter.datasource.instances;

import java.util.ArrayList;
import java.util.List;

import ohd.hseb.charter.ChartConstants;
import ohd.hseb.charter.datasource.DataSourceGenerator;
import ohd.hseb.charter.datasource.XYChartDataSource;
import ohd.hseb.charter.datasource.XYChartDataSourceException;

/**
 * A type of {@link NumericalXYChartDataSource} that includes a list of categories corresponding to the numerical values
 * 1, 2, 3, ... The only significant change is to the method {@link #getXAxisType()}, which returns
 * {@link ChartConstants#AXIS_IS_CATEGORICAL} for this data source. Note that the dataset displayed here assumes the
 * categorical domain values, which are numbers, are all identical. However, the plotter will plot them staggered if the
 * the bars are not completely overlapping, leading to the use of a special data set derived from this data set in the
 * chart itself.
 * 
 * @author Hank.Herr
 */
public class CategoricalXYChartDataSource extends NumericalXYChartDataSource
{
    private final String[] _xCategories;

    /**
     * @param generator Null is allowed.
     * @param dataSourceOrderIndex the order index of this data source relative to other plotted data sources.
     * @param xAxisValuesBySeries Values to plot along domain axis.
     * @param yAxisValuesBySeries Values to plot along range axis.
     * @throws XYChartDataSourceException Not thrown, currently.
     */
    public CategoricalXYChartDataSource(final DataSourceGenerator generator,
                                        final int dataSourceOrderIndex,
                                        final String[] xCategories,
                                        final List<double[]> yAxisValuesBySeries) throws XYChartDataSourceException
    {
        super(generator,
              dataSourceOrderIndex,
              generateXValues(yAxisValuesBySeries.size(), xCategories.length),
              yAxisValuesBySeries);
        this._xCategories = xCategories;
    }

    public String[] getXCategories()
    {
        return _xCategories;
    }

    /**
     * x-values start counting at one!
     * 
     * @param seriesCount
     * @param categoryCount
     * @return
     */
    private static List<double[]> generateXValues(final int seriesCount, final int categoryCount)
    {
        final List<double[]> builtXValues = new ArrayList<>();
        for(int i = 0; i < seriesCount; i++)
        {
            final double[] values = new double[categoryCount];
            for(int j = 0; j < categoryCount; j++)
            {
                values[j] = j + 1;
            }
            builtXValues.add(values);
        }
        return builtXValues;
    }

    @Override
    public XYChartDataSource returnNewInstanceWithCopyOfInitialParameters() throws XYChartDataSourceException
    {
        final CategoricalXYChartDataSource newInstance = new CategoricalXYChartDataSource(this.getGenerator(),
                                                                                          getDataSourceOrderIndex(),
                                                                                          _xCategories,
                                                                                          getYValues());
        copyTheseParametersIntoDataSource(newInstance);
        return newInstance;
    }

    @Override
    public Object getSeriesValue(final int seriesIndex, final int valueCount, final boolean xValue)
    {
        if(seriesIndex >= this._xCategories.length)
        {
            return null;
        }
        if(xValue)
        {
            if(valueCount >= _xCategories.length)
            {
                return null;
            }
            return _xCategories[valueCount];
        }
        return super.getSeriesValue(seriesIndex, valueCount, xValue);
    }

    @Override
    public int getXAxisType()
    {
        return ChartConstants.AXIS_IS_CATEGORICAL;
    }

}
