package ohd.hseb.charter.datasource.instances;

import java.util.List;

import com.google.common.collect.Lists;

import ohd.hseb.charter.datasource.DataSourceGenerator;
import ohd.hseb.charter.datasource.XYChartDataSourceException;
import ohd.hseb.util.data.DataSet;

/**
 * A {@link NumericalXYChartDataSource} that is constructed using a {@link DataSet} as a starting point.
 * 
 * @author hankherr
 */
public class DataSetXYChartDataSource extends NumericalXYChartDataSource
{

    /**
     * The data.
     */
    private final DataSet _data;

    /**
     * @param generator The {@link DataSourceGenerator} that generated this data.
     * @param dataSourceOrderIndex The index of the data source in the chart.
     * @param data The data underlying this source.
     * @param xVariables The indices of the variables that correspond to the x-axis.
     * @param yVariables The matching indicex of the variables that correspond to the y-axis.
     * @throws XYChartDataSourceException See
     *             {@link NumericalXYChartDataSource#NumericalXYChartDataSource(DataSourceGenerator, int, List, List)}
     *             which this calls.
     */
    public DataSetXYChartDataSource(final DataSourceGenerator generator,
                                    final int dataSourceOrderIndex,
                                    final DataSet data,
                                    final int[] xVariables,
                                    final int[] yVariables) throws XYChartDataSourceException
    {
        super(generator,
              dataSourceOrderIndex,
              getXAxisValues(data, xVariables, yVariables),
              getYAxisValues(data, xVariables, yVariables));
        _data = data;
    }

    public DataSet getData()
    {
        return _data;
    }

    private static List<double[]> getXAxisValues(final DataSet data, final int[] xVariables, final int[] yVariables)
    {
        if(xVariables.length != yVariables.length)
        {
            throw new IllegalArgumentException("The number of x variables must match the number of y variables.");
        }
        final List<double[]> xAxisValuesBySeries = Lists.newArrayList();
        for(int i = 0; i < xVariables.length; i++)
        {
            xAxisValuesBySeries.add(data.getSampleSizedVariable(xVariables[i]));
        }
        return xAxisValuesBySeries;
    }

    private static List<double[]> getYAxisValues(final DataSet data, final int[] xVariables, final int[] yVariables)
    {
        if(xVariables.length != yVariables.length)
        {
            throw new IllegalArgumentException("The number of x variables must match the number of y variables.");
        }
        final List<double[]> yAxisValuesBySeries = Lists.newArrayList();
        for(int i = 0; i < yVariables.length; i++)
        {
            yAxisValuesBySeries.add(data.getSampleSizedVariable(yVariables[i]));
        }
        return yAxisValuesBySeries;
    }
}
