package ohd.hseb.charter.datasource.instances;

import java.awt.Color;
import java.util.ArrayList;
import java.util.List;

import org.jfree.chart.plot.DefaultDrawingSupplier;
import org.jfree.data.xy.XYDataset;
import org.jfree.data.xy.XYSeries;
import org.jfree.data.xy.XYSeriesCollection;

import ohd.hseb.charter.ChartConstants;
import ohd.hseb.charter.datasource.DataSourceGenerator;
import ohd.hseb.charter.datasource.DefaultXYChartDataSource;
import ohd.hseb.charter.datasource.XYChartDataSource;
import ohd.hseb.charter.datasource.XYChartDataSourceException;
import ohd.hseb.charter.parameters.DataSourceDrawingParameters;
import ohd.hseb.charter.parameters.SeriesDrawingParameters;
import ohd.hseb.hefs.utils.gui.tools.ColorTools;

/**
 * Makes use of {@link XYSeriesCollection}, which requires duplicating the data provided to the constructor and recorded
 * in {@link #_xValues} and {@link #_yValues}. In general, this is not a good idea, particularly for larger plots. <br>
 * <br>
 * TODO -- Update so that it can make use of the attributes {@link #_xValues} and {@link #_yValues} directly without
 * duplicating. Then it is up to the caller to determine if creating a clone of the data is needed. This would require
 * creating my own {@link XYDataset} implementation.
 * 
 * @author Hank.Herr
 */
public class NumericalXYChartDataSource extends DefaultXYChartDataSource
{
    private final List<double[]> _xValues;
    private final List<double[]> _yValues;

    /**
     * @param generator Null is allowed.
     * @param dataSourceOrderIndex the order index of this data source relative to other plotted data sources.
     * @param xAxisValuesBySeries Values to plot along domain axis.
     * @param yAxisValuesBySeries Values to plot along range axis.
     * @throws XYChartDataSourceException Not thrown, currently.
     */
    public NumericalXYChartDataSource(final DataSourceGenerator generator,
                                      final int dataSourceOrderIndex,
                                      final List<double[]> xAxisValuesBySeries,
                                      final List<double[]> yAxisValuesBySeries) throws XYChartDataSourceException
    {
        this.setGenerator(generator);
        this._xValues = xAxisValuesBySeries;
        this._yValues = yAxisValuesBySeries;
        buildInitialParameters(dataSourceOrderIndex);
        this.setXAxisType(ChartConstants.AXIS_IS_NUMERICAL);
        this.setComputedDataType(ChartConstants.AXIS_IS_NUMERICAL);
    }

    /**
     * @return The y-values so that subclasses can see them.
     */
    protected List<double[]> getYValues()
    {
        return _yValues;
    }
    
    /**
     * @return Search the source for the largest plotted value.
     */
    public double getLargestPlottedValueOverall()
    {
        double maxValue = Double.MIN_VALUE;
        for(final double[] values: _xValues)
        {
            for(int i = 0; i < values.length; i++)
            {

                if(!Double.isNaN(values[i]) && values[i] > maxValue)
                {
                    maxValue = values[i];
                }
            }
        }
        for(final double[] values: _yValues)
        {
            for(int i = 0; i < values.length; i++)
            {
                if(!Double.isNaN(values[i]) && values[i] > maxValue)
                {
                    maxValue = values[i];
                }
            }
        }
        return maxValue;
    }

    /**
     * @return Search the source for the smallest plotted value.
     */
    public double getSmallestPlottedValueOverall()
    {
        double minValue = Double.MAX_VALUE;
        for(final double[] values: _xValues)
        {
            for(int i = 0; i < values.length; i++)
            {
                if(!Double.isNaN(values[i]) && values[i] < minValue)
                {
                    minValue = values[i];
                }
            }
        }
        for(final double[] values: _yValues)
        {
            for(int i = 0; i < values.length; i++)
            {
                if(!Double.isNaN(values[i]) && values[i] < minValue)
                {
                    minValue = values[i];
                }
            }
        }
        return minValue;
    }

    /**
     * Default parameters will display all time series as red, with only the first one shown in the legend. This
     * populates the return of {@link #getDefaultFullySpecifiedDataSourceDrawingParameters()} with default values.
     */
    protected void buildInitialParameters(final int dataSourceOrderIndex)
    {
        getDefaultFullySpecifiedDataSourceDrawingParameters().setDataSourceOrderIndex(dataSourceOrderIndex);
        getDefaultFullySpecifiedDataSourceDrawingParameters().setPlotterName("LineAndScatter");
        getDefaultFullySpecifiedDataSourceDrawingParameters().setSubPlotIndex(0);
        getDefaultFullySpecifiedDataSourceDrawingParameters().setYAxisIndex(0);

        constructAllSeriesDrawingParameters(_xValues.size());

        final DefaultDrawingSupplier supplier = new DefaultDrawingSupplier();
        for(int i = 0; i < _xValues.size(); i++)
        {
            final SeriesDrawingParameters seriesParms = this.getDefaultFullySpecifiedDataSourceDrawingParameters()
                                                            .getSeriesDrawingParametersForSeriesIndex(i);
            seriesParms.setNameInLegend("Series " + i);
            seriesParms.setLineColor((Color)supplier.getNextPaint());

            seriesParms.setShowInLegend(false);
            seriesParms.setShapeName(ChartConstants.NO_SHAPE);
            seriesParms.setShapeSize(ChartConstants.DEFAULT_SHAPE_SIZE);
            seriesParms.setBoxWidth(-1.0d);
            seriesParms.setBarWidth(0.9f);
            seriesParms.setFillColor(ColorTools.TRANSPARENT_WHITE);
            seriesParms.setIncludeInPlot(true);
            seriesParms.setLineWidth(0.5f);
            seriesParms.setShapeFilled(true);
        }
    }

    @Override
    protected XYDataset buildXYDataset(final DataSourceDrawingParameters parameters) throws XYChartDataSourceException
    {
        final XYSeriesCollection dataset = new XYSeriesCollection();
        for(int i = 0; i < this._xValues.size(); i++)
        {
            final String nameInLegend =
                                      parameters.getArguments()
                                                .replaceArgumentsInString(parameters.getSeriesDrawingParametersForSeriesIndex(i)
                                                                                    .getArgumentReplacedNameInLegend());

            //It is critical that auto sorting be turned off (false flag) in order for the constructed JFreeChart
            //data set to correspond properly to the x and y values contained herein.
            final XYSeries series = new XYSeries(nameInLegend, false);

            if(_xValues.get(i).length != _yValues.get(i).length)
            {
                throw new XYChartDataSourceException("Series " + i + " (name " + nameInLegend + ") "
                    + " does not have the same number of x-values as y-values.");
            }

            for(int j = 0; j < this._xValues.get(i).length; j++)
            {
                series.add(_xValues.get(i)[j], _yValues.get(i)[j]);
            }

            dataset.addSeries(series);
        }
        return dataset;
    }

    @Override
    public XYChartDataSource returnNewInstanceWithCopyOfInitialParameters() throws XYChartDataSourceException
    {
        final NumericalXYChartDataSource newInstance = new NumericalXYChartDataSource(this.getGenerator(),
                                                                                      getDataSourceOrderIndex(),
                                                                                      _xValues,
                                                                                      _yValues);
        copyTheseParametersIntoDataSource(newInstance);
        return newInstance;
    }

    @Override
    public int getNumberOfSeries()
    {
        return this._xValues.size();
    }

    @Override
    public Object getSeriesValue(final int seriesIndex, final int valueCount, final boolean xValue)
    {
        if(seriesIndex >= this._xValues.size())
        {
            return null;
        }
        double[] values = _xValues.get(seriesIndex);
        if(!xValue)
        {
            values = _yValues.get(seriesIndex);
        }

        if(valueCount >= values.length)
        {
            return null;
        }
        return values[valueCount];
    }

    @Override
    public int getSeriesValueCount(final int seriesIndex)
    {
        if(seriesIndex >= this._xValues.size())
        {
            return -1;
        }
        return _xValues.get(seriesIndex).length;
    }

    /**
     * @return An empty instance of {@link NumericalXYChartDataSource}; i.e, one with empty arrays.
     */
    public static NumericalXYChartDataSource createEmptySource()
    {
        try
        {
            return new NumericalXYChartDataSource(null, -1, new ArrayList<double[]>(), new ArrayList<double[]>());
        }
        catch(final XYChartDataSourceException e)
        {
            e.printStackTrace();
        }
        return null;
    }
}
