package ohd.hseb.charter.datasource.instances;

import java.awt.Color;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.TimeZone;

import org.jfree.data.xy.XYDataset;

import com.google.common.collect.Lists;

import nl.wldelft.util.timeseries.DefaultTimeSeriesHeader;
import nl.wldelft.util.timeseries.TimeSeriesArray;
import nl.wldelft.util.timeseries.TimeSeriesArrays;
import nl.wldelft.util.timeseries.TimeSeriesHeader;
import ohd.hseb.charter.ChartConstants;
import ohd.hseb.charter.datasource.DataSourceGenerator;
import ohd.hseb.charter.datasource.DefaultXYChartDataSource;
import ohd.hseb.charter.datasource.XYChartDataSource;
import ohd.hseb.charter.datasource.XYChartDataSourceException;
import ohd.hseb.charter.panel.AbstractChartEngineTableModel;
import ohd.hseb.charter.parameters.ChartDrawingParameters;
import ohd.hseb.charter.parameters.DataSourceDrawingParameters;
import ohd.hseb.charter.parameters.SeriesDrawingParameters;
import ohd.hseb.util.misc.HCalendar;

/**
 * {@link XYChartDataSource} to display time series contained in a {@link TimeSeriesArrays}. After constructed be sure
 * to call {@link #setReturnForecastTimeAsTableColumnHeader(boolean)} if necessary to turn off forecast time headers.
 * 
 * @author hankherr
 */
public class TimeSeriesArraysXYChartDataSource extends DefaultXYChartDataSource
{
    private TimeSeriesArrays _timeSeries = null;

    private boolean _returnForecastTimeAsTableColumnHeader = false;

    /**
     * @param generator The generator of this data source. Null is allowed.
     * @param dataSourceOrderIndex The index of this data source relative to other plotted sources.
     * @param timeSeries The time series to plot.
     * @throws XYChartDataSourceException Currently this is not thrown.
     */
    public TimeSeriesArraysXYChartDataSource(final DataSourceGenerator generator,
                                             final int dataSourceOrderIndex,
                                             final TimeSeriesArrays timeSeries) throws XYChartDataSourceException
    {
        setGenerator(generator);
        _timeSeries = timeSeries;
        buildInitialParameters(dataSourceOrderIndex);
        setXAxisType(ChartConstants.AXIS_IS_TIME);
        setComputedDataType(ChartConstants.AXIS_IS_NUMERICAL);
    }

    /**
     * @param generator The generator of this data source. Null is allowed.
     * @param dataSourceOrderIndex The index of this data source relative to other plotted sources.
     * @param timeSeries The time series to plot.
     * @param defaultDrawingParameters Initialized default drawing parameter to use.
     * @throws XYChartDataSourceException Thrown if the number of time series to plot is not identical to the number of
     *             series parameters in the defaultDrawingParameters.
     */
    public TimeSeriesArraysXYChartDataSource(final DataSourceGenerator generator,
                                             final int dataSourceOrderIndex,
                                             final TimeSeriesArrays timeSeries,
                                             final DataSourceDrawingParameters defaultDrawingParameters) throws XYChartDataSourceException
    {
        setGenerator(generator);
        if(timeSeries.size() != defaultDrawingParameters.getSeriesParametersCount())
        {
            throw new XYChartDataSourceException("Source being initialized with " + timeSeries.size()
                + " time series, but " + defaultDrawingParameters.getSeriesParametersCount()
                + " series drawing parameters.");
        }
        _timeSeries = timeSeries;
        setInitialDatasetDrawingParameters(defaultDrawingParameters);
        setXAxisType(ChartConstants.AXIS_IS_TIME);
        setComputedDataType(ChartConstants.AXIS_IS_NUMERICAL);
    }

    public TimeSeriesArrays getTimeSeries()
    {
        return _timeSeries;
    }
    
    /**
     * Converts this data source to one intended to display lead times (or lead durations).  That turns this into a numerical
     * data source instead of a time series data source.
     * @return The numerical data source.
     * @throws XYChartDataSourceException If a problem occurs in translation.
     */
    public NumericalXYChartDataSource convertToLeadDurationDataSource() throws XYChartDataSourceException
    {
        final List<double[]> xAxisValuesBySeries = Lists.newArrayList();
        final List<double[]> yAxisValuesBySeries = Lists.newArrayList();
        
        for (int i = 0; i < _timeSeries.size(); i ++)
        {
            final TimeSeriesArray<TimeSeriesHeader> ts = _timeSeries.get(i);
            final double[] leadDurationValues = new double[ts.size()];
            final double[] measurementValues = new double[ts.size()];
            for (int index = 0; index < ts.size(); index ++)
            {
                measurementValues[index] = ts.getValue(index);
                leadDurationValues[index] = (ts.getTime(index) - ts.getHeader().getForecastTime())/HCalendar.MILLIS_IN_HR;
            }
            xAxisValuesBySeries.add(leadDurationValues);
            yAxisValuesBySeries.add(measurementValues);
        }
        
        final NumericalXYChartDataSource source = new NumericalXYChartDataSource(getGenerator(), getDataSourceOrderIndex(), xAxisValuesBySeries, yAxisValuesBySeries);
        source.getDefaultFullySpecifiedDataSourceDrawingParameters().copyFrom(getDefaultFullySpecifiedDataSourceDrawingParameters());
        return source;
    }

    /**
     * @param returnForecastTimeAsTableColumnHeader If true, then the forecast time becomes a column header when the
     *            source is used in a {@link AbstractChartEngineTableModel}. If false, then the return of
     *            {@link #getChartTableColumnHeader(int)} will be null so that legend entries are used.
     */
    public void setReturnForecastTimeAsTableColumnHeader(final boolean returnForecastTimeAsTableColumnHeader)
    {
        _returnForecastTimeAsTableColumnHeader = returnForecastTimeAsTableColumnHeader;
    }

    /**
     * Default parameters will display all time series as red, with only the first one shown in the legend.
     * 
     * @return
     */
    protected void buildInitialParameters(final int dataSourceOrderIndex)
    {
        getDefaultFullySpecifiedDataSourceDrawingParameters().setDataSourceOrderIndex(dataSourceOrderIndex);
        getDefaultFullySpecifiedDataSourceDrawingParameters().setPlotterName("LineAndScatter");
        getDefaultFullySpecifiedDataSourceDrawingParameters().setSubPlotIndex(0);
        getDefaultFullySpecifiedDataSourceDrawingParameters().setYAxisIndex(0);

        constructAllSeriesDrawingParameters(_timeSeries.size());

        for(int i = 0; i < _timeSeries.size(); i++)
        {
            final SeriesDrawingParameters seriesParms = this.getDefaultFullySpecifiedDataSourceDrawingParameters()
                                                            .getSeriesDrawingParametersForSeriesIndex(i);

            seriesParms.setNameInLegend("Series " + i);
            if(i == 0)
            {
                seriesParms.setNameInLegend("Series");
            }
            seriesParms.setShowInLegend(false);
            if(i == 0)
            {
                seriesParms.setShowInLegend(true);
            }
            seriesParms.setLineColor(Color.RED);

            //The default shape is NO_SHAPE unless there is only one value in the time series, in which case
            //it becomes 0.
            if(_timeSeries.get(i).size() > 2)
            {
                seriesParms.setShapeName(ChartConstants.NO_SHAPE);
            }
            else
            {
                seriesParms.setShapeName(ChartConstants.SHAPE_NAMES[1]);
            }
            seriesParms.setShapeSize(ChartConstants.DEFAULT_SHAPE_SIZE);
            seriesParms.setBoxWidth(-1.0d);
            seriesParms.setBarWidth(0.9f);
            seriesParms.setFillColor(Color.red);
            seriesParms.setIncludeInPlot(true);
            seriesParms.setLineWidth(0.5f);
            seriesParms.setShapeFilled(true);
        }
    }

    private XYDataset buildXYDataSet(final DataSourceDrawingParameters parameters, final TimeZone timeZone)
    {
        final List<String> legendNames = new ArrayList<String>();
        for(int i = 0; i < _timeSeries.size(); i++)
        {
            legendNames.add(parameters.getArguments()
                                      .replaceArgumentsInString(parameters.getSeriesDrawingParametersForSeriesIndex(i)
                                                                          .getArgumentReplacedNameInLegend()));
        }
        return new TimeSeriesArraysXYDataSet(_timeSeries, timeZone, legendNames);
    }

    @Override
    protected XYDataset buildXYDataset(final DataSourceDrawingParameters parameters) throws XYChartDataSourceException
    {
        return buildXYDataSet(parameters, TimeZone.getTimeZone("GMT"));
    }

    @Override
    protected XYDataset buildXYDataset(final ChartDrawingParameters chartParameters, final int dataSourceIndex) throws XYChartDataSourceException
    {
        return buildXYDataSet(chartParameters.getDataSourceParameters(dataSourceIndex),
                              chartParameters.getGeneralParameters().getTimeZone());
    }

    @Override
    public XYChartDataSource returnNewInstanceWithCopyOfInitialParameters() throws XYChartDataSourceException
    {
        final TimeSeriesArraysXYChartDataSource newInstance = new TimeSeriesArraysXYChartDataSource(this.getGenerator(),
                                                                                                    getDataSourceOrderIndex(),
                                                                                                    getTimeSeries());
        copyTheseParametersIntoDataSource(newInstance);
        return newInstance;
    }

    @Override
    public int getNumberOfSeries()
    {
        return this._timeSeries.size();
    }

    @Override
    public Object getSeriesValue(final int seriesIndex, final int valueCount, final boolean xValue)
    {
        if(seriesIndex >= _timeSeries.size())
        {
            return null;
        }
        if(valueCount >= _timeSeries.get(seriesIndex).size())
        {
            return null;
        }
        if(xValue)
        {
            return new Date(_timeSeries.get(seriesIndex).getTime(valueCount));
        }
        else
        {
            return _timeSeries.get(seriesIndex).getValue(valueCount);
        }
    }

    @Override
    public int getSeriesValueCount(final int seriesIndex)
    {
        if(seriesIndex >= this._timeSeries.size())
        {
            return -1;
        }
        return _timeSeries.get(seriesIndex).size();
    }

    @Override
    public String getChartTableColumnHeader(final int seriesIndex)
    {
        if((!_returnForecastTimeAsTableColumnHeader) || (seriesIndex >= this._timeSeries.size()))
        {
            return null;
        }

        //If the forecast time is min or max value, use the start time instead.
        String timeSrc = "T0";
        long displayedTime = _timeSeries.get(seriesIndex).getHeader().getForecastTime();
        if((displayedTime == Long.MIN_VALUE) || (displayedTime == Long.MAX_VALUE))
        {
            timeSrc = "start";
            displayedTime = _timeSeries.get(seriesIndex).getStartTime();
        }

        //May need to return the start time if the forecast time is invalid, but how do I know if the forecast time is invalid?
        return HCalendar.buildDateStr(displayedTime, "CCYY-MM-DD hh'h'") + " (" + timeSrc + ")";
    }

    public static TimeSeriesArraysXYChartDataSource createEmptySource()
    {
        try
        {
            return new TimeSeriesArraysXYChartDataSource(null, -1, new TimeSeriesArrays(DefaultTimeSeriesHeader.class,
                                                                                        1));
        }
        catch(final XYChartDataSourceException e)
        {
            e.printStackTrace();
        }
        return null;
    }
}
