package ohd.hseb.charter.datasource.instances;

import java.util.List;
import java.util.TimeZone;

import nl.wldelft.util.timeseries.ParameterType;
import nl.wldelft.util.timeseries.TimeSeriesArray;
import nl.wldelft.util.timeseries.TimeSeriesArrays;
import ohd.hseb.util.misc.HCalendar;

import org.jfree.data.DomainInfo;
import org.jfree.data.DomainOrder;
import org.jfree.data.Range;
import org.jfree.data.time.TimeSeriesCollection;
import org.jfree.data.xy.AbstractIntervalXYDataset;
import org.jfree.data.xy.XisSymbolic;

/**
 * Class wraps a {@link TimeSeriesArrays} instance inside of classes that allow JFreeChart to display it without
 * duplicating data. We used to use {@link TimeSeriesCollection}, which required data duplication.
 * 
 * @author hank.herr
 */
@SuppressWarnings("serial")
public class TimeSeriesArraysXYDataSet extends AbstractIntervalXYDataset implements DomainInfo, XisSymbolic
{
    private final List<String> _seriesKeys;
    private final TimeZone _domainTimeZone;
    private final TimeSeriesArrays _timeSeries;
    private double _smallestRangeValue;
    private double _largestRangeValue;

    /**
     * Used to identify the lower bound on the period of the smallest item in the entire set of time series. Identifies
     * the lower bound of the domain axis.
     */
    private final int[] _smallestTimeSeriesItemPair = new int[2];

    /**
     * Used to identify the upper bound on the period of the largest item in the entire set of time series. Identifies
     * the upper bound of the domain axis.
     */
    private final int[] _largestTimeSeriesItemPair = new int[2];

    /**
     * @param timeSeries The time series that will be plotted.
     * @param domainTimeZone The time zone of the domain axis, for the purposes of tool tips.
     * @param seriesKeys The series keys, which are typically the legend names.
     */
    public TimeSeriesArraysXYDataSet(final TimeSeriesArrays timeSeries,
                                     final TimeZone domainTimeZone,
                                     final List<String> seriesKeys)
    {
        _domainTimeZone = domainTimeZone;
        _timeSeries = timeSeries;
        _seriesKeys = seriesKeys;
        if(timeSeries.isEmpty())
        {
            _smallestRangeValue = Float.NaN;
            _largestRangeValue = Float.NaN;
            return;
        }

        _smallestRangeValue = Float.MAX_VALUE;
        _largestRangeValue = Float.MIN_VALUE;
        long smallestTime = Long.MAX_VALUE;
        long largestTime = Long.MIN_VALUE;
        for(int i = 0; i < timeSeries.size(); i++)
        {
            for(int j = 0; j < timeSeries.get(i).size(); j++)
            {
                if(timeSeries.get(i).getValue(j) < _smallestRangeValue)
                {
                    _smallestRangeValue = timeSeries.get(i).getValue(j);
                }
                if(timeSeries.get(i).getValue(j) > _largestRangeValue)
                {
                    _largestRangeValue = timeSeries.get(i).getValue(j);
                }
                if(timeSeries.get(i).getTime(j) < smallestTime)
                {
                    smallestTime = timeSeries.get(i).getTime(j);
                    _smallestTimeSeriesItemPair[0] = i;
                    _smallestTimeSeriesItemPair[1] = j;
                }
                if(timeSeries.get(i).getTime(j) > largestTime)
                {
                    largestTime = timeSeries.get(i).getTime(j);
                    _largestTimeSeriesItemPair[0] = i;
                    _largestTimeSeriesItemPair[1] = j;
                }
            }
        }
    }

    public boolean isInstantaneousData()
    {
        if(_timeSeries.isEmpty())
        {
            return false;
        }
        return _timeSeries.get(0).getHeader().getParameterType().equals(ParameterType.INSTANTANEOUS);
    }

    public TimeSeriesArray getSeries(final int series)
    {
        return _timeSeries.get(series);
    }

    public TimeZone getDomainTimeZone()
    {
        return _domainTimeZone;
    }

    @Override
    public DomainOrder getDomainOrder()
    {
        return DomainOrder.ASCENDING;
    }

    @Override
    public Range getDomainBounds(final boolean includeInterval)
    {
        final double start = getStartX(_smallestTimeSeriesItemPair[0], _smallestTimeSeriesItemPair[1]).doubleValue();
        final double end = getEndX(_largestTimeSeriesItemPair[0], _largestTimeSeriesItemPair[1]).doubleValue();
        if((Double.isNaN(start)) || (Double.isNaN(end)))
        {
            return null;
        }
        return new Range(start, end);
    }

    @Override
    public double getDomainLowerBound(final boolean includeInterval)
    {
        return getStartX(_smallestTimeSeriesItemPair[0], _smallestTimeSeriesItemPair[1]).doubleValue();
    }

    @Override
    public double getDomainUpperBound(final boolean includeInterval)
    {
        return getEndX(_largestTimeSeriesItemPair[0], _largestTimeSeriesItemPair[1]).doubleValue();
    }

    @Override
    public int getItemCount(final int series)
    {
        if(_timeSeries.isEmpty())
        {
            return -1;
        }
        return _timeSeries.get(series).size();
    }

    @Override
    public Number getX(final int series, final int item)
    {
        if(_timeSeries.isEmpty())
        {
            return Double.NaN;
        }
        return _timeSeries.get(series).getTime(item);
    }

    @Override
    public Number getY(final int series, final int item)
    {
        if(_timeSeries.isEmpty())
        {
            return Double.NaN;
        }
        return _timeSeries.get(series).getValue(item);
    }

    @Override
    public Number getEndX(final int series, final int item)
    {
        if(_timeSeries.isEmpty())
        {
            return Double.NaN;
        }
        if(isInstantaneousData())
        {
            //Equidistant time series, use the time step.
            if(_timeSeries.get(series).getHeader().getTimeStep().isEquidistantMillis())
            {
                return _timeSeries.get(series).getTime(item)
                    + (long)(0.5 * _timeSeries.get(series).getHeader().getTimeStep().getStepMillis());
            }

            //If the series has only one point...
            if(_timeSeries.get(series).size() == 1)
            {
                //If there are no surrounding points, choose 1 day period arbitrarily.
                return _timeSeries.get(series).getTime(item) + 12 * HCalendar.MILLIS_IN_HR;
            }
            //For multiple points, use the surroundings for a non-equidistant time series.  Choose a mid point.
            if(item == _timeSeries.get(series).size() - 1)
            {
                //At the end of a series, assume that the end of this interval is the time + one half of the distance to the previous interval.
                return _timeSeries.get(series).getTime(item)
                    + (long)((_timeSeries.get(series).getTime(item) - _timeSeries.get(series).getTime(item - 1)) / 2.0);
            }
            return (long)((_timeSeries.get(series).getTime(item) + _timeSeries.get(series).getTime(item + 1)) / 2.0);
        }
        else
        {
            //Accumulative and mean time series always use the item time for the end.
            return _timeSeries.get(series).getTime(item);
        }
    }

    @Override
    public Number getEndY(final int series, final int item)
    {
        if(_timeSeries.isEmpty())
        {
            return Double.NaN;
        }
        return getY(series, item);
    }

    @Override
    public Number getStartX(final int series, final int item)
    {
        if(_timeSeries.isEmpty())
        {
            return Double.NaN;
        }
        if(isInstantaneousData())
        {
            //Equidistant time series, use the time step.
            if(_timeSeries.get(series).getHeader().getTimeStep().isEquidistantMillis())
            {
                return _timeSeries.get(series).getTime(item)
                    - (long)(0.5 * _timeSeries.get(series).getHeader().getTimeStep().getStepMillis());
            }

            //If the series has only one point...
            if(_timeSeries.get(series).size() == 1)
            {
                //If there are no surrounding points, choose 1 day period arbitrarily.
                return _timeSeries.get(series).getTime(item) - 12 * HCalendar.MILLIS_IN_HR;
            }
            //For multiple points, use the surroundings for a non-equidistant time series.  Choose a mid point.
            if(item == 0)
            {
                //At the end of a series, assume that the end of this interval is the time + one half of the distance to the previous interval.
                return _timeSeries.get(series).getTime(item)
                    - (long)((_timeSeries.get(series).getTime(item + 1) - _timeSeries.get(series).getTime(item)) / 2.0);
            }
            return (long)((_timeSeries.get(series).getTime(item) + _timeSeries.get(series).getTime(item - 1)) / 2.0);
        }
        else
        {
            //Equidistant time series, use the time step.
            if(_timeSeries.get(series).getHeader().getTimeStep().isEquidistantMillis())
            {
                return _timeSeries.get(series).getTime(item)
                    - _timeSeries.get(series).getHeader().getTimeStep().getStepMillis();
            }

            //If the series has only one point...
            if(_timeSeries.get(series).size() == 1)
            {
                //If there are no surrounding points, choose 1 day period arbitrarily.
                return _timeSeries.get(series).getTime(item) - 24 * HCalendar.MILLIS_IN_HR;
            }
            //For multiple points, use the surroundings for a non-equidistant time series.
            if(item == 0)
            {
                //At the beginning of a series, assume that the start of this interval is the time + the distance to the next interval.
                return _timeSeries.get(series).getTime(item)
                    - (_timeSeries.get(series).getTime(item + 1) - _timeSeries.get(series).getTime(item));
            }
            return _timeSeries.get(series).getTime(item - 1);
        }
    }

    @Override
    public Number getStartY(final int series, final int item)
    {
        if(_timeSeries.isEmpty())
        {
            return Double.NaN;
        }
        return getY(series, item);
    }

    @Override
    public int getSeriesCount()
    {
        return _timeSeries.size();
    }

    @Override
    public Comparable getSeriesKey(final int series)
    {
        if(_timeSeries.isEmpty())
        {
            return "";
        }
        return _seriesKeys.get(series);
    }

    @Override
    public String getXSymbolicValue(final Integer val)
    {
        return "DO NOT USE!!!";
    }

    /**
     * NOTE: The returned key would normally be used in a legend. However, I think that GraphGen does something
     * different that avoids using this method and, instead, does its own thing.
     */
    @Override
    public String getXSymbolicValue(final int series, final int item)
    {
        if(_timeSeries.isEmpty())
        {
            return "";
        }
        return HCalendar.buildDateStr(HCalendar.computeCalendarFromMilliseconds(_timeSeries.get(series).getTime(item),
                                                                                _domainTimeZone));
    }

    @Override
    public String[] getXSymbolicValues()
    {
        return new String[]{"DO NOT USE!!!"};
    }
}
