package ohd.hseb.charter.jfreechartoverride;

import java.awt.Graphics2D;
import java.awt.geom.Point2D;
import java.awt.geom.Rectangle2D;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import ohd.hseb.charter.tools.TranslatedAxis;

import org.jfree.chart.LegendItemCollection;
import org.jfree.chart.axis.ValueAxis;
import org.jfree.chart.plot.IntervalMarker;
import org.jfree.chart.plot.Marker;
import org.jfree.chart.plot.PlotRenderingInfo;
import org.jfree.chart.plot.PlotState;
import org.jfree.chart.plot.ValueMarker;
import org.jfree.chart.plot.XYPlot;
import org.jfree.chart.renderer.xy.XYItemRenderer;
import org.jfree.data.Range;
import org.jfree.ui.Layer;

public class ExtendedXYPlot extends XYPlot
{
    private static final long serialVersionUID = 1L;

    /**
     * Allows for reversing the order of the legend items for the items in this plot instance, which may be a subplot.
     */
    private boolean _reverseLegendItemOrder = false;

    public void setReverseLegendItemOrder(final boolean b)
    {
        _reverseLegendItemOrder = b;
    }

    @Override
    public LegendItemCollection getLegendItems()
    {
        final LegendItemCollection items = super.getLegendItems();
        if(!_reverseLegendItemOrder)
        {
            return items;
        }

        //Reverse the legend order here!
        final LegendItemCollection reversedItems = new LegendItemCollection();
        for(int i = items.getItemCount() - 1; i >= 0; i--)
        {
            reversedItems.add(items.get(i));
        }
        return reversedItems;
    }

    //XXX In the future this overridden method can be gotten rid of.  For a new version of JFreeChart, markers have their own entities.
    @Override
    public void draw(final Graphics2D arg0,
                     final Rectangle2D arg1,
                     final Point2D arg2,
                     final PlotState arg3,
                     final PlotRenderingInfo arg4)
    {
        super.draw(arg0, arg1, arg2, arg3, arg4);

        //Add entities for each drawn marker.  
        XYItemRenderer renderer;
        int rendererIndex = 0;
        while((renderer = getRenderer(rendererIndex)) != null)
        {
            if(renderer instanceof GraphGenXYLineAndShapeRenderer)
            {
                ((GraphGenXYLineAndShapeRenderer)renderer).addMarkerEntities(arg3, arg4);
            }

            rendererIndex++;
        }
    }

    /**
     * Overridden to return the first found range axis that is visible and not translated. If no such axis exists (i.e.,
     * user decided to create a translated axis and then make the baseline axis not visible), then the axis tick mark
     * grid lines will not show up.
     */
    @Override
    public ValueAxis getRangeAxis()
    {
        ValueAxis axis = null;

        //First choice is the first visible AND non-translated axis, which is returned within 
        //this for loop.
        //Second choice is first non-translated axis regardless of visibility, which is recorded
        //within this for loop and returned afterwards if no first choice was found.
        ValueAxis secondChoice = null;
        for(int i = 0; i < getRangeAxisCount(); i++)
        {
            axis = getRangeAxis(i);
            if(!(axis instanceof TranslatedAxis))
            {
                if(axis.isVisible())
                {
                    return axis;
                }
                secondChoice = axis;
            }
        }

        //Return the second choice, if found.
        if(secondChoice != null)
        {
            return secondChoice;
        }

        //If there is no axis that is NOT translated and is visible, then just return getRangeAxis()
        //If the only visible axis is a translated axis, chart grids will not draw properly.
        return super.getRangeAxis();
    }

    @Override
    protected void drawRangeGridlines(final Graphics2D g2,
                                      final Rectangle2D area,
                                      @SuppressWarnings("rawtypes") final List ticks)
    {
        super.drawRangeGridlines(g2, area, ticks);
    }

    @SuppressWarnings("unchecked")
    @Override
    public Range getDataRange(final ValueAxis axis)
    {
        //XXX This calls the super class version, which does not allow for ignoring data sets.  We may want to add that feature in the future:
        //Determine the range but ignoring certain data sets based on a flag.
        Range result = super.getDataRange(axis);
        double lo;
        double hi;

        //If desired, we can perform a check here based on a flag indicating if the limits should include
        //thresholds and that flag can be modified via parameters.  If false, return Range.  If true, recompute
        //Range to include markers.

        final List<Marker> markers = new ArrayList<Marker>();

        final int domainIndex = getDomainAxisIndex(axis);
        @SuppressWarnings("rawtypes")
        Collection definedMarkers = null;

        if(domainIndex >= 0)
        {
            definedMarkers = this.getDomainMarkers(Layer.FOREGROUND);
            if(definedMarkers != null)
            {
                markers.addAll(definedMarkers);
            }
        }
        else
        {
            for(int i = 0; i < this.getDatasetCount(); i++)
            {
                if(this.getRangeAxisForDataset(i) == axis)
                {
                    definedMarkers = getRangeMarkers(i, Layer.FOREGROUND);
                }
                if(definedMarkers != null)
                {
                    markers.addAll(definedMarkers);
                }
            }
        }

        for(int i = 0; i < markers.size(); i++)
        {
            //Get the lo and hi values.
            if(markers.get(i) instanceof ValueMarker)
            {
                lo = ((ValueMarker)markers.get(i)).getValue();
                hi = ((ValueMarker)markers.get(i)).getValue();
            }
            else if(markers.get(i) instanceof IntervalMarker)
            {
                lo = Math.min(((IntervalMarker)markers.get(i)).getStartValue(),
                              ((IntervalMarker)markers.get(i)).getEndValue());
                hi = Math.max(((IntervalMarker)markers.get(i)).getStartValue(),
                              ((IntervalMarker)markers.get(i)).getEndValue());
            }
            else
            {
                lo = result.getLowerBound();
                hi = result.getUpperBound();
            }

            //Account for MIN_VALUE, MAX_VALUE, and NaN, each of which specifies no lower/upper bound
            //if assigned to an interval end point.
            if((lo == Double.MIN_VALUE) || (lo == Double.MAX_VALUE) || (lo == Double.NaN))
            {
                lo = result.getLowerBound();
            }
            if((hi == Double.MIN_VALUE) || (hi == Double.MAX_VALUE) || (hi == Double.NaN))
            {
                hi = result.getUpperBound();
            }

            //Define the new range based on the order of lo and hi.
            Range newRange = null;
            if(lo > hi)
            {
                newRange = new Range(hi, lo);
            }
            else
            {
                newRange = new Range(lo, hi);
            }

            //Update the result.
            result = Range.combine(result, newRange);
        }

        return result;
    }
}
