package ohd.hseb.charter.jfreechartoverride;

import java.awt.Graphics2D;
import java.awt.geom.GeneralPath;
import java.awt.geom.Rectangle2D;
import java.util.Collections;
import java.util.LinkedList;

import org.jfree.chart.axis.ValueAxis;
import org.jfree.chart.plot.CrosshairState;
import org.jfree.chart.plot.PlotOrientation;
import org.jfree.chart.plot.PlotRenderingInfo;
import org.jfree.chart.plot.XYPlot;
import org.jfree.chart.renderer.xy.XYDifferenceRenderer;
import org.jfree.chart.renderer.xy.XYItemRendererState;
import org.jfree.data.xy.XYDataset;
import org.jfree.ui.RectangleEdge;

/**
 * Fixes an issue with the {@link XYDifferenceRenderer}. If a series visible is set to false, then I do not want the
 * line to be drawn. The difference can still be painted (it can be made invisible by setting the fill color to
 * transparent), but the line outlining the difference region should not be painted. I've added an appropriate if check
 * to the drawItem method of {@link XYDifferenceRenderer}.
 * 
 * @author hank.herr
 */
public class GraphGenXYDifferenceRenderer extends XYDifferenceRenderer
{
    private static final long serialVersionUID = 1L;

    @Override
    public void drawItem(final Graphics2D g2,
                         final XYItemRendererState state,
                         final Rectangle2D dataArea,
                         final PlotRenderingInfo info,
                         final XYPlot plot,
                         final ValueAxis domainAxis,
                         final ValueAxis rangeAxis,
                         final XYDataset dataset,
                         final int series,
                         final int item,
                         final CrosshairState crosshairState,
                         final int pass)
    {

        if(pass == 0)
        {
            drawItemPass0(g2, dataArea, info, plot, domainAxis, rangeAxis, dataset, series, item, crosshairState);
        }
        else if((pass == 1) && (this.getSeriesVisible(series))) //HANK: Second part of if was added.
        {
            drawItemPass1(g2, dataArea, info, plot, domainAxis, rangeAxis, dataset, series, item, crosshairState);
        }

    }

    @SuppressWarnings("unchecked")
    @Override
    /**
     * Draws the visual representation of a single data item, first pass.
     *
     * @param x_graphics  the graphics device.
     * @param x_dataArea  the area within which the data is being drawn.
     * @param x_info  collects information about the drawing.
     * @param x_plot  the plot (can be used to obtain standard color 
     *                information etc).
     * @param x_domainAxis  the domain (horizontal) axis.
     * @param x_rangeAxis  the range (vertical) axis.
     * @param x_dataset  the dataset.
     * @param x_series  the series index (zero-based).
     * @param x_item  the item index (zero-based).
     * @param x_crosshairState  crosshair information for the plot 
     *                          (<code>null</code> permitted).
     */
    protected void drawItemPass0(final Graphics2D x_graphics,
                                 final Rectangle2D x_dataArea,
                                 final PlotRenderingInfo x_info,
                                 final XYPlot x_plot,
                                 final ValueAxis x_domainAxis,
                                 final ValueAxis x_rangeAxis,
                                 final XYDataset x_dataset,
                                 final int x_series,
                                 final int x_item,
                                 final CrosshairState x_crosshairState)
    {

        if(!((0 == x_series) && (0 == x_item)))
        {
            return;
        }

        final boolean b_impliedZeroSubtrahend = (1 == x_dataset.getSeriesCount());

        // check if either series is a degenerate case (i.e. less than 2 points)
        if(isEitherSeriesDegenerate(x_dataset, b_impliedZeroSubtrahend))
        {
            return;
        }

        // check if series are disjoint (i.e. domain-spans do not overlap)
        if(!b_impliedZeroSubtrahend && areSeriesDisjoint(x_dataset))
        {
            return;
        }

        // polygon definitions
        @SuppressWarnings("rawtypes")
        final LinkedList l_minuendXs = new LinkedList();
        @SuppressWarnings("rawtypes")
        final LinkedList l_minuendYs = new LinkedList();
        @SuppressWarnings("rawtypes")
        final LinkedList l_subtrahendXs = new LinkedList();
        @SuppressWarnings("rawtypes")
        final LinkedList l_subtrahendYs = new LinkedList();
        @SuppressWarnings("rawtypes")
        final LinkedList l_polygonXs = new LinkedList();
        @SuppressWarnings("rawtypes")
        final LinkedList l_polygonYs = new LinkedList();

        // state
        int l_minuendItem = 0;
        final int l_minuendItemCount = x_dataset.getItemCount(0);
        Double l_minuendCurX = null;
        Double l_minuendNextX = null;
        Double l_minuendCurY = null;
        Double l_minuendNextY = null;
        double l_minuendMaxY = Double.NEGATIVE_INFINITY;
        double l_minuendMinY = Double.POSITIVE_INFINITY;

        int l_subtrahendItem = 0;
        int l_subtrahendItemCount = 0; // actual value set below
        Double l_subtrahendCurX = null;
        Double l_subtrahendNextX = null;
        Double l_subtrahendCurY = null;
        Double l_subtrahendNextY = null;
        double l_subtrahendMaxY = Double.NEGATIVE_INFINITY;
        double l_subtrahendMinY = Double.POSITIVE_INFINITY;

        // if a subtrahend is not specified, assume it is zero
        if(b_impliedZeroSubtrahend)
        {
            l_subtrahendItem = 0;
            l_subtrahendItemCount = 2;
            l_subtrahendCurX = Double.valueOf(x_dataset.getXValue(0, 0));
            l_subtrahendNextX = Double.valueOf(x_dataset.getXValue(0, (l_minuendItemCount - 1)));
            l_subtrahendCurY = Double.valueOf(0.0);
            l_subtrahendNextY = Double.valueOf(0.0);
            l_subtrahendMaxY = 0.0;
            l_subtrahendMinY = 0.0;

            l_subtrahendXs.add(l_subtrahendCurX);
            l_subtrahendYs.add(l_subtrahendCurY);
        }
        else
        {
            l_subtrahendItemCount = x_dataset.getItemCount(1);
        }

        boolean b_minuendDone = false;
        boolean b_minuendAdvanced = true;
        boolean b_minuendAtIntersect = false;
        boolean b_minuendFastForward = false;
        boolean b_minuendNaN = false;
        boolean b_subtrahendDone = false;
        boolean b_subtrahendAdvanced = true;
        boolean b_subtrahendAtIntersect = false;
        boolean b_subtrahendFastForward = false;
        boolean b_subtrahendNaN = false;
        boolean b_colinear = false;

        boolean b_positive;

        // coordinate pairs
        double l_x1 = 0.0, l_y1 = 0.0; // current minuend point
        double l_x2 = 0.0, l_y2 = 0.0; // next minuend point
        double l_x3 = 0.0, l_y3 = 0.0; // current subtrahend point
        double l_x4 = 0.0, l_y4 = 0.0; // next subtrahend point

        // fast-forward through leading tails
        boolean b_fastForwardDone = false;
        while(!b_fastForwardDone)
        {
            // get the x and y coordinates
            l_x1 = x_dataset.getXValue(0, l_minuendItem);
            l_y1 = x_dataset.getYValue(0, l_minuendItem);
            l_x2 = x_dataset.getXValue(0, l_minuendItem + 1);
            l_y2 = x_dataset.getYValue(0, l_minuendItem + 1);

            l_minuendCurX = Double.valueOf(l_x1);
            l_minuendCurY = Double.valueOf(l_y1);
            l_minuendNextX = Double.valueOf(l_x2);
            l_minuendNextY = Double.valueOf(l_y2);

            if(b_impliedZeroSubtrahend)
            {
                l_x3 = l_subtrahendCurX.doubleValue();
                l_y3 = l_subtrahendCurY.doubleValue();
                l_x4 = l_subtrahendNextX.doubleValue();
                l_y4 = l_subtrahendNextY.doubleValue();
            }
            else
            {
                l_x3 = x_dataset.getXValue(1, l_subtrahendItem);
                l_y3 = x_dataset.getYValue(1, l_subtrahendItem);
                l_x4 = x_dataset.getXValue(1, l_subtrahendItem + 1);
                l_y4 = x_dataset.getYValue(1, l_subtrahendItem + 1);

                l_subtrahendCurX = Double.valueOf(l_x3);
                l_subtrahendCurY = Double.valueOf(l_y3);
                l_subtrahendNextX = Double.valueOf(l_x4);
                l_subtrahendNextY = Double.valueOf(l_y4);
            }

            if(l_x2 <= l_x3)
            {
                // minuend needs to be fast forwarded
                l_minuendItem++;
                b_minuendFastForward = true;
                continue;
            }

            if(l_x4 <= l_x1)
            {
                // subtrahend needs to be fast forwarded
                l_subtrahendItem++;
                b_subtrahendFastForward = true;
                continue;
            }

            // check if initial polygon needs to be clipped
            if((l_x3 < l_x1) && (l_x1 < l_x4))
            {
                // project onto subtrahend
                final double l_slope = (l_y4 - l_y3) / (l_x4 - l_x3);
                l_subtrahendCurX = l_minuendCurX;
                l_subtrahendCurY = Double.valueOf((l_slope * l_x1) + (l_y3 - (l_slope * l_x3)));

                l_subtrahendXs.add(l_subtrahendCurX);
                l_subtrahendYs.add(l_subtrahendCurY);
            }

            if((l_x1 < l_x3) && (l_x3 < l_x2))
            {
                // project onto minuend
                final double l_slope = (l_y2 - l_y1) / (l_x2 - l_x1);
                l_minuendCurX = l_subtrahendCurX;
                l_minuendCurY = Double.valueOf((l_slope * l_x3) + (l_y1 - (l_slope * l_x1)));

                l_minuendXs.add(l_minuendCurX);
                l_minuendYs.add(l_minuendCurY);
            }

            l_minuendMaxY = l_minuendCurY.doubleValue();
            l_minuendMinY = l_minuendCurY.doubleValue();
            l_subtrahendMaxY = l_subtrahendCurY.doubleValue();
            l_subtrahendMinY = l_subtrahendCurY.doubleValue();

            b_fastForwardDone = true;
        }

        // start of algorithm
        while(!b_minuendDone && !b_subtrahendDone)
        {
            if(!b_minuendDone && !b_minuendFastForward && b_minuendAdvanced)
            {
                l_x1 = x_dataset.getXValue(0, l_minuendItem);
                l_y1 = x_dataset.getYValue(0, l_minuendItem);
                l_minuendCurX = Double.valueOf(l_x1);
                l_minuendCurY = Double.valueOf(l_y1);

                if(!b_minuendAtIntersect && !b_subtrahendNaN)
                {
                    l_minuendXs.add(l_minuendCurX);
                    l_minuendYs.add(l_minuendCurY);
                }

                l_minuendMaxY = Math.max(l_minuendMaxY, l_y1);
                l_minuendMinY = Math.min(l_minuendMinY, l_y1);

                l_x2 = x_dataset.getXValue(0, l_minuendItem + 1);
                l_y2 = x_dataset.getYValue(0, l_minuendItem + 1);
                l_minuendNextX = Double.valueOf(l_x2);
                l_minuendNextY = Double.valueOf(l_y2);
            }

            // never updated the subtrahend if it is implied to be zero
            if(!b_impliedZeroSubtrahend && !b_subtrahendDone && !b_subtrahendFastForward && b_subtrahendAdvanced)
            {
                l_x3 = x_dataset.getXValue(1, l_subtrahendItem);
                l_y3 = x_dataset.getYValue(1, l_subtrahendItem);
                l_subtrahendCurX = Double.valueOf(l_x3);
                l_subtrahendCurY = Double.valueOf(l_y3);

                if(!b_subtrahendAtIntersect && !b_minuendNaN)
                {
                    l_subtrahendXs.add(l_subtrahendCurX);
                    l_subtrahendYs.add(l_subtrahendCurY);
                }

                l_subtrahendMaxY = Math.max(l_subtrahendMaxY, l_y3);
                l_subtrahendMinY = Math.min(l_subtrahendMinY, l_y3);

                l_x4 = x_dataset.getXValue(1, l_subtrahendItem + 1);
                l_y4 = x_dataset.getYValue(1, l_subtrahendItem + 1);
                l_subtrahendNextX = Double.valueOf(l_x4);
                l_subtrahendNextY = Double.valueOf(l_y4);
            }

            // deassert b_*FastForward (only matters for 1st time through loop)
            b_minuendFastForward = false;
            b_subtrahendFastForward = false;

            b_minuendNaN = false;
            b_subtrahendNaN = false;

            Double l_intersectX = null;
            Double l_intersectY = null;
            boolean b_intersect = false;

            b_minuendAtIntersect = false;
            b_subtrahendAtIntersect = false;

            if(Double.isNaN(l_y2))
            {
                b_minuendNaN = true;
            }
            else if(Double.isNaN(l_y4))
            {
                b_subtrahendNaN = true;
            }
            else if((l_x2 == l_x4) && (l_y2 == l_y4))
            {
                // check if line segments are colinear
                if((l_x1 == l_x3) && (l_y1 == l_y3))
                {
                    b_colinear = true;
                }
                else
                {
                    // the intersect is at the next point for both the minuend 
                    // and subtrahend
                    l_intersectX = Double.valueOf(l_x2);
                    l_intersectY = Double.valueOf(l_y2);

                    b_intersect = true;
                    b_minuendAtIntersect = true;
                    b_subtrahendAtIntersect = true;
                }
            }
            else
            {
                // compute common denominator
                final double l_denominator = ((l_y4 - l_y3) * (l_x2 - l_x1)) - ((l_x4 - l_x3) * (l_y2 - l_y1));

                // compute common deltas
                final double l_deltaY = l_y1 - l_y3;
                final double l_deltaX = l_x1 - l_x3;

                // compute numerators
                final double l_numeratorA = ((l_x4 - l_x3) * l_deltaY) - ((l_y4 - l_y3) * l_deltaX);
                final double l_numeratorB = ((l_x2 - l_x1) * l_deltaY) - ((l_y2 - l_y1) * l_deltaX);

                // check if line segments are colinear
                if((0 == l_numeratorA) && (0 == l_numeratorB) && (0 == l_denominator))
                {
                    b_colinear = true;
                }
                else
                {
                    // check if previously colinear
                    if(b_colinear)
                    {
                        // clear colinear points and flag
                        l_minuendXs.clear();
                        l_minuendYs.clear();
                        l_subtrahendXs.clear();
                        l_subtrahendYs.clear();
                        l_polygonXs.clear();
                        l_polygonYs.clear();

                        b_colinear = false;

                        // set new starting point for the polygon
                        final boolean b_useMinuend = ((l_x3 <= l_x1) && (l_x1 <= l_x4));
                        l_polygonXs.add(b_useMinuend ? l_minuendCurX : l_subtrahendCurX);
                        l_polygonYs.add(b_useMinuend ? l_minuendCurY : l_subtrahendCurY);
                    }

                    // compute slope components
                    final double l_slopeA = l_numeratorA / l_denominator;
                    final double l_slopeB = l_numeratorB / l_denominator;

                    // check if the line segments intersect
                    if((0 < l_slopeA) && (l_slopeA <= 1) && (0 < l_slopeB) && (l_slopeB <= 1))
                    {
                        // compute the point of intersection
                        final double l_xi = l_x1 + (l_slopeA * (l_x2 - l_x1));
                        final double l_yi = l_y1 + (l_slopeA * (l_y2 - l_y1));

                        l_intersectX = Double.valueOf(l_xi);
                        l_intersectY = Double.valueOf(l_yi);
                        b_intersect = true;
                        b_minuendAtIntersect = ((l_xi == l_x2) && (l_yi == l_y2));
                        b_subtrahendAtIntersect = ((l_xi == l_x4) && (l_yi == l_y4));

                        // advance minuend and subtrahend to intesect
                        l_minuendCurX = l_intersectX;
                        l_minuendCurY = l_intersectY;
                        l_subtrahendCurX = l_intersectX;
                        l_subtrahendCurY = l_intersectY;
                    }
                }
            }

            if(b_intersect || b_minuendNaN || b_subtrahendNaN)
            {
                // create the polygon
                // add the minuend's points to polygon
                l_polygonXs.addAll(l_minuendXs);
                l_polygonYs.addAll(l_minuendYs);

                // add intersection point to the polygon
                if(!(b_minuendNaN || b_subtrahendNaN))
                {
                    l_polygonXs.add(l_intersectX);
                    l_polygonYs.add(l_intersectY);
                }

                // add the subtrahend's points to the polygon in reverse
                Collections.reverse(l_subtrahendXs);
                Collections.reverse(l_subtrahendYs);
                l_polygonXs.addAll(l_subtrahendXs);
                l_polygonYs.addAll(l_subtrahendYs);

                // create an actual polygon
                b_positive = (l_subtrahendMaxY <= l_minuendMaxY) && (l_subtrahendMinY <= l_minuendMinY);
                createPolygon(x_graphics,
                              x_dataArea,
                              x_plot,
                              x_domainAxis,
                              x_rangeAxis,
                              b_positive,
                              l_polygonXs,
                              l_polygonYs);

                // clear the point vectors
                l_minuendXs.clear();
                l_minuendYs.clear();
                l_subtrahendXs.clear();
                l_subtrahendYs.clear();
                l_polygonXs.clear();
                l_polygonYs.clear();

                // add interection point to new polygon
                if(!(b_minuendNaN || b_subtrahendNaN))
                {
                    // set the maxY and minY values to intersect y-value
                    final double l_y = l_intersectY.doubleValue();
                    l_minuendMaxY = l_y;
                    l_subtrahendMaxY = l_y;
                    l_minuendMinY = l_y;
                    l_subtrahendMinY = l_y;

                    l_polygonXs.add(l_intersectX);
                    l_polygonYs.add(l_intersectY);
                }
            }

            // advance the minuend if needed
            if(l_x2 <= l_x4)
            {
                l_minuendItem++;
                b_minuendAdvanced = true;
            }
            else
            {
                b_minuendAdvanced = false;
            }

            // advance the subtrahend if needed
            if(l_x4 <= l_x2)
            {
                l_subtrahendItem++;
                b_subtrahendAdvanced = true;
            }
            else
            {
                b_subtrahendAdvanced = false;
            }

            b_minuendDone = (l_minuendItem == (l_minuendItemCount - 1));
            b_subtrahendDone = (l_subtrahendItem == (l_subtrahendItemCount - 1));
        }

        // check if the final polygon needs to be clipped
        if(b_minuendDone && (l_x3 < l_x2) && (l_x2 < l_x4))
        {
            // project onto subtrahend
            final double l_slope = (l_y4 - l_y3) / (l_x4 - l_x3);
            l_subtrahendNextX = l_minuendNextX;
            l_subtrahendNextY = Double.valueOf((l_slope * l_x2) + (l_y3 - (l_slope * l_x3)));
        }

        if(b_subtrahendDone && (l_x1 < l_x4) && (l_x4 < l_x2))
        {
            // project onto minuend
            final double l_slope = (l_y2 - l_y1) / (l_x2 - l_x1);
            l_minuendNextX = l_subtrahendNextX;
            l_minuendNextY = Double.valueOf((l_slope * l_x4) + (l_y1 - (l_slope * l_x1)));
        }

        // consider last point of minuend and subtrahend for determining 
        // positivity
        l_minuendMaxY = Math.max(l_minuendMaxY, l_minuendNextY.doubleValue());
        l_subtrahendMaxY = Math.max(l_subtrahendMaxY, l_subtrahendNextY.doubleValue());
        l_minuendMinY = Math.min(l_minuendMinY, l_minuendNextY.doubleValue());
        l_subtrahendMinY = Math.min(l_subtrahendMinY, l_subtrahendNextY.doubleValue());

        // add the last point of the minuned and subtrahend
        l_minuendXs.add(l_minuendNextX);
        l_minuendYs.add(l_minuendNextY);
        l_subtrahendXs.add(l_subtrahendNextX);
        l_subtrahendYs.add(l_subtrahendNextY);

        // create the polygon
        // add the minuend's points to polygon
        l_polygonXs.addAll(l_minuendXs);
        l_polygonYs.addAll(l_minuendYs);

        // add the subtrahend's points to the polygon in reverse
        Collections.reverse(l_subtrahendXs);
        Collections.reverse(l_subtrahendYs);
        l_polygonXs.addAll(l_subtrahendXs);
        l_polygonYs.addAll(l_subtrahendYs);

        // create an actual polygon
        b_positive = (l_subtrahendMaxY <= l_minuendMaxY) && (l_subtrahendMinY <= l_minuendMinY);
        createPolygon(x_graphics, x_dataArea, x_plot, x_domainAxis, x_rangeAxis, b_positive, l_polygonXs, l_polygonYs);
    }

    /**
     * Determines if a dataset is degenerate. A degenerate dataset is a dataset where either series has less than two
     * (2) points.
     * 
     * @param x_dataset the dataset.
     * @param x_impliedZeroSubtrahend if false, do not check the subtrahend
     * @return true if the dataset is degenerate.
     */
    private boolean isEitherSeriesDegenerate(final XYDataset x_dataset, final boolean x_impliedZeroSubtrahend)
    {

        if(x_impliedZeroSubtrahend)
        {
            return (x_dataset.getItemCount(0) < 2);
        }

        return ((x_dataset.getItemCount(0) < 2) || (x_dataset.getItemCount(1) < 2));
    }

    /**
     * Determines if the two (2) series are disjoint. Disjoint series do not overlap in the domain space.
     * 
     * @param x_dataset the dataset.
     * @return true if the dataset is degenerate.
     */
    private boolean areSeriesDisjoint(final XYDataset x_dataset)
    {

        final int l_minuendItemCount = x_dataset.getItemCount(0);
        final double l_minuendFirst = x_dataset.getXValue(0, 0);
        final double l_minuendLast = x_dataset.getXValue(0, l_minuendItemCount - 1);

        final int l_subtrahendItemCount = x_dataset.getItemCount(1);
        final double l_subtrahendFirst = x_dataset.getXValue(1, 0);
        final double l_subtrahendLast = x_dataset.getXValue(1, l_subtrahendItemCount - 1);

        return ((l_minuendLast < l_subtrahendFirst) || (l_subtrahendLast < l_minuendFirst));
    }

    /**
     * Draws the visual representation of a polygon
     * 
     * @param x_graphics the graphics device.
     * @param x_dataArea the area within which the data is being drawn.
     * @param x_plot the plot (can be used to obtain standard color information etc).
     * @param x_domainAxis the domain (horizontal) axis.
     * @param x_rangeAxis the range (vertical) axis.
     * @param x_positive indicates if the polygon is positive (true) or negative (false).
     * @param x_xValues a linked list of the x values (expects values to be of type Double).
     * @param x_yValues a linked list of the y values (expects values to be of type Double).
     */
    private void createPolygon(final Graphics2D x_graphics,
                               final Rectangle2D x_dataArea,
                               final XYPlot x_plot,
                               final ValueAxis x_domainAxis,
                               final ValueAxis x_rangeAxis,
                               final boolean x_positive,
                               @SuppressWarnings("rawtypes") final LinkedList x_xValues,
                               @SuppressWarnings("rawtypes") final LinkedList x_yValues)
    {

        final PlotOrientation l_orientation = x_plot.getOrientation();
        final RectangleEdge l_domainAxisLocation = x_plot.getDomainAxisEdge();
        final RectangleEdge l_rangeAxisLocation = x_plot.getRangeAxisEdge();

        final Object[] l_xValues = x_xValues.toArray();
        final Object[] l_yValues = x_yValues.toArray();

        final GeneralPath l_path = new GeneralPath();

        if(PlotOrientation.VERTICAL == l_orientation)
        {
            double l_x = x_domainAxis.valueToJava2D(((Double)l_xValues[0]).doubleValue(),
                                                    x_dataArea,
                                                    l_domainAxisLocation);
            if(this.getRoundXCoordinates())
            {
                l_x = Math.rint(l_x);
            }

            double l_y = x_rangeAxis.valueToJava2D(((Double)l_yValues[0]).doubleValue(),
                                                   x_dataArea,
                                                   l_rangeAxisLocation);

            l_path.moveTo((float)l_x, (float)l_y);
            for(int i = 1; i < l_xValues.length; i++)
            {
                l_x = x_domainAxis.valueToJava2D(((Double)l_xValues[i]).doubleValue(), x_dataArea, l_domainAxisLocation);
                if(this.getRoundXCoordinates())
                {
                    l_x = Math.rint(l_x);
                }

                l_y = x_rangeAxis.valueToJava2D(((Double)l_yValues[i]).doubleValue(), x_dataArea, l_rangeAxisLocation);
                l_path.lineTo((float)l_x, (float)l_y);
            }
            l_path.closePath();
        }
        else
        {
            double l_x = x_domainAxis.valueToJava2D(((Double)l_xValues[0]).doubleValue(),
                                                    x_dataArea,
                                                    l_domainAxisLocation);
            if(this.getRoundXCoordinates())
            {
                l_x = Math.rint(l_x);
            }

            double l_y = x_rangeAxis.valueToJava2D(((Double)l_yValues[0]).doubleValue(),
                                                   x_dataArea,
                                                   l_rangeAxisLocation);

            l_path.moveTo((float)l_y, (float)l_x);
            for(int i = 1; i < l_xValues.length; i++)
            {
                l_x = x_domainAxis.valueToJava2D(((Double)l_xValues[i]).doubleValue(), x_dataArea, l_domainAxisLocation);
                if(this.getRoundXCoordinates())
                {
                    l_x = Math.rint(l_x);
                }

                l_y = x_rangeAxis.valueToJava2D(((Double)l_yValues[i]).doubleValue(), x_dataArea, l_rangeAxisLocation);
                l_path.lineTo((float)l_y, (float)l_x);
            }
            l_path.closePath();
        }

        if(l_path.intersects(x_dataArea))
        {
            x_graphics.setPaint(x_positive ? getPositivePaint() : getNegativePaint());
            x_graphics.fill(l_path);
        }
    }

}
