package ohd.hseb.charter.jfreechartoverride;

import java.awt.AlphaComposite;
import java.awt.Composite;
import java.awt.Font;
import java.awt.GradientPaint;
import java.awt.Graphics2D;
import java.awt.Paint;
import java.awt.Shape;
import java.awt.geom.Line2D;
import java.awt.geom.Point2D;
import java.awt.geom.Rectangle2D;
import java.util.HashMap;
import java.util.List;

import ohd.hseb.charter.ChartConstants;
import ohd.hseb.charter.parameters.ThresholdParameters;
import ohd.hseb.hefs.utils.gui.tools.ColorTools;

import org.jfree.chart.JFreeChart;
import org.jfree.chart.LegendItem;
import org.jfree.chart.axis.ValueAxis;
import org.jfree.chart.entity.ChartEntity;
import org.jfree.chart.entity.EntityCollection;
import org.jfree.chart.plot.CrosshairState;
import org.jfree.chart.plot.IntervalMarker;
import org.jfree.chart.plot.Marker;
import org.jfree.chart.plot.PlotOrientation;
import org.jfree.chart.plot.PlotRenderingInfo;
import org.jfree.chart.plot.PlotState;
import org.jfree.chart.plot.ValueMarker;
import org.jfree.chart.plot.XYPlot;
import org.jfree.chart.renderer.xy.XYItemRendererState;
import org.jfree.chart.renderer.xy.XYLineAndShapeRenderer;
import org.jfree.data.Range;
import org.jfree.data.xy.XYDataset;
import org.jfree.text.TextUtilities;
import org.jfree.ui.GradientPaintTransformer;
import org.jfree.ui.LengthAdjustmentType;
import org.jfree.ui.RectangleAnchor;
import org.jfree.ui.RectangleInsets;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;

/**
 * Wrapper renderer. See javadoc for each method for an explanation of why each method is subclassed. Note that the
 * marker methods are temporarily overridden to allow for adding entities pending use of a newer version of {@link
 * JFreeChart#}.
 * 
 * @author hankherr
 */
public class GraphGenXYLineAndShapeRenderer extends XYLineAndShapeRenderer
{
    private static final long serialVersionUID = 1L;

    /**
     * Flag indicates if the legend item should include a square in cases where it has no defined shape, to represent
     * that it is filled. The shape will be drawn only if the "shape" (i.e., area) is filled.
     */
    private boolean _alwaysDrawLegendColorSquare = false;

    public void setAlwaysDrawLegendColorSquare(final boolean drawFillRectangleInLegendIfNoShapeOrLine)
    {
        _alwaysDrawLegendColorSquare = drawFillRectangleInLegendIfNoShapeOrLine;
    }

    /**
     * This attribute is populated when the overridden method
     * {@link #drawDomainMarker(Graphics2D, XYPlot, ValueAxis, Marker, Rectangle2D)} and
     * {@link #drawRangeMarker(Graphics2D, XYPlot, ValueAxis, Marker, Rectangle2D)} are called. It stores {@link Shape}s
     * to allow for clicking the thresholds.
     */
    private final HashMap<Marker, Shape> _markerToShapeUsed = Maps.newHashMap();

    @Override
    public LegendItem getLegendItem(final int arg0, final int arg1)
    {
        LegendItem item = super.getLegendItem(arg0, arg1);
        if(!item.isShapeVisible() && _alwaysDrawLegendColorSquare)
        {
            item = new LegendItem(item.getLabel(),
                                  item.getDescription(),
                                  item.getToolTipText(),
                                  item.getURLText(),
                                  item.isShapeFilled(), //shape visible
                                  ChartConstants.getShape("square"),
                                  item.isShapeFilled(), //shape filled
                                  item.getFillPaint(),
                                  item.isShapeFilled(), // outline visible
                                  item.getOutlinePaint(),
                                  item.getOutlineStroke(),
                                  item.isLineVisible(),
                                  item.getLine(),
                                  item.getLineStroke(),
                                  item.getLinePaint());
            item.setDataset(item.getDataset());
            item.setDatasetIndex(item.getDatasetIndex());
            item.setFillPaintTransformer(item.getFillPaintTransformer());
            item.setSeriesIndex(item.getSeriesIndex());
            item.setSeriesKey(item.getSeriesKey());
        }
        return item;
    }

    @Override
    public int getPassCount()
    {
        return 1;
    }

    @Override
    public void drawItem(final Graphics2D g2,
                         final XYItemRendererState state,
                         final Rectangle2D dataArea,
                         final PlotRenderingInfo info,
                         final XYPlot plot,
                         final ValueAxis domainAxis,
                         final ValueAxis rangeAxis,
                         final XYDataset dataset,
                         final int series,
                         final int item,
                         final CrosshairState crosshairState,
                         final int pass)
    {
        // do nothing if item is not visible
        if(!getItemVisible(series, item))
        {
            return;
        }

        // first pass draws the background (lines, for instance)
        if(getItemLineVisible(series, item))
        {
            if(this.getDrawSeriesLineAsPath())
            {
                drawPrimaryLineAsPath(state, g2, plot, dataset, pass, series, item, domainAxis, rangeAxis, dataArea);
            }
            else
            {
                drawPrimaryLine(state, g2, plot, dataset, pass, series, item, domainAxis, rangeAxis, dataArea);
            }
        }

        // setup for collecting optional entity info...
        EntityCollection entities = null;
        if(info != null)
        {
            entities = info.getOwner().getEntityCollection();
        }

        drawSecondaryPass(g2,
                          plot,
                          dataset,
                          pass,
                          series,
                          item,
                          domainAxis,
                          dataArea,
                          rangeAxis,
                          crosshairState,
                          entities);
    }

    //XXX This method should go away if the two draw marker methods below are removed.
    /**
     * Adds entities for the {@link Marker} objects created by this renderer. The {@link ThresholdParameters} creates
     * thresholds using a renderer of this type.
     * 
     * @param state
     * @param info
     */
    public void addMarkerEntities(final PlotState state, final PlotRenderingInfo info)
    {
        EntityCollection entities = null;
        if(info != null)
        {
            entities = info.getOwner().getEntityCollection();
        }
        if(entities == null)
        {
            return;
        }

        final List<ChartEntity> addedEntities = Lists.newArrayList();
        for(final Marker marker: _markerToShapeUsed.keySet())
        {
            if(!marker.getPaint().equals(ColorTools.TRANSPARENT_WHITE))
            {
                //The shape used will be a rectangle of width 2 if the marker is a single line.
                Shape usedShape = _markerToShapeUsed.get(marker);
                if(usedShape instanceof Line2D)
                {
                    final Line2D line = (Line2D)usedShape;
                    if(line.getX1() == line.getX2())
                    {
                        usedShape = new Rectangle2D.Double(line.getX1() - 1, line.getY1(), 2, line.getY2()
                            - line.getY1());
                    }
                    else if(line.getY1() == line.getY2())
                    {
                        usedShape = new Rectangle2D.Double(line.getX1(),
                                                           line.getY1() - 1,
                                                           line.getX2() - line.getX1(),
                                                           2);
                    }
                }
                String toolTip = marker.getLabel();
                if((toolTip != null) && (toolTip.isEmpty()))
                {
                    toolTip = null;
                }
                final ChartEntity entity = new MarkerEntity(marker, usedShape, toolTip);
                addedEntities.add(entity);
            }
        }

        //I need to insert the entity at the beginning of the list, but JFreeChart is a pain the butt.
        //Hence I need to create a new list that is a copy of entities, add all new entities in slot 0, and
        //copy the new list to the old list.
        @SuppressWarnings("unchecked")
        final List<ChartEntity> newList = Lists.newArrayList(entities.getEntities());
        newList.addAll(0, addedEntities);
        entities.clear();
        for(int i = 0; i < newList.size(); i++)
        {
            entities.add(newList.get(i));
        }
    }

    /**
     * From a new version of {@link JFreeChart} used in
     * {@link #drawRangeMarker(Graphics2D, XYPlot, ValueAxis, Marker, Rectangle2D)} overridden method, below.
     */
    private Point2D calculateRangeMarkerTextAnchorPoint(final Graphics2D g2,
                                                        final PlotOrientation orientation,

                                                        final Rectangle2D dataArea,
                                                        final Rectangle2D markerArea,
                                                        final RectangleInsets markerOffset,
                                                        final LengthAdjustmentType labelOffsetForRange,
                                                        final RectangleAnchor anchor)
    {

        Rectangle2D anchorRect = null;
        if(orientation == PlotOrientation.HORIZONTAL)
        {
            anchorRect = markerOffset.createAdjustedRectangle(markerArea,
                                                              labelOffsetForRange,
                                                              LengthAdjustmentType.CONTRACT);
        }
        else if(orientation == PlotOrientation.VERTICAL)
        {
            anchorRect = markerOffset.createAdjustedRectangle(markerArea,
                                                              LengthAdjustmentType.CONTRACT,
                                                              labelOffsetForRange);
        }
        return RectangleAnchor.coordinates(anchorRect, anchor);

    }

    //XXX The draw marker methods should not need to be overridden if new JFreeChart is used.  The current version does not draw
    //entities for Markers.  Hence, I've added one line to each method below to record the Marker and its Shape so that it can 
    //be later added to the entities.  Just look for _markerToShapeUsed.  In a newer release, Markers will draw themselves and create
    //entities as needed.

    @Override
    public void drawDomainMarker(final Graphics2D g2,
                                 final XYPlot plot,
                                 final ValueAxis domainAxis,
                                 final Marker marker,
                                 final Rectangle2D dataArea)
    {

        if(marker instanceof ValueMarker)
        {
            final ValueMarker vm = (ValueMarker)marker;
            final double value = vm.getValue();
            final Range range = domainAxis.getRange();
            if(!range.contains(value))
            {
                return;
            }

            final double v = domainAxis.valueToJava2D(value, dataArea, plot.getDomainAxisEdge());

            final PlotOrientation orientation = plot.getOrientation();
            Line2D line = null;
            if(orientation == PlotOrientation.HORIZONTAL)
            {
                line = new Line2D.Double(dataArea.getMinX(), v, dataArea.getMaxX(), v);
            }
            else if(orientation == PlotOrientation.VERTICAL)
            {
                line = new Line2D.Double(v, dataArea.getMinY(), v, dataArea.getMaxY());
            }

            final Composite originalComposite = g2.getComposite();
            g2.setComposite(AlphaComposite.getInstance(AlphaComposite.SRC_OVER, marker.getAlpha()));
            g2.setPaint(marker.getPaint());
            g2.setStroke(marker.getStroke());
            g2.draw(line);

            _markerToShapeUsed.put(marker, line);

            final String label = marker.getLabel();
            final RectangleAnchor anchor = marker.getLabelAnchor();
            if((label != null) && (!marker.getLabelPaint().equals(ColorTools.TRANSPARENT_WHITE))) //XXX Added check... only draw label if color provided; see ThresholdParameters.
            {
                final Font labelFont = marker.getLabelFont();
                g2.setFont(labelFont);
                g2.setPaint(marker.getLabelPaint());
                final Point2D coordinates = calculateDomainMarkerTextAnchorPoint(g2,
                                                                                 orientation,
                                                                                 dataArea,
                                                                                 line.getBounds2D(),
                                                                                 marker.getLabelOffset(),
                                                                                 LengthAdjustmentType.EXPAND,
                                                                                 anchor);
                TextUtilities.drawAlignedString(label,
                                                g2,
                                                (float)coordinates.getX(),
                                                (float)coordinates.getY(),
                                                marker.getLabelTextAnchor());
            }
            g2.setComposite(originalComposite);
        }
        else if(marker instanceof IntervalMarker)
        {
            final IntervalMarker im = (IntervalMarker)marker;
            final double start = im.getStartValue();
            final double end = im.getEndValue();
            final Range range = domainAxis.getRange();
            if(!(range.intersects(start, end)))
            {
                return;
            }

            final double start2d = domainAxis.valueToJava2D(start, dataArea, plot.getDomainAxisEdge());
            final double end2d = domainAxis.valueToJava2D(end, dataArea, plot.getDomainAxisEdge());
            double low = Math.min(start2d, end2d);
            double high = Math.max(start2d, end2d);

            final PlotOrientation orientation = plot.getOrientation();
            Rectangle2D rect = null;
            if(orientation == PlotOrientation.HORIZONTAL)
            {
                // clip top and bottom bounds to data area
                low = Math.max(low, dataArea.getMinY());
                high = Math.min(high, dataArea.getMaxY());
                rect = new Rectangle2D.Double(dataArea.getMinX(), low, dataArea.getWidth(), high - low);
            }
            else if(orientation == PlotOrientation.VERTICAL)
            {
                // clip left and right bounds to data area
                low = Math.max(low, dataArea.getMinX());
                high = Math.min(high, dataArea.getMaxX());
                rect = new Rectangle2D.Double(low, dataArea.getMinY(), high - low, dataArea.getHeight());
            }

            final Composite originalComposite = g2.getComposite();
            g2.setComposite(AlphaComposite.getInstance(AlphaComposite.SRC_OVER, marker.getAlpha()));
            final Paint p = marker.getPaint();
            if(p instanceof GradientPaint)
            {
                GradientPaint gp = (GradientPaint)p;
                final GradientPaintTransformer t = im.getGradientPaintTransformer();
                if(t != null)
                {
                    gp = t.transform(gp, rect);
                }
                g2.setPaint(gp);
            }
            else
            {
                g2.setPaint(p);
            }
            g2.fill(rect);

            _markerToShapeUsed.put(marker, rect);

            // now draw the outlines, if visible...
            if(im.getOutlinePaint() != null && im.getOutlineStroke() != null)
            {
                if(orientation == PlotOrientation.VERTICAL)
                {
                    final Line2D line = new Line2D.Double();
                    final double y0 = dataArea.getMinY();
                    final double y1 = dataArea.getMaxY();
                    g2.setPaint(im.getOutlinePaint());
                    g2.setStroke(im.getOutlineStroke());
                    if(range.contains(start))
                    {
                        line.setLine(start2d, y0, start2d, y1);
                        g2.draw(line);
                    }
                    if(range.contains(end))
                    {
                        line.setLine(end2d, y0, end2d, y1);
                        g2.draw(line);
                    }
                }
                else
                { // PlotOrientation.HORIZONTAL
                    final Line2D line = new Line2D.Double();
                    final double x0 = dataArea.getMinX();
                    final double x1 = dataArea.getMaxX();
                    g2.setPaint(im.getOutlinePaint());
                    g2.setStroke(im.getOutlineStroke());
                    if(range.contains(start))
                    {
                        line.setLine(x0, start2d, x1, start2d);
                        g2.draw(line);
                    }
                    if(range.contains(end))
                    {
                        line.setLine(x0, end2d, x1, end2d);
                        g2.draw(line);
                    }
                }
            }

            final String label = marker.getLabel();
            final RectangleAnchor anchor = marker.getLabelAnchor();
            if((label != null) && (!marker.getLabelPaint().equals(ColorTools.TRANSPARENT_WHITE))) //XXX Added check... only draw label if color provided; see ThresholdParameters.
            {
                final Font labelFont = marker.getLabelFont();
                g2.setFont(labelFont);
                g2.setPaint(marker.getLabelPaint());
                final Point2D coordinates = calculateDomainMarkerTextAnchorPoint(g2,
                                                                                 orientation,
                                                                                 dataArea,
                                                                                 rect,
                                                                                 marker.getLabelOffset(),
                                                                                 marker.getLabelOffsetType(),
                                                                                 anchor);
                TextUtilities.drawAlignedString(label,
                                                g2,
                                                (float)coordinates.getX(),
                                                (float)coordinates.getY(),
                                                marker.getLabelTextAnchor());
            }
            g2.setComposite(originalComposite);

        }

    }

    @Override
    public void drawRangeMarker(final Graphics2D g2,
                                final XYPlot plot,
                                final ValueAxis rangeAxis,
                                final Marker marker,
                                final Rectangle2D dataArea)
    {

        if(marker instanceof ValueMarker)
        {
            final ValueMarker vm = (ValueMarker)marker;
            final double value = vm.getValue();
            final Range range = rangeAxis.getRange();
            if(!range.contains(value))
            {
                return;
            }

            final double v = rangeAxis.valueToJava2D(value, dataArea, plot.getRangeAxisEdge());
            final PlotOrientation orientation = plot.getOrientation();
            Line2D line = null;
            if(orientation == PlotOrientation.HORIZONTAL)
            {
                line = new Line2D.Double(v, dataArea.getMinY(), v, dataArea.getMaxY());
            }
            else if(orientation == PlotOrientation.VERTICAL)
            {
                line = new Line2D.Double(dataArea.getMinX(), v, dataArea.getMaxX(), v);
            }

            final Composite originalComposite = g2.getComposite();
            g2.setComposite(AlphaComposite.getInstance(AlphaComposite.SRC_OVER, marker.getAlpha()));
            g2.setPaint(marker.getPaint());
            g2.setStroke(marker.getStroke());
            g2.draw(line);

            _markerToShapeUsed.put(marker, line);

            final String label = marker.getLabel();
            final RectangleAnchor anchor = marker.getLabelAnchor();
            if((label != null) && (!marker.getLabelPaint().equals(ColorTools.TRANSPARENT_WHITE))) //XXX Added check... only draw label if color provided; see ThresholdParameters.
            {
                final Font labelFont = marker.getLabelFont();
                g2.setFont(labelFont);
                g2.setPaint(marker.getLabelPaint());
                final Point2D coordinates = calculateRangeMarkerTextAnchorPoint(g2,
                                                                                orientation,
                                                                                dataArea,
                                                                                line.getBounds2D(),
                                                                                marker.getLabelOffset(),
                                                                                LengthAdjustmentType.EXPAND,
                                                                                anchor);
                TextUtilities.drawAlignedString(label,
                                                g2,
                                                (float)coordinates.getX(),
                                                (float)coordinates.getY(),
                                                marker.getLabelTextAnchor());
            }
            g2.setComposite(originalComposite);
        }
        else if(marker instanceof IntervalMarker)
        {
            final IntervalMarker im = (IntervalMarker)marker;
            final double start = im.getStartValue();
            final double end = im.getEndValue();
            final Range range = rangeAxis.getRange();
            if(!(range.intersects(start, end)))
            {
                return;
            }

            final double start2d = rangeAxis.valueToJava2D(start, dataArea, plot.getRangeAxisEdge());
            final double end2d = rangeAxis.valueToJava2D(end, dataArea, plot.getRangeAxisEdge());
            double low = Math.min(start2d, end2d);
            double high = Math.max(start2d, end2d);

            final PlotOrientation orientation = plot.getOrientation();
            Rectangle2D rect = null;
            if(orientation == PlotOrientation.HORIZONTAL)
            {
                // clip left and right bounds to data area
                low = Math.max(low, dataArea.getMinX());
                high = Math.min(high, dataArea.getMaxX());
                rect = new Rectangle2D.Double(low, dataArea.getMinY(), high - low, dataArea.getHeight());
            }
            else if(orientation == PlotOrientation.VERTICAL)
            {
                // clip top and bottom bounds to data area
                low = Math.max(low, dataArea.getMinY());
                high = Math.min(high, dataArea.getMaxY());
                rect = new Rectangle2D.Double(dataArea.getMinX(), low, dataArea.getWidth(), high - low);
            }

            final Composite originalComposite = g2.getComposite();
            g2.setComposite(AlphaComposite.getInstance(AlphaComposite.SRC_OVER, marker.getAlpha()));
            final Paint p = marker.getPaint();
            if(p instanceof GradientPaint)
            {
                GradientPaint gp = (GradientPaint)p;
                final GradientPaintTransformer t = im.getGradientPaintTransformer();
                if(t != null)
                {
                    gp = t.transform(gp, rect);
                }
                g2.setPaint(gp);
            }
            else
            {
                g2.setPaint(p);
            }
            g2.fill(rect);

            _markerToShapeUsed.put(marker, rect);

            // now draw the outlines, if visible...
            if(im.getOutlinePaint() != null && im.getOutlineStroke() != null)
            {
                if(orientation == PlotOrientation.VERTICAL)
                {
                    final Line2D line = new Line2D.Double();
                    final double x0 = dataArea.getMinX();
                    final double x1 = dataArea.getMaxX();
                    g2.setPaint(im.getOutlinePaint());
                    g2.setStroke(im.getOutlineStroke());
                    if(range.contains(start))
                    {
                        line.setLine(x0, start2d, x1, start2d);
                        g2.draw(line);
                    }
                    if(range.contains(end))
                    {
                        line.setLine(x0, end2d, x1, end2d);
                        g2.draw(line);
                    }
                }
                else
                { // PlotOrientation.HORIZONTAL
                    final Line2D line = new Line2D.Double();
                    final double y0 = dataArea.getMinY();
                    final double y1 = dataArea.getMaxY();
                    g2.setPaint(im.getOutlinePaint());
                    g2.setStroke(im.getOutlineStroke());
                    if(range.contains(start))
                    {
                        line.setLine(start2d, y0, start2d, y1);
                        g2.draw(line);
                    }
                    if(range.contains(end))
                    {
                        line.setLine(end2d, y0, end2d, y1);
                        g2.draw(line);
                    }
                }
            }

            final String label = marker.getLabel();
            final RectangleAnchor anchor = marker.getLabelAnchor();
            if((label != null) && (!marker.getLabelPaint().equals(ColorTools.TRANSPARENT_WHITE))) //XXX Added check... only draw label if color provided; see ThresholdParameters.
            {
                final Font labelFont = marker.getLabelFont();
                g2.setFont(labelFont);
                g2.setPaint(marker.getLabelPaint());
                final Point2D coordinates = calculateRangeMarkerTextAnchorPoint(g2,
                                                                                orientation,
                                                                                dataArea,
                                                                                rect,
                                                                                marker.getLabelOffset(),
                                                                                marker.getLabelOffsetType(),
                                                                                anchor);
                TextUtilities.drawAlignedString(label,
                                                g2,
                                                (float)coordinates.getX(),
                                                (float)coordinates.getY(),
                                                marker.getLabelTextAnchor());
            }
            g2.setComposite(originalComposite);
        }
    }

}
