package ohd.hseb.charter.jfreechartoverride;

import java.awt.Graphics2D;
import java.awt.Paint;
import java.awt.Shape;
import java.awt.Stroke;
import java.awt.geom.Line2D;
import java.awt.geom.Rectangle2D;

import ohd.hseb.charter.ChartConstants;

import org.jfree.chart.LegendItem;
import org.jfree.chart.axis.ValueAxis;
import org.jfree.chart.entity.EntityCollection;
import org.jfree.chart.entity.XYItemEntity;
import org.jfree.chart.labels.XYToolTipGenerator;
import org.jfree.chart.plot.CrosshairState;
import org.jfree.chart.plot.PlotOrientation;
import org.jfree.chart.plot.PlotRenderingInfo;
import org.jfree.chart.plot.XYPlot;
import org.jfree.chart.renderer.xy.XYItemRendererState;
import org.jfree.chart.renderer.xy.XYLineAndShapeRenderer;
import org.jfree.chart.renderer.xy.XYStepRenderer;
import org.jfree.data.xy.XYDataset;
import org.jfree.ui.RectangleEdge;

/**
 * Allows for an {@link XYLineAndShapeRenderer} to be connected to the {@link XYStepRenderer}, drawing shapes on top of
 * the steps. The connected renderer is used to determine the legend entries if the series shape is visible in the
 * connected renderer. <br>
 * <br>
 * This class also provides a new attribute, {@link #_stepAtEnd}, that controls if the step occurs at the end of an
 * interval (i.e., it first draws a horizontal line before stepping down or up; the default behavior for
 * {@link XYStepRenderer}) or at the beginning (i.e., the step down or up is done first). The default is true, so that
 * this behaves as a standard {@link XYStepRenderer}.
 * 
 * @author herrhd
 */
public class GraphGenXYStepRenderer extends XYStepRenderer
{
    private static final long serialVersionUID = 1L;

    private XYLineAndShapeRenderer _connectedLineAndShapeRenderer = null;
    private int _datasetIndexOfConnectedLineAndShapeRenderer = -1;

    /**
     * Default is true. If true, the step is performed at the end. Otherwise, it is performed at the beginning.
     */
    private boolean _stepAtEnd = true;

    /**
     * Flag indicates if the legend item should include a square in cases where it has no defined shape, to represent
     * that it is filled. The shape will be drawn only if the "shape" (i.e., area) is filled.
     */
    private boolean _alwaysDrawLegendColorSquare = false;

    public void setConnectedLineAndShapeRenderer(final int datasetIndex, final XYLineAndShapeRenderer renderer)
    {
        _datasetIndexOfConnectedLineAndShapeRenderer = datasetIndex;
        this._connectedLineAndShapeRenderer = renderer;
    }

    public void setAlwaysDrawLegendColorSquare(final boolean drawFillRectangleInLegendIfNoShapeOrLine)
    {
        _alwaysDrawLegendColorSquare = drawFillRectangleInLegendIfNoShapeOrLine;
    }

    /**
     * If true, then the step portion of the rendering is done at the end of the interval; i.e., it goes horizontally
     * first and then steps. Otherwise, the step occurs immediately; i.e, it steps first and then goes horizontally.
     */
    public void setStepAtEnd(final boolean b)
    {
        _stepAtEnd = b;
    }

    @Override
    public LegendItem getLegendItem(final int datasetIndex, final int series)
    {
        LegendItem item = null;
        if(_connectedLineAndShapeRenderer != null)
        {
            if(_connectedLineAndShapeRenderer.getSeriesShapesVisible(series))
            {
                //For a step renderer, the lines are handled within this renderer, not the connected line and shape renderer.
                //Hence, I need the connected renderer to render legend items that include the line based on the flag within
                //this.  That is what the line visibility stuff is for.
                final boolean originalLineVis = _connectedLineAndShapeRenderer.getSeriesLinesVisible(series);
                final boolean originalPtVis = _connectedLineAndShapeRenderer.getSeriesShapesVisible(series);
                _connectedLineAndShapeRenderer.setSeriesLinesVisible(series, this.getSeriesLinesVisible(series));
                _connectedLineAndShapeRenderer.setSeriesShapesVisible(series, this.getSeriesShapesVisible(series));
                item = _connectedLineAndShapeRenderer.getLegendItem(_datasetIndexOfConnectedLineAndShapeRenderer,
                                                                    series);
                _connectedLineAndShapeRenderer.setSeriesLinesVisible(series, originalLineVis);
                _connectedLineAndShapeRenderer.setSeriesShapesVisible(series, originalPtVis);
            }
        }
        if(item == null)
        {
            item = super.getLegendItem(datasetIndex, series);
        }
        if(!item.isShapeVisible() && _alwaysDrawLegendColorSquare)
        {
            item = new LegendItem(item.getLabel(),
                                  item.getDescription(),
                                  item.getToolTipText(),
                                  item.getURLText(),
                                  item.isShapeFilled(), //shape visible
                                  ChartConstants.getShape("square"),
                                  item.isShapeFilled(), //shape filled
                                  item.getFillPaint(),
                                  item.isShapeFilled(), // outline visible
                                  item.getOutlinePaint(),
                                  item.getOutlineStroke(),
                                  item.isLineVisible(),
                                  item.getLine(),
                                  item.getLineStroke(),
                                  item.getLinePaint());
            item.setDataset(item.getDataset());
            item.setDatasetIndex(item.getDatasetIndex());
            item.setFillPaintTransformer(item.getFillPaintTransformer());
            item.setSeriesIndex(item.getSeriesIndex());
            item.setSeriesKey(item.getSeriesKey());
        }
        return item;
    }

    /**
     * Overridden to account for the {@link #_stepAtEnd} flag.
     */
    @Override
    public void drawItem(final Graphics2D g2,
                         final XYItemRendererState state,
                         final Rectangle2D dataArea,
                         final PlotRenderingInfo info,
                         final XYPlot plot,
                         final ValueAxis domainAxis,
                         final ValueAxis rangeAxis,
                         final XYDataset dataset,
                         final int series,
                         final int item,
                         final CrosshairState crosshairState,
                         final int pass)
    {

        // do nothing if item is not visible
        if(!getItemVisible(series, item))
        {
            return;
        }

        final PlotOrientation orientation = plot.getOrientation();

        final Paint seriesPaint = getItemPaint(series, item);
        final Stroke seriesStroke = getItemStroke(series, item);
        g2.setPaint(seriesPaint);
        g2.setStroke(seriesStroke);

        // get the data point...
        final double x1 = dataset.getXValue(series, item);
        final double y1 = dataset.getYValue(series, item);
        if(Double.isNaN(y1))
        {
            return;
        }

        final RectangleEdge xAxisLocation = plot.getDomainAxisEdge();
        final RectangleEdge yAxisLocation = plot.getRangeAxisEdge();
        final double transX1 = domainAxis.valueToJava2D(x1, dataArea, xAxisLocation);
        final double transY1 = rangeAxis.valueToJava2D(y1, dataArea, yAxisLocation);

        if(item > 0)
        {
            // get the previous data point...
            final double x0 = dataset.getXValue(series, item - 1);
            final double y0 = dataset.getYValue(series, item - 1);
            if(!Double.isNaN(y0))
            {
                final double transX0 = domainAxis.valueToJava2D(x0, dataArea, xAxisLocation);
                final double transY0 = rangeAxis.valueToJava2D(y0, dataArea, yAxisLocation);

                final Line2D line = state.workingLine;
                if(orientation == PlotOrientation.HORIZONTAL)
                {
                    if(transY0 == transY1)
                    { //this represents the situation 
                      // for drawing a horizontal bar.
                        line.setLine(transY0, transX0, transY1, transX1);
                        g2.draw(line);
                    }
                    else
                    { //this handles the need to perform a 'step'.
                        if(_stepAtEnd)
                        {
                            line.setLine(transY0, transX0, transY0, transX1);
                            g2.draw(line);
                            line.setLine(transY0, transX1, transY1, transX1);
                            g2.draw(line);
                        }
                        else
                        {
                            line.setLine(transY0, transX0, transY1, transX0);
                            g2.draw(line);
                            line.setLine(transY1, transX0, transY1, transX1);
                            g2.draw(line);
                        }
                    }
                }
                else if(orientation == PlotOrientation.VERTICAL)
                {
                    if(transY0 == transY1)
                    { // this represents the situation 
                      // for drawing a horizontal bar.
                        line.setLine(transX0, transY0, transX1, transY1);
                        g2.draw(line);
                    }
                    else
                    { //this handles the need to perform a 'step'.
                        if(_stepAtEnd)
                        {
                            line.setLine(transX0, transY0, transX1, transY0);
                            g2.draw(line);
                            line.setLine(transX1, transY0, transX1, transY1);
                            g2.draw(line);
                        }
                        else
                        {
                            line.setLine(transX0, transY0, transX0, transY1);
                            g2.draw(line);
                            line.setLine(transX0, transY1, transX1, transY1);
                            g2.draw(line);
                        }
                    }
                }

            }
        }

        // draw the item label if there is one...
        if(isItemLabelVisible(series, item))
        {
            double xx = transX1;
            double yy = transY1;
            if(orientation == PlotOrientation.HORIZONTAL)
            {
                xx = transY1;
                yy = transX1;
            }
            drawItemLabel(g2, orientation, dataset, series, item, xx, yy, (y1 < 0.0));
        }

        final int domainAxisIndex = plot.getDomainAxisIndex(domainAxis);
        final int rangeAxisIndex = plot.getRangeAxisIndex(rangeAxis);
        updateCrosshairValues(crosshairState, x1, y1, domainAxisIndex, rangeAxisIndex, transX1, transY1, orientation);

        // collect entity and tool tip information...
        if(state.getInfo() != null)
        {
            final EntityCollection entities = state.getEntityCollection();
            if(entities != null)
            {
                final int r = getDefaultEntityRadius();
                final Shape shape = orientation == PlotOrientation.VERTICAL ? new Rectangle2D.Double(transX1 - r,
                                                                                                     transY1 - r,
                                                                                                     2 * r,
                                                                                                     2 * r) : new Rectangle2D.Double(transY1
                                                                                                                                         - r,
                                                                                                                                     transX1
                                                                                                                                         - r,
                                                                                                                                     2 * r,
                                                                                                                                     2 * r);
                if(shape != null)
                {
                    String tip = null;
                    final XYToolTipGenerator generator = getToolTipGenerator(series, item);
                    if(generator != null)
                    {
                        tip = generator.generateToolTip(dataset, series, item);
                    }
                    String url = null;
                    if(getURLGenerator() != null)
                    {
                        url = getURLGenerator().generateURL(dataset, series, item);
                    }
                    final XYItemEntity entity = new XYItemEntity(shape, dataset, series, item, tip, url);
                    entities.add(entity);
                }
            }
        }
    }
}
