package ohd.hseb.charter.jfreechartoverride;

import java.text.DateFormat;
import java.text.NumberFormat;
import java.text.SimpleDateFormat;
import java.util.Date;

import org.jfree.chart.JFreeChart;
import org.jfree.chart.labels.StandardXYToolTipGenerator;
import org.jfree.data.xy.XYDataset;

import ohd.hseb.charter.ChartTools;
import ohd.hseb.charter.translator.AxisTranslator;
import ohd.hseb.hefs.utils.datetime.DateTools;

/**
 * Special tool tip handler for Graphics Generator. It is used for series that are not box-and-whisker; those are
 * handled through standard JFreeChart tools. This class allows for either one or two y-axes being displayed in the
 * plot, displaying a '-' for an axis if the entity value is not plotted against it. For translated axes, it performs
 * the translation and displays values for both the left and right axes.
 * 
 * @author herrhd
 */
public class GraphGenXYToolTipGenerator extends StandardXYToolTipGenerator
{
    private static final long serialVersionUID = 1L;

    /**
     * The _yFormat is recorded here so that its value can be null. In the super class, the y-axis formatter can never
     * be null.
     */
    private NumberFormat _yFormat = null;

    /**
     * Same for _yFormat2.
     */
    private NumberFormat _yFormat2 = null;

    /**
     * If there is a translator, this will be non-null.
     */
    private AxisTranslator _translatorForYAxis2 = null;

    /**
     * Used to translate from numbers to categories via
     * {@link ChartTools#computeCategoryForNumericalValue(JFreeChart, double)} when the domain axis is categorical. If
     * null, then it is assumed to not be categorical.
     */
    private JFreeChart _displayingChart = null;

    public GraphGenXYToolTipGenerator(final NumberFormat xFormat, final NumberFormat yFormat)
    {
        super("{0}: ({1}, {2})", xFormat, (yFormat == null) ? NumberFormat.getInstance() : yFormat);
        _yFormat = yFormat;
    }

    public GraphGenXYToolTipGenerator(final DateFormat xFormat, final NumberFormat yFormat)
    {
        super("{0}: ({1}, {2})", xFormat, (yFormat == null) ? NumberFormat.getInstance() : yFormat);
        _yFormat = yFormat;
    }

    public GraphGenXYToolTipGenerator(final JFreeChart chartContainingDomainAxisForCategoryTranslation,
                                      final NumberFormat yFormat)
    {
        //For a categorical domain axis, the x-axis formatter is not used, so just give it filler.
        super("{0}: ({1}, {2})", NumberFormat.getInstance(), (yFormat == null) ? NumberFormat.getInstance() : yFormat);
        _yFormat = yFormat;
        _displayingChart = chartContainingDomainAxisForCategoryTranslation;
    }

    public GraphGenXYToolTipGenerator(final NumberFormat xFormat,
                                      final NumberFormat yFormat,
                                      final NumberFormat yFormat2,
                                      final AxisTranslator translator)
    {
        super("{0}: ({1}, {2}, {3})", xFormat, (yFormat == null) ? NumberFormat.getInstance() : yFormat);
        _yFormat2 = yFormat2;
        _translatorForYAxis2 = translator;
        _yFormat = yFormat;
    }

    public GraphGenXYToolTipGenerator(final DateFormat xFormat,
                                      final NumberFormat yFormat,
                                      final NumberFormat yFormat2,
                                      final AxisTranslator translator)
    {
        super("{0}: ({1}, {2}, {3})", xFormat, (yFormat == null) ? NumberFormat.getInstance() : yFormat);
        _yFormat2 = yFormat2;
        _translatorForYAxis2 = translator;
        _yFormat = yFormat;
    }

    public GraphGenXYToolTipGenerator(final JFreeChart chartContainingDomainAxisForCategoryTranslation,
                                      final NumberFormat yFormat,
                                      final NumberFormat yFormat2,
                                      final AxisTranslator translator)
    {
        super("{0}: ({1}, {2}, {3})",
              NumberFormat.getInstance(),
              (yFormat == null) ? NumberFormat.getInstance() : yFormat);
        _yFormat2 = yFormat2;
        _translatorForYAxis2 = translator;
        _yFormat = yFormat;
        _displayingChart = chartContainingDomainAxisForCategoryTranslation;
    }

    @Override
    protected Object[] createItemArray(final XYDataset dataset, final int series, final int item)
    {
        Object[] result = null;

        result = new Object[4];

        result[0] = dataset.getSeriesKey(series).toString();

        //X-axis is simple.
        final double x = dataset.getXValue(series, item);
        if(Double.isNaN(x) && dataset.getX(series, item) == null)
        {
            result[1] = "null";
        }
        else if(_displayingChart != null)
        {
            result[1] = ChartTools.computeCategoryForNumericalValue(_displayingChart, x);
        }
        else
        {
            if(this.getXDateFormat() != null)
            {
                result[1] = DateTools.getThreadSafeSimpleDateFormat((SimpleDateFormat)getXDateFormat())
                                     .format(new Date((long)x));
            }
            else
            {
                result[1] = this.getXFormat().format(x);
            }
        }

        //For the y-axis on the left, return '-' if the data is plotted against the other axis.  Otherwise,
        //return the formatted value.
        if(_yFormat == null)
        {
            result[2] = "-";
        }
        else
        {
            final double y = dataset.getYValue(series, item);
            if(Double.isNaN(y) && dataset.getY(series, item) == null)
            {
                result[2] = "null";
            }
            else
            {
                result[2] = this._yFormat.format(y);
            }
        }

        //For the second y-axis another complication is added.  If the format is null, then set the label to '-'.
        //If the result is not null, then if the translator is null, use the y-axis value.  Otherwise, translate and
        //use that value.
        if(this._yFormat2 == null)
        {
            result[3] = "-";
        }
        else
        {
            if(this._translatorForYAxis2 != null)
            {
                //XXX The commented out code below will yield translated values, but may cause the software
                //to slow significantly during chart drawing.  Hence, it is commented out.
                //double y = this._translatorForYAxis2.computeTranslatedValue(dataset.getYValue(series, item));
                //if(Double.isNaN(y) && dataset.getY(series, item) == null)
                //{
                //    result[3] = "null";
                //}
                //else
                //{
                //result[3] = this._yFormat2.format(y);
                //}
                result[3] = "trans";
            }
            else
            {
                final double y = dataset.getYValue(series, item);
                if(Double.isNaN(y) && dataset.getY(series, item) == null)
                {
                    result[3] = "null";
                }
                else
                {
                    result[3] = this._yFormat2.format(y);
                }
            }
        }

        return result;
    }
}
