package ohd.hseb.charter.jfreechartoverride;

// Java util dependencies
import java.util.ArrayList;
import java.util.List;

// JFreeChart dependencies
import org.jfree.data.Range;
import org.jfree.data.RangeInfo;
import org.jfree.data.xy.AbstractXYDataset;

/**
 * A simple implementation of the BoxPlotDataset. The dataset can hold only one series.
 *
 * @author evs@hydrosolved.com
 * @version 4.0
 */

public class ScatterBoxPlotDataset extends AbstractXYDataset implements RangeInfo
{

    /********************************************************************************
     * * INSTANCE VARIABLES * *
     *******************************************************************************/

    /**
     * The series key.
     */

    private final Comparable<String> seriesKey;

    /**
     * Storage for the x values.
     */

    private final List<Number> xVals;

    /**
     * Storage for the box and whisker statistics.
     */

    private final List<ScatterBoxPlotItem> items;

    /**
     * The minimum range value.
     */

    private Number minimumRangeValue;

    /**
     * The maximum range value.
     */

    private Number maximumRangeValue;

    /**
     * The range of values.
     */

    private Range rangeBounds;

    /*******************************************************************************
     * * CONSTRUCTOR * *
     ******************************************************************************/

    /**
     * Constructs a new box dataset.
     * <p>
     * The current implementation allows only one series in the dataset. This may be extended in a future version.
     *
     * @param seriesKey the key for the series.
     */
    public ScatterBoxPlotDataset(final Comparable<String> seriesKey)
    {
        this.seriesKey = seriesKey;
        this.xVals = new ArrayList<>();
        this.items = new ArrayList<>();
        this.minimumRangeValue = null;
        this.maximumRangeValue = null;
        this.rangeBounds = null;
    }

    /*******************************************************************************
     * * ACCESSOR METHODS * *
     ******************************************************************************/

    /**
     * Returns the name of the series stored in this dataset.
     *
     * @param i the index of the series. Currently ignored.
     * @return the name of this series.
     */
    @Override
    public Comparable<String> getSeriesKey(final int i)
    {
        return this.seriesKey;
    }

    /**
     * Return an item from within the dataset.
     * 
     * @param series the series index (ignored, since this dataset contains only one series).
     * @param item the item within the series (zero-based index)
     * @return the item.
     */
    public ScatterBoxPlotItem getItem(final int series, final int item)
    {
        return this.items.get(item);
    }

    /**
     * Returns the x-value for one item in a series.
     * <p>
     *
     * @param series the series (zero-based index).
     * @param item the item (zero-based index).
     * @return The x-value.
     */
    @Override
    public Number getX(final int series, final int item)
    {
        return xVals.get(item);
    }

    /**
     * Returns the y-value for one item in a series.
     * <p>
     * This method (from the XYDataset interface) is mapped to the getMaxNonOutlierValue() method.
     *
     * @param series the series (zero-based index).
     * @param item the item (zero-based index).
     * @return The y-value.
     */
    @Override
    public Number getY(final int series, final int item)
    {
        return null;
    }

    /**
     * Returns the min-value for the specified series and item.
     *
     * @param series the series (zero-based index).
     * @param item the item (zero-based index).
     * @return The min-value for the specified series and item.
     */
    public Number getMinRegularValue(final int series, final int item)
    {
        Number result = null;
        final ScatterBoxPlotItem stats = this.items.get(item);
        if(stats != null)
        {
            result = stats.getMinRegularValue();
        }
        return result;
    }

    /**
     * Returns the max-value for the specified series and item.
     *
     * @param series the series (zero-based index).
     * @param item the item (zero-based index).
     * @return The max-value for the specified series and item.
     */
    public Number getMaxRegularValue(final int series, final int item)
    {
        Number result = null;
        final ScatterBoxPlotItem stats = this.items.get(item);
        if(stats != null)
        {
            result = stats.getMaxRegularValue();
        }
        return result;
    }

    /**
     * Returns the number of series in the dataset.
     * <p>
     * This implementation only allows one series.
     *
     * @return The number of series.
     */
    @Override
    public int getSeriesCount()
    {
        return 1;
    }

    /**
     * Returns the number of items in the specified series.
     *
     * @param series the index (zero-based) of the series.
     * @return The number of items in the specified series.
     */
    @Override
    public int getItemCount(final int series)
    {
        return xVals.size();
    }

    /**
     * Returns the minimum y-value in the dataset.
     *
     * @param includeInterval a flag that determines whether or not the y-interval is taken into account.
     * @return The minimum value.
     */
    @Override
    public double getRangeLowerBound(final boolean includeInterval)
    {
        double result = Double.NaN;
        if(this.minimumRangeValue != null)
        {
            result = this.minimumRangeValue.doubleValue();
        }
        return result;
    }

    /**
     * Returns the maximum y-value in the dataset.
     *
     * @param includeInterval a flag that determines whether or not the y-interval is taken into account.
     * @return The maximum value.
     */
    @Override
    public double getRangeUpperBound(final boolean includeInterval)
    {
        double result = Double.NaN;
        if(this.maximumRangeValue != null)
        {
            result = this.maximumRangeValue.doubleValue();
        }
        return result;
    }

    /**
     * Returns the range of the values in the range of this dataset.
     *
     * @param includeInterval a flag that determines whether or not the y-interval is taken into account.
     * @return The range.
     */
    @Override
    public Range getRangeBounds(final boolean includeInterval)
    {
        return this.rangeBounds;
    }

    /*******************************************************************************
     * * MUTATOR METHODS * *
     ******************************************************************************/

    /**
     * Adds an item to the dataset.
     * 
     * @param xVal the x value.
     * @param item the item.
     */

    public void add(final Number xVal, final ScatterBoxPlotItem item)
    {
        this.xVals.add(xVal);
        this.items.add(item);
        if(this.minimumRangeValue == null)
        {
            this.minimumRangeValue = item.getMinRegularValue();
        }
        else
        {
            if(item.getMinRegularValue().doubleValue() < this.minimumRangeValue.doubleValue())
            {
                this.minimumRangeValue = item.getMinRegularValue();
            }
        }
        if(this.maximumRangeValue == null)
        {
            this.maximumRangeValue = item.getMaxRegularValue();
        }
        else
        {
            if(item.getMaxRegularValue().doubleValue() > this.maximumRangeValue.doubleValue())
            {
                this.maximumRangeValue = item.getMaxRegularValue();
            }
        }
        this.rangeBounds = new Range(this.minimumRangeValue.doubleValue(), this.maximumRangeValue.doubleValue());
    }

}
