package ohd.hseb.charter.jfreechartoverride;

// Java util dependencies
import java.util.Arrays;

/**
 * Represents one data item within a modified box plot dataset.
 *
 * @author evs@hydrosolved.com
 * @version 4.0
 */

public final class ScatterBoxPlotItem
{

    /********************************************************************************
     * * INSTANCE VARIABLES * *
     *******************************************************************************/

    /**
     * The whisker items.
     */

    private final double[] points;

    /**
     * The center line.
     */

    private double center;

    /*******************************************************************************
     * * CONSTRUCTOR * *
     ******************************************************************************/

    /**
     * Creates a new box plot item with the plot points comprising the horizontal elements (box boundaries or whiskers)
     * in order of increasing size.
     * 
     * @param points the plot points
     */

    public ScatterBoxPlotItem(final double[] points)
    {
        if(points == null || points.length == 0)
        {
            throw new IllegalArgumentException("Specify valid points for the box plot item.");
        }
        this.points = new double[points.length];
        System.arraycopy(points, 0, this.points, 0, points.length);
    }

    /**
     * Creates a new box plot item with the plot points comprising the horizontal elements (box boundaries or whiskers)
     * in order of increasing size.
     * 
     * @param points the plot points
     * @param center the center line
     */

    public ScatterBoxPlotItem(final double[] points, final double center)
    {
        this(points);
        this.center = center;
    }

    /*******************************************************************************
     * * ACCESSOR METHODS * *
     ******************************************************************************/

    /**
     * Returns the plot points.
     * 
     * @return the plot points
     */
    public double[] getPoints()
    {
        return Arrays.copyOf(points, points.length);
    }

    /**
     * Returns the center line.
     * 
     * @return the center line
     */
    public double getCenter()
    {
        return center;
    }

    /**
     * Returns the minimum regular value.
     * 
     * @return The minimum regular value (possibly <code>null</code>).
     */
    public Number getMinRegularValue()
    {
        return points[0];
    }

    /**
     * Returns the maximum regular value.
     * 
     * @return The maximum regular value (possibly <code>null</code>).
     */
    public Number getMaxRegularValue()
    {
        return points[points.length - 1];
    }

    /**
     * Tests this object for equality with an arbitrary object.
     * 
     * @param obj the object to test against (<code>null</code> permitted).
     * @return a boolean.
     */
    @Override
    public boolean equals(final Object obj)
    {
        if(obj == this)
        {
            return true;
        }
        if(!(obj instanceof ScatterBoxPlotItem))
        {
            return false;
        }
        final ScatterBoxPlotItem that = (ScatterBoxPlotItem)obj;
        return Arrays.equals(that.getPoints(), points);
    }

    /**
     * Override hashcode: not implemented.
     * 
     * @return a hashcode
     */
    @Override
    public int hashCode()
    {
        assert false : "hashCode not implemented for BoxPlotItem.";
        return 1;
    }

}
