package ohd.hseb.charter.jfreechartoverride;

// Java awt dependencies
import java.awt.Color;
import java.awt.Graphics2D;
import java.awt.Paint;
import java.awt.Shape;
import java.awt.Stroke;
import java.awt.geom.Line2D;
import java.awt.geom.Rectangle2D;

// JFreeChart dependencies
import org.jfree.chart.axis.ValueAxis;
import org.jfree.chart.event.RendererChangeEvent;
import org.jfree.chart.plot.CrosshairState;
import org.jfree.chart.plot.PlotOrientation;
import org.jfree.chart.plot.PlotRenderingInfo;
import org.jfree.chart.plot.XYPlot;
import org.jfree.chart.renderer.xy.AbstractXYItemRenderer;
import org.jfree.chart.renderer.xy.XYItemRendererState;
import org.jfree.data.xy.XYDataset;
import org.jfree.util.PaintUtilities;

/**
 * A renderer that draws box plot items on an {@link org.jfree.chart.plot.XYPlot}.
 * 
 * @author evs@hydrosolved.com
 * @version 4.0
 */

public final class ScatterBoxPlotRenderer extends AbstractXYItemRenderer
{

    /**
     * The box width.
     */
    private double boxWidth;

    /**
     * The paint used to draw various artifacts.
     */
    private Paint artifactPaint = Color.black;

    /**
     * Indicates if the box drawn will be filled. Otherwise it is just a box.
     */
    private boolean boxFilled = true;

    public ScatterBoxPlotRenderer()
    {
        this(-1.0);
        setBaseFillPaint(Color.green);
    }

    /**
     * Creates a new renderer.
     * <P>
     * Use -1 for the box width if you prefer the width to be calculated automatically.
     *
     * @param boxWidth the box width.
     */
    public ScatterBoxPlotRenderer(final double boxWidth)
    {
        super();
        this.boxWidth = boxWidth;
        setBaseFillPaint(Color.green);
    }

    public double getBoxWidth()
    {
        return boxWidth;
    }

    public Paint getArtifactPaint()
    {
        return artifactPaint;
    }

    public void setBoxFilled(final boolean b)
    {
        boxFilled = b;
    }

    public boolean getBoxFilled()
    {
        return boxFilled;
    }

    /**
     * Tests this renderer for equality with another object.
     *
     * @param obj the object (<code>null</code> permitted).
     * @return <code>true</code> or <code>false</code>.
     */
    @Override
    public boolean equals(final Object obj)
    {
        if(obj == this)
        {
            return true;
        }
        if(!(obj instanceof ScatterBoxPlotRenderer))
        {
            return false;
        }
        if(!super.equals(obj))
        {
            return false;
        }
        final ScatterBoxPlotRenderer that = (ScatterBoxPlotRenderer)obj;
        if(Math.abs(this.boxWidth - that.getBoxWidth()) > .0000001)
        {
            return false;
        }
        if(!PaintUtilities.equal(this.artifactPaint, that.artifactPaint))
        {
            return false;
        }
        return true;
    }

    /**
     * Override hashcode: not implemented.
     * 
     * @return a hashcode
     */
    @Override
    public int hashCode()
    {
        assert false : "hashCode not implemented for BoxPlotRenderer.";
        return 1;
    }

    /**
     * Sets the box width.
     * <P>
     * If you set the width to a negative value, the renderer will calculate the box width automatically based on the
     * space available on the chart.
     *
     * @param boxWidth the box width.
     */
    public void setBoxWidth(final double boxWidth)
    {
        if(Math.abs(boxWidth - this.boxWidth) > .0000001)
        {
            this.boxWidth = boxWidth;
            notifyListeners(new RendererChangeEvent(this));
        }
    }

    /**
     * Sets the paint used to paint the various artifacts.
     * 
     * @param artifactPaint the paint.
     */
    public void setArtifactPaint(final Paint artifactPaint)
    {
        this.artifactPaint = artifactPaint;
    }

    /**
     * Draws the visual representation of a single data item.
     *
     * @param g2 the graphics device.
     * @param state the renderer state.
     * @param dataArea the area within which the plot is being drawn.
     * @param info collects info about the drawing.
     * @param plot the plot (can be used to obtain standard color information etc).
     * @param domainAxis the domain axis.
     * @param rangeAxis the range axis.
     * @param dataset the dataset.
     * @param series the series index (zero-based).
     * @param item the item index (zero-based).
     * @param crosshairState crosshair information for the plot (<code>null</code> permitted).
     * @param pass the pass index.
     */
    @Override
    public void drawItem(final Graphics2D g2,
                         final XYItemRendererState state,
                         final Rectangle2D dataArea,
                         final PlotRenderingInfo info,
                         final XYPlot plot,
                         final ValueAxis domainAxis,
                         final ValueAxis rangeAxis,
                         final XYDataset dataset,
                         final int series,
                         final int item,
                         final CrosshairState crosshairState,
                         final int pass)
    {

        final PlotOrientation orientation = plot.getOrientation();

        if(orientation == PlotOrientation.HORIZONTAL)
        {
            drawHorizontalItem(g2,
                               dataArea,
                               info,
                               plot,
                               domainAxis,
                               rangeAxis,
                               dataset,
                               series,
                               item,
                               crosshairState,
                               pass);
        }
        else if(orientation == PlotOrientation.VERTICAL)
        {
            drawVerticalItem(g2,
                             dataArea,
                             info,
                             plot,
                             domainAxis,
                             rangeAxis,
                             dataset,
                             series,
                             item,
                             crosshairState,
                             pass);
        }

    }

    /**
     * Draws the visual representation of a single data item.
     *
     * @param g2 the graphics device.
     * @param dataArea the area within which the plot is being drawn.
     * @param info collects info about the drawing.
     * @param plot the plot (can be used to obtain standard color information etc).
     * @param domainAxis the domain axis.
     * @param rangeAxis the range axis.
     * @param dataset the dataset.
     * @param series the series index (zero-based).
     * @param item the item index (zero-based).
     * @param crosshairState crosshair information for the plot (<code>null</code> permitted).
     * @param pass the pass index.
     */
    public void drawHorizontalItem(final Graphics2D g2,
                                   final Rectangle2D dataArea,
                                   final PlotRenderingInfo info,
                                   final XYPlot plot,
                                   final ValueAxis domainAxis,
                                   final ValueAxis rangeAxis,
                                   final XYDataset dataset,
                                   final int series,
                                   final int item,
                                   final CrosshairState crosshairState,
                                   final int pass)
    {

        throw new IllegalStateException("Horizontal scatter box-and-whisker plots are not implemented.");
    }

    /**
     * Draws the visual representation of a single data item.
     *
     * @param g2 the graphics device.
     * @param dataArea the area within which the plot is being drawn.
     * @param info collects info about the drawing.
     * @param plot the plot (can be used to obtain standard color information etc).
     * @param domainAxis the domain axis.
     * @param rangeAxis the range axis.
     * @param dataset the dataset.
     * @param series the series index (zero-based).
     * @param item the item index (zero-based).
     * @param crosshairState crosshair information for the plot (<code>null</code> permitted).
     * @param pass the pass index.
     */
    public void drawVerticalItem(final Graphics2D g2,
                                 final Rectangle2D dataArea,
                                 final PlotRenderingInfo info,
                                 final XYPlot plot,
                                 final ValueAxis domainAxis,
                                 final ValueAxis rangeAxis,
                                 final XYDataset dataset,
                                 final int series,
                                 final int item,
                                 final CrosshairState crosshairState,
                                 final int pass)
    {
        //The data set.
        final ScatterBoxPlotDataset boxData = (ScatterBoxPlotDataset)dataset;

        //The x-axis value.
        final Number x = boxData.getX(series, item);

        //Converts the x-value to pixel location.
        final double xx = domainAxis.valueToJava2D(x.doubleValue(), dataArea, plot.getDomainAxisEdge());

        //Various stuff involved with the drawing.
        double exactboxWidth = getBoxWidth();
        double width = exactboxWidth;
        final double dataAreaX = dataArea.getMaxX() - dataArea.getMinX();
        final double maxboxPercent = 0.1;
        final double maxboxWidth = dataAreaX * maxboxPercent;

        //A default box width value?  Unclear what this does... ask James.
        if(exactboxWidth <= 0.0)
        {
            final int itemCount = boxData.getItemCount(series);
            exactboxWidth = dataAreaX / itemCount * 4.5 / 7;
            if(exactboxWidth < 3)
            {
                width = 3;
            }
            else if(exactboxWidth > maxboxWidth)
            {
                width = maxboxWidth;
            }
            else
            {
                width = exactboxWidth;
            }
        }

        //Determine color for line drawing.
        final Paint p = getArtifactPaint();
        if(p != null)
        {
            g2.setPaint(p);
        }

        //Determine the stroke using the line width.
        final Stroke s = getItemStroke(series, item);
        g2.setStroke(s);

//      // add an entity for the item...
//      String url = null;
//      String tip = null;
//      if (entities != null) {
//          XYToolTipGenerator generator = getToolTipGenerator(series, item);
//          if (generator != null) {
//              tip = generator.generateToolTip(dataset, series, item);
//          }
//          if (getURLGenerator() != null) {
//              url = getURLGenerator().generateURL(dataset, series, item);
//          }
//      }

        final double[] data = boxData.getItem(series, item).getPoints();
        //=================================
        //Draw the box 
        //=================================

        //Only draw if there are at least 4 items, and then assume that it is the first four items we use!
        int boxLBIndex = -1;
        int boxUBIndex = -1;
        if(data.length > 3)
        {
            boxLBIndex = (int)Math.round(0.2 * (data.length - 1)); //20% 
            boxUBIndex = (int)Math.round(0.8 * (data.length - 1)); //80%
            
            //Size and position the box.
            final double yLower = data[boxLBIndex]; //used to use bBound
            final double yUpper = data[boxUBIndex]; //used to use tBound
            
            //Only drw the box if it is defined!
            if(!Double.isNaN(yLower) && !Double.isNaN(yUpper))
            {
                final double yyLower = rangeAxis.valueToJava2D(yLower, dataArea, plot.getRangeAxisEdge());
                final double yyUpper = rangeAxis.valueToJava2D(yUpper, dataArea, plot.getRangeAxisEdge());
                
                final Shape box = new Rectangle2D.Double(xx - width / 2, yyUpper, width, Math.abs(yyUpper - yyLower));

                //Draw the box in the color of the fill.  
                g2.setPaint(this.getBaseFillPaint());
                g2.draw(box);

                //Fill the box only if indicated.
                if(boxFilled)
                {
                    g2.fill(box);
                }
            }
        }

        //=====================================
        //Draw the whiskers
        //=====================================
        for(int i = 0; i < data.length - 1; i++)
        {
            final double yLower = data[i];
            final double yUpper = data[i + 1];
            if(!Double.isNaN(yLower) && !Double.isNaN(yUpper))
            {
                final double yyLower = rangeAxis.valueToJava2D(yLower, dataArea, plot.getRangeAxisEdge());
                final double yyUpper = rangeAxis.valueToJava2D(yUpper, dataArea, plot.getRangeAxisEdge());
                final Shape line1 = new Line2D.Double(xx - width / 2, yyUpper, xx + width / 2, yyUpper); //Horizontal line top
                final Shape line2 = new Line2D.Double(xx - width / 2, yyLower, xx + width / 2, yyLower); //Horizontal line bottom

                //Lines are drawn in the line color pain, which is the artifact paint.  
                g2.setPaint(p);
                g2.draw(line1);
                g2.draw(line2);

                //The vertical line.
                if((boxLBIndex < 0) || (i < boxLBIndex) || (i >= boxUBIndex))
                {
                    final Shape line3 = new Line2D.Double(xx, yyLower, xx, yyUpper);
                    g2.draw(line3);
                }
            }
        }

        //======================================
        //Draw the center line 
        //======================================
        final ScatterBoxPlotItem b = boxData.getItem(series, item);
        final double cent = b.getCenter();
        if(!Double.isNaN(cent))
        {
            final double cent2 = rangeAxis.valueToJava2D(cent, dataArea, plot.getRangeAxisEdge());
            final Shape line = new Line2D.Double(xx - width / 2, cent2, xx + width / 2, cent2);
            g2.setPaint(Color.black);
            g2.draw(line);
            g2.setPaint(p);
        }

    }

}