package ohd.hseb.charter.panel;

import java.awt.Color;
import java.awt.Component;
import java.util.Date;
import java.util.List;

import javax.swing.JLabel;
import javax.swing.JTable;
import javax.swing.table.AbstractTableModel;
import javax.swing.table.TableCellRenderer;

import org.jfree.chart.JFreeChart;
import org.jfree.chart.axis.ValueAxis;
import org.jfree.chart.plot.XYPlot;

import com.google.common.base.Predicate;

import ohd.hseb.charter.ChartConstants;
import ohd.hseb.charter.ChartEngine;
import ohd.hseb.charter.datasource.XYChartDataSource;
import ohd.hseb.charter.parameters.ChartDrawingParameters;
import ohd.hseb.charter.tools.NumberAxisOverride;
import ohd.hseb.hefs.utils.Dyad;
import ohd.hseb.hefs.utils.gui.jtable.EventPostingCellSelectableTable;
import ohd.hseb.hefs.utils.gui.jtable.GenericTable;
import ohd.hseb.hefs.utils.gui.jtable.MarkPanel;
import ohd.hseb.hefs.utils.gui.jtable.models.MinWidthsTableModel;
import ohd.hseb.hefs.utils.gui.jtable.models.PreferredWidthsTableModel;
import ohd.hseb.hefs.utils.gui.jtable.models.RowColumnMarkingTableModel;
import ohd.hseb.hefs.utils.gui.jtable.models.RowHeaderTableModel;
import ohd.hseb.hefs.utils.gui.jtable.models.TableAwareTableModel;
import ohd.hseb.hefs.utils.gui.jtable.models.WrapRendererTableModel;
import ohd.hseb.hefs.utils.gui.jtable.renderers.AlternatingColumnColorTableCellRenderer;
import ohd.hseb.hefs.utils.gui.jtable.renderers.PredicateTableCellRenderer;
import ohd.hseb.hefs.utils.gui.tools.ColorTools;

/**
 * The table model for the chart. It uses a {@link List} of {@link XYChartDataSource} objects and
 * {@link ChartDrawingParameters} to determine what to put in the table. This assumes that if the domain values are
 * identical for all series, then the domain value need only be included once. It implements
 * {@link WrapRendererTableModel} making use of an {@link AlternatingColumnColorTableCellRenderer} that wraps the
 * default rendering; see {@link GenericTable}.<br>
 * <br>
 * See the method headers for the default return values of standard methods.<br>
 * <br>
 * Methods that are prefixed by "abstract" are written in order to expose these top level methods to subclasses that may
 * be several layers deep, but still want to call the top level method defined here. For example,
 * {@link #abstractGetColumnName(int)} will call a default algorithm for determining a column name based on the return
 * value of {@link #areAllSeriesXaxisValuesSame()} and information in {@link #_chartEngine}. A subclass may override
 * that algorithm (example: {@link DomainSharingTimeSeriesChartEngineTableModel}), but then a subclass of a subclass may
 * want to call the original algorithm used here (example: see CFSv2MonthlyChartDiagnosticsTableModel in hefsplugins).
 * There are currently four such abstract methods, but others may be added if needed.
 * 
 * @author zcui
 * @author hank
 */
@SuppressWarnings("serial")
public abstract class AbstractChartEngineTableModel extends AbstractTableModel implements ChartEngineTableModel,
MinWidthsTableModel, PreferredWidthsTableModel, WrapRendererTableModel, TableAwareTableModel,
RowColumnMarkingTableModel, RowHeaderTableModel
{

    public static Color WITHIN_LIMITS_MARK_COLOR = Color.black;
    public static Color DEFAULT_SELECTED_CELL_COLOR = new Color(40, 89, 208);

    private ChartEngine _chartEngine;

    private JFreeChart _chart;

    /**
     * For the {@link TableAwareTableModel} implementation.
     */
    private EventPostingCellSelectableTable _table;

    /**
     * Call the {@link #setDataSourceIndex(int)} method to change this value initially before drawing the table. The
     * initial value of 0 may cause problems if not properly set.
     */
    private int _dataSourceIndex = 0;

    /**
     * Remembers the results of a call to {@link #areAllSeriesXaxisValuesSame()} so that the algorithm need only be
     * performed the first time it is called. This is critical to saving time, because the method is called a lot
     * throughout this model.
     */
    private Boolean _areAllSeriesXAxisValuesSame = null;

    public AbstractChartEngineTableModel()
    {
        _chartEngine = null;
    }

    /**
     * @param dataSources List of data sources.
     * @param chartDrawingParameters Corresponding drawing parameters.
     */
    public AbstractChartEngineTableModel(final ChartEngine chartEngine)
    {
        setChartEngineWithoutFiringEvent(chartEngine);
    }

    /**
     * Sets {@link #_chartEngine} without calling {@link #fireTableStructureChanged()}. Useful for subclasses that must
     * do some processing after the chart engine is known and before the table is redrawn. This may need to be
     * overridden if checking of the engine is required for a subclass; for example if all data sources must be of one
     * time, override this method and check the types before calling the super class.
     * 
     * @param engine The new chart engine, which means new data to display.
     */
    public void setChartEngineWithoutFiringEvent(final ChartEngine engine)
    {
        abstractSetChartEngineWithoutFiringEvent(engine);
    }

    /**
     * @return The domain axis within {@link #_chart} that corresponds to the current source.
     */
    public ValueAxis getDomainAxis()
    {
        return _chart.getXYPlot().getDomainAxis();
    }

    /**
     * @return The range axis within {@link #_chart} that corresponds to the current source. It may possible for this to
     *         return null if somehow the subplot for the current data source is null. I'm not sure, but this may happen
     *         if the subplot is made not visible. Just in case, be sure to be able to handle a null return.
     */
    public ValueAxis getRangeAxis()
    {
        //Gather the information needed.
        final int subPlotIndex = _chartEngine.getChartParameters()
                                             .getDataSourceParameters(getDataSourceIndex())
                                             .getSubPlotIndex();
        final int axisIndex = _chartEngine.getChartParameters()
                                          .getDataSourceParameters(getDataSourceIndex())
                                          .getYAxisIndex();
        final XYPlot subPlot = _chartEngine.getSubPlot(subPlotIndex);
        if(subPlot != null)
        {
            final ValueAxis rangeAxis = subPlot.getRangeAxis(axisIndex);
            return rangeAxis;
        }
        return null;
    }

    /**
     * @return The results of {@link ChartEngine#getDataSources()} for {@link #_chartEngine}.
     */
    public List<XYChartDataSource> getDataSources()
    {
        return _chartEngine.getDataSources();
    }

    /**
     * @return Calls {@link #getDataSources()} and returns the one at index.
     */
    public XYChartDataSource getDataSource(final int index)
    {
        return getDataSources().get(index);
    }

    /**
     * @return The results of {@link ChartEngine#getChartParameters()} for {@link #_chartEngine}.
     */
    public ChartDrawingParameters getChartDrawingParameters()
    {
        return _chartEngine.getChartParameters();
    }

    /**
     * Sets the data source index without calling {@link #fireTableStructureChanged()}.
     */
    public void setDataSourceIndexWithoutFiringEvent(final int dataSourceIndex)
    {
        _areAllSeriesXAxisValuesSame = null;
        _dataSourceIndex = dataSourceIndex;
    }

    /**
     * @return Calls {@link ChartConstants#isAxisTypeTime(int)} given {@link XYChartDataSource#getXAxisType()}
     *         and returns the result. If false, the domain is numerical.
     */
    public boolean isDomainTime()
    {
        if(getDataSources().isEmpty())
        {
            return false;
        }
        return ChartConstants.isAxisTypeTime(getDataSources().get(0).getXAxisType());
    }

    /**
     * Override as needed.<br>
     * <br>
     * If the method {@link XYChartDataSource#getChartTableColumnHeader(int)} returns a non null value for the current
     * source and series, it is returned. Otherwise, it returns the legend entry for the source and series.
     * 
     * @param dataSourceIdx The index of the data source for which to acquire a header.
     * @param seriesIdx The index of the series.
     * @return The header to use (see above).
     */
    public String buildSeriesColumnHeader(final int dataSourceIdx, final int seriesIdx)
    {
        final String name = getCurrentDataSource().getChartTableColumnHeader(seriesIdx);
        if(name != null)
        {
            if(getCurrentDataSource().getNumberOfSeries() > 1)
            {
                return "<html>Series " + seriesIdx + "<br>" + name + "</html>";
            }
            return "<html>" + name + "</html>";
        }

        final String legendEntry = getChartDrawingParameters().getDataSourceParameters(_dataSourceIndex)
                                                              .getSeriesDrawingParametersForSeriesIndex(seriesIdx)
                                                              .getArgumentReplacedNameInLegend();
        return "<html>Series " + seriesIdx + "<br>" + legendEntry + "</html>";
    }

    /**
     * Do NOT override this!!! This exposes the top level version of the {@link #areAllSeriesXaxisValuesSame()} method
     * algorithm to subclasses, no matter how deep those subclasses are. See {@link #areAllSeriesXaxisValuesSame()} for
     * more information. This method makes use of {@link #_areAllSeriesXAxisValuesSame} in order to remember the
     * algorithm result after the first time it is called so that the algorithm need not be performed again. This saves
     * lots and lots of time.
     */
    protected boolean abstractAreAllSeriesXaxisValuesSame()
    {
        if(_areAllSeriesXAxisValuesSame != null)
        {
            return _areAllSeriesXAxisValuesSame;
        }

        if(this.getCurrentDataSource().getNumberOfSeries() <= 1)
        {
            _areAllSeriesXAxisValuesSame = true;
            return true;
        }

        //check if they have the same counts
        final int numberOfDataPoints = this.getCurrentDataSource().getSeriesValueCount(0);

        for(int seriesIndex = 1; seriesIndex < this.getCurrentDataSource().getNumberOfSeries(); ++seriesIndex)
        {
            if(numberOfDataPoints != this.getCurrentDataSource().getSeriesValueCount(seriesIndex))
            {
                _areAllSeriesXAxisValuesSame = false;
                return false;
            }
        }

        //Check the values one index at a time.
        for(int seriesValueIndex = 0; seriesValueIndex < this.getCurrentDataSource().getSeriesValueCount(0); ++seriesValueIndex)
        {
            final Object xValue = this.getCurrentDataSource().getSeriesValue(0, seriesValueIndex, true);

            for(int seriesIndex = 1; seriesIndex < this.getCurrentDataSource().getNumberOfSeries(); ++seriesIndex)
            {
                if(!xValue.equals(this.getCurrentDataSource().getSeriesValue(seriesIndex, seriesValueIndex, true)))
                {
                    _areAllSeriesXAxisValuesSame = false;
                    return false;
                }
            }

        }

        _areAllSeriesXAxisValuesSame = true;
        return true;
    }

    /**
     * Do NOT override this!!! This exposes the top level version of the {@link #getRawValueAt(int, int)} method
     * algorithm to subclasses, not matter how deep those subclasses are. See {@link #getRawValueAt(int, int)} for more
     * details.
     */
    protected Object abstractGetRawValueAt(final int modelRow, final int modelColumn)
    {
        if(modelColumn == 0)
        {
            return getCurrentDataSource().getSeriesValue(modelColumn, modelRow, true);
        }
        else
        {
            final int seriesIndex = computeSeriesIndex(modelColumn);
            if(seriesIndex < 0)
            {
                return -1;
            }

            if(areAllSeriesXaxisValuesSame())
            {
                return getCurrentDataSource().getSeriesValue(seriesIndex, modelRow, false);
            }
            else
            {
                if(modelColumn % 2 == 0)
                {
                    return getCurrentDataSource().getSeriesValue(seriesIndex, modelRow, true);
                }
                else
                {
                    return getCurrentDataSource().getSeriesValue(seriesIndex, modelRow, false);
                }
            }
        }
    }

    /**
     * Do NOT override this!!! This exposes the default mechanism for determining a column name for subclasses, no
     * matter how deep.
     */
    protected String abstractGetColumnName(final int col)
    {
        if(getDataSources().size() < 1)
        {
            return "UNDEFINED";
        }
        if(col == 0)
        {
            if(areAllSeriesXaxisValuesSame())
            {
                return this.buildDomainColumnHeader(-1);
            }
            return this.buildDomainColumnHeader(col / 2);
        }
        else
        {
            if(areAllSeriesXaxisValuesSame())
            {
                return buildSeriesColumnHeader(_dataSourceIndex, col - 1);
            }
            else
            {
                if(col % 2 == 0)
                {
                    return this.buildDomainColumnHeader(col / 2);
                }
                else
                {
                    return buildSeriesColumnHeader(_dataSourceIndex, col / 2);
                }
            }
        }
    }

    /**
     * Do NOT override this!!! This exposes the default mechanism for determining a row count, no matter how deep.
     */
    protected int abstractGetRowCount()
    {
        if(areAllSeriesXaxisValuesSame())
        {
            return this.getCurrentDataSource().getSeriesValueCount(0);
        }
        else
        {
            //check if they have the same counts
            int numberOfDataPoints = this.getCurrentDataSource().getSeriesValueCount(0);

            for(int seriesIndex = 1; seriesIndex < this.getCurrentDataSource().getNumberOfSeries(); ++seriesIndex)
            {
                if(numberOfDataPoints < this.getCurrentDataSource().getSeriesValueCount(seriesIndex))
                {
                    numberOfDataPoints = this.getCurrentDataSource().getSeriesValueCount(seriesIndex);
                }
            }
            return numberOfDataPoints;
        }
    }

    /**
     * Do NOT override this!!! This exposes the default implementation to subclasses no matter how deep.
     * 
     * @param engine
     */
    protected void abstractSetChartEngineWithoutFiringEvent(final ChartEngine engine)
    {
        _chartEngine = engine;
    }

    /**
     * @return The currently displayed data soure.
     */
    protected XYChartDataSource getCurrentDataSource()
    {
        return getDataSources().get(_dataSourceIndex);
    }

    /**
     * @param seriesIndex Index of the series for which to acquire the domain axis column header.
     * @return The domain axis column header, which is general across all data sources.
     */
    protected String buildDomainColumnHeader(final int seriesIndex)
    {
        String baseName;

        //Provided
        final String name = getCurrentDataSource().getChartTableDomainHeader();
        if(getCurrentDataSource().getChartTableDomainHeader() != null)
        {
            if(!areAllSeriesXaxisValuesSame())
            {
                return "<html>Series " + seriesIndex + "<br>" + name + "</html>";
            }
            return "<html>" + name + "</html>";
        }

        //Time...
        else if(ChartConstants.isAxisTypeTime(this.getCurrentDataSource().getXAxisType()))
        {
            baseName = "time (" + getChartDrawingParameters().getGeneralParameters().getTimeZoneUsedInGraphic() + ")";
        }
        //Numerical...
        else if(this.getCurrentDataSource().getXAxisType() == ChartConstants.AXIS_IS_NUMERICAL)
        {
            baseName = getChartDrawingParameters().getDomainAxis().getLabel().getText();
            if(baseName.isEmpty())
            {
                baseName = "X";
            }
        }
        //Unknown...
        else
        {
            baseName = "UNDEFINED";
        }

        //Just return the baseName.
        if(seriesIndex < 0)
        {
            return baseName;
        }
        //Otherwise add the series index.
        return "<html>Series " + seriesIndex + "<br>" + baseName + "</html>";
    }

    /**
     * This calls {@link #abstractAreAllSeriesXaxisValuesSame()} by default.
     * 
     * @return True if all domain axis values are identical for the series in the current data source. This checks for
     *         exactly identical series, including order and values.
     */
    protected boolean areAllSeriesXaxisValuesSame()
    {
        return abstractAreAllSeriesXaxisValuesSame();
    }

    protected JTable getTable()
    {
        return _table;
    }

    /**
     * Used within {@link DefaultChartEngineCellRenderer}.
     * 
     * @param modelRow The row in terms of the model... not the viewed row.
     * @param modelColumn The column in terms of the model... not the viewed column.
     * @return True if the the value for the provided row and column is within the axis limits of the current chart.
     *         False if it is NOT within the limits.
     */
    protected boolean isWithinAxisLimit(final int modelRow, final int modelColumn)
    {
        int seriesIndex = computeSeriesIndex(modelColumn);

        //If the column is 0 then it is always a domain column, but...
        if(modelColumn == 0)
        {
            //If all series share the same domain column and the number of those series exceeds 1, then the domain
            //column is treated independently of the series.
            if((areAllSeriesXaxisValuesSame()) && (getCurrentDataSource().getNumberOfSeries() > 1))
            {
                Object rawValue = getRawValueAt(modelRow, modelColumn);
                if(rawValue == null)
                {
                    return false;
                }
                if(rawValue instanceof Date)
                {
                    rawValue = ((Date)rawValue).getTime();
                }
                //XXX For WRES:
                //Translate category to number. There has to be a better way to handle this!
                //A string raw value is a category and must be translated to a number by the associated axis.
                if (rawValue instanceof String) //Translate cateogry to number.
                {
                    rawValue = ((NumberAxisOverride)getDomainAxis()).getXValueForCategory((String)rawValue);
                }
                if(!getDomainAxis().getRange().contains(((Number)rawValue).doubleValue()))
                {
                    return false;
                }
            }
            //Otherwise, its treated identically to the series, but the seriesIndex must be forced to be 0.
            else
            {
                seriesIndex = 0;
            }
        }

        if(seriesIndex >= 0)
        {
            //Gather the information needed.
            final ValueAxis rangeAxis = getRangeAxis();
            if(rangeAxis == null)
            {
                return false;
            }

            //If either the domain value or the range value are outside their limits, fade out the row.
            Object domainRawValue = getRawValueAt(modelRow, computeDomainColumn(seriesIndex));
            if(domainRawValue == null)
            {
                return false;
            }
            if(domainRawValue instanceof Date)
            {
                domainRawValue = ((Date)domainRawValue).getTime();
            }
            //XXX For WRES:
            //Translate category to number.
            //A string raw value is a category and must be translated to a number by the associated axis.
            if (domainRawValue instanceof String) 
            {
                domainRawValue = ((NumberAxisOverride)getDomainAxis()).getXValueForCategory((String)domainRawValue);
            }
            if(!_chart.getXYPlot().getDomainAxis().getRange().contains(((Number)domainRawValue).doubleValue()))
            {
                return false;
            }

            final Object rangeRawValue = getRawValueAt(modelRow, computeRangeColumn(seriesIndex));
            if(rangeRawValue == null)
            {
                return false;
            }
            if(!rangeAxis.getRange().contains(((Number)rangeRawValue).doubleValue()))
            {
                return false;
            }
        }
        return true;
    }

    /**
     * Override as needed. For example, if a subclass reorders the series sorting them by forecast time, overriding may
     * be necessary (see {@link DomainSharingTimeSeriesChartEngineTableModel}).<br>
     * <br>
     * This calls {@link #abstractGetRawValueAt(int, int)} by default.
     */
    @Override
    public Object getRawValueAt(final int modelRow, final int modelColumn)
    {
        return abstractGetRawValueAt(modelRow, modelColumn);
    }

    /**
     * See interface javadoc. You will need to override this method if the series are displayed in an order different
     * from the datasource.
     * 
     * @return The series index relative to the current data source, {@link #getCurrentDataSource()}, given a column. A
     *         -1 is returned if column 0 is provided and all series share the same axis.
     */
    @Override
    public int computeSeriesIndex(final int modelColumn)
    {
        if(getCurrentDataSource().getNumberOfSeries() <= 0)
        {
            return -1;
        }
        if(modelColumn == 0)
        {
            if(areAllSeriesXaxisValuesSame() && (getCurrentDataSource().getNumberOfSeries() > 1))
            {
                return -1;
            }
            else
            {
                return 0; //First series.
            }
        }
        else
        {
            if(areAllSeriesXaxisValuesSame())
            {
                return modelColumn - 1;
            }
            return modelColumn / 2;
        }
    }

    /**
     * Only include a row header if {@link #areAllSeriesXaxisValuesSame()} is true; in that case 0 is returned. -1 is
     * returned otherwise.
     */
    @Override
    public int getRowHeaderColumn()
    {
        if(areAllSeriesXaxisValuesSame())
        {
            return 0;
        }
        return -1;
    }

    /**
     * Calls {@link #setChartEngineWithoutFiringEvent(ChartEngine)} followed by {@link #fireTableStructureChanged()}
     * after setting.
     * 
     * @param engine The new chart engine, which means new data to display.
     */
    @Override
    public void setChartEngine(final ChartEngine engine)
    {
        setChartEngineWithoutFiringEvent(engine);
        this.fireTableStructureChanged();
    }

    /**
     * Checks all columns to see if any one of them is within the axis limits.
     */
    @Override
    public boolean isRowVisibleInChart(final int modelRow)
    {
        for(int seriesIndex = 0; seriesIndex < getCurrentDataSource().getNumberOfSeries(); seriesIndex++)
        {
            if(isWithinAxisLimit(modelRow, computeRangeColumn(seriesIndex)))
            {
                return true;
            }
        }
        return false;
    }

    /**
     * Checks all rows to see if any one of them is within the axis limits.
     */
    @Override
    public boolean isColumnVisibleInChart(final int modelCol)
    {
        //Don't count the overall domain column if the domain column is shared
        if(areAllSeriesXaxisValuesSame() && (modelCol == 0))
        {
            return false;
        }

        //Find the series corresponding to the col.
        final int seriesIndex = computeSeriesIndex(modelCol);
        for(int modelRow = 0; modelRow < getRowCount(); modelRow++)
        {
            if(isWithinAxisLimit(modelRow, computeRangeColumn(seriesIndex)))
            {
                return true;
            }
        }
        return false;
    }

    /**
     * You will need to override this if the table displays values that do not exactly match underlying sources.
     */
    @Override
    public int computeRowForItem(final int seriesIndex, final int seriesItemNumber)
    {
        if((seriesItemNumber < 0) || (seriesItemNumber >= getCurrentDataSource().getSeriesValueCount(seriesIndex)))
        {
            return -1;
        }
        return seriesItemNumber;
    }

    /**
     * You will need to override this method if the series are displayed in an order different from the datasource.
     * 
     * @return The column index corresponding to the domain variable of a series.
     */
    @Override
    public int computeDomainColumn(final int seriesIndex)
    {
        if(areAllSeriesXaxisValuesSame())
        {
            return 0;
        }
        return seriesIndex * 2;
    }

    /**
     * You will need to override this method if the series are displayed in an order different from the datasource.
     * 
     * @return The column index corresponding to the range variable of a series.
     */
    @Override
    public int computeRangeColumn(final int seriesIndex)
    {
        if(areAllSeriesXaxisValuesSame())
        {
            return seriesIndex + 1;
        }
        return computeDomainColumn(seriesIndex) + 1;
    }

    /**
     * Override as needed.
     * 
     * @return If false, then the data source choice box will not be included in the GUI.
     */
    @Override
    public boolean allowForDataSourceSwitching()
    {
        return true;
    }

    /**
     * @return The currently displayed data source index.
     */
    @Override
    public int getDataSourceIndex()
    {
        return _dataSourceIndex;
    }

    /**
     * Calls {@link #fireTableStructureChanged()} after setting.
     */
    @Override
    public void setDataSourceIndex(final int dataSourceIndex)
    {
        setDataSourceIndexWithoutFiringEvent(dataSourceIndex);
        this.fireTableStructureChanged();
    }

    /**
     * Sets the chart for reference, but does nothing else.
     */
    @Override
    public void setChart(final JFreeChart chart)
    {
        _chart = chart;
    }

    @Override
    public void addTable(final JTable table)
    {
        if(!(table instanceof EventPostingCellSelectableTable))
        {
            throw new IllegalArgumentException("Incompatible table provided to AbstractChartEngineTableModel; table must be a ChartEngineTable.");
        }

        _table = (EventPostingCellSelectableTable)table;
    }

    @Override
    public void removeTable(final JTable table)
    {
        _table = null;
    }

    @Override
    public Class<?> getColumnClass(final int col)
    {
        if(col == 0)
        {
            if(ChartConstants.isAxisTypeTime(getCurrentDataSource().getXAxisType()))
            {
                return Date.class;
            }
        }
        else if(!areAllSeriesXaxisValuesSame() && (col % 2 == 0))
        {
            if(ChartConstants.isAxisTypeTime(getCurrentDataSource().getXAxisType()))
            {
                return Date.class;
            }
        }
        return Double.class;
    }

    @Override
    public int getColumnCount()
    {
        if(getCurrentDataSource().getNumberOfSeries() <= 0)
        {
            return 0;
        }
        if(this.areAllSeriesXaxisValuesSame())
        {
            return this.getCurrentDataSource().getNumberOfSeries() + 1;
        }
        else
        {
            return this.getCurrentDataSource().getNumberOfSeries() * 2;
        }
    }

    /**
     * This calls {@link #abstractGetColumnName(int)} by default.
     */
    @Override
    public String getColumnName(final int col)
    {
        return abstractGetColumnName(col);
    }

    /**
     * This calls {@link #abstractGetRowCount()} by default.
     */
    @Override
    public int getRowCount()
    {
        return abstractGetRowCount();
    }

    /**
     * It should not be necessary to override this method. Instead, look at {@link #getRawValueAt(int, int)}. This
     * method must always return a {@link String}.
     */
    @Override
    public Object getValueAt(final int rowIndex, final int columnIndex)
    {
        final Object value = getRawValueAt(rowIndex, columnIndex);
        if(value instanceof Float) //Time series data may be float values.  We only display double values for simplicity...
        {
            return ((Float)value).doubleValue();
        }
        return value;
    }

    @Override
    public boolean applyAfterAllOtherRenderers()
    {
        return false;
    }

    /**
     * @return Default return is the within-axis-limits mark which has color {@link #WITHIN_LIMITS_MARK_COLOR} and
     *         priority 0 (background), and a mark for the selected cell that uses {@link #DEFAULT_SELECTED_CELL_COLOR}
     *         mixed with white with a priority of .
     */
    @Override
    public Dyad<Color, Integer> getRowMarkColor(final int modelRow)
    {
        //Handle selected cell, first
        final int selectedRow = getTable().getSelectedRow();
        if((selectedRow >= 0) && (modelRow == getTable().convertRowIndexToModel(selectedRow)))
        {
            return new Dyad<Color, Integer>(ColorTools.mixColors(DEFAULT_SELECTED_CELL_COLOR, Color.white),
                                            MarkPanel.TIER_5_PRIORITY);
        }

        //Axis limit check
        for(int col = 0; col < getColumnCount(); col++)
        {
            if(isWithinAxisLimit(modelRow, col))
            {
                return new Dyad<Color, Integer>(WITHIN_LIMITS_MARK_COLOR, MarkPanel.TIER_0_PRIORITY);
            }
        }

        return null;
    }

    @Override
    public Dyad<Color, Integer> getColumnMarkColor(final int modelCol)
    {
        //Handle selected cell, first.
        final int selectedCol = getTable().getSelectedColumn();
        if((selectedCol >= 0) && (modelCol == getTable().convertColumnIndexToModel(selectedCol)))
        {
            return new Dyad<Color, Integer>(ColorTools.mixColors(DEFAULT_SELECTED_CELL_COLOR, Color.white),
                                            MarkPanel.TIER_5_PRIORITY);
        }

        //If any value is within axis limit for column.
        for(int modelRow = 0; modelRow < getRowCount(); modelRow++)
        {
            if(isWithinAxisLimit(modelRow, modelCol))
            {
                return new Dyad<Color, Integer>(WITHIN_LIMITS_MARK_COLOR, MarkPanel.TIER_0_PRIORITY);
            }
        }
        return null;
    }

    @Override
    public Integer getMinWidth(final int column)
    {
        return getPreferredWidth(column);
    }

    /**
     * It will use a {@link JLabel} to approximate the required column widths.
     */
    @Override
    public Integer getPreferredWidth(final int column)
    {
        try
        {
            final Class rawDataClass = getColumnClass(column);

            final JLabel testLabel = new JLabel();
            if(Date.class.isAssignableFrom(rawDataClass))
            {
                testLabel.setText("yyyy-mm-dd hh:mm:ss");
            }
            else if(Number.class.isAssignableFrom(rawDataClass))
            {
                testLabel.setText("0000.00000");
            }
            else
            {
                testLabel.setText("xxxxxxxxxxxxx"); //Generic string used for sizing
            }
            return (int)testLabel.getPreferredSize().getWidth() + 10;
        }
        catch(final Exception e)
        {
            return -1;
        }
    }

    @Override
    public TableCellRenderer wrapRenderer(final TableCellRenderer baseRenderer)
    {
        //Compute the number of columns for each color.
        int numberOfColumnsPerColor = 1;
        if(!areAllSeriesXaxisValuesSame())
        {
            numberOfColumnsPerColor = 2; //Each series is a pair of x,y coords.  There is never more than 2 columns per series.
        }

        return new DefaultChartEngineCellRenderer(baseRenderer, numberOfColumnsPerColor);
    }

    /**
     * Renderer is a {@link PredicateTableCellRenderer} wrapping an {@link AlternatingColumnColorTableCellRenderer}
     * which, itself, wraps the base renderer. It will fade the foreground of table cells if the value at that row and
     * column is not within the axis limits. It will use a foreground of black if it is. It also creates table
     * crosshairs for selected cell, coloring the cells in the same row and column based on
     * {@link JTable#getSelectionBackground()}.
     * 
     * @author hankherr
     */
    protected class DefaultChartEngineCellRenderer extends PredicateTableCellRenderer
    {
        public DefaultChartEngineCellRenderer(final TableCellRenderer baseRenderer, final int numberOfColumnsPerColor)
        {
            super(new AlternatingColumnColorTableCellRenderer(baseRenderer, 0, numberOfColumnsPerColor, new Color[]{
                new Color(232, 232, 232), Color.white}));

            //Faded foreground for cells not within axis limits.
            addEffect(new Predicate<Dyad<Integer, Integer>>()
            {
                public boolean apply(final Dyad<Integer, Integer> input)
                {
                    final int modelRow = getTable().convertRowIndexToModel(input.getFirst());
                    final int modelCol = getTable().convertColumnIndexToModel(input.getSecond());
                    return !isWithinAxisLimit(modelRow, modelCol);
                }
            }, new PredicateTableCellRenderer.ForegroundColorEffect(Color.WHITE, true), null);

            //Faded selection color predicate for selected row and column.
            final Color fadedSelectionColor = ColorTools.mixColors(DEFAULT_SELECTED_CELL_COLOR, Color.WHITE);
            addEffect(new Predicate<Dyad<Integer, Integer>>()
            {
                public boolean apply(final Dyad<Integer, Integer> input)
                {
                    //Do not change the color for the selected cell.  Only the selected row and column cells that are not the selected cell.
                    if(getTable().isCellSelected(input.getFirst(), input.getSecond()))
                    {
                        return false;
                    }
                    return (getTable().isRowSelected(input.getFirst()) || (getTable().isColumnSelected(input.getSecond())));
                }
            },
                      PredicateTableCellRenderer.createBackgroundEffect(fadedSelectionColor, false),
                      null);
        }

        @Override
        public Component getTableCellRendererComponent(final JTable table,
                                                       final Object value,
                                                       final boolean isSelected,
                                                       final boolean hasFocus,
                                                       final int row,
                                                       final int column)
        {
            return super.getTableCellRendererComponent(table, value, isSelected, hasFocus, row, column);
        }

    }
}