package ohd.hseb.charter.panel;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.HierarchyEvent;
import java.awt.event.HierarchyListener;
import java.util.Vector;

import javax.swing.BorderFactory;
import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.DefaultComboBoxModel;
import javax.swing.JComboBox;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JSplitPane;
import javax.swing.SwingUtilities;

import org.apache.commons.lang.ArrayUtils;
import org.jfree.chart.ChartPanel;
import org.jfree.chart.JFreeChart;

import com.google.common.base.Supplier;
import com.google.common.eventbus.EventBus;
import com.google.common.eventbus.Subscribe;

import ohd.hseb.charter.ChartEngine;
import ohd.hseb.charter.ChartEngineException;
import ohd.hseb.charter.ChartPanelTools;
import ohd.hseb.charter.datasource.XYChartDataSource;
import ohd.hseb.charter.datasource.XYChartDataSourceException;
import ohd.hseb.charter.panel.notices.ChangeChartEngineAndChartNotice;
import ohd.hseb.charter.panel.notices.ChartDataPointClickedNotice;
import ohd.hseb.charter.panel.notices.ChartEngineTableCellSelectedNotice;
import ohd.hseb.charter.panel.notices.ChartTableVisibilityChangedNotice;
import ohd.hseb.charter.panel.notices.DetachDataTableNotice;
import ohd.hseb.charter.panel.notices.ThresholdClickedNotice;
import ohd.hseb.charter.parameters.DataSourceDrawingParameters;
import ohd.hseb.hefs.utils.gui.components.DetachablePanel;
import ohd.hseb.hefs.utils.gui.jtable.EventPostingCellSelectableTable;
import ohd.hseb.hefs.utils.gui.jtable.MarkedTableScrollPanel;
import ohd.hseb.hefs.utils.gui.jtable.OutputDataToCSVFileMenuItem;
import ohd.hseb.hefs.utils.gui.jtable.RecomputeMarksNotice;
import ohd.hseb.hefs.utils.gui.jtable.models.RowColumnMarkingTableModel;
import ohd.hseb.hefs.utils.gui.tools.HSwingFactory;
import ohd.hseb.hefs.utils.gui.tools.SwingTools;

/**
 * This is the entry point for a panel that enables viewing a chart and data table. Underlying it is a
 * {@link ChartEngine} that must be passed to its constructor. After construction, the underlying {@link ChartEngine}
 * can be changed by calling {@link #setChartEngineAndChart(ChartEngine, JFreeChart, boolean)} (or one of its wrappers
 * {@link #setChartEngine(ChartEngine)} or {@link #setChartEngineAndChart(ChartEngine, JFreeChart)}), which reconstructs
 * {@link #_tableModel} (see below), updates {@link #_dataSourceChoiceBox}, and broadcasts the change to all components
 * within this panel by posting a {@link ChangeChartEngineAndChartNotice} notice to the {@link #_centralBus}.<br>
 * <br>
 * The chart is the primary panel displayed, and it is shown within a {@link ChartEngineChartPanel} (which displays an
 * {@link OHDFixedChartPanel} connected to a {@link ChartEngine} and including a show table menu item in its popup
 * menu). The attribute specifying the panel is {@link #_chartEnginePanel} and it is always displayed within this panel.<br>
 * <br>
 * The secondary panel within this is the {@link #_tablePanel}. It displays {@link #_table} within {@link #_scrollPanel}
 * , which is a {@link MarkedTableScrollPanel} ({@link #_scrollPanel}). The {@link MarkedTableScrollPanel} allows for
 * marking regions of the scrollbars as dictated by the {@link #_tableModel} if it implements the
 * {@link RowColumnMarkingTableModel} interface. Though the table is constructed when this is constructed, it is only
 * made visible by selecting the show table menu item from the popup menu for the {@link #_chartEnginePanel}, or by
 * calling {@link #setTableShowing(boolean)} (which triggers that menu item). The table when shown will be displayed by
 * default to the left of the {@link #_chartEnginePanel} within {@link #_splitPane}. The displayed {@link #_table} is an
 * {@link EventPostingCellSelectableTable} that displays data contained in the table model returned by
 * {@link #constructTableModel(ChartEngine)}.<br>
 * <br>
 * The {@link #_modelSupplier} can be set in the constructor in order to use custom table models (for example, those
 * that include different scroll bar marks), so long as the {@link Supplier} returns a {@link ChartEngineTableModel}
 * instance via its {@link Supplier#get()} method. If not set, the default model used is a
 * {@link DefaultChartEngineTableModel}, which provides selected cell and basic axis limits marking to the scroll panel.
 * If a different table model must be used, you should define the supplier appropriately and call
 * {@link ChartPanelTools#createPanelForChartAndEngine(JFreeChart, ChartEngine, boolean, Supplier)} in order to
 * construct an instance of this panel as it allows for specifying the table model's {@link Supplier}.<br>
 * <br>
 * The {@link #_scrollPanel} is displayed within a {@link DetachablePanel}, {@link #_tablePanel}. This allows for the
 * table to be detached from the split pane by right clicking to open up a popup menu with the detach/reattach menu
 * item. <br>
 * <br>
 * An {@link EventBus}, {@link #_centralBus}, is used for handling communication between different components, each of
 * which keeps a pointer to the {@link #_centralBus} internally for posting notices. <br>
 * <br>
 * See the TimeSeriesChartDiagnosticPanel in HEFS (MEFPPE) for an example of how to associate a
 * {@link CombinedDomainChartNavigationPanel} with this {@link ChartEngineChartAndTablePanel}. <br>
 * 
 * @author hankherr
 */
public class ChartEngineChartAndTablePanel extends JPanel implements ChartTableVisibilityChangedNotice.Subscriber,
DetachDataTableNotice.Subscriber, ChartDataPointClickedNotice.Subscriber, ThresholdClickedNotice.Subscriber
{
    private static final long serialVersionUID = 1L;
    private static final int MAXIMIUM_ITEM_LENGTH = 50;

    /**
     * Passed to {@link #_tablePanel}.
     */
    private static int SPLITPANE_DIVIDER_STANDARD_SIZE = 5;

    /**
     * The {@link ChartEngine} used to construct the chart displayed in {@link #_chartEnginePanel} and provide the data
     * to {@link #_tableModel}.
     */
    private ChartEngine _chartEngine = null;

    /**
     * The panel used for displaying chart constructed via {@link #_chartEngine}.
     */
    private final ChartEngineChartPanel _chartEnginePanel;

    /**
     * The model specifying data to display in {@link #_table}.
     */
    private ChartEngineTableModel _tableModel = null;

    /**
     * The data table.
     */
    private ChartEngineTable _table = null;

    /**
     * Panel to display {@link #_table}.
     */
    private DetachablePanel _tablePanel = null;

    /**
     * Displays choice of data sources.
     */
    private JComboBox _dataSourceChoiceBox = null;

    /**
     * Panel that displays the choice box. This panel can be removed from the interface by call
     * {@link #removeDataSourceChoiceBoxForTable()}. In that case, only one data source can be displayed in the table.
     */
    private JPanel _dataSourceComboBoxPanel = null;

    /**
     * The split pane for displaying {@link #_table} next to the {@link #_chartEnginePanel}.
     */
    private JSplitPane _splitPane = null;

    /**
     * The scroll panel within {@link #_tablePanel} for viewing {@link #_table}. It is tied to the {@link #_tableModel}
     * which must implement {@link RowColumnMarkingTableModel} in order for marks to displayed.
     */
    private MarkedTableScrollPanel _scrollPanel;

    /**
     * {@link Vector} index is the combo box index, while the {@link Integer} value is the data source index.
     */
    private Vector<Integer> _comboBoxItemIndexToDataSourceIndex = null;

    /**
     * The bus used by all subcomponents within this panel.
     */
    private final EventBus _centralBus = new EventBus();

    /**
     * Called to provide the {@link ChartEngineTableModel} as needed via a call to {@link #buildTableModel(JFreeChart)}.
     * If this must be overridden so that the default it used, the supplier to use must be made passed into the
     * constructor and this must be set before {@link #initialize(boolean)} is called.
     */
    private Supplier<ChartEngineTableModel> _modelSupplier = new Supplier<ChartEngineTableModel>()
    {
        @Override
        public ChartEngineTableModel get()
        {
            return new DefaultChartEngineTableModel(getChartEngine());
        }
    };

    /**
     * Sets {@link #_chartEnginePanel}, {@link #_chartEngine} (from whats in the panel), the {@link #_modelSupplier} (if
     * not null), and calls {@link #initialize(boolean)} with the specified boolean.
     * 
     * @param chartEnginePanel The panel for that engine, which may be constructed in multiple ways.
     * @param modelSupplier The {@link ChartEngineTableModel} {@link Supplier}, or null if the default supplier which
     *            uses {@link DefaultChartEngineTableModel} is to be used.
     * @param showTableInitially True to show the table initially, false otherwise.
     */
    private ChartEngineChartAndTablePanel(final ChartEngineChartPanel chartEnginePanel,
                                          final Supplier<ChartEngineTableModel> modelSupplier,
                                          final boolean showTableInitially)
    {
        _chartEnginePanel = chartEnginePanel;
        _chartEngine = chartEnginePanel.getChartEngine();
        if(modelSupplier != null)
        {
            _modelSupplier = modelSupplier;
        }
        initialize(showTableInitially);
    }

    /**
     * Calls {@link ChartEngineChartPanel#ChartEngineChartPanel(ChartEngine, JFreeChart)} and
     * {@link #ChartEngineChartAndTablePanel(ChartEngineChartPanel, Supplier, boolean)} with false passed in so that the
     * table does not show initially. The default {@link #_modelSupplier} will be used.
     */
    public ChartEngineChartAndTablePanel(final ChartEngine chartEngine, final JFreeChart chart)
    {
        this(new ChartEngineChartPanel(chartEngine, chart), null, false);
    }

    /**
     * Calls
     * {@link ChartEngineChartPanel#ChartEngineChartPanel(JFreeChart, boolean, boolean, boolean, boolean, boolean, ChartEngine)}
     * and passes that to {@link #ChartEngineChartAndTablePanel(ChartEngineChartPanel, Supplier, boolean)}.
     */
    public ChartEngineChartAndTablePanel(final ChartEngine chartEngine,
                                         final JFreeChart chart,
                                         final Supplier<ChartEngineTableModel> modelSupplier,
                                         final boolean showTableInitially,
                                         final boolean properties,
                                         final boolean save,
                                         final boolean print,
                                         final boolean zoom,
                                         final boolean tooltips)
    {
        this(new ChartEngineChartPanel(chart, properties, save, print, zoom, tooltips, chartEngine),
             modelSupplier,
             showTableInitially);
    }

    /**
     * Calls
     * {@link ChartEngineChartPanel#ChartEngineChartPanel(JFreeChart, int, int, int, int, int, int, boolean, boolean, boolean, boolean, boolean, boolean, ChartEngine)}
     * and passes that into {@link #ChartEngineChartAndTablePanel(ChartEngineChartPanel, Supplier, boolean)}.
     */
    public ChartEngineChartAndTablePanel(final ChartEngine chartEngine,
                                         final JFreeChart chart,
                                         final Supplier<ChartEngineTableModel> modelSupplier,
                                         final boolean showTableInitially,
                                         final int width,
                                         final int height,
                                         final int minimumDrawWidth,
                                         final int minimumDrawHeight,
                                         final int maximumDrawWidth,
                                         final int maximumDrawHeight,
                                         final boolean useBuffer,
                                         final boolean properties,
                                         final boolean save,
                                         final boolean print,
                                         final boolean zoom,
                                         final boolean tooltips)
    {
        this(new ChartEngineChartPanel(chart,
                                       width,
                                       height,
                                       minimumDrawWidth,
                                       minimumDrawHeight,
                                       maximumDrawWidth,
                                       maximumDrawHeight,
                                       useBuffer,
                                       properties,
                                       save,
                                       print,
                                       zoom,
                                       tooltips,
                                       chartEngine), modelSupplier, showTableInitially);
    }

    /**
     * Calls {@link ChartEngineChartPanel#ChartEngineChartPanel(ChartEngine, JFreeChart)} and
     * {@link #ChartEngineChartAndTablePanel(ChartEngineChartPanel, Supplier, boolean)} with false passed in so that the
     * table does not show initially.
     */
    public ChartEngineChartAndTablePanel(final ChartEngine chartEngine,
                                         final JFreeChart chart,
                                         final Supplier<ChartEngineTableModel> modelSupplier)
    {
        this(new ChartEngineChartPanel(chartEngine, chart), modelSupplier, false);
    }

    /**
     * Calls
     * {@link ChartEngineChartPanel#ChartEngineChartPanel(JFreeChart, boolean, boolean, boolean, boolean, boolean, ChartEngine)}
     * and {@link #ChartEngineChartAndTablePanel(ChartEngine, JFreeChart, Supplier)} so that the default supplier is
     * used.
     */
    public ChartEngineChartAndTablePanel(final ChartEngine chartEngine,
                                         final JFreeChart chart,
                                         final boolean showTableInitially,
                                         final boolean properties,
                                         final boolean save,
                                         final boolean print,
                                         final boolean zoom,
                                         final boolean tooltips)
    {
        this(new ChartEngineChartPanel(chart, properties, save, print, zoom, tooltips, chartEngine),
             null,
             showTableInitially);
    }

    /**
     * Calls
     * {@link ChartEngineChartPanel#ChartEngineChartPanel(JFreeChart, int, int, int, int, int, int, boolean, boolean, boolean, boolean, boolean, boolean, ChartEngine)}
     * and {@link #ChartEngineChartAndTablePanel(ChartEngine, JFreeChart, Supplier)} so that the default supplier is
     * used.
     */
    public ChartEngineChartAndTablePanel(final ChartEngine chartEngine,
                                         final JFreeChart chart,
                                         final boolean showTableInitially,
                                         final int width,
                                         final int height,
                                         final int minimumDrawWidth,
                                         final int minimumDrawHeight,
                                         final int maximumDrawWidth,
                                         final int maximumDrawHeight,
                                         final boolean useBuffer,
                                         final boolean properties,
                                         final boolean save,
                                         final boolean print,
                                         final boolean zoom,
                                         final boolean tooltips)
    {
        this(new ChartEngineChartPanel(chart,
                                       width,
                                       height,
                                       minimumDrawWidth,
                                       minimumDrawHeight,
                                       maximumDrawWidth,
                                       maximumDrawHeight,
                                       useBuffer,
                                       properties,
                                       save,
                                       print,
                                       zoom,
                                       tooltips,
                                       chartEngine), null, showTableInitially);
    }

    /**
     * Creates the table panel (registering it with the {@link #_centralBus}), registers this and the
     * {@link #_chartEnginePanel} with the {@link #_centralBus}, and makes the table showing according to the parameter.
     */
    private void initialize(final boolean showTableInitially)
    {
        createDisplay();

        _centralBus.register(this);
        _chartEnginePanel.setCentralBus(_centralBus);

        _chartEnginePanel.setTableShowing(showTableInitially);

        //Make the table panel not visible if either this panel's parent
        //has been set to null or this panel is not visible.
        this.addHierarchyListener(new HierarchyListener()
        {
            @Override
            public void hierarchyChanged(final HierarchyEvent e)
            {
                if(e.getChangeFlags() == HierarchyEvent.PARENT_CHANGED)
                {
                    //If the parent is not define or this panel is not showing, then if the table is detached,
                    //make sure it is not showing.  If the table is attached, then change nothing, because this
                    //not showing will also apply to the table.
                    if((getParent() == null) || (!isShowing()))
                    {
                        if(_tablePanel.isDetached())
                        {
                            _tablePanel.setVisible(false);
                        }
                    }
                    //Otherwise, the parent exists and this panel is showing, so the table must be showing.
                    else
                    {
                        _tablePanel.setVisible(true);
                    }
                }
                else if(e.getChangeFlags() == HierarchyEvent.SHOWING_CHANGED)
                {
                    //If this is showing, then make sure the table panel is showing and brought to front.
                    if(isShowing())
                    {
                        _tablePanel.setVisible(true);
                        if(_tablePanel.isDetached())
                        {
                            _tablePanel.bringToFront();
                        }
                    }
                }
            }
        });
    }

    /**
     * Method creates the display. It calls {@link #createTablePanel()} to create {@link #_table} given the model
     * constructed by {@link #constructTableModel(ChartEngine)}. It also creates the {@link #_scrollPanel} to view the
     * table and puts it within {@link #_tablePanel}, which is detachable. By default, the table is not visible.
     */
    private void createDisplay()
    {
        //Construct the split pane
        _splitPane = HSwingFactory.createJSPlitPane(JSplitPane.HORIZONTAL_SPLIT, new JPanel(), //filler until the table can be built
                                                    _chartEnginePanel,
                                                    false);
        _splitPane.setTopComponent(null);
        createTablePanel();
        _tablePanel.setVisible(false);
        _splitPane.setDividerSize(0);

        //Construct the table panel.
        _splitPane.setDividerLocation(0.25);
        if(_splitPane.getDividerLocation() < 200)
        {
            _splitPane.setDividerLocation(200);
        }
        add(_splitPane, BorderLayout.CENTER);
    }

    /**
     * Calls {@link #_modelSupplier} {@link Supplier#get()} method to get a {@link ChartEngineTableModel}, and then sets
     * its chart engine to be {@link #_chartEngine} and chart to be the passed in chart.
     * 
     * @param chart The displayed chart corresponding to the table model.
     */
    private void buildTableModel(final JFreeChart chart)
    {
        _tableModel = _modelSupplier.get();
        _tableModel.setChartEngine(_chartEngine);
        _tableModel.setChart(chart);
    }

    /**
     * This method assumes that the checked state of {@link #_showTableCheckBoxMenuItem} has already been set. This
     * method will not change that state. This is only called during construction. After that, call
     * {@link #setChartEngine(ChartEngine)} or {@link #setChartEngineAndChart(ChartEngine, JFreeChart)} to update the
     * displayed table.
     */
    private void createTablePanel()
    {
        _tablePanel = new DetachablePanel("Chart Data Table", _splitPane, true, SPLITPANE_DIVIDER_STANDARD_SIZE);
        _tablePanel.setLayout(new BorderLayout());
        _tablePanel.setMinimumSize(new Dimension(100, 100));

        //Construct a table model
        buildTableModel(_chartEnginePanel.getCurrentChart());

        //Only include the series choice box if data source switching is allowed by the model.  Otherwise,
        //the model will tell the table what to display with no user control.
        _dataSourceChoiceBox = null;
        if(getChartEngineTableModel().allowForDataSourceSwitching())
        {
            //Combo box and its panel.  The action listener reacts whenever the combo box is clicked, even if the
            //selection is not changed.
            final JLabel choiceBoxLabel = new JLabel("Select Data:");
            _dataSourceChoiceBox = HSwingFactory.createJComboBox(constructDataSourceChoiceListItems(), null);
            _dataSourceChoiceBox.addActionListener(new ActionListener()
            {
                @Override
                public void actionPerformed(final ActionEvent e)
                {
                    if(getDataSourceIndexFromComboBoxIndex() != getChartEngineTableModel().getDataSourceIndex())
                    {
                        getChartEngineTableModel().setDataSourceIndex(getDataSourceIndexFromComboBoxIndex());
                        _centralBus.post(new ChartEngineTableCellSelectedNotice(this));
                    }
                }
            });
            _dataSourceChoiceBox.setPrototypeDisplayValue("");

            _dataSourceComboBoxPanel = new JPanel();
            _dataSourceComboBoxPanel.setLayout(new BoxLayout(_dataSourceComboBoxPanel, BoxLayout.X_AXIS));
            _dataSourceComboBoxPanel.setBorder(BorderFactory.createEmptyBorder(10, 10, 10, 10));
            _dataSourceComboBoxPanel.add(choiceBoxLabel);
            _dataSourceComboBoxPanel.add(Box.createRigidArea(new Dimension(10, 10)));
            _dataSourceComboBoxPanel.add(_dataSourceChoiceBox);

            _tablePanel.add(_dataSourceComboBoxPanel, BorderLayout.NORTH);
        }

        //When the table is constructed, its valueChanged method is overridden
        //so that the column and row are repainted as well.  This is because cell selection is allowed, and the 
        //AbstractChartEngineTableModel colors the column and row of the selected cell to faded blue. 
        //NOTE that GenericTable only allows for single selection, so that only the rectangles for one row and
        //column need to be made dirty.
        _table = new ChartEngineTable(_tableModel);
        _table.setCentralBus(_centralBus);

        //Setup the scroll panel.  Note that the recomputeMarks() method inside this panel is called when the table model is
        //set for the table, so there is no need to call it here (as part of setting it up).
        _scrollPanel = new MarkedTableScrollPanel(_table, _centralBus);
        _scrollPanel.setPreferredSize(new Dimension(400, 200));

        //This makes sure the background color is consistent between header and table.
        _scrollPanel.getScrollPane().getViewport().setBackground(ChartEngineTable.TABLE_BG_COLOR);
        _tablePanel.add(_scrollPanel, BorderLayout.CENTER);

        //Setup the table's popup menu.
        //NOTE: On a mac, a ctrl-click opens the popup menu.  I don't believe this is the case for Linux or PC, so don't worry about i
        //All I'm doing  here is concatenating the array returned by _scrollPanel.constructTablePanelAdditionalComponents() with an instance of 
        //  OutputDataToCSVFileMenuItem and turning the array into a component array.  I can't find a convenient way to do that using Java 
        //  utilities, though I'm sure there must be one.
        final Object[] compObjs = ArrayUtils.add(_scrollPanel.constructTablePanelAdditionalComponents(),
                                                 new OutputDataToCSVFileMenuItem(_table,
                                                                                 SwingTools.getGlobalDialogParent(_table)));
        final Component[] comps = new Component[compObjs.length];
        for(int i = 0; i < compObjs.length; i++)
        {
            comps[i] = (Component)compObjs[i];
        }
        _tablePanel.setComponentPopupMenu(_tablePanel.constructPopupMenu(comps));

        //The overall panel is a JSplitPane.
        setLayout(new BorderLayout());

        //Initialize the source selection.  This must be done manually because it is possible for the _seriesChoiceBox contents to NOT
        //match the data sources, since empty data sources are not included.
        if((_dataSourceChoiceBox != null) && (_dataSourceChoiceBox.getModel().getSize() > 0))
        {
            //Select the first item in the list.  Since that will not trigger an event, call the _tableModel's 
            //setDataSourceIndex to manually force it to react.
            _dataSourceChoiceBox.setSelectedIndex(0);
            getChartEngineTableModel().setDataSourceIndex(getDataSourceIndexFromComboBoxIndex());
        }
    }

    /**
     * @return {@link Vector} of {@link String}s to display in the chart series selection combo box. NOTE that this may
     *         not yield one string per data source: empty sources are not included in the selectable list! The name
     *         displayed is either that specified by the {@link XYChartDataSource#getChartTableSourceName()} or, if that
     *         is null, then it is a default constructed using the descriptive name of the data source genenator
     *         returned by {@link XYChartDataSource#getGenerator()} and the legend names of all series in the data
     *         source (comma delimited list).
     */
    private Vector<String> constructDataSourceChoiceListItems()
    {
        final Vector<String> listItems = new Vector<String>();
        _comboBoxItemIndexToDataSourceIndex = new Vector<Integer>();
        Integer count = 0;
        //Loop through the chart engine data sources...
        for(final XYChartDataSource source: _chartEngine.getDataSources())
        {
            //If the source yielded series...
            if(source.getNumberOfSeries() > 0)
            {
                if(source.getChartTableSourceName() != null)
                {
                    listItems.add(source.getChartTableSourceName());
                }
                else
                {
                    //The item begins with the generator's descriptive name.
                    String item = "Source " + source.getDataSourceOrderIndex() + ": ";
                    if(source.getGenerator() != null)
                    {
                        item = source.getGenerator().getDescriptiveName() + ": ";
                    }

                    //Add the name in legend for each series associated with source.  Add the first one and then append.
                    final DataSourceDrawingParameters drawingParam = source.getDefaultFullySpecifiedDataSourceDrawingParameters();
                    item += drawingParam.getSeriesDrawingParametersForSeriesIndex(0).getArgumentReplacedNameInLegend();
                    int seriesIndex = 1;
                    while(seriesIndex < source.getNumberOfSeries() && item.length() < MAXIMIUM_ITEM_LENGTH)
                    {
                        final String s = drawingParam.getSeriesDrawingParametersForSeriesIndex(seriesIndex)
                                                     .getArgumentReplacedNameInLegend();
                        if(!s.isEmpty())
                        {
                            item += new String("; ") + s;
                        }
                        seriesIndex++;
                    }

                    //Add ... if needed.
                    if((seriesIndex < source.getNumberOfSeries() - 1) && (item.length() >= MAXIMIUM_ITEM_LENGTH))
                    {
                        item += "; ... "
                            + drawingParam.getSeriesDrawingParametersForSeriesIndex(source.getNumberOfSeries() - 1)
                                          .getArgumentReplacedNameInLegend();
                    }

                    String usedItem = item;
                    int index = 1;
                    while(listItems.contains(usedItem))
                    {
                        usedItem = item + " (" + index + ")";
                        index++;
                    }
                    listItems.add(usedItem);
                }
                _comboBoxItemIndexToDataSourceIndex.add(count);
            }

            count++;
        }
        return listItems;

    }

    /**
     * Equivalent to providing a model that has its {@link AbstractChartEngineTableModel#allowForDataSourceSwitching()}
     * method return false. This removes the data source combo box.
     */
    public void removeDataSourceChoiceBoxForTable()
    {
        _dataSourceComboBoxPanel.setVisible(false);
    }

    /**
     * Registers the provided {@link Object} with the {@link #_centralBus} so that notice postings can be heard by that
     * {@link Object}.
     */
    public void registerWithCentralBus(final Object o)
    {
        _centralBus.register(o);
    }

    /**
     * @return The {@link ChartEngine} underlying this panel.
     */
    public ChartEngine getChartEngine()
    {
        return _chartEngine;
    }

    /**
     * @return The {@link OHDFixedChartPanel} that displays the chart.
     */
    public OHDFixedChartPanel getChartPanel()
    {
        return _chartEnginePanel.getChartPanel();
    }

    /**
     * Set the panel to display the chart and update the display. This calls {@link #_chartEnginePanel} method
     * {@link ChartEngineChartPanel#setChartPanel(OHDFixedChartPanel)}.
     */
    public void setChartPanel(final OHDFixedChartPanel chartPanel)
    {
        _chartEnginePanel.setChartPanel(chartPanel);
        SwingTools.forceComponentRedraw(this);
    }

    /**
     * @return The table model underlying the table.
     */
    public ChartEngineTableModel getTableModel()
    {
        return _tableModel;
    }

    /**
     * @return The table.
     */
    public ChartEngineTable getTable()
    {
        return _table;
    }

    /**
     * Convenience method.
     * 
     * @return Casts the table model into a {@link ChartEngineTableModel} for return. This is a convenience method.
     */
    public ChartEngineTableModel getChartEngineTableModel()
    {
        return _tableModel;
    }

    /**
     * Sets the table model but does NOT update the display.
     */
    public void setTableModel(final AbstractChartEngineTableModel tableModel)
    {
        this._tableModel = tableModel;
        _table.setModel(_tableModel);
    }

    /**
     * Calls {@link #setChartEngineAndChart(ChartEngine, JFreeChart)}, calling {@link ChartEngine#buildChart()} to
     * acquire the chart.
     * 
     * @param chartEngine The {@link ChartEngine} used to draw the chart and populate the table.
     * @throws ChartEngineException Generated by {@link ChartEngine#buildChart()}.
     * @throws XYChartDataSourceException Generated by {@link ChartEngine#buildChart()}.
     */
    public void setChartEngine(final ChartEngine chartEngine) throws ChartEngineException, XYChartDataSourceException
    {
        setChartEngineAndChart(chartEngine, _chartEngine.buildChart());

    }

    /**
     * Calls {@link #setChartEngineAndChart(ChartEngine, JFreeChart, boolean)} passing in false so that the selected
     * source is not maintained.
     */
    public void setChartEngineAndChart(final ChartEngine chartEngine, final JFreeChart chart)
    {
        setChartEngineAndChart(chartEngine, chart, false);
    }

    /**
     * Sets {@link #_chartEngine}. Then for {@link #_chartPanel}, it calls {@link ChartPanel#setChart(JFreeChart)}. That
     * is followed by reconstructing the model for {@link #_dataSourceChoiceBox}, calling
     * {@link #constructTableModel(ChartEngine)}, and setting the model underlying {@link #_table}.
     * 
     * @param chartEngine The new engine.
     * @param chart The new chart to use, which better be properly consistent with the engine or the table won't display
     *            the right numbers.
     * @param maintainSelectedSource If true, this will attempt to maintain the selected source if it exists in the new
     *            choice box.
     */
    public void setChartEngineAndChart(final ChartEngine chartEngine,
                                       final JFreeChart chart,
                                       final boolean maintainSelectedSource)
    {
        _chartEngine = chartEngine;
        final Object item = _dataSourceChoiceBox.getSelectedItem();
        _dataSourceChoiceBox.setModel(new DefaultComboBoxModel(constructDataSourceChoiceListItems()));
        buildTableModel(chart);

        //There are three times below where the marks are recomputed.  I want to reduce it to one.  So, ignore them...
        _scrollPanel.setIgnoreRequestsToRecomputeMarks(true);

        _table.setModel(_tableModel); //triggers a mark recompute...

        //Initialize the source selection.  This must be done manually because it is possible for the _seriesChoiceBox contents to NOT
        //match the data sources, since empty data sources are not included.
        if((_dataSourceChoiceBox != null) && (_dataSourceChoiceBox.getModel().getSize() > 0))
        {
            //Select the first item in the list.  Since that will not trigger an event, call the _tableModel's 
            //setDataSourceIndex to manually force it to react.
            if(maintainSelectedSource)
            {
                _dataSourceChoiceBox.setSelectedItem(item); //triggers a mark recompute
            }
            else
            {
                _dataSourceChoiceBox.setSelectedIndex(0); //triggers a mark recompute
            }
            getChartEngineTableModel().setDataSourceIndex(getDataSourceIndexFromComboBoxIndex()); //triggers a mark recompute
        }

        //Manually recompute marks now one time.  Note that it 
        _scrollPanel.setIgnoreRequestsToRecomputeMarks(false);
        _scrollPanel.recomputeMarks();

        _centralBus.post(new ChangeChartEngineAndChartNotice(this, _chartEngine, chart));
    }

    /**
     * @return Result of call to {@link #_tablePanel} method {@link JPanel#isShowing()} method.
     */
    public boolean isTableShowing()
    {
        return _tablePanel.isShowing();
    }

    /**
     * Method passes through call to {@link #_chartEnginePanel}, calling
     * {@link ChartEngineChartPanel#setTableShowing(boolean)}.
     * 
     * @param b The visibility state that table is to take.
     */
    public void setTableShowing(final boolean b)
    {
        _chartEnginePanel.setTableShowing(b);
    }

    /**
     * @return Index of the data source selected from the {@link #_dataSourceChoiceBox}. It will make use of
     *         {@link #_comboBoxItemIndexToDataSourceIndex} unless it is null, in which case it uses the select index.
     */
    private int getDataSourceIndexFromComboBoxIndex()
    {
        if(_dataSourceChoiceBox == null)
        {
            throw new IllegalStateException("If _dataSourceChoiceBox is null, why are you calling this method?");
        }
        if(_comboBoxItemIndexToDataSourceIndex == null)
        {
            return _dataSourceChoiceBox.getSelectedIndex();
        }
        else
        {
            return _comboBoxItemIndexToDataSourceIndex.get(_dataSourceChoiceBox.getSelectedIndex()).intValue();
        }
    }

    /**
     * If the {@link #_tablePanel} is currently showing, it will be detached to a frame. This posts a
     * {@link DetachDataTableNotice} to the {@link #_centralBus} so that the standard subscriber method can react
     * forcing all detaches through the same mechanism.
     * 
     * @param source Pass in the caller to this method; i.e., 'detachDataTable(this)' from wherever it is called.
     */
    public void detachDataTable(final Object source)
    {
        _centralBus.post(new DetachDataTableNotice(source));
    }

    @Override
    public void setVisible(final boolean b)
    {
        super.setVisible(b);
        if(_tablePanel.isDetached())
        {
            _tablePanel.setVisible(b);
        }
    }

    @Override
    @Subscribe
    public void reactToChartTableVisibilityChangedNotice(final ChartTableVisibilityChangedNotice evt)
    {
        //User wants it visible...
        if(evt.isVisible())
        {
            _splitPane.setDividerSize(SPLITPANE_DIVIDER_STANDARD_SIZE);

            //If its attached, set the top component to _tablePanel and make it visible.
            if(!_tablePanel.isDetached())
            {
                _splitPane.setTopComponent(_tablePanel);
            }
        }
        //User wants it NOT visible...
        else
        {
            _splitPane.setDividerSize(0);

            //If its attached, then remove it from the split pane.
            if(!_tablePanel.isDetached())
            {
                _splitPane.setTopComponent(null);
            }
        }

        //Make the panel visible.
        _tablePanel.setVisible(evt.isVisible());
    }

    @Override
    @Subscribe
    public void reactToDetachDataTable(final DetachDataTableNotice evt)
    {
        if(_tablePanel.isShowing())
        {
            _tablePanel.detach();
        }
    }

    @Override
    @Subscribe
    public void reactToChartEntityClicked(final ChartDataPointClickedNotice evt)
    {
        //Select the data source from the combo box and update the model.
        _table.clearSelection();
        int i;
        for(i = 0; i < _comboBoxItemIndexToDataSourceIndex.size(); i++)
        {
            if(_chartEngine.getDataSources().get(_comboBoxItemIndexToDataSourceIndex.get(i)) == evt.getDataSource())
            {
                _dataSourceChoiceBox.setSelectedIndex(i);
                if(getDataSourceIndexFromComboBoxIndex() != getChartEngineTableModel().getDataSourceIndex())
                {
                    getChartEngineTableModel().setDataSourceIndex(getDataSourceIndexFromComboBoxIndex());
                }
                break;
            }
        }
        if(i == _comboBoxItemIndexToDataSourceIndex.size())
        {
            return; //Not Found!!!
        }

        //Identify the column and row of the cell to select.
        final int rangeModelColumn = _tableModel.computeRangeColumn(evt.getChartSeriesIndex());
        final int modelRow = _tableModel.computeRowForItem(evt.getChartSeriesIndex(), evt.getChartItemIndex());

        //Select the cell, but do this in an invokeLater so that the table can be drawn before selecting the cell.
        //This is necessary for the changeSelection to scroll to the proper rectangle when scrolling.
        SwingUtilities.invokeLater(new Runnable()
        {
            @Override
            public void run()
            {
                _table.changeSelection(_table.convertRowIndexToView(modelRow),
                                       _table.convertColumnIndexToView(rangeModelColumn),
                                       false,
                                       false);
            }
        });

        _centralBus.post(new RecomputeMarksNotice(this));
    }

    @Override
    @Subscribe
    public void reactToThresholdClicked(final ThresholdClickedNotice evt)
    {
        //TODO Is there anything to do here???
    }
}
