package ohd.hseb.charter.panel;

import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.geom.Point2D;
import java.util.Arrays;

import javax.swing.JCheckBoxMenuItem;
import javax.swing.JMenuItem;
import javax.swing.JRootPane;
import javax.swing.JSeparator;
import javax.swing.SwingConstants;

import org.jfree.chart.ChartMouseEvent;
import org.jfree.chart.ChartMouseListener;
import org.jfree.chart.JFreeChart;
import org.jfree.chart.entity.XYItemEntity;
import org.jfree.chart.plot.CombinedDomainXYPlot;
import org.jfree.chart.plot.Marker;
import org.jfree.chart.plot.XYPlot;
import org.jfree.data.xy.XYDataset;

import com.google.common.eventbus.EventBus;
import com.google.common.eventbus.Subscribe;

import ohd.hseb.charter.ChartEngine;
import ohd.hseb.charter.datasource.XYChartDataSource;
import ohd.hseb.charter.datasource.instances.CategoricalXYBarDataset;
import ohd.hseb.charter.jfreechartoverride.MarkerEntity;
import ohd.hseb.charter.panel.notices.ChangeChartEngineAndChartNotice;
import ohd.hseb.charter.panel.notices.ChartDataPointClickedNotice;
import ohd.hseb.charter.panel.notices.ChartTableVisibilityChangedNotice;
import ohd.hseb.charter.panel.notices.GeneralAxisLimitsChangedNotice;
import ohd.hseb.charter.panel.notices.ThresholdClickedNotice;
import ohd.hseb.charter.parameters.ThresholdParameters;
import ohd.hseb.hefs.utils.notify.NoticePoster;

/**
 * This wraps an {@link OHDFixedChartPanel} within a {@link JRootPane} (this) so that a
 * {@link ChartEngineChartPanelGlassPane} can be displayed on top for displaying selected table cell crosshairs or other
 * features. This panel requires a {@link ChartEngine} and a corresponding {@link JFreeChart} (typically constructed
 * from the {@link ChartEngine}). The {@link ChartEngine} is not recorded herein, but is recorded within the
 * {@link ChartEngineChartPanelGlassPane} which uses it to compute cross hairs. Though this panel can be used
 * stand-alone, it is recommend that it be used within a {@link ChartEngineChartAndTablePanel}.<br>
 * <br>
 * External user should call {@link #setTableShowing(boolean)} after constructing this if the table is to be showing
 * initially (it will trigger a {@link ChartTableVisibilityChangedNotice} that the {@link ChartEngineChartAndTablePanel}
 * should pickup on to make the table showing). By default, the check box is unchecked, but not event is fired.<br>
 * <br>
 * The {@link ChartEngine} used in this panel cannot be null!
 * 
 * @author hankherr
 */
@SuppressWarnings("serial")
public class ChartEngineChartPanel extends JRootPane implements ChangeChartEngineAndChartNotice.Subscriber,
ChartTableVisibilityChangedNotice.Subscriber, NoticePoster
{

    private ChartEngine _chartEngine;
    private OHDFixedChartPanel _chartPanel;

    /**
     * A special equals is created to make it easier to search to see if it is already included in the popup menu of a
     * provided chart when {@link #setChartPanel(OHDFixedChartPanel)} is called.
     */
    private final JCheckBoxMenuItem _showTableCheckBoxMenuItem = new JCheckBoxMenuItem("Show table")
    {
        @Override
        public boolean equals(final Object obj)
        {
            return ((obj instanceof JCheckBoxMenuItem) && (((JCheckBoxMenuItem)obj).getText().equals(getText())));
        }
    };

    /**
     * Event bus to use for posting notices.
     */
    private EventBus _centralBus;

    public ChartEngineChartPanel(final ChartEngine chartEngine, final JFreeChart chart)
    {
        _chartPanel = new OHDFixedChartPanel(chart);
        addShowTableCheckBoxToPopupMenu(false);
        createDisplay(chartEngine);
    }

    public ChartEngineChartPanel(final JFreeChart chart,
                                 final boolean properties,
                                 final boolean save,
                                 final boolean print,
                                 final boolean zoom,
                                 final boolean tooltips,
                                 final ChartEngine chartEngine)
    {
        _chartPanel = new OHDFixedChartPanel(chart, properties, save, print, zoom, tooltips);
        addShowTableCheckBoxToPopupMenu(false);
        createDisplay(chartEngine);
    }

    public ChartEngineChartPanel(final JFreeChart chart,
                                 final int width,
                                 final int height,
                                 final int minimumDrawWidth,
                                 final int minimumDrawHeight,
                                 final int maximumDrawWidth,
                                 final int maximumDrawHeight,
                                 final boolean useBuffer,
                                 final boolean properties,
                                 final boolean save,
                                 final boolean print,
                                 final boolean zoom,
                                 final boolean tooltips,
                                 final ChartEngine chartEngine)
    {
        _chartPanel = new OHDFixedChartPanel(chart,
                                             width,
                                             height,
                                             minimumDrawWidth,
                                             minimumDrawHeight,
                                             maximumDrawWidth,
                                             maximumDrawHeight,
                                             useBuffer,
                                             properties,
                                             save,
                                             print,
                                             zoom,
                                             tooltips);
        addShowTableCheckBoxToPopupMenu(false);
        createDisplay(chartEngine);
    }

    private void createDisplay(final ChartEngine chartEngine)
    {
        setContentPane(_chartPanel);
        setChartEngine(chartEngine);

        addAxisLimitsChangeListenerToAllAxes();
        addChartMouseListener();
    }

    /**
     * Adds the checkbox to the chart popup menu.
     * 
     * @param showTableInitially The state of the checkbox to create.
     */
    private void addShowTableCheckBoxToPopupMenu(final boolean showTableInitially)
    {
        _showTableCheckBoxMenuItem.setSelected(showTableInitially);
        _showTableCheckBoxMenuItem.addActionListener(new ActionListener()
        {
            @Override
            public void actionPerformed(final ActionEvent e)
            {
                post(new ChartTableVisibilityChangedNotice(this, _showTableCheckBoxMenuItem.isSelected()));
            }
        });

        _chartPanel.getPopupMenu().insert(_showTableCheckBoxMenuItem, 0);
        _chartPanel.getPopupMenu().insert(new JSeparator(SwingConstants.HORIZONTAL), 1);
    }

    /**
     * Adds listeners to all axes within the chart in {@link #_chartPanel}. The listener posts
     * {@link GeneralAxisLimitsChangedNotice} to the {@link #_centralBus}. This is used to communicate axis limit
     * changes in the chart to the data table and anything else that must hear. Since the {@link OHDFixedChartPanel}
     * knows nothing of the {@link #_centralBus}, these listeners must be added to relay messages.
     */
    private void addAxisLimitsChangeListenerToAllAxes()
    {
        //This populates the _listeners in _chartPanel.  I don't think I want this constructor adding itself as a listener to _chartPanel.
        //Surely there must be a better way!
        new GeneralAxisLimitsChangedNotice.PostingListener(_chartPanel.getChart().getXYPlot().getDomainAxis(),
                                                           _chartPanel,
                                                           this,
                                                           this,
                                                           true);
        final CombinedDomainXYPlot plot = (CombinedDomainXYPlot)_chartPanel.getChart().getXYPlot();
        for(int i = 0; i < plot.getSubplots().size(); i++)
        {
            final XYPlot subplot = (XYPlot)plot.getSubplots().get(i);
            for(int j = 0; j < plot.getRangeAxisCount(); j++)
            {
                new GeneralAxisLimitsChangedNotice.PostingListener(subplot.getRangeAxis(j),
                                                                   _chartPanel,
                                                                   this,
                                                                   this,
                                                                   true);
            }
        }
    }

    private void addChartMouseListener()
    {
        _chartPanel.addChartMouseListener(new ChartMouseListener()
        {
            @Override
            public void chartMouseClicked(final ChartMouseEvent event)
            {
                //The legend was clicked.
                if(event.getEntity() instanceof XYItemEntity)
                {
                    final XYItemEntity ent = (XYItemEntity)event.getEntity();
                    XYDataset datasetToCheck = ent.getDataset();
                    //XXX For XYBarDataset instances, the bar has a root, but the root is constructed based on the original
                    //data set shifting the x-values toa ccount for overlap proportion.  How to get back to that 
                    //original root?  
                    if ( (datasetToCheck instanceof CategoricalXYBarDataset))
                    {
                        datasetToCheck = ((CategoricalXYBarDataset)datasetToCheck).getOriginalDataset();  //Need to go one level further to get the data set underlying this one!
                    }
                    final XYChartDataSource source = _chartEngine.getDataSourceThatProducedDataSet(datasetToCheck);
                    _centralBus.post(new ChartDataPointClickedNotice(this, source, ent.getSeriesIndex(), ent.getItem()));
                }
                else if(event.getEntity() instanceof MarkerEntity)
                {
                    final Marker marker = ((MarkerEntity)event.getEntity()).getMarker();
                    final ThresholdParameters parms = getChartEngine().getChartParameters()
                                                                      .getThresholdList()
                                                                      .findParametersForMarker(marker);
                    _centralBus.post(new ThresholdClickedNotice(this, parms));
                }
            }

            @Override
            public void chartMouseMoved(final ChartMouseEvent arg0)
            {
            }
        });
    }

    /**
     * Passes the call through to {@link #_chartPanel}.
     */
    public void zoomFromPoint(final Point2D zoomPoint, final boolean zoomIn)
    {
        this._chartPanel.zoomFromPoint(zoomPoint, zoomIn);
    }

    /**
     * Passes the call through to {@link #_chartPanel}.
     */
    public void setHorizontalZoom(final boolean flag)
    {
        this._chartPanel.setHorizontalZoom(flag);
    }

    /**
     * Passes the call through to {@link #_chartPanel}.
     */
    public void setVerticalZoom(final boolean flag)
    {
        this._chartPanel.setVerticalZoom(flag);
    }

    /**
     * Passes call through to {@link #_chartPanel}.
     */
    public void addPopupMenuItems(final JMenuItem[] menuItems)
    {
        this._chartPanel.addPopupMenuItems(menuItems);
    }

    /**
     * Calls {@link #_chartPanel} method {@link OHDFixedChartPanel#setChart(JFreeChart)}. Also calls
     * {@link #addAxisLimitsChangeListenerToAllAxes()} so that this can hear axis changes to the chart.
     * 
     * @param chart
     */
    public void setChart(final JFreeChart chart)
    {
        //Set the chart displayed.
        _chartPanel.setChart(chart);

        //First, since the chart changed, I need to clear the listeners in the _chartPanel.  This will allow the chart
        //to be completely forgotten and, later, garbage collected.
        _chartPanel.clearOHDFixedChartListeners();

        //Then I need to re-add new axis limits listeners for the new chart.
        addAxisLimitsChangeListenerToAllAxes();
    }

    /**
     * Passes the {@link ChartEngine} through to the {@link ChartEngineChartPanelGlassPane}. This will switch the
     * current glass pane to an instance of ChartEngineChartPanelGlassPane if not already an instance of that class.
     * 
     * @param chartEngine Cannot be null.
     */
    public void setChartEngine(final ChartEngine chartEngine)
    {
        if(chartEngine == null)
        {
            throw new IllegalArgumentException("A ChartEngine is required for use of ChartEngineChartPanel; it cannot be null.");
        }
        _chartEngine = chartEngine;
        if((getGlassPane() != null) && (getGlassPane() instanceof ChartEngineChartPanelGlassPane))
        {
            ((ChartEngineChartPanelGlassPane)getGlassPane()).setChartEngine(chartEngine);
        }
        else
        {
            final ChartEngineChartPanelGlassPane glassPane = new ChartEngineChartPanelGlassPane(chartEngine,
                                                                                                _chartPanel);
            setGlassPane(glassPane);
            glassPane.setVisible(true);
        }
    }

    /**
     * @return Calls {@link #_chartPanel} method {@link OHDFixedChartPanel#getChart()}.
     */
    public JFreeChart getCurrentChart()
    {
        return _chartPanel.getChart();
    }

    public OHDFixedChartPanel getChartPanel()
    {
        return _chartPanel;
    }

    public ChartEngine getChartEngine()
    {
        return _chartEngine;
    }

    /**
     * Calls {@link #setContentPane(java.awt.Container)} and sets the glass pane, which is a
     * {@link ChartEngineChartPanelGlassPane}, to use the new panel. If the panel's popup menu does not yet include a
     * {@link #_showTableCheckBoxMenuItem}, this will add one.
     */
    public void setChartPanel(final OHDFixedChartPanel panel)
    {
        _chartPanel = panel;
        setContentPane(_chartPanel);

        ((ChartEngineChartPanelGlassPane)getGlassPane()).setChartPanel(_chartPanel);

        //This has not be tested yet to see if the equals in the _showTableCheckBoxMenuItem will allow for the
        //menu item to be properly identified if present.
        if(!Arrays.asList(_chartPanel.getPopupMenu().getComponents()).contains(_showTableCheckBoxMenuItem))
        {
            addShowTableCheckBoxToPopupMenu(false);
        }

        addChartMouseListener();
    }

    /**
     * Sets the visibility of the table. This sets the state of the {@link #_showTableCheckBoxMenuItem}, which then
     * determines visibility of the {@link #_tablePanel}.
     */
    public void setTableShowing(final boolean showing)
    {
        if(showing != _showTableCheckBoxMenuItem.isSelected())
        {
            _showTableCheckBoxMenuItem.doClick(); //Causes an appropriate action event to be fired.
        }
    }

    /**
     * Registers itself with the bus. Passes bus through to the glass pane;
     * {@link ChartEngineChartPanelGlassPane#setCentralBus(EventBus)}.
     */
    public void setCentralBus(final EventBus bus)
    {
        _centralBus = bus;
        _centralBus.register(this);

        if(getGlassPane() instanceof ChartEngineChartPanelGlassPane)
        {
            ((ChartEngineChartPanelGlassPane)getGlassPane()).setCentralBus(_centralBus);
        }
    }

    @Override
    @Subscribe
    public void reactToChangedChartEngineAndChart(final ChangeChartEngineAndChartNotice evt)
    {
        setChartEngine(evt.getChartEngine());
        setChart(evt.getChart());
    }

    @Override
    @Subscribe
    public void reactToChartTableVisibilityChangedNotice(final ChartTableVisibilityChangedNotice evt)
    {
        if(evt.getSource() != this)
        {
            if(evt.isVisible())
            {
                _showTableCheckBoxMenuItem.setSelected(true);
            }
            else
            {
                _showTableCheckBoxMenuItem.setSelected(false);
            }
        }
    }

    @Override
    public void post(final Object event)
    {
        if(_centralBus != null)
        {
            _centralBus.post(event);
        }
    }

}
