package ohd.hseb.charter.panel;

import java.awt.BasicStroke;
import java.awt.Color;
import java.awt.Component;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Point;
import java.awt.geom.Point2D;
import java.awt.geom.Rectangle2D;
import java.util.Date;
import java.util.List;

import javax.swing.JRootPane;

import com.google.common.collect.Lists;
import com.google.common.eventbus.EventBus;
import com.google.common.eventbus.Subscribe;

import ohd.hseb.charter.ChartEngine;
import ohd.hseb.charter.ChartPanelTools;
import ohd.hseb.charter.ChartTools;
import ohd.hseb.charter.panel.notices.ChartEngineTableCellSelectedNotice;
import ohd.hseb.hefs.utils.gui.components.ComponentPanelGlassPane;
import ohd.hseb.hefs.utils.gui.jtable.EventPostingCellSelectableTable;

/**
 * This component is used as a glass pane for a {@link ChartEngineChartPanel}. It draws on top of the chart, making it
 * so that the chart need not be repainted often. Currently, this is used to draw crosshairs over points in the chart
 * selected from a {@link EventPostingCellSelectableTable} with which it shares an {@link EventBus}. See
 * {@link ChartEngineChartAndTablePanel} for the {@link EventBus} generally used. The notice listened to is
 * {@link ChartEngineTableCellSelectedNotice}.<br>
 * <br>
 * This extends {@link ComponentPanelGlassPane} in order to take advantage of its overridden
 * {@link Component#contains(Point)} method that returns false if the point is not over one of the components.
 * 
 * @author hankherr
 */
@SuppressWarnings("serial")
public class ChartEngineChartPanelGlassPane extends ComponentPanelGlassPane implements
ChartEngineTableCellSelectedNotice.Subscriber
{

    /**
     * The {@link ChartEngine} used to determine cross hairs.
     */
    private ChartEngine _chartEngine;

    /**
     * The panel that displays the chart over which the crosshairs must be drawn.
     */
    private OHDFixedChartPanel _chartPanel;

    /**
     * Stores selected points for display as cross pairs. Each cross hair is specified by a
     * {@link ChartEngineTableCellSelectedNotice}.
     */
    private final List<ChartEngineTableCellSelectedNotice> _crossHairsToSubplotMap = Lists.newArrayList();

    /**
     * @param chartEngine The chart engine, which never changes.
     * @param chartPanel
     */
    public ChartEngineChartPanelGlassPane(final ChartEngine chartEngine, final OHDFixedChartPanel chartPanel)
    {
        setChartEngine(chartEngine);
        setChartPanel(chartPanel);
        setOpaque(false);
    }

    /**
     * Registers itself with the provided {@link EventBus}. This does not remember the event bus (and won't unless it
     * needs to post in the future).
     */
    public void setCentralBus(final EventBus bus)
    {
        bus.register(this);
    }

    /**
     * This must be kept in synch with the containing {@link JRootPane}, specifically {@link ChartEngineChartPanel}.
     * 
     * @param chartPanel The new displayed chart panel.
     */
    public void setChartPanel(final OHDFixedChartPanel chartPanel)
    {
        _chartPanel = chartPanel;
    }

    /**
     * Set the {@link ChartEngine} backing the displayed chart. Presumably, this will be paired with a change to either
     * {@link #_chartPanel} or the chart displayed with {@link #_chartPanel}.
     * 
     * @param engine The new {@link ChartEngine}.
     */
    public void setChartEngine(final ChartEngine engine)
    {
        _chartEngine = engine;
    }

    /**
     * @param evt The event for which the point selected needs to be converted to pixes.
     * @return The {@link Point} specifying the pixels corresponding to the
     *         {@link ChartEngineTableCellSelectedNotice#getRawDomainValue()} and
     *         {@link ChartEngineTableCellSelectedNotice#getRawRangeValue()}. If the domain or range are null, this
     *         returns null.
     */
    public Point convertDataPointToPixels(final ChartEngineTableCellSelectedNotice evt)
    {
        final Point2D.Double dataPoint = new Point2D.Double();

        final Object rawDomainValue = evt.getRawDomainValue();
        final Object rawRangeValue = evt.getRawRangeValue();

        if((rawDomainValue == null) || (rawRangeValue == null))
        {
            return null;
        }
        if(rawDomainValue instanceof Date)
        {
            dataPoint.x = ((Date)rawDomainValue).getTime();
        }
        //XXX For WRES: 
        //Strings - Categories.  Need to convert to number to draw the cross hairs.
        else if (rawDomainValue instanceof String) 
        {
            dataPoint.x = ChartTools.computeNumericalValueForCategory(_chartPanel.getChart(), (String)rawDomainValue);
        }
        else
        {
            dataPoint.x = ((Number)rawDomainValue).doubleValue();
        }
        dataPoint.y = Double.NaN;

        if(rawRangeValue instanceof Number)
        {
            dataPoint.y = ((Number)rawRangeValue).doubleValue();
        }
        final Point pt = ChartPanelTools.convertDataPointToPixels(_chartPanel,
                                                                         _chartEngine,
                                                                         evt.getDataSourceIndex(),
                                                                         dataPoint);
        return pt;
    }

    @Override
    public void paint(final Graphics g)
    {
        super.paint(g);

        //Draw the crosshairs in magenta.
        g.setColor(Color.MAGENTA);
        ((Graphics2D)g).setStroke(new BasicStroke(2.0f));
        for(final ChartEngineTableCellSelectedNotice evt: _crossHairsToSubplotMap)
        {
            final int trueSubPlotIndex = ChartTools.detetermineTrueSubplotIndex(_chartEngine,
                                                                                            _chartPanel.getChart(),
                                                                                            evt.getDataSourceIndex());

            final Rectangle2D bounds = _chartPanel.getChartRenderingInfo()
                                                  .getPlotInfo()
                                                  .getSubplotInfo(trueSubPlotIndex)
                                                  .getDataArea();

            //Scale bounds if the chart has a maximum size and its been exceeded.
            final Rectangle2D scaledBounds = new Rectangle2D.Double(_chartPanel.getScaleX() * bounds.getX(),
                                                                    _chartPanel.getScaleY() * bounds.getY(),
                                                                    _chartPanel.getScaleX() * bounds.getWidth(),
                                                                    _chartPanel.getScaleY() * bounds.getHeight());

            //Convert to pixels the data point selected.
            final Point pt = convertDataPointToPixels(evt);
            if(pt != null)
            {
                if((scaledBounds.getMinX() <= pt.getX()) && (pt.getX() <= scaledBounds.getMaxX()))
                {
                    //Draw cross hairs.
                    g.drawLine((int)pt.getX(), (int)scaledBounds.getMinY(), (int)pt.getX(), (int)scaledBounds.getMaxY());
                }
                if((scaledBounds.getMinY() <= pt.getY()) && (pt.getY() <= scaledBounds.getMaxY()))
                {
                    g.drawLine((int)scaledBounds.getMinX(), (int)pt.getY(), (int)scaledBounds.getMaxX(), (int)pt.getY());
                }
            }
        }

    }

    @Override
    @Subscribe
    public void reactToChartEngineTableCellSelected(final ChartEngineTableCellSelectedNotice evt)
    {
        _crossHairsToSubplotMap.clear();
        if(evt.getDataSourceIndex() >= 0) //Negative indicates an empty event, or no cells selected.
        {
            _crossHairsToSubplotMap.add(evt);
        }

        //This needs to be only a partial paint.
        _chartPanel.setForceFullRepaint(false);

        repaint();
    }
}
