package ohd.hseb.charter.panel;

import java.awt.Color;
import java.awt.Rectangle;

import javax.swing.JTable;
import javax.swing.table.TableModel;

import ohd.hseb.charter.panel.notices.ChartEngineTableCellSelectedNotice;
import ohd.hseb.charter.panel.notices.GeneralAxisLimitsChangedNotice;
import ohd.hseb.charter.panel.notices.GeneralTableCellSelectedNotice;
import ohd.hseb.hefs.utils.gui.jtable.EventPostingCellSelectableTable;
import ohd.hseb.hefs.utils.gui.jtable.RecomputeMarksNotice;
import ohd.hseb.hefs.utils.gui.tools.SwingTools;
import ohd.hseb.hefs.utils.tools.IconTools;

import org.jdesktop.swingx.table.ColumnControlButton;

import com.google.common.eventbus.Subscribe;

/**
 * An {@link EventPostingCellSelectableTable} that is designed for using a {@link ChartEngineTableModel} model. In
 * addition to some GUI settings being modified when constructed, it also implements a
 * {@link GeneralAxisLimitsChangedNotice.Subscriber} for {@link GeneralAxisLimitsChangedNotice} instances that will
 * scroll to rows and columns when the chart axis limits change, typically due to zooming. See
 * {@link #_scrollForAxisLimitsChanges}.
 * 
 * @author hankherr
 */
@SuppressWarnings("serial")
public class ChartEngineTable extends EventPostingCellSelectableTable implements
GeneralAxisLimitsChangedNotice.Subscriber, GeneralTableCellSelectedNotice.Subscriber
{
    public static final Color TABLE_BG_COLOR = new Color(212, 212, 212);

    /**
     * If true, then {@link GeneralAxisLimitsChangedNotice} will result in a scrolling of the table to the area of the
     * table dictated by the axis limits. By default, this flag will be true so that the table does scroll.
     */
    private boolean _scrollForAxisLimitsChanges = true;

    public ChartEngineTable(final ChartEngineTableModel model)
    {
        super(model);

        final ColumnControlButton controlButton = new ColumnControlButton(this,
                                                                          IconTools.getHSEBIcon("yellowGear14x14"));
        controlButton.setBackground(new Color(128, 128, 255));
        controlButton.setToolTipText("Open table control tools");
        setColumnControl(controlButton);
        setAutoResizeMode(JTable.AUTO_RESIZE_OFF);
        setReturnCellContentsByDefault(true);
        getTableHeader().setBackground(TABLE_BG_COLOR);
        setColumnControlVisible(false);//Taking out column control button... doesn't really serve a purpose since detach table was moved to a popup.

    }

    public void setScrollForxisLimitsChanges(final boolean scrollForAxisLimitsChanges)
    {
        _scrollForAxisLimitsChanges = scrollForAxisLimitsChanges;
    }

    /**
     * Convenience wrapper converts {@link TableModel} to {@link ChartEngineTableModel}.
     */
    @Override
    public ChartEngineTableModel getModel()
    {
        return (ChartEngineTableModel)super.getModel();
    }

    @Override
    public void scrollCellToVisible(final int row, final int column)
    {
        final Rectangle cellRect = getCellRect(row, column, false);
        final Rectangle visibleRect = getVisibleRect();

        if(!visibleRect.contains(cellRect))
        {
            //For the row... If the cell is below the visible rect, then I want to target a row 
            //adjusted by the number of visible rows.
            int usedY = cellRect.y;
            if(cellRect.y > visibleRect.y)
            {
                usedY = cellRect.y + (visibleRect.height - cellRect.height);
            }

            //For the column... If the cell is to the right of the visible rect, then adjust.
            int usedX = cellRect.x;
            if(cellRect.x > visibleRect.x)
            {
                usedX = cellRect.x + (visibleRect.width - cellRect.width);
            }

            scrollRectToVisible(new Rectangle(usedX, usedY, cellRect.width, cellRect.height));
        }
    }

    @Override
    @Subscribe
    public void reactToGeneralAxisLimitsChanged(final GeneralAxisLimitsChangedNotice evt)
    {
        getCentralBus().post(new RecomputeMarksNotice(this));

        //If we are to scroll for axis limits changed events...
        if(_scrollForAxisLimitsChanges)
        {
            //Scroll to the first view row found with data visible within the current axis limits.
            int viewRow;
            for(viewRow = 0; viewRow < getRowCount(); viewRow++)
            {
                if(getModel().isRowVisibleInChart(convertRowIndexToModel(viewRow)))
                {
                    break;
                }
            }

            //Scroll to the first view column found with data visible within the current axis limits.
            int viewCol;
            for(viewCol = 0; viewCol < getColumnCount(); viewCol++)
            {
                if(getModel().isColumnVisibleInChart(convertColumnIndexToModel(viewCol)))
                {
                    break;
                }
            }

            scrollCellToVisible(viewRow - 1, viewCol - 1);
        }

        SwingTools.forceComponentRedraw(this);
    }

    /**
     * Maps {@link GeneralTableCellSelectedNotice} to a {@link ChartEngineTableCellSelectedNotice}.
     */
    @Override
    @Subscribe
    public void reactToTableCellSelection(final GeneralTableCellSelectedNotice evt)
    {
        final int selectedRow = evt.getModelRow();
        final int selectedCol = evt.getModelCol();

        if((selectedRow >= 0) && (selectedCol >= 0))
        {
            final ChartEngineTableModel wrappedModel = getModel();
            final int dataSourceIndex = wrappedModel.getDataSourceIndex();
            final int seriesIndex = wrappedModel.computeSeriesIndex(selectedCol);
            final int domainCol = wrappedModel.computeDomainColumn(seriesIndex);
            final Object domainValue = wrappedModel.getRawValueAt(selectedRow, domainCol);
            int rangeCol = -1;
            Object rangeValue = null;
            if(seriesIndex >= 0)
            {
                rangeCol = wrappedModel.computeRangeColumn(seriesIndex);
                rangeValue = wrappedModel.getRawValueAt(selectedRow, rangeCol);
            }
            getCentralBus().post(new ChartEngineTableCellSelectedNotice(this,
                                                                        dataSourceIndex,
                                                                        seriesIndex,
                                                                        domainValue,
                                                                        rangeValue));
        }
        else if(selectedRow >= 0)
        {
            final ChartEngineTableModel wrappedModel = getModel();
            final int dataSourceIndex = wrappedModel.getDataSourceIndex();
            getCentralBus().post(new ChartEngineTableCellSelectedNotice(this, dataSourceIndex, -1, null, null));
        }
        else if(selectedCol >= 0)
        {
            final AbstractChartEngineTableModel wrappedModel = (AbstractChartEngineTableModel)getModel();
            final int dataSourceIndex = wrappedModel.getDataSourceIndex();
            final int seriesIndex = wrappedModel.computeSeriesIndex(selectedCol);
            getCentralBus().post(new ChartEngineTableCellSelectedNotice(this, dataSourceIndex, seriesIndex, null, null));
        }
        else
        {
            getCentralBus().post(new ChartEngineTableCellSelectedNotice(this));
        }

    }

}
