package ohd.hseb.charter.panel;

import javax.swing.table.TableModel;

import ohd.hseb.charter.ChartEngine;
import ohd.hseb.hefs.utils.gui.jtable.EventPostingCellSelectableTable;

import org.jfree.chart.JFreeChart;

/**
 * Superclass of table models used within {@link ChartEngineChartAndTablePanel}'s
 * {@link EventPostingCellSelectableTable} table instance.
 * 
 * @author hankherr
 */
public interface ChartEngineTableModel extends TableModel
{

    /**
     * @return True if a user is allowed to selected between different sources. If false, the source selection combo box
     *         displayed above the table within {@link ChartEngineChartAndTablePanel} will not be visible.
     */
    public boolean allowForDataSourceSwitching();

    /**
     * @return The index of the data source for the data currently displayed within the model.
     */
    public int getDataSourceIndex();

    /**
     * Set the index of the data source for the data currently displayed within the model.
     * 
     * @param index
     */
    public void setDataSourceIndex(int index);

    /**
     * Sets the chart currently being viewed by the user. This is called immediately after the
     * {@link ChartEngineChartAndTablePanel} builds the table model via its supplier, and after the
     * {@link #setChartEngine(ChartEngine)} is called.
     */
    public void setChart(JFreeChart chart);

    /**
     * @param engine The {@link ChartEngine} corresponding to the model. This is called immediately after the
     *            {@link ChartEngineChartAndTablePanel} builds the table model via its supplier.
     */
    public void setChartEngine(ChartEngine engine);

    /**
     * @return The model column index corresponding to the domain variable of a series for the current data source (with
     *         index set via {@link #setDataSourceIndex(int)}). Returns -1 if the index does not appear to be valid.
     */
    public int computeDomainColumn(final int seriesIndex);

    /**
     * @return The model column index corresponding to the range variable of a series for the current data source (with
     *         index set via {@link #setDataSourceIndex(int)}). Returns -1 if the index does not appear to be valid.
     */
    public int computeRangeColumn(final int seriesIndex);

    /**
     * @param seriesIndex The index of the series for which to acquire the item number.
     * @param seriesItemNumber The item number in the series with the specified index.
     * @return The model row corresponding to the series and item. Returns -1 if the series or item does not seem valid.
     */
    public int computeRowForItem(int seriesIndex, int seriesItemNumber);

    /**
     * @return True if the data in the row is visible within the current chart.
     */
    public boolean isRowVisibleInChart(int modelRow);

    /**
     * @return True if the data in the column is visible within the current chart.
     */
    public boolean isColumnVisibleInChart(int modelColumn);

    /**
     * @return The series index relative to the current data source given a column. Return -1 if column 0 is provided
     *         and all series share the same axis.
     */
    public int computeSeriesIndex(final int modelColumn);

    /**
     * {@link #getValueAt(int, int)} should call this method in its implementation. This can also be used by other
     * methods as needed to get the raw value for a cell, instead of a possibly translated value; for example if a
     * cell's display type is {@link String}, but the original (raw) data is something different.
     * 
     * @param modelRow The row for which to acquire the raw value.
     * @param modelColumn The column.
     * @return Object The raw value before formatting for display within a cell of the table.
     */
    public Object getRawValueAt(final int modelRow, final int modelColumn);
}
