package ohd.hseb.charter.panel;

import java.awt.Color;
import java.awt.Composite;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Image;
import java.awt.Insets;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.event.ComponentAdapter;
import java.awt.event.ComponentEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;

import javax.swing.BorderFactory;
import javax.swing.JPanel;

import ohd.hseb.charter.ChartEngine;
import ohd.hseb.charter.ChartTools;
import ohd.hseb.charter.panel.notices.NavigatedSubplotAxisLimitsChangedNotice;
import ohd.hseb.charter.panel.notices.NavigationCanvasRectangleChangedNotice;
import ohd.hseb.hefs.utils.gui.tools.SwingTools;
import ohd.hseb.util.data.DataPoint;

import org.jfree.chart.ChartPanel;
import org.jfree.chart.JFreeChart;
import org.jfree.chart.event.AxisChangeEvent;
import org.jfree.chart.event.AxisChangeListener;
import org.jfree.chart.plot.CombinedDomainXYPlot;
import org.jfree.chart.plot.XYPlot;
import org.jfree.data.Range;

import com.google.common.eventbus.EventBus;
import com.google.common.eventbus.Subscribe;

/**
 * A {@link JPanel} that wraps a {@link NavigationCanvas} for use in navigating a {@link ChartEngine} constructed
 * {@link JFreeChart}. That means the provided chart must use a {@link CombinedDomainXYPlot} with only one domain axis
 * and either one or two range axes for the monitored subplot, which is ALWAYS subplot 0 (code will need to be changed
 * if this is ever not the case).<br>
 * <br>
 * Whenever axis limits are altered, the provided chart is updated automatically if this is the source of the change and
 * a {@link NavigatedSubplotAxisLimitsChangedNotice} is posted to the {@link #_eventBus}. A listener must implement
 * {@link NavigatedSubplotAxisLimitsChangedNotice.Subscriber} and register itself via
 * {@link #registerWithBus(ohd.hseb.charter.panel.notices.NavigatedSubplotAxisLimitsChangedNotice.Subscriber)}. Axis
 * limits change notices are posted whenever any axis limit changed in the wrapped chart, regardless of whether or not
 * it originated from navigator panel interaction.<br>
 * <br>
 * The {@link #_eventBus} contained herein is also passed down to the {@link #_navigatorCanvas}. This means it will also
 * carry {@link NavigationCanvasRectangleChangedNotice} instances posted by {@link #_navigatorCanvas}, but no one other
 * than this should need to listen for those events. <br>
 * This panel will react to a changed chart it the recorded {@link #_viewingPanel}.
 * 
 * @author hankherr
 */
public class CombinedDomainChartNavigationPanel extends JPanel implements
NavigationCanvasRectangleChangedNotice.Subscriber
{
    private static final long serialVersionUID = 1L;

    private static final int MINIMUM_BUFFER_AROUND_CANVAS = 15;

    /**
     * The NavigatorCanvas around which this panel wraps.
     */
    private final NavigationCanvas _navigatorCanvas;

    /**
     * The navigated {@link JFreeChart} created using the {@link ChartEngine} framework, meaning its plot is a
     * {@link CombinedDomainXYPlot} and and only the subplot with index 0 is used for navigating.
     */
    private JFreeChart _navigatedChart;

    /**
     * The instance of {@link ChartPanel} used to view the chart. It is likely an instance of {@link OHDFixedChartPanel}
     * .
     */
    private final OHDFixedChartPanel _viewingPanel;

    /**
     * The original axes lower bounds, acquired when the navigator is constructed based on the {@link #_navigatedChart}.
     */
    private DataPoint _originalAxesLowerBounds;

    /**
     * The original axes upper bounds, acquired when the navigator is constructed based on the {@link #_navigatedChart}.
     */
    private DataPoint _originalAxesUpperBounds;

    /**
     * The largest dimensions allowed for this panel. Used to determine this panel's preferred size.
     */
    private final Dimension _allowedPanelDimensions;

    /**
     * Event bus for firing axis limits changed events. This class will handle adjusting the {@link #_navigatedChart}
     * axis limits, but leaves the other limits alone. This bus is also passed to the {@link NavigationCanvas} for
     * posting {@link NavigationCanvasRectangleChangedNotice} instances, which this is subscribed for.
     */
    private final EventBus _eventBus = new EventBus();

    private boolean _maintainAspectRatio = true;

    /**
     * General axis change listener for listening to changes in any of three valid axes; see
     * {@link #addAxisChangeListener()} and {@link #removeAxisChangeListener()}. This is an attribute because it needs
     * to listen to two or three axes.
     */
    private final AxisChangeListener _axisChangeListener = new AxisChangeListener()
    {
        @Override
        public void axisChanged(final AxisChangeEvent arg0)
        {
            updateCanvasCurrentViewedRectangle();
            postAxisLimitsChangeEventForCurrentLimits();
            repaint();
        }
    };

    /**
     * Listens for mouse presses in the border surrounding the {@link #_navigatorCanvas}. This is an attribute because
     * it must be removable.
     */
    private final MouseAdapter _mouseAdapter = new MouseAdapter()
    {
        @Override
        public void mousePressed(final MouseEvent e)
        {
            Point canvasLocation = _navigatorCanvas.getLocation();
            if(e.getSource() == _navigatorCanvas)
            {
                canvasLocation = new Point(0, 0);
            }
            final Dimension canvasSize = _navigatorCanvas.getBackgroundImageSize();
            double xMovement = 0.0D;
            double yMovement = 0.0D;
            if(e.getPoint().x < canvasLocation.x)
            {
                //Increase y-axis value.
                xMovement = -0.10D;
            }
            if(e.getPoint().x > canvasLocation.x + canvasSize.width)
            {
                //Increase x-axis value.
                xMovement = 0.10D;
            }
            if(e.getPoint().y < canvasLocation.y)
            {
                //Increase y-axis value.
                yMovement = 0.10D;
            }
            if(e.getPoint().y > canvasLocation.y + canvasSize.height)
            {
                //Decrease y-axis value.
                yMovement = -0.10D;
            }

            shiftAxisLimits(xMovement, yMovement);
        }
    };

    /**
     * @param navigatedChart The chart being navigated. Its plot must be a {@link CombinedDomainXYPlot} and only the
     *            subplot with index 0 is navigated.
     * @param viewingPanel The {@link ChartPanel} instance used for viewing, most liked {@link OHDFixedChartPanel}.
     * @param allowedPanelDimensions The largest dimensions allowed for this panel.
     */
    public CombinedDomainChartNavigationPanel(final OHDFixedChartPanel viewingPanel,
                                              final Dimension allowedPanelDimensions)
    {
        checkChartForValidity(viewingPanel.getChart());
        _eventBus.register(this);

        //Record the variables.
        _viewingPanel = viewingPanel;
        _navigatedChart = _viewingPanel.getChart();
        _allowedPanelDimensions = allowedPanelDimensions;

        //Initialize the canvas and set its background.
        _navigatorCanvas = new NavigationCanvas();
        _navigatorCanvas.setReactToMouseEvents(true);
        _navigatorCanvas.setEventBus(_eventBus);

        //Add the change listeners to the axes.
        addAxisChangeListener();

        //Get the original axis limits.
        acquireOriginalAxisLimitsFromChart();

        //Add listeners to the _viewingPanel so that this can hear resizes and property changes for when the viewed chart changes.
        _viewingPanel.addComponentListener(new ComponentAdapter()
        {
            @Override
            public void componentResized(final ComponentEvent e)
            {
                resizeBackgroundImage();
            }
        });
        _viewingPanel.addPropertyChangeListener(new PropertyChangeListener()
        {
            @Override
            public void propertyChange(final PropertyChangeEvent evt)
            {
                if(evt.getPropertyName().equals(OHDFixedChartPanel.PROPERTY_CHART_CHANGED))
                {
                    clearDisplay();
                    checkChartForValidity(_viewingPanel.getChart());
                    _navigatedChart = _viewingPanel.getChart();
                    acquireOriginalAxisLimitsFromChart();
                    refreshBackgroundImage();
                    addAxisChangeListener();
                }
            }
        });

        //Create the display.
        createDisplay();

        //Make this a mouse listener for both the navigator canvas and itself.
        this.addMouseListener(_mouseAdapter);
    }

    private void createDisplay()
    {
        setLayout(new GridBagLayout());
        GridBagConstraints constraints;
        final Insets theinsets = new Insets(0, 0, 0, 0);
        constraints = SwingTools.returnGridBagConstraints(0,
                                                          0,
                                                          1,
                                                          1,
                                                          1,
                                                          1,
                                                          GridBagConstraints.CENTER,
                                                          GridBagConstraints.NONE,
                                                          theinsets,
                                                          0,
                                                          0);
        add(_navigatorCanvas, constraints);

        setBorder(BorderFactory.createLineBorder(Color.BLACK));
        setBackground(Color.GRAY);
    }

    /**
     * Clears the navigator canvas likely due to a problem with the chart to display.
     */
    private void clearDisplay()
    {
        _navigatorCanvas.setNavigatedImageComponents(null, null);
    }

    /**
     * Resets the navigated chart so that it uses that currently displayed in the {@link #_viewingPanel}:
     * {@link OHDFixedChartPanel#getChart()}. This should be called whenever that panel changes its displayed chart.
     * This will call {@link #addAxisChangeListener()}, {@link #acquireOriginalAxisLimitsFromChart()}, and finally
     * {@link #resizeBackgroundImage()} in order to reset the nav panel image.
     */
    public void resetNavigatedChart()
    {
        //Record the variables.
        _navigatedChart = _viewingPanel.getChart();

        //Add the change listeners to the axes for the chart.
        addAxisChangeListener();

        //Get the original axis limits from the reset chart.
        acquireOriginalAxisLimitsFromChart();

        //Call to refresh the nav image from scratch.
        resizeBackgroundImage();
    }

    /**
     * @param chart Chart to check for validity of use.
     * @throws IllegalArgumentException If a problem is found.
     */
    private void checkChartForValidity(final JFreeChart chart) throws IllegalArgumentException
    {
        if(!(chart.getXYPlot() instanceof CombinedDomainXYPlot))
        {
            throw new IllegalArgumentException("The plot within the provided chart is not a CombinedDomainXYPlot.");
        }
        final XYPlot navPlot = (XYPlot)((CombinedDomainXYPlot)chart.getXYPlot()).getSubplots().get(0);
        if(navPlot.getRangeAxisCount() > 2)
        {
            throw new IllegalArgumentException("There are too many range axes on the first subplot within the chart, "
                + navPlot.getRangeAxisCount() + "; only one or two is allowed.");
        }
        if(navPlot.getDomainAxisCount() != 1)
        {
            throw new IllegalArgumentException("There are too many domain axes on the first subplot within the chart, "
                + navPlot.getRangeAxisCount() + "; only one is allowed.");
        }
    }

    /**
     * Shift the limits of the viewed panel, calling {@link #updateChartForNewLimits(DataPoint, DataPoint)}.
     * 
     * @param xMovement Movement factor for x-axis; a proportion relative to to the current axis limits of the plot
     *            (data units; not pixels).
     * @param yMovement Movement factor for y-axes; a proportion relative to to the current axis limits of the plot
     *            (data units; not pixels).
     */
    private void shiftAxisLimits(final double xMovement, final double yMovement)
    {
        //Shift the axis limits.
        final XYPlot plot = getNavigatedXYPlot();
        final DataPoint lowerBounds = new DataPoint(plot.getRangeAxisCount());
        final DataPoint upperBounds = new DataPoint(plot.getRangeAxisCount());

        lowerBounds.setX(plot.getDomainAxis().getLowerBound() + xMovement
            * (plot.getDomainAxis().getUpperBound() - plot.getDomainAxis().getLowerBound()));
        upperBounds.setX(plot.getDomainAxis().getUpperBound() + xMovement
            * (plot.getDomainAxis().getUpperBound() - plot.getDomainAxis().getLowerBound()));

        lowerBounds.setY(0, plot.getRangeAxis(0).getLowerBound() + yMovement
            * (plot.getRangeAxis(0).getUpperBound() - plot.getRangeAxis(0).getLowerBound()));
        upperBounds.setY(0, plot.getRangeAxis(0).getUpperBound() + yMovement
            * (plot.getRangeAxis(0).getUpperBound() - plot.getRangeAxis(0).getLowerBound()));

        if(plot.getRangeAxisCount() == 2)
        {
            lowerBounds.setY(1, plot.getRangeAxis(1).getLowerBound() + yMovement
                * (plot.getRangeAxis(1).getUpperBound() - plot.getRangeAxis(1).getLowerBound()));
            upperBounds.setY(1, plot.getRangeAxis(1).getUpperBound() + yMovement
                * (plot.getRangeAxis(1).getUpperBound() - plot.getRangeAxis(1).getLowerBound()));
        }

        //Update the chart
        updateChartForNewLimits(lowerBounds, upperBounds);
    }

    /**
     * Posts an {@link NavigatedSubplotAxisLimitsChangedNotice} using the current limits.
     */
    private void postAxisLimitsChangeEventForCurrentLimits()
    {
        //Shift the axis limits.
        final DataPoint lowerBounds = ChartTools.constructLowerBoundsDataPoint(_navigatedChart, 0);
        final DataPoint upperBounds = ChartTools.constructUpperBoundsDataPoint(_navigatedChart, 0);
        _eventBus.post(new NavigatedSubplotAxisLimitsChangedNotice(this, lowerBounds, upperBounds));
    }

    /**
     * Refreshes the background image when it has been resized. In this case, it assumes that the chart contained within
     * {@link #_viewingPanel} is still correct, but may have had its axis limits changed due to zooming. It records the
     * current limits, recovers the original limits, draws the chart image according to the current dimensions of the
     * viewing panel, and then resets the recorded limits.
     */
    private void resizeBackgroundImage()
    {
        //Record the limits, then set them to the _originals*, draw the image, and recover the recorded limits.
        final XYPlot plot = getNavigatedXYPlot();
        final Range domainRange = new Range(plot.getDomainAxis().getLowerBound(), plot.getDomainAxis().getUpperBound());
        plot.getDomainAxis().setRange(_originalAxesLowerBounds.getX(), _originalAxesUpperBounds.getX());
        final Range range1Range = new Range(plot.getRangeAxis(0).getLowerBound(), plot.getRangeAxis(0).getUpperBound());
        plot.getRangeAxis(0).setRange(_originalAxesLowerBounds.getY(0), _originalAxesUpperBounds.getY(0));
        Range range2Range = null;
        if(plot.getRangeAxisCount() == 2)
        {
            range2Range = new Range(plot.getRangeAxis(1).getLowerBound(), plot.getRangeAxis(1).getUpperBound());
            plot.getRangeAxis(1).setRange(_originalAxesLowerBounds.getY(1), _originalAxesUpperBounds.getY(1));
        }

        //Refresh the image.
        refreshBackgroundImage();

        //Recover the recorded axis limits.
        plot.getDomainAxis().setRange(domainRange);
        plot.getRangeAxis(0).setRange(range1Range);
        if(range2Range != null)
        {
            plot.getRangeAxis(1).setRange(range2Range);
        }
    }

    /**
     * Whenever the background image must be redrawn, call this method. It acquires a new background image by calling
     * the {@link #_navigatedChart}'s
     * {@link JFreeChart#createBufferedImage(int, int, org.jfree.chart.ChartRenderingInfo)} method using the rendering
     * info contained within {@link #_viewingPanel}. It then pulls the plot area rectangle from the same rendering
     * information. The preferred size is computed and all info is passed to the {@link #_navigatorCanvas} for updating.
     */
    private void refreshBackgroundImage()
    {
        //TODO The below sizes image and plot area rectangle to be consistent with navigated image (_viewingPanel).  
        //Need to update to allow for drawing a fresh image to fill in dimensions consistent with size of this
        //panel adjusted to maximum size allowed for chart.  

        //Refresh the plot background image
        Image newBackgroundImage = null;
        Rectangle plotAreaRectangle = null;
        if(((int)_viewingPanel.getChartRenderingInfo().getChartArea().getWidth() > 0)
            && ((int)_viewingPanel.getChartRenderingInfo().getChartArea().getHeight() > 0))
        {
            newBackgroundImage = _navigatedChart.createBufferedImage((int)_viewingPanel.getChartRenderingInfo()
                                                                                       .getChartArea()
                                                                                       .getWidth(),
                                                                     (int)_viewingPanel.getChartRenderingInfo()
                                                                                       .getChartArea()
                                                                                       .getHeight(),
                                                                     _viewingPanel.getChartRenderingInfo());
            plotAreaRectangle = _viewingPanel.getChartRenderingInfo().getPlotInfo().getDataArea().getBounds();
        }

        computePreferredSize(plotAreaRectangle);

        _navigatorCanvas.setReactToMouseEvents(true);

        _navigatorCanvas.setNavigatedImageComponents(newBackgroundImage, plotAreaRectangle);
        _navigatorCanvas.computeScaleFactorAndPreferredSize(new Dimension((int)getPreferredSize().getWidth()
            - MINIMUM_BUFFER_AROUND_CANVAS, (int)getPreferredSize().getHeight() - MINIMUM_BUFFER_AROUND_CANVAS));

        updateCanvasCurrentViewedRectangle();
        SwingTools.forceComponentRedraw(this); //Force a full reconstruct due to the panel needing to potentially be resized.
    }

    /**
     * Computes the preferred size for this navigator panel based on the plot area rectangle and the allowed panel
     * width. After computation, {@link SwingTools#forceComponentRedraw(java.awt.Component)} is called on this panel to
     * force a repainting.
     * 
     * @param plotAreaRectangle The plot area rectangle that applies to the background image in the navigator canvas.
     *            Null is allowed and results in this doing nothing.
     */
    private void computePreferredSize(final Rectangle plotAreaRectangle)
    {
        //Determine the preferred size for this panel and the canvas.
        if(plotAreaRectangle != null)
        {
            Dimension usedDim = this.getSize();
            if(_allowedPanelDimensions != null)
            {
                usedDim = _allowedPanelDimensions;
            }

            //Compute scaling factors
            double widthScalingFactor = (usedDim.getWidth() - 20) / plotAreaRectangle.getWidth();
            double heightScalingFactor = (usedDim.getHeight() - 10) / plotAreaRectangle.getHeight();
            if(_maintainAspectRatio)
            {
                widthScalingFactor = Math.min(widthScalingFactor, heightScalingFactor);
                heightScalingFactor = Math.min(widthScalingFactor, heightScalingFactor);
            }

            //Set sizes
            setPreferredSize(new Dimension((int)(widthScalingFactor * plotAreaRectangle.getWidth()) + 5,
                                           (int)(heightScalingFactor * plotAreaRectangle.getHeight()) + 10));
            setMinimumSize(getPreferredSize());
            setMaximumSize(getPreferredSize());

            SwingTools.forceComponentRedraw(this);
        }
    }

    /**
     * Updates the currently viewed rectangle within the {@link #_navigatorCanvas} to match those implied by bounds
     * within the navigated {@link XYPlot} returned by {@link #getNavigatedXYPlot()}.
     */
    private void updateCanvasCurrentViewedRectangle()
    {
        if(!_navigatorCanvas.getReactToMouseEvents())
        {
            _navigatorCanvas.setCurrentViewedRectangle(null);
            return;
        }

        //This cannot use the construct*BoundsDataPoint method, because the range is inverted relative to pixel points.  
        //Compute the points below:
        final XYPlot plot = getNavigatedXYPlot();
        final Point minimumPoint = computeDrawnPixelPointInNavigatorScaleFromDataPoint(new DataPoint(plot.getDomainAxis()
                                                                                                         .getLowerBound(),
                                                                                                     plot.getRangeAxis(0)
                                                                                                         .getUpperBound()));
        final Point maximumPoint = computeDrawnPixelPointInNavigatorScaleFromDataPoint(new DataPoint(plot.getDomainAxis()
                                                                                                         .getUpperBound(),
                                                                                                     plot.getRangeAxis(0)
                                                                                                         .getLowerBound()));

        _navigatorCanvas.setCurrentViewedRectangle(new Rectangle(minimumPoint.x, minimumPoint.y, maximumPoint.x
            - minimumPoint.x, maximumPoint.y - minimumPoint.y));
    }

    /**
     * Updates the bounds on the navigated {@link XYPlot} to match those provided.
     */
    private void updateChartForNewLimits(final DataPoint lowerBounds, final DataPoint upperBounds)
    {
        //Don't listen for changes which are about to be made here.
        removeAxisChangeListener();

        //Set the range for the 2 or 3 axes and update the currently viewed rectangle.
        final XYPlot plot = this.getNavigatedXYPlot();
        plot.getDomainAxis().setRange(lowerBounds.getX(), upperBounds.getX());
        plot.getRangeAxis(0).setRange(lowerBounds.getY(0), upperBounds.getY(0));
        if(plot.getRangeAxisCount() == 2)
        {
            plot.getRangeAxis(1).setRange(lowerBounds.getY(1), upperBounds.getY(1));
        }
        updateCanvasCurrentViewedRectangle();

        //Add the listener back in.
        addAxisChangeListener();

        _eventBus.post(new NavigatedSubplotAxisLimitsChangedNotice(this, lowerBounds, upperBounds));
    }

    /**
     * Update chart limits and post a {@link NavigatedSubplotAxisLimitsChangedNotice} event to the bus.
     */
    private void updateChartFromNavigatorCanvas()
    {
        //Lower bounds computed from pixels of lower bound in nav canvas.
        final Point lowerLimitsPixels = new Point(_navigatorCanvas.getCurrentViewedRectangle().x,
                                                  _navigatorCanvas.getCurrentViewedRectangle().y
                                                      + _navigatorCanvas.getCurrentViewedRectangle().height);
        final DataPoint lowerBounds = computeDataPointFromPixelPointInNavigatorScale(lowerLimitsPixels);

        //Upper bounds computed using nav canvas.
        final Point upperLimitsPixels = new Point(_navigatorCanvas.getCurrentViewedRectangle().x
            + _navigatorCanvas.getCurrentViewedRectangle().width, _navigatorCanvas.getCurrentViewedRectangle().y);
        final DataPoint upperBounds = computeDataPointFromPixelPointInNavigatorScale(upperLimitsPixels);

        //Update the chart bounds
        updateChartForNewLimits(lowerBounds, upperBounds);
    }

    /**
     * Computes a pixel point given a data point, based on the {@link #_originalAxesLowerBounds} and
     * {@link #_originalAxesUpperBounds} attributes. It will not allow a pixel to go outside the _navigatorCanvas's
     * preferred dimensions.
     * 
     * @param pointInOriginalScale Data point in data scale.
     * @return {@link Point} specifying pixels.
     */
    private Point computeDrawnPixelPointInNavigatorScaleFromDataPoint(final DataPoint pointInOriginalScale)
    {
        final Point results = new Point();

        results.x = (int)((_navigatorCanvas.getBackgroundImageSize().getWidth())
            * (pointInOriginalScale.getX() - _originalAxesLowerBounds.getX()) / (_originalAxesUpperBounds.getX() - _originalAxesLowerBounds.getX()));

        //The y-axis is reversed: the smallest pixel is at the top, not the bottom.
        results.y = (int)(_navigatorCanvas.getBackgroundImageSize().getHeight())
            - (int)((_navigatorCanvas.getBackgroundImageSize().getHeight())
                * (pointInOriginalScale.getY(0) - _originalAxesLowerBounds.getY(0)) / (_originalAxesUpperBounds.getY(0) - _originalAxesLowerBounds.getY(0)));

        return results;
    }

    /**
     * Computes a data point given a pixel point, based on the {@link #_originalAxesLowerBounds} and
     * {@link #_originalAxesUpperBounds} attributes. It will not allow a pixel to go outside the axis limits.
     * 
     * @param pointInOriginalScale Data point.
     * @return {@link Point} specifying pixels relative to the {@link #_viewingPanel}.
     */
    private DataPoint computeDataPointFromPixelPointInNavigatorScale(final Point pointInViewer)
    {
        final DataPoint results = new DataPoint(2);

        final XYPlot plot = getNavigatedXYPlot();

        final double domainValue = _originalAxesLowerBounds.getX()
            + (_originalAxesUpperBounds.getX() - _originalAxesLowerBounds.getX())
            * (pointInViewer.getX() / _navigatorCanvas.getBackgroundImageSize().width);
        results.setX(domainValue);

        double rangeValue = _originalAxesLowerBounds.getY(0)
            + (_originalAxesUpperBounds.getY(0) - _originalAxesLowerBounds.getY(0))
            * (_navigatorCanvas.getBackgroundImageSize().height - pointInViewer.getY())
            / _navigatorCanvas.getBackgroundImageSize().height;
        results.setY(0, rangeValue);

        if(plot.getRangeAxisCount() == 2)
        {
            rangeValue = _originalAxesLowerBounds.getY(1)
                + (_originalAxesUpperBounds.getY(1) - _originalAxesLowerBounds.getY(1))
                * (_navigatorCanvas.getBackgroundImageSize().height - pointInViewer.getY())
                / _navigatorCanvas.getBackgroundImageSize().height;
            results.setY(1, rangeValue);
        }
        else
        {
            results.setY(1, Double.NaN);
        }
        return results;
    }

    /**
     * Looks at the {@link #getNavigatedXYPlot()} and pulls the axis limits for storage in
     * {@link #_originalAxesLowerBounds} and {@link #_originalAxesUpperBounds}. It can handle either one or two range
     * axes and only one domain axis.
     */
    private void acquireOriginalAxisLimitsFromChart()
    {

        final XYPlot plot = getNavigatedXYPlot();
        if((plot.getRangeAxisCount() == 1) || (plot.getRangeAxisCount() == 2))
        {
            _originalAxesLowerBounds = ChartTools.constructLowerBoundsDataPoint(_navigatedChart, 0);
            _originalAxesUpperBounds = ChartTools.constructUpperBoundsDataPoint(_navigatedChart, 0);
        }
        else
        {
            throw new IllegalArgumentException("Expected only one or two range axes, but found "
                + plot.getRangeAxisCount() + ".");
        }
    }

    /**
     * Adds {@link #_axisChangeListener} to the domain axis and any (one or two) range axes.
     */
    private void addAxisChangeListener()
    {
        getNavigatedXYPlot().getDomainAxis().addChangeListener(_axisChangeListener);
        getNavigatedXYPlot().getRangeAxis(0).addChangeListener(_axisChangeListener);
        if(getNavigatedXYPlot().getRangeAxisCount() > 1)
        {
            getNavigatedXYPlot().getRangeAxis(1).addChangeListener(_axisChangeListener);
        }
    }

    /**
     * Removes {@link #_axisChangeListener} from the domain axis and any (one or two) range axes.
     */
    private void removeAxisChangeListener()
    {
        getNavigatedXYPlot().getDomainAxis().removeChangeListener(_axisChangeListener);
        getNavigatedXYPlot().getRangeAxis(0).removeChangeListener(_axisChangeListener);
        if(getNavigatedXYPlot().getRangeAxisCount() > 1)
        {
            getNavigatedXYPlot().getRangeAxis(1).removeChangeListener(_axisChangeListener);
        }
    }

    /**
     * @return The {@link CombinedDomainXYPlot} that must underly the {@link #_navigatedChart}.
     */
    private XYPlot getNavigatedXYPlot()
    {
        final CombinedDomainXYPlot plot = (CombinedDomainXYPlot)_navigatedChart.getXYPlot();
        return (XYPlot)plot.getSubplots().get(0);
    }

    /**
     * Passes call through to {@link #_navigatedCanvas} to set the rectangle line color.
     */
    public void setSelectionRectangleColor(final Color selectionRectangleColor)
    {
        _navigatorCanvas.setSelectionRectangleColor(selectionRectangleColor);
    }

    /**
     * Passes call through to {@link #_navigatedCanvas} to set the rectangle fill color.
     */
    public void setSelectionRectangleFillColor(final Color selectionRectangleFillColor)
    {
        _navigatorCanvas.setSelectionRectangleFillColor(selectionRectangleFillColor);
    }

    /**
     * Passes the {@link Composite} to the {@link #_navigatorCanvas} for use when it draws its image.
     */
    public void setNavigationConvasAppliedComposite(final Composite c)
    {
        _navigatorCanvas.setAppliedComposite(c);
    }

    public void setMaintainAspectRatio(final boolean maintainAspectRatio)
    {
        _maintainAspectRatio = maintainAspectRatio;
    }

    public Image getCurrentFullSizeChartImage()
    {
        return _navigatorCanvas.getFullSizeImage();
    }

    public Rectangle getCurrentPlotAreaRectangle()
    {
        return _navigatorCanvas.getNavigatedPartOfImageRectangle();
    }

    /**
     * Register the supplied Object to the {@link EventBus} that underlies this panel. This is public because a testing
     * tool needs to register from the GraphGen package.
     */
    public void registerWithBus(final NavigatedSubplotAxisLimitsChangedNotice.Subscriber registree)
    {
        _eventBus.register(registree);
    }

    /**
     * Unregister the provided subscriber. This is public to be consistent with
     * {@link #registerWithBus(ohd.hseb.charter.panel.notices.NavigatedSubplotAxisLimitsChangedNotice.Subscriber)}.
     */
    public void unregisterWithBus(final NavigatedSubplotAxisLimitsChangedNotice.Subscriber registree)
    {
        _eventBus.unregister(registree);
    }

    /**
     * Overrides Component. Sets the preferred size and then turns on the _preferredSizeIsSet flag so that it is not set
     * again.
     */
    @Override
    public void setPreferredSize(final Dimension dim)
    {
        super.setPreferredSize(new Dimension((int)dim.getWidth(), (int)dim.getHeight()));
    }

    @Override
    @Subscribe
    public void reactToViewedRectangleChanged(final NavigationCanvasRectangleChangedNotice evt)
    {
        updateChartFromNavigatorCanvas();
        repaint();
    }

    @Override
    public void paint(final Graphics g)
    {
        super.paint(g);
    }

    /**
     * Constructs a {@link CombinedDomainChartNavigationPanel} tied to the provided {@link JFreeChart}, which is
     * displayed in the provided {@link OHDFixedChartPanel}. The background color for the panel will be set to a
     * partially transparent gray.<br>
     * <br>
     * If the displayAsNeeded flag will be false, it is best to just call the constructor,
     * {@link CombinedDomainChartNavigationPanel#CombinedDomainChartNavigationPanel(JFreeChart, OHDFixedChartPanel, Dimension)}
     * .
     * 
     * @param panelDisplayingChart The panel that displays the chart. The chart must display a
     *            {@link CombinedDomainXYPlot} as its primary plot. Only the subplot with index 0 will be tracked.
     * @param allowedPanelDimensions See {@link #_allowedPanelDimensions}. Can be null.
     * @param displayOnlyAsNeeded If false, the returned panel has its visible flag set to true and it will not change
     *            unless modified externally. If this argument is true, the returned panel will have its visible flag is
     *            set to false. Furthermore, a listener is created that listens to the returned
     *            {@link CombinedDomainChartNavigationPanel} for any axis limits changed (via the
     *            {@link NavigatedSubplotAxisLimitsChangedNotice}), either through the navigator or chart. If the axis
     *            limits do not match the original limits when this method was called, then the returned panel will
     *            become visible (i.e., setVisible(true)) wherever it is located. Otherwise, it will not be visible.
     */
    public static CombinedDomainChartNavigationPanel applyPopupNavigationPanelToChart(final OHDFixedChartPanel panelDisplayingChart,
                                                                                      final Dimension allowePanelDimensions,
                                                                                      final boolean displayOnlyAsNeeded)
    {

        final CombinedDomainChartNavigationPanel navPanel = new CombinedDomainChartNavigationPanel(panelDisplayingChart,
                                                                                                   allowePanelDimensions);
        navPanel.setBackground(new Color(64, 64, 64, 128)); //Partially transparent background. 

        //Makes the glass pane containing the panel visible only as needed.
        if(displayOnlyAsNeeded)
        {
            navPanel.registerWithBus(new NavigatedSubplotAxisLimitsChangedNotice.Subscriber()
            {
                private final DataPoint _originalLowerBounds = ChartTools.constructLowerBoundsDataPoint(panelDisplayingChart.getChart(),
                                                                                                                    0);
                private final DataPoint _originalUpperBounds = ChartTools.constructUpperBoundsDataPoint(panelDisplayingChart.getChart(),
                                                                                                                    0);

                @Override
                @Subscribe
                public void reactToAxisLimitsChanged(final NavigatedSubplotAxisLimitsChangedNotice evt)
                {
                    //If any part of the original limits is not completely included within the new limits, then the nav panel will be visible!

                    //I convert  to ranges because it is easier to think about the problem conceptually using
                    //the contains function within a Range.
                    final Range newDomainRange = new Range(evt.getNewDomainAxisLowerBound(),
                                                           evt.getNewDomainAxisUpperBound());
                    final Range[] newYRanges = new Range[_originalLowerBounds.getNumberOfYVariables()];
                    for(int i = 0; i < newYRanges.length; i++)
                    {
                        newYRanges[i] = new Range(evt.getNewRangeAxisLowerBound(i), evt.getNewRangeAxisUpperBound(i));
                    }

                    //Do the checks in the domain axis...
                    if(!newDomainRange.contains(_originalLowerBounds.getX())
                        || !newDomainRange.contains(_originalUpperBounds.getX()))
                    {
                        navPanel.setVisible(true);
                        return;
                    }
                    //Then the range axes...
                    for(int i = 0; i < newYRanges.length; i++)
                    {
                        if(!newYRanges[i].contains(_originalLowerBounds.getY(i))
                            || !newYRanges[i].contains(_originalUpperBounds.getY(i)))
                        {
                            navPanel.setVisible(true);
                            return;
                        }
                    }

                    //Made it this far, so it is not visible.
                    navPanel.setVisible(false);
                }
            });

            //Set vis to false by default.
            navPanel.setVisible(false);
        }
        //Set vis to true and return.
        else
        {
            navPanel.setVisible(true);
        }
        return navPanel;
    }
}
