package ohd.hseb.charter.panel;

import java.util.Date;
import java.util.HashMap;
import java.util.List;

import nl.wldelft.util.timeseries.TimeSeriesArray;
import ohd.hseb.charter.ChartEngine;
import ohd.hseb.charter.datasource.XYChartDataSource;
import ohd.hseb.charter.datasource.instances.TimeSeriesArraysXYChartDataSource;
import ohd.hseb.hefs.utils.tools.ListTools;
import ohd.hseb.hefs.utils.tools.NumberTools;
import ohd.hseb.hefs.utils.tsarrays.TimeSeriesArrayTools;
import ohd.hseb.util.misc.HCalendar;

import org.jfree.chart.axis.ValueAxis;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;

/**
 * This table model can only be applied if all sources are {@link TimeSeriesArraysXYChartDataSource} instances.
 * Furthermore, the time series provided by those sources must all have the same time step, and the start times of all
 * time series must differ by a factor of that time step. This will allow a common domain of times to be constructed for
 * all time series. If this requirement is not satisfied, {@link #_dataSourceToStartAndEndTimes} will not be populated,
 * and problems may occur.<br>
 * <br>
 * See the CFSv2 monthly chart diagnostic for an example of unsharing a
 * {@link DomainSharingTimeSeriesChartEngineTableModel}, if necessary. Basically, any method that assumes shared time
 * must be modified.
 * 
 * @author hankherr
 */
@SuppressWarnings("serial")
public class DomainSharingTimeSeriesChartEngineTableModel extends AbstractChartEngineTableModel
{

    /**
     * Records the start and end times for each data source.
     */
    private final HashMap<XYChartDataSource, long[]> _dataSourceToStartAndEndTimes = Maps.newHashMap();

    private long _currentStepSizeInMillis = -0L;

    public DomainSharingTimeSeriesChartEngineTableModel()
    {
        super();
    }

    public DomainSharingTimeSeriesChartEngineTableModel(final ChartEngine chartEngine)
    {
        super(chartEngine);
    }

    /**
     * Uses {@link NumberTools#computeGCD(Integer[])} to determine the step size.
     */
    private long determineStepSize(final TimeSeriesArraysXYChartDataSource source)
    {
        if(source.getTimeSeries().isEmpty())
        {
            return 0L;
        }
        else if(source.getTimeSeries().size() == 1)
        {
            return source.getTimeSeries().get(0).getHeader().getTimeStep().getStepMillis();
        }
        else
        {
            final List<Integer> stepSizesInHours = Lists.newArrayList();
            for(int i = 0; i < source.getTimeSeries().size(); i++)
            {
                final int hours = (int)((double)source.getTimeSeries().get(i).getHeader().getTimeStep().getStepMillis() / (double)HCalendar.MILLIS_IN_HR);
                ListTools.addItemIfNotAlreadyInList(stepSizesInHours, hours);
            }
            return NumberTools.computeGCD(stepSizesInHours.toArray(new Integer[1])) * HCalendar.MILLIS_IN_HR;
        }
    }

    /**
     * Assumes all time series for the current data source use the same step size, which is ensured by the call to
     * {@link #checkForValidityOfSource(TimeSeriesArraysXYChartDataSource)} made when building
     * {@link #_dataSourceToStartAndEndTimes}.
     * 
     * @return Shared step size.
     */
    public long getStepSizeInMillis()
    {
        return _currentStepSizeInMillis;
    }

    /**
     * @throws An {@link IllegalArgumentException} is throws if the start times of any two time series do not line up
     *             based on the step size to use computed via
     *             {@link #determineStepSize(TimeSeriesArraysXYChartDataSource)}. If they don't line up, it becomes
     *             impossible to display the time series in a single table.
     */
    private void checkForValidityOfSource(final TimeSeriesArraysXYChartDataSource source)
    {
        final long stepSizeMillis = determineStepSize(source);

        if(!source.getTimeSeries().isEmpty())
        {
            final TimeSeriesArray firstTS = source.getTimeSeries().get(0);
            //All time series must have identical time steps and compatible starts.
            for(int i = 1; i < source.getTimeSeries().size(); i++)
            {
                final TimeSeriesArray checkTS = source.getTimeSeries().get(i);
                if((checkTS.getStartTime() - firstTS.getStartTime()) % stepSizeMillis != 0)
                {
                    throw new IllegalArgumentException("Time series are not aligned relative to the time step, "
                        + (int)((double)stepSizeMillis / (double)HCalendar.MILLIS_IN_HR) + " hours: "
                        + HCalendar.buildDateStr(firstTS.getStartTime()) + " and "
                        + HCalendar.buildDateStr(checkTS.getStartTime()));
                }
            }
        }
    }

    /**
     * @return An array of two longs: smallest start and largest end times. Used for determining domain values for which
     *         to display table rows.
     */
    private long[] computeStartAndEndForDataSource(final TimeSeriesArraysXYChartDataSource source)
    {
        if(source.getTimeSeries().isEmpty())
        {
            return null;
        }

        //Find the smallest start time and largest end time.
        long startTime = Long.MAX_VALUE;
        long endTime = Long.MIN_VALUE;

        for(int i = 0; i < source.getTimeSeries().size(); i++)
        {
            final TimeSeriesArray checkTS = source.getTimeSeries().get(i);
            if(checkTS.getStartTime() < startTime)
            {
                startTime = checkTS.getStartTime();
            }
            if(checkTS.getEndTime() > endTime)
            {
                endTime = checkTS.getEndTime();
            }
        }

        return new long[]{startTime, endTime};
    }

    /**
     * Constructs the {@link #_dataSourceToStartAndEndTimes} mapping. It also calls the
     * {@link #checkForValidityOfSource(TimeSeriesArraysXYChartDataSource)} method to confirm validity of using this
     * model relative to the current chart engine.
     */
    private void buildDataSourceToStartAndEndTimesMap()
    {
        for(int i = 0; i < getDataSources().size(); i++)
        {
            if(!(getDataSources().get(i) instanceof TimeSeriesArraysXYChartDataSource))
            {
                throw new IllegalArgumentException("Data source provided at index " + i
                    + " is not a TimeSeriesArraysXYChartDataSource.");
            }

            final TimeSeriesArraysXYChartDataSource source = getDataSource(i);
            checkForValidityOfSource(source);
            final long[] times = computeStartAndEndForDataSource(source);
            _dataSourceToStartAndEndTimes.put(source, times);
        }
    }

    /**
     * @return The time series corresponding to the provided column. Null is returned for column 0, since that is the
     *         shared domain column.
     */
    public TimeSeriesArray getTimeSeries(final int column)
    {
        final int seriesIndex = computeSeriesIndex(column);
        if(seriesIndex < 0)
        {
            return null;
        }
        return getCurrentDataSource().getTimeSeries().get(seriesIndex);
    }

    /**
     * @param source The source for which to get the time for the given row. This need not be the current source.
     * @return The time correspond to the row in the form of a {@link Date}. If the return is null, then the underlying
     *         data is invalid and the row count should be 0.
     */
    public Date getTimeForRow(final int modelRow, final TimeSeriesArraysXYChartDataSource source)
    {
        final long[] times = _dataSourceToStartAndEndTimes.get(source);
        long stepMillis = getStepSizeInMillis();
        if(source != getCurrentDataSource())
        {
            stepMillis = determineStepSize(source);
        }
        if((times == null) || (stepMillis <= 0))
        {
            return null;
        }
        return new Date(times[0] + modelRow * stepMillis);
    }

    /**
     * @return True if any time series value is within the currently displayed chart limits.
     */
    protected boolean isAnyTimeSeriesValueDisplayedWithinLimits(final int column)
    {
        final ValueAxis domainAxis = getDomainAxis();
        final ValueAxis rangeAxis = getRangeAxis();
        if(rangeAxis == null)
        {
            return false;
        }
        final TimeSeriesArray ts = getTimeSeries(column);

        if(ts == null)
        {
            return true;
        }
        if(ts.getStartTime() > domainAxis.getRange().getUpperBound())
        {
            return false;
        }
        if(ts.getEndTime() < domainAxis.getRange().getLowerBound())
        {
            return false;
        }

        //If any ts value is within the range axis limits, return true.
        for(int i = 0; i < ts.size(); i++)
        {
            if(rangeAxis.getRange().contains(ts.getValue(i)))
            {
                return true;
            }
        }
        return false;

    }

    /**
     * @return Computes the number of rows that will be in the table for the given source.
     */
    protected int computeRowCount(final TimeSeriesArraysXYChartDataSource source)
    {
        final long[] times = _dataSourceToStartAndEndTimes.get(source);
        long stepMillis = getStepSizeInMillis();
        if(source != getCurrentDataSource())
        {
            stepMillis = determineStepSize(source);
        }
        if((times == null) || (stepMillis <= 0))
        {
            return 0;
        }
        return (int)((times[1] - times[0]) / stepMillis) + 1;
    }

    @Override
    public int computeRowForItem(final int seriesIndex, final int seriesItemNumber)
    {
        final long[] times = _dataSourceToStartAndEndTimes.get(getCurrentDataSource());
        if((times == null) || (getStepSizeInMillis() == 0))
        {
            return -1;
        }

        //Range ----------
        //First row with data is the difference between the start time and overall start time divided by the time step.
        TimeSeriesArray ts = null;
        ts = getCurrentDataSource().getTimeSeries().get(seriesIndex);
        final long startTimeDiff = ts.getStartTime() - times[0];
        final int firstRowWithData = (int)(startTimeDiff / getStepSizeInMillis());

        //Multiplying the seriesItemNumber by the number of rows between each displayed value and adding to the first row.
        return firstRowWithData + seriesItemNumber
            * (int)((double)ts.getHeader().getTimeStep().getStepMillis() / (double)getStepSizeInMillis());
    }

    @Override
    public void setChartEngineWithoutFiringEvent(final ChartEngine engine)
    {
        for(final XYChartDataSource source: engine.getDataSources())
        {
            if(!(source instanceof TimeSeriesArraysXYChartDataSource))
            {
                throw new IllegalArgumentException("At least once source within the provided chart engine is not a TimeSeriesArraysXYChartDataSource.");
            }
        }
        super.setChartEngineWithoutFiringEvent(engine);
        buildDataSourceToStartAndEndTimesMap();
        _currentStepSizeInMillis = determineStepSize(getCurrentDataSource());
    }

    /**
     * Convenience wrapper changes return class.
     */
    @Override
    protected TimeSeriesArraysXYChartDataSource getCurrentDataSource()
    {
        return (TimeSeriesArraysXYChartDataSource)super.getCurrentDataSource();
    }

    /**
     * Convenience wrapper changes return class.
     */
    @Override
    public TimeSeriesArraysXYChartDataSource getDataSource(final int index)
    {
        return (TimeSeriesArraysXYChartDataSource)super.getDataSource(index);
    }

    /**
     * Calls {@link #buildDataSourceToStartAndEndTimesMap()} each time the engine is changed.
     */
    @Override
    public void setChartEngine(final ChartEngine engine)
    {
        setChartEngineWithoutFiringEvent(engine);
        fireTableStructureChanged();
    }

    @Override
    public void setDataSourceIndexWithoutFiringEvent(final int dataSourceIndex)
    {
        super.setDataSourceIndexWithoutFiringEvent(dataSourceIndex);
        _currentStepSizeInMillis = determineStepSize(getCurrentDataSource());
    }

    @Override
    protected boolean areAllSeriesXaxisValuesSame()
    {
        return true;
    }

    @Override
    public String getColumnName(final int col)
    {
        if(col == 0)
        {
            return "time (GMT)";
        }
        if(getCurrentDataSource().getTimeSeries().isEmpty())
        {
            return "-none-";
        }
        final long forecastTime = getCurrentDataSource().getTimeSeries().get(col - 1).getHeader().getForecastTime();
        if((forecastTime == Long.MIN_VALUE) || (forecastTime == Long.MAX_VALUE))
        {
            //When its not a forecast time series, it is presumed it is not an ensemble.  Hence it is a single value time series.
            //The parameter id is the best way to identify between single time series.
            return "<html>Series " + (col - 1) + "<br>"
                + getCurrentDataSource().getTimeSeries().get(col - 1).getHeader().getParameterId() + "</html>";
        }
        return "<html>Series " + (col - 1) + "<br>" + HCalendar.buildDateStr(forecastTime, "CCYY-MM-DD hh'h'") + "<br>"
            + getCurrentDataSource().getTimeSeries().get(col - 1).getHeader().getParameterId() + "</html>";
    }

    @Override
    public int getRowCount()
    {
        return computeRowCount(getCurrentDataSource());
    }

    @Override
    public Object getRawValueAt(final int modelRow, final int modelColumn)
    {
        //Domain ----------
        if(modelColumn == 0)
        {
            return getTimeForRow(modelRow, getCurrentDataSource());
        }

        final long[] times = _dataSourceToStartAndEndTimes.get(getCurrentDataSource());
        if((times == null) || (getStepSizeInMillis() <= 0))
        {
            return null;
        }

        //Range ----------
        //First row with data is the difference between the start time and overall start time divided by the time step.
        TimeSeriesArray ts = null;
        ts = getCurrentDataSource().getTimeSeries().get(modelColumn - 1);
        if(ts != null)
        {
            //Use the time series array 
            return TimeSeriesArrayTools.getValueByTime(ts, times[0] + modelRow * getStepSizeInMillis());
        }
        return null;
    }
}
