package ohd.hseb.charter.panel;

import java.awt.Color;
import java.awt.Composite;
import java.awt.Cursor;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Image;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;

import javax.swing.JPanel;

import ohd.hseb.charter.panel.notices.NavigationCanvasRectangleChangedNotice;

import com.google.common.eventbus.EventBus;

/**
 * This generic class provides a mechanism that displays a shrunken down version of an image, and then draws rectangles
 * on top of that image as instructed via the {@link #setCurrentViewedRectangle(Rectangle)}. This is useful as a
 * navigation window, where the user can focus in on smaller areas of a larger graphic.
 * 
 * @author hank
 */
public class NavigationCanvas extends JPanel
{
    private static final long serialVersionUID = 1L;

    /**
     * The rectangle to draw showing the viewed region of the master image.
     */
    private Rectangle _currentViewedRectangle;

    /**
     * The full chart image, used to acquire the background.
     */
    private Image _fullSizeImage;

    /**
     * The navigated image, cut out from the {@link #_fullSizeImage} based on {@link #_navigatedPartOfImageRectangle}
     * and resized to fit this canvas.
     */
    private Image _navigatedImage;

    /**
     * The plot area specifying the portion of the full size image to draw into this canvas.
     */
    private Rectangle _navigatedPartOfImageRectangle;

    /**
     * Boolean tells the canvas if the navigator map is clickable and press-drag-releasable.
     */
    private boolean _reactToMouseEvents;

    /**
     * Boolean is true if the box is currently being dragged.
     */
    private boolean _isBoxBeingMoved;

    /**
     * Composite applied before painting the mini image. Good for making partially transparent images.
     */
    private Composite _appliedComposite = null;

    /**
     * {@link NavigationCanvasRectangleChangedNotice} instances are posted to this bus whenever the rectangle is moved.
     */
    private EventBus _eventBus = null;

    private Color _selectionRectangleColor = Color.black;
    private Color _selectionRectangleFillColor = new Color(128, 128, 128, 64);

    /**
     * Listens for press/releases and dragged mouse events.
     */
    private final MouseAdapter _mouseAdapter = new MouseAdapter()
    {
        @Override
        public void mousePressed(final MouseEvent e)
        {
            if(_reactToMouseEvents)
            {
                if(isCurrentViewedRectangleWholeImage())
                {
                    return;
                }
                if(!isClickWithinImage(e.getPoint()))
                {
                    return;
                }
                recenterDrawnRectangle(e.getPoint());
                repaint();
                _isBoxBeingMoved = true;
            }
        }

        @Override
        public void mouseReleased(final MouseEvent e)
        {
            if(_reactToMouseEvents)
            {
                if(!_isBoxBeingMoved)
                {
                    return;
                }
                if(isCurrentViewedRectangleWholeImage())
                {
                    return;
                }
                recenterDrawnRectangle(e.getPoint());
                repaint();
                _isBoxBeingMoved = false;
                NavigationCanvas.this.setCursor(Cursor.getPredefinedCursor(Cursor.DEFAULT_CURSOR));
                _eventBus.post(new NavigationCanvasRectangleChangedNotice(this, _currentViewedRectangle));
            }
        }

        @Override
        public void mouseDragged(final MouseEvent arg0)
        {
            if(_reactToMouseEvents)
            {
                if(!_isBoxBeingMoved)
                {
                    return;
                }
                if(isCurrentViewedRectangleWholeImage())
                {
                    return;
                }
                recenterDrawnRectangle(arg0.getPoint());
                repaint();
            }

        }
    };

    /**
     * Calls {@link #NavigationCanvas(boolean)} passing in true.
     */
    public NavigationCanvas()
    {
        this(true);
    }

    /**
     * @param includeMouseListener True if the canvas is interactive.
     */
    public NavigationCanvas(final boolean includeMouseListener)
    {
        _currentViewedRectangle = null;
        _fullSizeImage = null;
        _navigatedImage = null;
        _navigatedPartOfImageRectangle = null;
        if(includeMouseListener)
        {
            this.addMouseListener(_mouseAdapter);
            this.addMouseMotionListener(_mouseAdapter);
        }
    }

    /**
     * Set the rectangle line color.
     */
    public void setSelectionRectangleColor(final Color selectionRectangleColor)
    {
        _selectionRectangleColor = selectionRectangleColor;
    }

    /**
     * Set the rectangle fill color.
     */
    public void setSelectionRectangleFillColor(final Color selectionRectangleFillColor)
    {
        _selectionRectangleFillColor = selectionRectangleFillColor;
    }

    /**
     * The composite provided is applied during painting. I am not certain if this is working properly.
     */
    public void setAppliedComposite(final Composite composite)
    {
        _appliedComposite = composite;
    }

    /**
     * Compute the preferred size by looking at the plot area and shrinking it. The aspect ratio is always maintained
     * for the navigated image when shrinking.
     */
    public void computeScaleFactorAndPreferredSize(final Dimension containerPreferredSize)
    {
        setPreferredSize(containerPreferredSize);
    }

    /**
     * Set the components used to determine the background image.
     * 
     * @param fullSizeChartImage The complete initial image.
     * @param plotArea Rectangle specifying the plot region within the full size chart image.
     */
    public void setNavigatedImageComponents(final Image fullSizeChartImage, final Rectangle plotArea)
    {
        _navigatedPartOfImageRectangle = plotArea;
        _fullSizeImage = fullSizeChartImage;
        _navigatedImage = null;
        repaint();
    }

    /**
     * Creates the background image using the full size chart image and the scale factor.
     */
    private void createNavigatedImage()
    {
        if((_fullSizeImage == null) || (_navigatedPartOfImageRectangle == null)
            || (_navigatedPartOfImageRectangle.getWidth() == 0) || (_navigatedPartOfImageRectangle.getHeight() == 0))
        {
            _navigatedImage = null;
            return;
        }
        _navigatedImage = this.createImage((int)getPreferredSize().getWidth(), (int)getPreferredSize().getHeight());
        final Graphics g = _navigatedImage.getGraphics();

        g.drawImage(_fullSizeImage,
                    0,
                    0,
                    _navigatedImage.getWidth(this),
                    _navigatedImage.getHeight(this),
                    (int)_navigatedPartOfImageRectangle.getX(),
                    (int)_navigatedPartOfImageRectangle.getY(),
                    (int)(_navigatedPartOfImageRectangle.getX() + _navigatedPartOfImageRectangle.getWidth()),
                    (int)(_navigatedPartOfImageRectangle.getY() + _navigatedPartOfImageRectangle.getHeight()),
                    this);
    }

    /**
     * Recenter the drawn rectangle at the passed in coordinates. Event posting is handled outside of this method, so
     * when it calls {@link #shiftCurrentViewedRectangleByPixels(int, int, boolean)} to correct for moving outside the
     * bounds of the plot, it will NOT fire an event.
     */
    private void recenterDrawnRectangle(final Point point)
    {
        if(_navigatedImage == null)
        {
            return;
        }

        //Recenter the rectangle at the provided point.
        final int newXCoordinate = point.x - (_currentViewedRectangle.width / 2);
        final int newYCoordinate = point.y - (_currentViewedRectangle.height / 2);
        _currentViewedRectangle = new Rectangle(newXCoordinate,
                                                newYCoordinate,
                                                _currentViewedRectangle.width,
                                                _currentViewedRectangle.height);

        //Compute shift that keeps the rectangle within the image.  
        int widthShift = 0;
        if(_currentViewedRectangle.x < 0)
        {
            widthShift = -1 * _currentViewedRectangle.x;
        }
        if(_currentViewedRectangle.x + _currentViewedRectangle.width > _navigatedImage.getWidth(this))
        {
            widthShift = _navigatedImage.getWidth(this) - (_currentViewedRectangle.x + _currentViewedRectangle.width);
        }
        int heightShift = 0;
        if(_currentViewedRectangle.y < 0)
        {
            heightShift = -1 * _currentViewedRectangle.y;
        }
        if(_currentViewedRectangle.y + _currentViewedRectangle.height > _navigatedImage.getHeight(this))
        {
            heightShift = _navigatedImage.getHeight(this)
                - (_currentViewedRectangle.y + _currentViewedRectangle.height);
        }
        shiftCurrentViewedRectangleByPixels(widthShift, heightShift, false);
    }

    /**
     * @param widthNumPixels Number of pixels to shift along x-axis.
     * @param heightNumPixels Number of pixes to shift along y-axis.
     * @param postEvent If true, post a {@link NavigationCanvasRectangleChangedNotice} after shifting.
     */
    private void shiftCurrentViewedRectangleByPixels(final int widthNumPixels,
                                                     final int heightNumPixels,
                                                     final boolean postEvent)
    {
        if((_navigatedImage == null) || (isCurrentViewedRectangleWholeImage()))
        {
            return;
        }

        final Rectangle oldRect = _currentViewedRectangle;

        final int newXCoordinate = oldRect.x + widthNumPixels;
        final int newYCoordinate = oldRect.y + heightNumPixels;

        _currentViewedRectangle = new Rectangle(newXCoordinate, newYCoordinate, oldRect.width, oldRect.height);

        //Fire the property change only if necessary.
        if((_currentViewedRectangle.x != oldRect.x) || (_currentViewedRectangle.y != oldRect.y))
        {
            repaint();
            if(postEvent)
            {
                _eventBus.post(new NavigationCanvasRectangleChangedNotice(this, _currentViewedRectangle));
            }
        }
    }

    /**
     * CURRENTLY, this is not used!! Shifting is handled within the panel containing this in its mouse listener.<br>
     * <br>
     * Shifts the currently viewed rectangle based on the two passed in double. Each will be multiplied by the
     * corresponding dimension of the {@link #_currentViewedRectangle} and used to adjust the rectangle position. This
     * calls {@link #shiftCurrentViewedRectangleByPixels(int, int, boolean)}.
     * 
     * @parma widthMovementFactor Multiplied by width of rectangle to shift x coordinate.
     * @param heightMovementFactor Multiplied by height of rectangle to shift y coordinate.
     * @param postEvent If true, post a {@link NavigationCanvasRectangleChangedNotice} after shifting.
     */
    public void shiftCurrentViewedRectangle(final double widthMovementFactor,
                                            final double heightMovementFactor,
                                            final boolean postEvent)
    {
        shiftCurrentViewedRectangleByPixels((int)((widthMovementFactor) * _currentViewedRectangle.width),
                                            (int)((heightMovementFactor) * _currentViewedRectangle.height),
                                            postEvent);
    }

    /**
     * @return true if the current viewed rectangle is the whole image.
     */
    public boolean isCurrentViewedRectangleWholeImage()
    {
        if(_currentViewedRectangle == null)
        {
            return true;

        }
        if((_currentViewedRectangle.getX() == 0) && (_currentViewedRectangle.getY() == 0)
            && (_currentViewedRectangle.getWidth() == _navigatedImage.getWidth(this))
            && (_currentViewedRectangle.getHeight() == _navigatedImage.getHeight(this)))
        {
            return true;
        }
        return false;
    }

    /**
     * Return true if the passed in point
     */
    private boolean isClickWithinImage(final Point pt)
    {
        if((pt.x < 0) || (pt.y < 0) || (pt.x > _navigatedImage.getWidth(this))
            || (pt.y > _navigatedImage.getHeight(this)))
        {
            return false;
        }
        return true;
    }

    /**
     * Sets the drawn rectangle and sets {@link #_canvasNeedsPainting} to true. Makes sure the drawn rectangle has a
     * width and height of at least 1; if less than one, it forces it to one.
     * 
     * @param drawnRectangle The new rect. This is copied; it is not referred to by pointer. null is acceptable.
     */
    public void setCurrentViewedRectangle(final Rectangle drawnRectangle)
    {
        if(drawnRectangle == null)
        {
            _currentViewedRectangle = null;
        }
        else
        {
            _currentViewedRectangle = new Rectangle(drawnRectangle.x,
                                                    drawnRectangle.y,
                                                    Math.max(1, drawnRectangle.width),
                                                    Math.max(1, drawnRectangle.height));
        }
        repaint();
    }

    public void setFullSizeImage(final Image fullSizeImage)
    {
        _fullSizeImage = fullSizeImage;
    }

    public boolean hasFullSizeImageBeenSet()
    {
        return (_fullSizeImage != null);
    }

    public void setReactToMouseEvents(final boolean b)
    {
        _reactToMouseEvents = b;
    }

    public boolean getReactToMouseEvents()
    {
        return _reactToMouseEvents;
    }

    public Image getFullSizeImage()
    {
        return _fullSizeImage;
    }

    public Rectangle getCurrentViewedRectangle()
    {
        return _currentViewedRectangle;
    }

    public Dimension getBackgroundImageSize()
    {
        if(_navigatedImage == null)
        {
            return getPreferredSize();
        }
        return new Dimension(_navigatedImage.getWidth(this), _navigatedImage.getHeight(this));
    }

    public Rectangle getNavigatedPartOfImageRectangle()
    {
        return _navigatedPartOfImageRectangle;
    }

    /**
     * The {@link EventBus} is only needed for posting, but this will be registered just in case.
     * 
     * @param eventBus The event bus for posting {@link NavigationCanvasRectangleChangedNotice} instances.
     */
    public void setEventBus(final EventBus eventBus)
    {
        _eventBus = eventBus;
        _eventBus.register(this);
    }

    /**
     * Paints the image by first creating the background image, if it does not already exist, and then drawing the
     * background image (if it exists) and the drawn rectangle (if it exists).
     */
    @Override
    public void paint(final Graphics g)
    {
        g.clearRect(0, 0, this.getWidth(), this.getHeight());
        if(_appliedComposite != null)
        {
            ((Graphics2D)g).setComposite(_appliedComposite);
        }

        //Create the background image if it needs updating.
        if(_navigatedImage == null)
        {
            createNavigatedImage();
        }

        //If the background was successfully created, draw it into the graphics.
        if(_navigatedImage != null)
        {
            g.drawImage(_navigatedImage, 0, 0, this);
        }

        //Draw the rectangle.  If the rectangle is null, draw one around the entire background image.
        if(_currentViewedRectangle != null)
        {
            final Color oldC = g.getColor();
            g.setColor(_selectionRectangleFillColor);
            g.fillRect((int)_currentViewedRectangle.getX(),
                       (int)_currentViewedRectangle.getY(),
                       (int)_currentViewedRectangle.getWidth() - 1, //See the javadoc on this method: -1 comes from the edge computation
                       (int)_currentViewedRectangle.getHeight() - 1);
            g.setColor(_selectionRectangleColor);
            g.drawRect((int)_currentViewedRectangle.getX(),
                       (int)_currentViewedRectangle.getY(),
                       (int)_currentViewedRectangle.getWidth() - 1,
                       (int)_currentViewedRectangle.getHeight() - 1);
            g.setColor(oldC);
        }
        else if(_navigatedImage != null)
        {
            g.drawRect(0, 0, _navigatedImage.getWidth(this) - 1, _navigatedImage.getHeight(this) - 1);
        }
    }

}
