package ohd.hseb.charter.panel;

import java.awt.Cursor;
import java.awt.Graphics;
import java.awt.Image;
import java.awt.Point;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseEvent;
import java.awt.event.MouseWheelEvent;
import java.awt.event.MouseWheelListener;
import java.awt.geom.Point2D;
import java.awt.geom.Rectangle2D;
import java.awt.image.BufferedImage;
import java.io.File;
import java.io.IOException;
import java.text.DateFormat;
import java.text.NumberFormat;
import java.text.SimpleDateFormat;
import java.util.EventListener;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.swing.ButtonGroup;
import javax.swing.JComponent;
import javax.swing.JFileChooser;
import javax.swing.JMenuItem;
import javax.swing.JPopupMenu;
import javax.swing.JRadioButtonMenuItem;
import javax.swing.KeyStroke;
import javax.swing.SwingUtilities;

import org.jfree.chart.ChartMouseEvent;
import org.jfree.chart.ChartMouseListener;
import org.jfree.chart.ChartPanel;
import org.jfree.chart.ChartUtilities;
import org.jfree.chart.JFreeChart;
import org.jfree.chart.axis.NumberAxis;
import org.jfree.chart.axis.ValueAxis;
import org.jfree.chart.event.ChartChangeEvent;
import org.jfree.chart.event.ChartChangeListener;
import org.jfree.chart.labels.BoxAndWhiskerXYToolTipGenerator;
import org.jfree.chart.plot.CombinedDomainXYPlot;
import org.jfree.chart.plot.Plot;
import org.jfree.chart.plot.PlotOrientation;
import org.jfree.chart.plot.XYPlot;
import org.jfree.chart.plot.Zoomable;
import org.jfree.chart.renderer.xy.AbstractXYItemRenderer;
import org.jfree.data.xy.XYDataset;
import org.jfree.ui.ExtensionFileFilter;
import org.jfree.util.ShapeUtilities;

import com.google.common.collect.Lists;
import com.google.common.eventbus.EventBus;

import nl.wldelft.util.ClipboardUtils;
import ohd.hseb.charter.ChartTools;
import ohd.hseb.charter.jfreechartoverride.FixedChartPanel;
import ohd.hseb.charter.jfreechartoverride.GraphGenXYToolTipGenerator;
import ohd.hseb.charter.jfreechartoverride.XYBoxAndWhiskerRenderer;
import ohd.hseb.charter.tools.DateAxisPlus;
import ohd.hseb.charter.tools.TranslatedAxis;
import ohd.hseb.charter.translator.AxisTranslator;
import ohd.hseb.hefs.utils.gui.tools.SwingTools;

/**
 * Much of this code is copied from something provided to me by Deltares. Its been modified to suit the needs of
 * GraphGen. It merely wraps JFreeChart's {@link FixedChartPanel} (note that the FixedChartPanel has been modified, as
 * well, to fix bugs). This interface does not use {@link EventBus} in any way, meaning that it uses a traditional set
 * of listeners that implement {@link OHDFixedChartListener}. That listener is intended to relay messages from the
 * displayed {@link JFreeChart} that may otherwise go unheard. Currently, the only message relayed is a drag message.<br>
 * <br>
 * It is up to external stuff calling this to maintain {@link #_listeners} through the provided methods.
 * 
 * @author hankherr
 */
public class OHDFixedChartPanel extends FixedChartPanel
{
    private static final long serialVersionUID = 1L;

    /**
     * This is solely fired for convenience. Whenever {@link #setChart(JFreeChart)} is called, this property is fired.
     * This was the easiest way to signal listeners of such a change without adding much code, and it makes a certain
     * amount of sense, since a chart change is related to the GUI display.
     */
    public static final String PROPERTY_CHART_CHANGED = "CHART_CHANGED";

    /**
     * Zoom in domain axis only action command.
     */
    public static final String ALLOW_ZOOM_IN_DOMAIN_COMMAND = "ALLOW_ZOOM_IN_DOMAIN";

    /**
     * Zoom in range axis only action command.
     */
    public static final String ALLOW_ZOOM_IN_RANGE_COMMAND = "ALLOW_ZOOM_IN_RANGE";

    public static final String COPY_TO_CLIPBOARD = "COPY_TO_CLIPBOARD";

    //XXX All attributes that do not include an '_' at the beginning date to the original Deltares implementation.

    private JMenuItem allowDomainAxisZoomMenuItem = null;
    private JMenuItem allowRangeAxisZoomMenuItem = null;
    private JRadioButtonMenuItem copyToClipBoardMenuItem = null;
    private final Set<ButtonGroup> buttonGroups = new HashSet<ButtonGroup>();
    private boolean domainZoomable = false;
    private boolean rangeZoomable = false;
    private File lastSelectedDir = null;
    private Point2D startPanPoint = null;
    private boolean dragging = false;

    /**
     * Useful for external tools that may need to draw on top of this. This is recorded each time
     * {@link #paint(Graphics)} is called with {@link #_forceFullRepaint} being true.
     */
    private Image _mostRecentPaintedImage = null;

    /**
     * See {@link #setForceFullRepaint(boolean)}.
     */
    private boolean _forceFullRepaint = true;

    /**
     * Manages list of {@link OHDFixedChartListener} instances.
     */
    private final List<OHDFixedChartListener> _listeners = Lists.newArrayList();

    /**
     * Mouse wheel used for zooming.
     */
    private final MouseWheelListener mouseWheelListener = new MouseWheelListener()
    {
        @Override
        public void mouseWheelMoved(final MouseWheelEvent e)
        {
            final Point2D point = ShapeUtilities.getPointInRectangle(e.getX(), e.getY(), getScreenDataArea());
            final boolean zoomIn = e.getWheelRotation() < 0;
            zoomFromPoint(point, zoomIn);
        }
    };

    /**
     * Escape used to close the popup menu.
     */
    private final ActionListener escapeActionListener = new ActionListener()
    {
        @Override
        public void actionPerformed(final ActionEvent e)
        {
            getPopupMenu().setVisible(false);
        }
    };

    /**
     * Calls {@link ChartPanel#ChartPanel(JFreeChart)} for the given chart and then
     * {@link #setupToolTipsForChart(JFreeChart)}.
     * 
     * @param chart
     */
    public OHDFixedChartPanel(final JFreeChart chart)
    {
        super(chart);
        extendPopupMenu();
        setupToolTipsForChart(chart);
        addCursorListener();
        setMaximumDrawWidth(2400);
        setMaximumDrawHeight(1800);
    }

    /**
     * Calls {@link ChartPanel#ChartPanel(JFreeChart, boolean, boolean, boolean, boolean, boolean)}. It then calls
     * {@link #extendPopupMenu()}, {@link #addMouseWheelListener(MouseWheelListener)}, and
     * {@link #setupToolTipsForChart(JFreeChart)}.
     */
    public OHDFixedChartPanel(final JFreeChart chart,
                              final boolean properties,
                              final boolean save,
                              final boolean print,
                              final boolean zoom,
                              final boolean tooltips)
    {
        super(chart, properties, save, print, zoom, tooltips);
        extendPopupMenu();
        addMouseWheelListener(mouseWheelListener);
        addCursorListener();
        setupToolTipsForChart(chart);
        setMaximumDrawWidth(2400);
        setMaximumDrawHeight(1800);
    }

    /**
     * Calls {@link ChartPanel#ChartPanel(JFreeChart, boolean)}. It then calls {@link #extendPopupMenu()},
     * {@link #addMouseWheelListener(MouseWheelListener)}, and {@link #setupToolTipsForChart(JFreeChart)}.
     */
    public OHDFixedChartPanel(final JFreeChart chart, final boolean useBuffer)
    {
        super(chart, useBuffer);
        extendPopupMenu();
        addMouseWheelListener(mouseWheelListener);
        addCursorListener();
        setupToolTipsForChart(chart);
        setMaximumDrawWidth(2400);
        setMaximumDrawHeight(1800);
    }

    /**
     * Calls
     * {@link ChartPanel#ChartPanel(JFreeChart, int, int, int, int, int, int, boolean, boolean, boolean, boolean, boolean, boolean)}
     * . It then calls {@link #extendPopupMenu()}, {@link #addMouseWheelListener(MouseWheelListener)}, and
     * {@link #setupToolTipsForChart(JFreeChart)}.
     */
    public OHDFixedChartPanel(final JFreeChart chart,
                              final int width,
                              final int height,
                              final int minimumDrawWidth,
                              final int minimumDrawHeight,
                              final int maximumDrawWidth,
                              final int maximumDrawHeight,
                              final boolean useBuffer,
                              final boolean properties,
                              final boolean save,
                              final boolean print,
                              final boolean zoom,
                              final boolean tooltips)
    {
        super(chart,
              width,
              height,
              minimumDrawWidth,
              minimumDrawHeight,
              maximumDrawWidth,
              maximumDrawHeight,
              useBuffer,
              properties,
              save,
              print,
              zoom,
              tooltips);
        extendPopupMenu();
        addMouseWheelListener(mouseWheelListener);
        addCursorListener();
        setupToolTipsForChart(chart);
        setMaximumDrawWidth(2400);
        setMaximumDrawHeight(1800);
    }

    /**
     * @param listener The listener to add.
     */
    public void addOHDFixedChartListener(final OHDFixedChartListener listener)
    {
        _listeners.add(listener);
    }

    /**
     * @param listener The listener to remove.
     */
    public void removeOHDFixedChartListener(final OHDFixedChartListener listener)
    {
        _listeners.remove(listener);
    }

    /**
     * Clear all {@link OHDFixedChartListener} added to {@link #_listeners}.
     */
    public void clearOHDFixedChartListeners()
    {
        _listeners.clear();
    }

    /**
     * Calls the {@link OHDFixedChartListener#doneDraggingChart()} method for all listeners inside {@link #_listeners}.
     */
    private void fireDoneDraggingChart()
    {
        for(final OHDFixedChartListener listener: _listeners)
        {
            listener.doneDraggingChart();
        }
    }

    /**
     * Adds a {@link ChartMouseListener} that changes the cursor to a hand cursor if it is over a clickable entity.
     */
    private void addCursorListener()
    {
        addChartMouseListener(new ChartMouseListener()
        {
            @Override
            public void chartMouseClicked(final ChartMouseEvent event)
            {
            }

            @Override
            public void chartMouseMoved(final ChartMouseEvent arg0)
            {
                if(arg0.getEntity() != null)
                {
                    setCursor(Cursor.getPredefinedCursor(Cursor.HAND_CURSOR));
                }
                else
                {
                    setCursor(Cursor.getDefaultCursor());
                }
            }
        });
    }

    /**
     * Sets up the tool tips by setting the renderer for each subplot based on the data sets.
     * 
     * @param chart
     */
    private void setupToolTipsForChart(final JFreeChart chart)
    {
        final CombinedDomainXYPlot combinedPlot = (CombinedDomainXYPlot)chart.getXYPlot();

        //XXX If we ever do category plotting, I'll need to add a version of this method that uses CategoryPlot.
        for(int i = 0; i < combinedPlot.getSubplots().size(); i++)
        {
            final XYPlot subPlot = (XYPlot)combinedPlot.getSubplots().get(i);
            for(int j = 0; j < subPlot.getDatasetCount(); j++)
            {
                setupToolTipsForRenderer(chart, i, j);
            }
        }
    }

    /**
     * Unscales the provided point, reversing {@link #getScaleX()} and {@link #getScaleY()}.
     * 
     * @return The unscaled {@link Point2D}.
     */
    private Point2D unscale(final Point2D point)
    {
        final double x = (point.getX() - getInsets().left) / this.getScaleX();
        final double y = (point.getY() - getInsets().top) / this.getScaleY();
        return new Point2D.Double(x, y);
    }

    /**
     * Copies an image of the chart to the clipboard.
     */
    private void copyToClipboard()
    {
        final BufferedImage image = getChart().createBufferedImage(getWidth(), getHeight(), null);
        ClipboardUtils.copyImageToClipboard(image);
    }

    /**
     * Changes the zoom level based on a point.
     * 
     * @param point Point about which to zoom.
     * @param zoomIn True to zoom in, false to zoom out.
     */
    public void zoomFromPoint(final Point2D point, final boolean zoomIn)
    {
        final Point2D unscaledPoint = unscale(point);
        Rectangle2D scaledDataArea;
        if(getChartRenderingInfo().getPlotInfo().getSubplotCount() == 0)
        {
            scaledDataArea = getScreenDataArea();
        }
        else
        {
            final int subplotIndex = getChartRenderingInfo().getPlotInfo().getSubplotIndex(unscaledPoint);
            if(subplotIndex == -1)
                return;
            scaledDataArea = scale(getChartRenderingInfo().getPlotInfo().getSubplotInfo(subplotIndex).getDataArea());
        }
        final double currentPercentX = (point.getX() - scaledDataArea.getMinX()) / scaledDataArea.getWidth();
        final double currentPercentY = 1 - (point.getY() - scaledDataArea.getMinY()) / scaledDataArea.getHeight();

        final double factor = zoomIn ? 0.9 : 1.1;

        final double lowerPercentY = (1 - factor) * currentPercentY;
        final double upperPercentY = 1 - (1 - factor) * (1 - currentPercentY);
        final double lowerPercentX = (1 - factor) * currentPercentX;
        final double upperPercentX = 1 - (1 - factor) * (1 - currentPercentX);

        final Plot plot = getChart().getPlot();
        if(!(plot instanceof Zoomable))
            return;

        final Zoomable zoomable = (Zoomable)plot;

        if(zoomable.getOrientation() == PlotOrientation.VERTICAL)
        {
            if(rangeZoomable && zoomable.isRangeZoomable())
            {
                zoomable.zoomRangeAxes(lowerPercentY,
                                       upperPercentY,
                                       getChartRenderingInfo().getPlotInfo(),
                                       unscaledPoint);
            }

            if(domainZoomable && zoomable.isDomainZoomable())
            {
                zoomable.zoomDomainAxes(lowerPercentX,
                                        upperPercentX,
                                        getChartRenderingInfo().getPlotInfo(),
                                        unscaledPoint);
            }
        }
        else
        {
            if(rangeZoomable && zoomable.isRangeZoomable())
            {
                zoomable.zoomRangeAxes(lowerPercentX,
                                       upperPercentX,
                                       getChartRenderingInfo().getPlotInfo(),
                                       unscaledPoint);
            }

            if(domainZoomable && zoomable.isDomainZoomable())
            {
                zoomable.zoomDomainAxes(lowerPercentY,
                                        upperPercentY,
                                        getChartRenderingInfo().getPlotInfo(),
                                        unscaledPoint);
            }
        }
    }

    /**
     * Calls {@link #setDomainZoom(boolean)} and changes the selection state of {@link #allowDomainAxisZoomMenuItem}.
     */
    public void setHorizontalZoom(final boolean flag)
    {
        setDomainZoom(flag);
        allowDomainAxisZoomMenuItem.setSelected(flag);
    }

    /**
     * Calls {@link #setRangeZoom(boolean)} and changes the selection state of the {@link #allowRangeAxisZoomMenuItem}.
     */
    public void setVerticalZoom(final boolean flag)
    {
        setRangeZoom(flag);
        allowRangeAxisZoomMenuItem.setSelected(this.rangeZoomable);
    }

    /**
     * Sets the domain zoomability based on the flag, but the plot must be {@link Zoomable} in order to allow zooming.
     * 
     * @param flag True to allow for zooming, false if not.
     */
    private void setDomainZoom(final boolean flag)
    {
        if(flag)
        {
            final Plot plot = this.getChart().getPlot();
            if(plot instanceof Zoomable)
            {
                final Zoomable z = (Zoomable)plot;
                this.domainZoomable = z.isDomainZoomable();
            }
        }
        else
        {
            this.domainZoomable = false;
        }
        super.setDomainZoomable(flag);

    }

    /**
     * Sets the range zoomability based on the flag, but the plot must be {@link Zoomable} in order to allow zooming.
     * 
     * @param flag True to allow for zooming, false if not.
     */
    private void setRangeZoom(final boolean flag)
    {
        if(flag)
        {
            final Plot plot = this.getChart().getPlot();
            if(plot instanceof Zoomable)
            {
                final Zoomable z = (Zoomable)plot;
                this.rangeZoomable = z.isRangeZoomable();
            }
        }
        else
        {
            this.rangeZoomable = false;
        }
        super.setRangeZoomable(flag);
    }

    /**
     * Method to add additional menu itmes to the popup menu.
     */
    private void extendPopupMenu()
    {

        allowDomainAxisZoomMenuItem = new JRadioButtonMenuItem("Allow Domain Axis Zoom");
        allowRangeAxisZoomMenuItem = new JRadioButtonMenuItem("Allow Range Axis Zoom");
        copyToClipBoardMenuItem = new JRadioButtonMenuItem("Copy to Clipboard");

        allowDomainAxisZoomMenuItem.addActionListener(this);
        allowRangeAxisZoomMenuItem.addActionListener(this);
        copyToClipBoardMenuItem.addActionListener(this);

        allowDomainAxisZoomMenuItem.setActionCommand(ALLOW_ZOOM_IN_DOMAIN_COMMAND);
        allowRangeAxisZoomMenuItem.setActionCommand(ALLOW_ZOOM_IN_RANGE_COMMAND);
        copyToClipBoardMenuItem.setActionCommand(COPY_TO_CLIPBOARD);

        final JPopupMenu popupMenu = getPopupMenu();
        if(popupMenu == null)
            return;

        popupMenu.addSeparator();
        popupMenu.add(allowDomainAxisZoomMenuItem);
        popupMenu.add(allowRangeAxisZoomMenuItem);
        popupMenu.addSeparator();
        popupMenu.add(copyToClipBoardMenuItem);
        popupMenu.registerKeyboardAction(escapeActionListener,
                                         KeyStroke.getKeyStroke("ESCAPE"),
                                         JComponent.WHEN_IN_FOCUSED_WINDOW);

    }

    /**
     * Adds items to popup menu. All items are added after a new sparator
     * 
     * @param menuItems
     */
    public void addPopupMenuItems(final JMenuItem[] menuItems)
    {

        final JPopupMenu popupMenu = super.getPopupMenu();
        if(popupMenu == null)
            return;

        popupMenu.addSeparator();
        final ButtonGroup buttonGroup = new ButtonGroup();

        for(int i = 0; i < menuItems.length; i++)
        {
            final JMenuItem menuItem = menuItems[i];
            final EventListener[] listeners = menuItem.getListeners(ActionListener.class);
            if(listeners.length > 0 && menuItem.getAccelerator() != null)
            {
                super.registerKeyboardAction((ActionListener)listeners[0],
                                             menuItem.getAccelerator(),
                                             JComponent.WHEN_IN_FOCUSED_WINDOW);
            }
            popupMenu.add(menuItem);
            buttonGroup.add(menuItem);
        }

        buttonGroups.add(buttonGroup);
    }

    /**
     * @param b If true, a full repaint will be forced the next time {@link #paint(Graphics)} is called. Otherwise, it
     *            will only be a repating of {@link #_mostRecentPaintedImage}.
     */
    public void setForceFullRepaint(final boolean b)
    {
        _forceFullRepaint = b;
    }

    /**
     * @return {@link #_mostRecentPaintedImage}, which is the most recently drawn chart image.
     */
    public Image getMostRecentPaintedImage()
    {
        return this._mostRecentPaintedImage;
    }

    /**
     * @return True if the chart is currently being panned/dragged by the user.
     */
    public boolean isDragging()
    {
        return dragging;
    }

    @Override
    public void mouseDragged(final MouseEvent e)
    {
        if(!SwingUtilities.isRightMouseButton(e))
        {
            super.mouseDragged(e);
            return;
        }

        final Point2D point = ShapeUtilities.getPointInRectangle(e.getX(), e.getY(), getScreenDataArea());
        dragging = true;
        if(startPanPoint == null)
        {
            startPanPoint = point;
            e.consume();
            return;
        }

        final Point2D unscaledPoint = unscale(point);
        Rectangle2D scaledDataArea;
        if(this.getChartRenderingInfo().getPlotInfo().getSubplotCount() == 0)
        {
            scaledDataArea = getScreenDataArea();
        }
        else
        {
            final int subplotIndex = this.getChartRenderingInfo().getPlotInfo().getSubplotIndex(unscaledPoint);
            if(subplotIndex == -1)
                return;
            scaledDataArea = scale(this.getChartRenderingInfo()
                                       .getPlotInfo()
                                       .getSubplotInfo(subplotIndex)
                                       .getDataArea());
        }

        //If the popup is showing, remove it.
        if(getPopupMenu().isShowing())
        {
            getPopupMenu().setVisible(false);
        }

        final double deltaRelX = (e.getX() - startPanPoint.getX()) / scaledDataArea.getWidth();
        final double deltaRelY = (e.getY() - startPanPoint.getY()) / -scaledDataArea.getHeight();
        final Plot plot = getChart().getPlot();
        if(!(plot instanceof Zoomable))
            return;
        final Zoomable zoomable = (Zoomable)plot;

        if(zoomable.getOrientation() == PlotOrientation.VERTICAL)
        {
            if(domainZoomable && zoomable.isDomainZoomable())
            {
                zoomable.zoomDomainAxes(-deltaRelX, 1 - deltaRelX, getChartRenderingInfo().getPlotInfo(), unscaledPoint);
            }
            if(rangeZoomable && zoomable.isRangeZoomable())
            {
                zoomable.zoomRangeAxes(-deltaRelY, 1 - deltaRelY, getChartRenderingInfo().getPlotInfo(), unscaledPoint);
            }
        }
        else
        {
            if(domainZoomable && zoomable.isDomainZoomable())
            {
                zoomable.zoomDomainAxes(-deltaRelY, 1 - deltaRelY, getChartRenderingInfo().getPlotInfo(), unscaledPoint);
            }
            if(rangeZoomable && zoomable.isRangeZoomable())
            {
                zoomable.zoomRangeAxes(-deltaRelX, 1 - deltaRelX, getChartRenderingInfo().getPlotInfo(), unscaledPoint);
            }
        }

        startPanPoint = point;
        e.consume();
    }

    @Override
    public void mousePressed(MouseEvent e)
    {
        //On Linux and Mac, a mouse pressed triggers the popup.  This must be turned off as it does not work well with panning.
        e = SwingTools.setPopupTrigger(e, false);

        super.mousePressed(e);
        if(!SwingUtilities.isRightMouseButton(e))
        {
            return;
        }
        startPanPoint = e.getPoint();
    }

    @Override
    public void mouseClicked(final MouseEvent event)
    {
        if(SwingUtilities.isRightMouseButton(event))
        {
            return;
        }
        super.mouseClicked(event);
    }

    @Override
    public void mouseReleased(MouseEvent e)
    {
        //Force the popup to open if the right mouse button is released.
        e = SwingTools.setPopupTrigger(e, SwingUtilities.isRightMouseButton(e));

        if(!dragging || !SwingUtilities.isRightMouseButton(e))
        {
            super.mouseReleased(e);
            return;
        }

        if(dragging)
        {
            dragging = false;
            startPanPoint = null;
            fireDoneDraggingChart();
        }
    }

    @Override
    public void doSaveAs() throws IOException
    {

        final JFileChooser fileChooser = new JFileChooser();
        if(lastSelectedDir != null)
            fileChooser.setCurrentDirectory(lastSelectedDir);
        final ExtensionFileFilter filter = new ExtensionFileFilter(localizationResources.getString("PNG_Image_Files"),
                                                                   ".png");
        fileChooser.addChoosableFileFilter(filter);

        final int option = fileChooser.showSaveDialog(this);
        if(option == JFileChooser.APPROVE_OPTION)
        {
            lastSelectedDir = fileChooser.getSelectedFile().getParentFile();
            String filename = fileChooser.getSelectedFile().getPath();
            if(isEnforceFileExtensions())
            {
                if(!filename.endsWith(".png"))
                {
                    filename += ".png";
                }
            }
            ChartUtilities.saveChartAsPNG(new File(filename), getChart(), getWidth(), getHeight());
        }

    }

    @Override
    public void actionPerformed(final ActionEvent event)
    {

        final String command = event.getActionCommand();

        if(command.equals(ALLOW_ZOOM_IN_DOMAIN_COMMAND))
        {
            setDomainZoom(allowDomainAxisZoomMenuItem.isSelected());
        }
        else if(command.equals(ALLOW_ZOOM_IN_RANGE_COMMAND))
        {
            setRangeZoom(allowRangeAxisZoomMenuItem.isSelected());
        }
        else if(command.equals(COPY_TO_CLIPBOARD))
        {
            copyToClipboard();
        }
        else
        {
            super.actionPerformed(event);
        }
    }

    @Override
    protected void displayPopupMenu(final int x, final int y)
    {
        if(super.getPopupMenu() == null)
            return;
        super.displayPopupMenu(x, y);
    }

    /**
     * Overrules method zoomInRange due to the bug in {@link ChartPanel} The original method uses the wrong method
     * zoomRangeAxes(..)
     */
    @Override
    public void zoomInRange(final double x, final double y)
    {
        final Plot p = this.getChart().getPlot();
        if(p instanceof Zoomable)
        {
            final Zoomable z = (Zoomable)p;
            z.zoomRangeAxes(this.getZoomInFactor(),
                            getChartRenderingInfo().getPlotInfo(),
                            translateScreenToJava2D(new Point((int)x, (int)y)));
        }
    }

    /**
     * This forces a full repaint of the panel. It also adds a {@link ChartChangeListener} that forces a full repaint
     * whenever the chart is changed. It then sets the chart in the super class and fires a
     * {@link #PROPERTY_CHART_CHANGED} property change event that contains both the old chart and new chart.
     */
    @Override
    public void setChart(final JFreeChart chart)
    {
        setForceFullRepaint(true);
        final JFreeChart oldChart = getChart();
        chart.addChangeListener(new ChartChangeListener()
        {
            @Override
            public void chartChanged(final ChartChangeEvent arg0)
            {
                setForceFullRepaint(true);
            }
        });
        super.setChart(chart);
        setupToolTipsForChart(chart);
        this.firePropertyChange(PROPERTY_CHART_CHANGED, oldChart, getChart());
    }

    @Override
    public void paint(final Graphics g)
    {
        //This method receives calls for pretty much anything.  Most of those calls do not require a full repaint of the
        //chart.  Merely redrawing the previously generated image is enough.  As such I can speed it up somewhat by
        //telling it to redraw the previous image when possible.  Unfortunately, most repaints are triggered by chart change
        //events in the display chart, and I have no way to detect if those are meaningful changes.

        //Slow...
        if((_forceFullRepaint) || (_mostRecentPaintedImage == null)
            || (getWidth() != _mostRecentPaintedImage.getWidth(this))
            || (getHeight() != _mostRecentPaintedImage.getHeight(this)))
        {
            _mostRecentPaintedImage = this.createImage(getWidth(), getHeight());
            super.paint(_mostRecentPaintedImage.getGraphics());
            g.drawImage(_mostRecentPaintedImage, 0, 0, this);
            _forceFullRepaint = false;
        }
        //Fast...
        else
        {
            if(!g.drawImage(_mostRecentPaintedImage, 0, 0, this))
            {
                System.err.println("Unexpected error attempting to draw the most recent painted image to the OHDFixedChartPanel!!!");
            }
        }
    }

    //TODO For WRES: To implement categorical tool tips, the method below will need to be modified.  If the NumberAxis is a NumbeAxisOverride
    //then check the isCategoricalAxis.  If so, don't set the xFormat and the generator will need to get the category value, instead.
    //How to do that?
    
    /**
     * Sets the tool tips to use for the data set renderer.
     * 
     * @param chart The chart affected.
     * @param subPlotIndex The index of the subplot affected within the chart.
     * @param datasetIndex The index of the data set to look at to determine tool tips.
     */
    public static void setupToolTipsForRenderer(final JFreeChart chart, final int subPlotIndex, final int datasetIndex)
    {
        //Note that yFormat cannot be null because if it is passed into the GraphGenXYToolTipGenerator constructor as null, 
        //then it will cause a "Null 'yFormat' argument' exception wtihin the AbstractXYItemLabelGenerator.
        NumberFormat xFormat = null;
        DateFormat xDateFormat = null;
        NumberFormat yFormat = null;
        NumberFormat yFormat2 = null;
        AxisTranslator translator = null;

        final CombinedDomainXYPlot combinedPlot = (CombinedDomainXYPlot)chart.getXYPlot();
        final XYPlot subPlot = (XYPlot)combinedPlot.getSubplots().get(subPlotIndex);
        final XYDataset dataset = subPlot.getDataset(datasetIndex);
        final ValueAxis rangeAxisForDataset = subPlot.getRangeAxisForDataset(datasetIndex);
        final AbstractXYItemRenderer renderer = (AbstractXYItemRenderer)subPlot.getRendererForDataset(dataset);

        renderer.setBaseCreateEntities(true);

        //x-axis, first
        //If time-based...
        if(combinedPlot.getDomainAxis() instanceof DateAxisPlus)
        {
            xDateFormat = new SimpleDateFormat("MM-dd-yyyy HH:mm:ss zzz");
            xDateFormat.setTimeZone(((DateAxisPlus)combinedPlot.getDomainAxis()).getTimeZone());
        }
        //If not categorical, implying true numerical...
        else if (! ChartTools.isDomainAxisCategorical(chart))
        {
            xFormat = ((NumberAxis)combinedPlot.getDomainAxis()).getNumberFormatOverride();
            if(xFormat == null)
            {
                xFormat = NumberFormat.getNumberInstance();
            }
        }

        //Leave the y-axis formats null if the range axis for the dataset is not the left hand y-axis.
        if(rangeAxisForDataset == subPlot.getRangeAxis(0))
        {
            yFormat = ((NumberAxis)subPlot.getRangeAxis(0)).getNumberFormatOverride();
            if(yFormat == null)
            {
                yFormat = NumberFormat.getNumberInstance();
            }
        }

        //For the right hand y-axis, first check to see if it exists.  If so, then check to see if its plotted against.
        //If so, set the appropriate format.  Otherwise, check to see if it is translated.  If so, set the appropriate
        //format and record the translator.
        if(subPlot.getRangeAxisCount() > 1)
        {
            if(rangeAxisForDataset == subPlot.getRangeAxis(1))
            {
                yFormat2 = ((NumberAxis)subPlot.getRangeAxis(1)).getNumberFormatOverride();
                if(yFormat2 == null)
                {
                    yFormat2 = NumberFormat.getNumberInstance();
                }
            }
            else if(subPlot.getRangeAxis(1).isVisible() && (subPlot.getRangeAxis(1) instanceof TranslatedAxis))
            {
                yFormat2 = ((NumberAxis)subPlot.getRangeAxis(1)).getNumberFormatOverride();
                if(yFormat2 == null)
                {
                    yFormat2 = NumberFormat.getNumberInstance();
                }
                translator = ((TranslatedAxis)subPlot.getRangeAxis(1)).getTranslator();
            }
        }

        //Box and whisker must have a date x-axis. XXX This may not be a good thing.
        if(renderer instanceof XYBoxAndWhiskerRenderer)
        {
            if(yFormat == null)
            {
                yFormat = NumberFormat.getInstance();
            }
            renderer.setBaseToolTipGenerator(new BoxAndWhiskerXYToolTipGenerator("X: {1} Middle: {3} Min: {4} Max: {5} Q1: {6} Q3: {7} ",
                                                                                 xDateFormat,
                                                                                 yFormat));
        }

        //For number axes, it will use the special GraphGen version.
        else
        {
            if(xDateFormat != null)
            {
                renderer.setBaseToolTipGenerator(new GraphGenXYToolTipGenerator(xDateFormat,
                                                                                yFormat,
                                                                                yFormat2,
                                                                                translator));
            }
            else if (xFormat == null)//Implies categorical, so pass in the chart which is then used to translate numerical to categorical.
            {
                renderer.setBaseToolTipGenerator(new GraphGenXYToolTipGenerator(chart, yFormat, yFormat2, translator));
            }
            else
            {
                renderer.setBaseToolTipGenerator(new GraphGenXYToolTipGenerator(xFormat, yFormat, yFormat2, translator));
            }
        }
    }

}
