package ohd.hseb.charter.panel.notices;

import ohd.hseb.charter.panel.OHDFixedChartListener;
import ohd.hseb.charter.panel.OHDFixedChartPanel;
import ohd.hseb.hefs.utils.notify.Notice;
import ohd.hseb.hefs.utils.notify.NoticePoster;

import org.jfree.chart.JFreeChart;
import org.jfree.chart.axis.ValueAxis;
import org.jfree.chart.event.AxisChangeEvent;
import org.jfree.chart.event.AxisChangeListener;
import org.jfree.data.Range;

import com.google.common.eventbus.Subscribe;

/**
 * Posted to reflect a change to the limits of any axis displayed within a {@link JFreeChart}. See the
 * {@link PostingListener} listener for an {@link AxisChangeListener} to add to a {@link ValueAxis} that will detect
 * when changes are due to axis limts changes and post this event accordingly. If the chart is currently being dragged,
 * and that drag is what caused the axisl imits change, then the {@link #_chartDragging} flag will be true.
 * 
 * @author hankherr
 */
public class GeneralAxisLimitsChangedNotice extends Notice
{

    public ValueAxis _axis;
    public boolean _chartDragging;

    public GeneralAxisLimitsChangedNotice(final Object source, final ValueAxis axis, final boolean chartDragging)
    {
        super(source);
        _axis = axis;
        _chartDragging = chartDragging;
    }

    public ValueAxis getAxis()
    {
        return _axis;
    }

    public boolean isChartDragging()
    {
        return _chartDragging;
    }

    public static interface Subscriber
    {
        @Subscribe
        public void reactToGeneralAxisLimitsChanged(GeneralAxisLimitsChangedNotice evt);
    }

    /**
     * An {@link AxisChangeListener} that detects if the limits have been changed and posts
     * {@link GeneralAxisLimitsChangedNotice} instances to the bus it is provided. Some {@link AxisChangeListener}
     * events are thrown that are not related to axis limits changing; for example, if the size of the axis is changed
     * due to a chart resize.<br>
     * <br>
     * This class is somewhat dangerous because it remembers the {@link ValueAxis}, which is connected to a chart. If
     * the chart inside of the provided {@link OHDFixedChartPanel} is modified, the axis connection will result in the
     * chart remaining in memory until this listener is removed from the {@link OHDFixedChartPanel} listeners!
     * 
     * @author hankherr
     */
    public static class PostingListener implements AxisChangeListener, OHDFixedChartListener
    {
        private final ValueAxis _axis;
        private Range _lastLimits;
        private final NoticePoster _poster;
        private Object _source;
        private final OHDFixedChartPanel _chartViewingPanel;
        private final boolean _doNotPostWhenDragging;

        /**
         * @param axis Axis to which to listen.
         * @param chartViewingPanel The {@link OHDFixedChartPanel} that will view the chart. Null is allowed, in which
         *            case messages are always posted with a dragging flag of false. This is added as a listener to this
         *            chart viewing panel via a call to
         *            {@link OHDFixedChartPanel#addOHDFixedChartListener(OHDFixedChartListener)}.
         * @param poster The {@link NoticePoster} to call for posting the notice.
         * @param postingSource The source to associate with those events. If null, then the source will be this.
         * @param doNotPostWhenDragging If true, notices will NOT be posted while the mouse is being dragged.
         */
        public PostingListener(final ValueAxis axis,
                               final OHDFixedChartPanel chartViewingPanel,
                               final NoticePoster poster,
                               final Object postingSource,
                               final boolean doNotPostWhenDragging)
        {
            _axis = axis;
            _axis.addChangeListener(this);
            _lastLimits = axis.getRange();
            _poster = poster;
            _doNotPostWhenDragging = doNotPostWhenDragging;

            _chartViewingPanel = chartViewingPanel;
            _chartViewingPanel.addOHDFixedChartListener(this);

            if(postingSource != null)
            {
                _source = postingSource;
            }
            else
            {
                _source = this;
            }
        }

        /**
         * Only posts a notice if the limits do not match the last limits.
         */
        private void postNoticeIfNecessary()
        {
            //If the viewing panel is being dragged and we are told to not post while dragging, then do not post.
            if((_doNotPostWhenDragging) && (_chartViewingPanel != null) && _chartViewingPanel.isDragging())
            {
                return;
            }

            //We need to post a message only if the limits differ from the last posted limits.
            if(!_axis.getRange().equals(_lastLimits))
            {
                _lastLimits = _axis.getRange();
                boolean dragging = false;
                if(_chartViewingPanel != null)
                {
                    dragging = _chartViewingPanel.isDragging();
                }
                _poster.post(new GeneralAxisLimitsChangedNotice(_source, _axis, dragging));
            }
        }

        @Override
        public void axisChanged(final AxisChangeEvent arg0)
        {
            if(arg0.getAxis() == _axis)
            {
                postNoticeIfNecessary();
            }
        }

        @Override
        public void doneDraggingChart()
        {
            postNoticeIfNecessary();
        }

    }
}
