package ohd.hseb.charter.parameters;

import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;

import org.jfree.chart.axis.ValueAxis;
import org.jfree.chart.plot.CombinedDomainXYPlot;
import org.jfree.chart.plot.Marker;
import org.jfree.chart.plot.XYPlot;
import org.jfree.data.Range;
import org.jfree.ui.Layer;

import com.google.common.collect.Maps;

import ohd.hseb.charter.ChartConstants;
import ohd.hseb.hefs.utils.Dyad;

/**
 * Helper for calculating axis range for a single axis identified by {@link #_axisIndex} according to the
 * {@link ChartConstants} defined constants {@link ChartConstants#LEFT_YAXIS_INDEX},
 * {@link ChartConstants#RIGHT_YAXIS_INDEX}, and {@value ChartConstants#DOMAIN_AXIS}. That axis must be
 * part of the provided {@link #_xyplot}.
 * 
 * @author hank.herr
 */
public class AxisAutoRangeHelper
{

    private static final double MINIMUM_RANGE_DIFFERENCE = 0.0000001;
    private static final double RANGE_RESCALE_FACTOR = 0.05;

    /**
     * List of thresholds, needed within {@link #autoRangeWithExcludingThresholds()}.
     */
    private ThresholdListParameters _thresholds = null;

    /**
     * The containing plot for the axis for which help is being acquired.
     */
    private XYPlot _xyplot = null;

    /**
     * The index of the axis within the plot.
     */
    private int _axisIndex = ChartConstants.DOMAIN_AXIS;

    /**
     * Preps this to be used with auto range computations.
     * 
     * @param xyplot the plot containing the axis for which limits are being computed.
     * @param thresholds Thresholds that may be plotted against that axis and which must be accounted for.
     */
    public AxisAutoRangeHelper(final XYPlot xyplot, final ThresholdListParameters thresholds)
    {
        _xyplot = xyplot;
        _thresholds = thresholds;

    }

    public XYPlot getXYPlot()
    {
        return _xyplot;
    }

    public ThresholdListParameters getThresholdListParameters()
    {
        return _thresholds;
    }

    public void setAxisIndex(final int index)
    {
        _axisIndex = index;
    }

    public int getAxisIndex()
    {
        return _axisIndex;
    }

    /**
     * @return The {@link ValueAxis} corresponding to the {@link #_axisIndex} within the {@link #_xyplot}.
     */
    public ValueAxis getAxis()
    {
        if(_axisIndex == ChartConstants.DOMAIN_AXIS)
        {
            return _xyplot.getDomainAxis();
        }
        else
        {
            return _xyplot.getRangeAxis(_axisIndex);
        }
    }

    /**
     * Automatically adjusts the range for the axis corresponding to {@link #_axisIndex} within the plot
     * {@link #_xyplot}. This accounts for the flag {@link ThresholdParameters#getIncludeInAxisLimitsComputation()} for
     * all thresholds as specified in the {@link #_thresholds}.
     */
    public void autoRangeWithExcludingThresholds()
    {
        final int rendererOrSubplotCount = this.getDataSetOrSubplotCount();
        final java.util.List<Layer> layers = Arrays.asList(Layer.FOREGROUND, Layer.BACKGROUND);

        final HashMap<Marker, Dyad<Layer, Integer>> removedMarkers = Maps.newHashMap();

        try
        {
            //first remove the markers that are set not to be not included
            this.setAutoRange(false);
            for(final ThresholdParameters parameters: _thresholds.getThresholdParametersList())
            {
                //Threshold that have not been fully set will not be applied to the chart and, hence,
                //can be skipped for the purposes of axis limits.
                try
                {
                    parameters.haveAllParametersBeenSet();
                }
                catch(final Exception e)
                {
                    continue;
                }

//                System.err.println("####>> CHECKING ... " + p);
                if((!parameters.getIncludeInAxisLimitsComputation()) && (parameters.getVisible()))
                {
                    for(int rendererOrSubplotIndex =
                                                   0; rendererOrSubplotIndex < rendererOrSubplotCount; ++rendererOrSubplotIndex)
                    {
                        for(final Layer layer: layers)
                        {
                            final Collection<Marker> markers = this.getMarkers(rendererOrSubplotIndex, layer);
                            if(markers != null)
                            {
                                for(final Marker marker: markers)
                                {
                                    if(marker == parameters.getThresholdMarker())
                                    {
//                                        System.err.println("####>>    marker found for " + p.getLabel().getText());
                                        this.removeMarker(rendererOrSubplotIndex, marker, layer);
                                        removedMarkers.put(marker,
                                                           new Dyad<Layer, Integer>(layer, rendererOrSubplotIndex));
                                        break;
                                    }
                                }
                            }
                        }
                    }
                }
            }

            //Loop over all range axes and domain axes calling the below statements to compute the limits and set them.
            this.adjustRange();
            //now add the removed markers back to the subplots
            for(final Marker marker: removedMarkers.keySet())
            {
                this.addMarker(removedMarkers.get(marker).getSecond(), marker, removedMarkers.get(marker).getFirst());
            }
        }
        catch(final Exception e)
        {
            e.printStackTrace();
            return;
        }
    }

    /**
     * Passes the flag through to the {@link ValueAxis#setAutoRange(boolean)} method of the impacted axis within
     * {@link #_xyplot}.
     */
    private void setAutoRange(final Boolean autoRange)
    {
        if(_axisIndex == ChartConstants.DOMAIN_AXIS)
        {
            _xyplot.getDomainAxis().setAutoRange(autoRange);
            // _xyplot.getDomainAxis().setAutoRange(autoRange);
        }
        else
        {
            _xyplot.getRangeAxis(_axisIndex).setAutoRange(autoRange);
        }

    }

    @SuppressWarnings("unchecked")
    private Collection<Marker> getMarkers(final int subplotIndex, final Layer layer)
    {
        if(_axisIndex == ChartConstants.DOMAIN_AXIS)
        {
            //return _xyplot.getDomainMarkers(renderer, layer);
            //return _xyplot.getDomainMarkers(layer);
            final Object subplot = ((CombinedDomainXYPlot)_xyplot).getSubplots().get(subplotIndex);
            return ((XYPlot)subplot).getDomainMarkers(layer);
        }
        else
        {
            return _xyplot.getRangeMarkers(subplotIndex, layer);
        }
    }

    private void removeMarker(final int subplotIndex, final Marker marker, final Layer layer)
    {
        if(_axisIndex == ChartConstants.DOMAIN_AXIS)
        {
            final Object subplot = ((CombinedDomainXYPlot)_xyplot).getSubplots().get(subplotIndex);
            //_xyplot.removeDomainMarker(renderer, marker, layer);
            ((XYPlot)subplot).removeDomainMarker(marker, layer);
        }
        else
        {
            _xyplot.removeRangeMarker(subplotIndex, marker, layer);
        }
    }

    private void addMarker(final int subplotIndex, final Marker marker, final Layer layer)
    {
        if(_axisIndex == ChartConstants.DOMAIN_AXIS)
        {
            //_xyplot.addDomainMarker(renderer, marker, layer);
            final Object subplot = ((CombinedDomainXYPlot)_xyplot).getSubplots().get(subplotIndex);
            ((XYPlot)subplot).addDomainMarker(marker, layer);
        }
        else
        {
            _xyplot.addRangeMarker(subplotIndex, marker, layer);
        }
    }

    /**
     * @return The range corresponding to the axis within {@link #_xyplot}.
     */
    private Range getRange(final int axisIndex)
    {
        if(_axisIndex == ChartConstants.DOMAIN_AXIS)
        {
            return _xyplot.getDomainAxis(axisIndex).getRange();
        }
        else
        {
            return _xyplot.getRangeAxis(axisIndex).getRange();
        }

    }

    /**
     * Sets the range corresponding to the axis within {@link #_xyplot}.
     */
    private void setRange(final int axisIndex, final Range range)
    {
        if(_axisIndex == ChartConstants.DOMAIN_AXIS)
        {
            _xyplot.getDomainAxis(axisIndex).setRange(range);
            _xyplot.getDomainAxis().setRange(range);
        }
        else
        {
            _xyplot.getRangeAxis(axisIndex).setRange(range);
        }

    }

    /**
     * @return The count of the number of subplots for the domain axis or the data set count for range axes.
     */
    private int getDataSetOrSubplotCount()
    {
        if(_axisIndex == ChartConstants.DOMAIN_AXIS)
        {
            return ((CombinedDomainXYPlot)_xyplot).getSubplots().size();
        }
        return _xyplot.getDatasetCount();
    }

    /**
     * Auto-compute the range and set it for the appropriate axis corresponding to {@link #_axisIndex} within
     * {@link #_xyplot}.
     */
    private void adjustRange()
    {
        if(_axisIndex == ChartConstants.DOMAIN_AXIS)
        {
            _xyplot.getDomainAxis().setAutoRange(true);
            final Range range = _xyplot.getDomainAxis().getRange();
            _xyplot.getDomainAxis().setAutoRange(false);
            _xyplot.getDomainAxis().setRange(range);
        }
        else
        {
            this.setAutoRange(true);
            Range range = this.getRange(_axisIndex);

            // Expand the range if the upper and lower bounds are equal.
            // This is a fix for FogBugz 1332. When the values are constant
            // the ticks and tick labels are not displayed. the smaller difference
            // that does not cause the problems is 0.0000001. If the difference 
            // is less than 0.0000001, the ticks and labels are not displayed.
            if(Math.abs(range.getUpperBound() - range.getLowerBound()) < MINIMUM_RANGE_DIFFERENCE)
            {
                //if the upper and lower bounds are zero or close to zero,
                // add 1 and -1
                if(range.getCentralValue() == 0.0 || (Math.abs(range.getUpperBound()) < MINIMUM_RANGE_DIFFERENCE
                    && Math.abs(range.getLowerBound()) < MINIMUM_RANGE_DIFFERENCE))
                {
                    range = new Range(-1, 1);
                }
                //increase the upper and lower bound by a percentage
                else
                {
                    range = new Range(range.getLowerBound() - Math.abs(range.getLowerBound()) * RANGE_RESCALE_FACTOR,
                                      range.getUpperBound() + Math.abs(range.getUpperBound()) * RANGE_RESCALE_FACTOR);
                }
            }
            this.setAutoRange(false);
            this.setRange(_axisIndex, range);
        }

    }

//
//This method is not used. I first wrote it and then
//found that it is not needed. 
//    public void setCalculatedTickLabelFormat()
//    {
//        if(_axisIndex != ChartToolsAndConstants.DOMAIN_AXIS
//            && getAxis() instanceof NumberAxis)
//        {
//            final Range range = this.getRange(_axisIndex);
//            final NumberAxis numberAxis = (NumberAxis)(this.getAxis());
//            if(Math.log10(range.getLength()) < -1)
//            {
//                String formatstring = new String("0.0");
//                for(int i =
//                          1; i < Math.ceil(Math.abs(Math.log10(range.getLength()))); ++i)
//                {
//                    formatstring += '0';
//                }
//                final DecimalFormat format = new DecimalFormat(formatstring);
//                numberAxis.setNumberFormatOverride(format);
//            }
//            else
//            {
//                numberAxis.setNumberFormatOverride(null);
//            }
//        }
//    }
}
