package ohd.hseb.charter.parameters;

import org.jfree.chart.JFreeChart;
import org.jfree.chart.axis.Axis;
import org.jfree.chart.axis.NumberAxis;
import org.jfree.chart.axis.ValueAxis;
import org.jfree.chart.plot.CombinedDomainXYPlot;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.xml.sax.Attributes;

import ohd.hseb.charter.ChartConstants;
import ohd.hseb.charter.ChartEngine;
import ohd.hseb.charter.ChartParameters;
import ohd.hseb.charter.DefaultChartParameters;
import ohd.hseb.charter.tools.TranslatedAxis;
import ohd.hseb.charter.translator.AxisTranslatorParameters;
import ohd.hseb.hefs.utils.arguments.ArgumentsProcessor;
import ohd.hseb.hefs.utils.plugins.GeneralPlugInParameters;
import ohd.hseb.hefs.utils.tools.GeneralTools;
import ohd.hseb.hefs.utils.tools.StringTools;
import ohd.hseb.hefs.utils.xml.XMLReader;
import ohd.hseb.hefs.utils.xml.XMLReaderException;
import ohd.hseb.hefs.utils.xml.XMLTools;
import ohd.hseb.hefs.utils.xml.XMLWriterException;

/**
 * Records parameters pertaining to an axis, include label, tick mark info, limits info, and visibility. Note that
 * {@link #_visible} can never be null and is recorded as an attribute of the main element of this within XML. Note that
 * when an axis element is found, it is assumed that the axis is to be visible, unless the visible attribute is found
 * and it specifies false.<br>
 * <br>
 * Currently, this single {@link AxisParameters} class can record the parameters either for a date axis or numeric axis.
 * It does so through {@link #_dateAxisParameters} and {@link #_numericAxisParameters}, both of which are contained
 * herein though only one is actively used. This is probably a poor design; there should be one superclass for axis
 * parameters and then subclasses for date and number axes... or something like that.
 * 
 * @author herrhd
 */
public class AxisParameters extends DefaultChartParameters
{
    public static final String[] DEFAULT_YES_NO_DISPLAY_STRINGS_YES_DEFAULT = {"Default (Yes)", "Yes", "No"};
    public static final String[] DEFAULT_YES_NO_DISPLAY_STRINGS_NO_DEFAULT = {"Default (No)", "Yes", "No"};
    public static final Boolean[] DEFAULT_INVERTED_BOOLEAN = new Boolean[]{null, true, false};
    public static final String[] UNIT_ARRAY = new String[]{"years", "months", "weeks", "days", "hours"};

    /**
     * A fixed flag specifying if this axis is a range or domain axis.
     */
    private final boolean _isRangeAxis;

    /**
     * This MUST NOT be edited via XML!!! Only the outside owning {@link SubPlotParameters} instance should set it. When
     * the subplot is created, it will let this know what its index is and change it if needed.
     */
    private int _subPlotIndex = -1;

    /**
     * This MUST NOT be edited via XML!!! Only the outside owning SubPlotParameters instance should set it. When the
     * subplot is created, it initialize this value, which must remain fixed!
     */
    private final int _axisIndex;

    /**
     * The parameters for the axis label.
     */
    private final LabelChartParameters _labelParameters = new LabelChartParameters("label");

    /**
     * If this is a date axis, the parameters that are date specific.
     */
    private final DateAxisParameters _dateAxisParameters = new DateAxisParameters();

    /**
     * If this is a numeric axis, the parameters that are number specific.
     */
    private final NumericAxisParameters _numericAxisParameters = new NumericAxisParameters();

    /**
     * If the axis is translated, the parameters for the translation.
     */
    private AxisTranslatorParameters _translatorParameters = null;

    /**
     * Indicates if the axis is visible or not. An XML attribute.
     */
    private Boolean _visible = null;

    /**
     * The type of axis; see {@link ChartConstants#AXIS_CHOICES}.
     */
    private String _axisType = null;

    /**
     * Tracks the inverted axis parameter.
     */
    private Boolean _inverted = null;

    /**
     * Base constructor that other constructors call.
     * 
     * @param subPlotIndex The index of the subplot to which the affected axis is associated.
     * @param axisIndex The index of the affected axis within the subplot.
     * @param isRangeAxis True if it is a range axis, false if domain.
     */
    public AxisParameters(final int subPlotIndex, final int axisIndex, final boolean isRangeAxis)
    {
        _subPlotIndex = subPlotIndex;
        _axisIndex = axisIndex;
        _isRangeAxis = isRangeAxis;

        if(_isRangeAxis)
        {
            setXMLTagName(ChartConstants.YAXIS_XML_STRINGS[_axisIndex] + "RangeAxis");
        }
        else
        {
            setXMLTagName("domainAxis");
        }
    }

    /**
     * @param dateParameters The parameters for a date axis to use. Note that it is copied into place using
     *            {@link DateAxisParameters#copyOverriddenParameters(ChartParameters)}, so that null parameters are
     *            allowed and indicate that the default should be used.
     */
    public void overrideDateParameters(final DateAxisParameters dateParameters)
    {
        _dateAxisParameters.copyOverriddenParameters(dateParameters);
    }

    /**
     * @param numericParameters The parameters for a date axis to use. Note that it is copied into place using
     *            {@link NumericAxisParameters#copyOverriddenParameters(ChartParameters)}, so that null parameters are
     *            allowed and indicate that the default should be used.
     */
    public void overrideNumericAxisParameters(final NumericAxisParameters numericParameters)
    {
        _numericAxisParameters.copyOverriddenParameters(numericParameters);
    }

    /**
     * Set the {@link #_translatorParameters} to an empty instance of {@link AxisTranslatorParameters} and hands it a
     * copy of the arguments.
     */
    public AxisTranslatorParameters addAxisTranslator()
    {
        _translatorParameters = new AxisTranslatorParameters();
        _translatorParameters.setArguments(getArguments());
        return _translatorParameters;
    }

    /**
     * @return The axis from the provided {@link JFreeChart} that corresponds to this parameter object based on
     *         information in the provided {@link ChartEngine}.
     */
    public ValueAxis getAxisFromChart(final ChartEngine engine, final JFreeChart builtChart)
    {
        //Return null if the engine is null or, if this is a range axis, the subplot corresponding
        //to the subPlotIndex is null.
        if(engine == null || ((getIsRangeAxis() && (engine.getSubPlot(_subPlotIndex) == null))))
        {
            return null;
        }

        final CombinedDomainXYPlot plot = (CombinedDomainXYPlot)builtChart.getPlot();
        if(this.getIsRangeAxis())
        {
            //return the range axis based on the _subPlotIndex and the _axisIndex
            return engine.getSubPlot(_subPlotIndex).getRangeAxis(getAxisIndex());
        }
        else
        {
            //return the domain axis
            return plot.getDomainAxis();
        }
    }

    public int getAxisIndex()
    {
        return this._axisIndex;
    }

    public AxisTranslatorParameters getAxisTranslator()
    {
        return this._translatorParameters;
    }

    public String getAxisType()
    {
        return _axisType;
    }

    public int getAxisTypeInt()
    {
        if(_axisType == null)
        {
            return -1;
        }
        return ChartConstants.determineAxisTypeIntFromAxisTypeString(_axisType);
    }

    public DateAxisParameters getDateAxisParameters()
    {
        return this._dateAxisParameters;
    }

    public Boolean getInverted()
    {
        return _inverted;
    }

    public boolean getIsRangeAxis()
    {
        return this._isRangeAxis;
    }

    public LabelChartParameters getLabel()
    {
        return this._labelParameters;
    }

    public NumericAxisParameters getNumericAxisParameters()
    {
        return _numericAxisParameters;
    }

    public int getSubPlotIndex()
    {
        return this._subPlotIndex;
    }

    public Boolean getVisible()
    {
        return _visible;
    }

    public void removeAxisTranslator()
    {
        this._translatorParameters = null;
    }

    public void setAxisType(final int type)
    {
        _axisType = ChartConstants.AXIS_CHOICES[type];
    }

    public void setAxisType(final String type)
    {
        _axisType = type;
    }

    public void setInverted(final Boolean inverted)
    {
        this._inverted = inverted;
    }

    public void setSubPlotIndex(final int index)
    {
        this._subPlotIndex = index;
    }

    public void setVisible(final Boolean b)
    {
        _visible = b;
    }

    @Override
    public void applyParametersToChart(final Object objectAppliedTo)
    {
        //System.out.println("\nApplying axis parameters: \n" + this);

        //This assumes the axis type has already been applied via the ChartEngine!
        final Axis axis = ((AxisAutoRangeHelper)objectAppliedTo).getAxis();
        axis.setVisible(_visible);

        if(_inverted != null)
        {
            if(axis instanceof ValueAxis)
            {
                ((ValueAxis)axis).setInverted(_inverted);
            }
        }

        //Do not apply to a translated axis that has error... the label is important.
        if(axis instanceof TranslatedAxis)
        {
            if(((TranslatedAxis)axis).getErroneousAxis())
            {
                axis.setVisible(false);
                return;
            }
        }

        if(_visible)
        {
            //This call, if or else, triggers a recompute of the axis limits, if auto.  I need that recompute to happen
            //here is order to make sure markers are accounted for.
            if(axis instanceof NumberAxis)
            {
                final NumberAxis numAxis = (NumberAxis)axis;

                numAxis.setAutoRangeIncludesZero(false);
            }
            else
            {
                axis.configure();
            }
            axis.setLabel(this.getLabel().getArgumentReplacedText());
            axis.setLabelFont(this.getLabel().getFont());
            axis.setLabelPaint(this.getLabel().getColor());

            //XXX Presently, the tick font matches the label font.  This should be something that can
            //be indepenendently controlled.
            axis.setTickLabelFont(this.getLabel().getFont());

            if(axis instanceof NumberAxis)
            {
                _numericAxisParameters.applyParametersToChart(objectAppliedTo);
            }
            else
            {
                _dateAxisParameters.applyParametersToChart(objectAppliedTo);
            }
        }
    }

    @Override
    public void argumentsChanged()
    {
        _labelParameters.argumentsChanged();
        _dateAxisParameters.argumentsChanged();
        _numericAxisParameters.argumentsChanged();
        if(_translatorParameters != null)
        {
            _translatorParameters.argumentsChanged();
        }
    }

    @Override
    public void clearParameters()
    {
        _visible = null;
        _inverted = null;
        _axisType = null;
        _translatorParameters = null;
        _labelParameters.clearParameters();
        _dateAxisParameters.clearParameters();
        _numericAxisParameters.clearParameters();
    }

    @Override
    public Object clone()
    {
        final AxisParameters result = new AxisParameters(this._subPlotIndex, this._axisIndex, this._isRangeAxis);
        result.copyFrom(this);
        return result;
    }

    @Override
    public void copyFrom(final GeneralPlugInParameters parameters)
    {
        super.copyFrom(parameters);
        clearParameters();
        copyOverriddenParameters((ChartParameters)parameters);
//        final AxisParameters base = (AxisParameters)parameters;
//        this._visible = base.getVisible();
//        this._inverted = base.getInverted();
//        _axisType = base.getAxisType();
//        _labelParameters.copyFrom(base.getLabel());
//        _dateAxisParameters.copyFrom(base.getDateAxisParameters());
//        _numericAxisParameters.copyFrom(base.getNumericAxisParameters());
//        _translatorParameters = null;
//        if(base.getAxisTranslator() != null)
//        {
//            _translatorParameters = (AxisTranslatorParameters)base.getAxisTranslator().clone();
//        }
    }

    @Override
    public void copyOverriddenParameters(final ChartParameters override)
    {
        final AxisParameters base = (AxisParameters)override;
        if(base.getVisible() != null)
        {
            this._visible = base.getVisible();
        }
        if(base.getInverted() != null)
        {
            this._inverted = base.getInverted();
        }
        if(base.getAxisType() != null)
        {
            this._axisType = base.getAxisType();
        }
        this._labelParameters.copyOverriddenParameters(base.getLabel());
        this._dateAxisParameters.copyOverriddenParameters(base.getDateAxisParameters());
        this._numericAxisParameters.copyOverriddenParameters(base.getNumericAxisParameters());

        //The translator is copied in total.  All parameters required must be non-null.
        if(base.getAxisTranslator() != null)
        {
            _translatorParameters = (AxisTranslatorParameters)base.getAxisTranslator().clone();
        }
    }

    @Override
    public boolean equals(final Object parameters)
    {
        if(!(parameters instanceof ohd.hseb.charter.parameters.AxisParameters))
        {
            return false;
        }
        final AxisParameters other = (AxisParameters)parameters;

        if((this._axisIndex != other._axisIndex) || (this._isRangeAxis != other._isRangeAxis)
            || (this._subPlotIndex != other._subPlotIndex))
        {
            return false;
        }
        if(!GeneralTools.checkForFullEqualityOfObjects(this._visible, other.getVisible()))
        {
            return false;
        }
        if(!GeneralTools.checkForFullEqualityOfObjects(this._inverted, other.getInverted()))
        {
            return false;
        }
        if(!StringTools.checkForFullEqualityOfStrings(_axisType, other.getAxisType(), false))
        {
            return false;
        }
        if(!this._labelParameters.equals(other.getLabel()))
        {
            return false;
        }
        if(!this._dateAxisParameters.equals(other.getDateAxisParameters()))
        {
            return false;
        }
        if(!this._numericAxisParameters.equals(other.getNumericAxisParameters()))
        {
            return false;
        }
        if(!ohd.hseb.hefs.utils.tools.GeneralTools.checkForFullEqualityOfObjects(_translatorParameters,
                                                                                 other.getAxisTranslator()))
        {
            return false;
        }
        return true;
    }

    @Override
    public void finalizeReading() throws XMLReaderException
    {
    }

    @Override
    public void validate() throws XMLReaderException
    {
        if(!ChartConstants.isAxisTypeTranslated(getAxisTypeInt()))
        {
            if(this._translatorParameters != null)
            {
                throw new XMLReaderException("Axis type is not translated, but the XML specified a translator.");
            }
        }
        else
        {
            if(this._translatorParameters == null)
            {
                throw new XMLReaderException("Axis type is translated, but the XML did not specify a translator.");
            }
        }
    }

    @Override
    public String getShortGUIDisplayableParametersSummary()
    {
        return null;
    }

    @Override
    public void haveAllParametersBeenSet() throws ChartParametersException
    {
        if(ChartConstants.isAxisTypeTranslated(getAxisTypeInt()))
        {
            if(this._translatorParameters == null)
            {
                throw new ChartParametersException("For axis " + this.getXMLTagName()
                    + ": Axis type is translated but translator parameters not given.");
            }
        }

        try
        {
            _labelParameters.haveAllParametersBeenSet();
            _dateAxisParameters.haveAllParametersBeenSet();
        }
        catch(final ChartParametersException e)
        {
            throw new ChartParametersException("For axis " + this.getXMLTagName() + ": " + e.getMessage());
        }
    }

    @Override
    public XMLReader readInPropertyFromXMLElement(final String elementName,
                                                  final Attributes attr) throws XMLReaderException
    {
        final AxisTranslatorParameters transParms = new AxisTranslatorParameters();
        if(elementName.equals(this.getXMLTagName()))
        {
            clearParameters();
            this._visible = null;
            final String visStr = attr.getValue("visible");
            if(visStr != null)
            {
                try
                {
                    _visible = Boolean.parseBoolean(visStr);
                }
                catch(final NumberFormatException e)
                {
                    throw new XMLReaderException("Visible attribute is neither true nor false.");
                }
            }
            this._inverted = null;
            final String invertedStr = attr.getValue("inverted");
            if(invertedStr != null)
            {
                try
                {
                    _inverted = Boolean.parseBoolean(invertedStr);
                }
                catch(final NumberFormatException e)
                {
                    throw new XMLReaderException("Inverted attribute is neither true nor false.");
                }
            }
        }
        else if(elementName.equals(this._labelParameters.getXMLTagName()))
        {
            return _labelParameters;
        }
        else if(elementName.equals(this._dateAxisParameters.getXMLTagName()))
        {
            return _dateAxisParameters;
        }
        else if(elementName.equals(this._numericAxisParameters.getXMLTagName()))
        {
            return _numericAxisParameters;
        }
        else if(elementName.equals(transParms.getXMLTagName()))
        {
            _translatorParameters = transParms;
            return this._translatorParameters;
        }

        return null;
    }

    @Override
    public void setArguments(final ArgumentsProcessor arguments)
    {
        super.setArguments(arguments);
        if(_translatorParameters != null)
        {
            _translatorParameters.setArguments(arguments);
        }
        this._labelParameters.setArguments(arguments);
        this._dateAxisParameters.setArguments(arguments);
    }

    @Override
    public void setupDefaultParameters()
    {
        _visible = false;
        if(!_isRangeAxis)
        {
            _visible = true;
        }

        _inverted = false;
        _axisType = null;

        _labelParameters.setupDefaultParameters();
        _dateAxisParameters.setupDefaultParameters();
        _numericAxisParameters.setupDefaultParameters();
        _translatorParameters = null;
    }

    @Override
    public void setValueOfElement(final String elementName, final String value) throws XMLReaderException
    {
        if(elementName.equals("axisType"))
        {
            if(ChartConstants.determineAxisTypeIntFromAxisTypeString(value) < 0)
            {
                throw new XMLReaderException("Axis type of '" + value + "' is invalid.");
            }
            this._axisType = value;
        }
    }

    @Override
    public String toString()
    {
        String results = "AxisParameters: ";
        results += "xmlTagName = '" + this.getXMLTagName() + "'; ";
        results += "subPlotIndex = " + this._subPlotIndex + "; ";
        results += "axisIndex = " + this._axisIndex + "; ";
        results += "isRangeAxis = " + this._isRangeAxis + "; ";
        results += "visible = " + this._visible + "; ";
        results += "inverted = " + this._inverted + "; ";
        results += "axisType = " + this._axisType + "; ";
        results += "label = {" + this._labelParameters + "}; ";
        results += "dateAxis = {" + this._dateAxisParameters + "}; ";
        results += "numericAxis = {" + this._numericAxisParameters + "}; ";

        if(this._translatorParameters != null)
        {
            results += "translator = {" + this._translatorParameters + "}; ";
        }
        return results;
    }

    @Override
    public Element writePropertyToXMLElement(final Document request) throws XMLWriterException
    {
        final Element mainElement = request.createElement(getXMLTagName());

        if(_visible != null)
        {
            mainElement.setAttribute("visible", "" + _visible);
        }

        if(_inverted != null)
        {
            mainElement.setAttribute("inverted", "" + _inverted);
        }

        if(_axisType != null)
        {
            mainElement.appendChild(XMLTools.createTextNodeElement(request, "axisType", _axisType));
        }
        if(_translatorParameters != null)
        {
            mainElement.appendChild(this._translatorParameters.writePropertyToXMLElement(request));
        }

        XMLTools.appendElementIfNotEmpty(mainElement, _labelParameters.writePropertyToXMLElement(request));
        XMLTools.appendElementIfNotEmpty(mainElement, _dateAxisParameters.writePropertyToXMLElement(request));
        XMLTools.appendElementIfNotEmpty(mainElement, _numericAxisParameters.writePropertyToXMLElement(request));

        return mainElement;
    }

    /**
     * This should use enums, but whatever.
     * 
     * @return The appropriate yes/no choice {@link String} given a flag indicating if the choice is true or false,
     *         where true is always 'Yes' . This uses {@link #DEFAULT_INVERTED_BOOLEAN},
     *         {@link #DEFAULT_YES_NO_DISPLAY_STRINGS_NO_DEFAULT}
     */
    public static String getYesNoChoiceString(final Boolean trueIsYes)
    {
        for(int index = 0; index < DEFAULT_INVERTED_BOOLEAN.length; index++)
        {
            if(trueIsYes == DEFAULT_INVERTED_BOOLEAN[index])
            {
                return DEFAULT_YES_NO_DISPLAY_STRINGS_NO_DEFAULT[index];
            }
        }
        return null;
    }

    /**
     * This should use enums, but whatever.
     * 
     * @param yesIsTrue Converts a chosen value to the appropriate {@link Boolean}.
     */
    public static Boolean getYesNoChoiceValue(final String yesIsTrue)
    {
        for(int index = 0; index < DEFAULT_YES_NO_DISPLAY_STRINGS_NO_DEFAULT.length; index++)
        {
            if(yesIsTrue.equalsIgnoreCase(DEFAULT_YES_NO_DISPLAY_STRINGS_NO_DEFAULT[index]))
            {
                return DEFAULT_INVERTED_BOOLEAN[index];
            }
        }

        return null;
    }
}
